/*
 * Copyright 2014 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 18.09.2014 - [JR] - creation
 */
package com.sibvisions.rad.server.http.rest.service;

import javax.rad.server.UnknownObjectException;

import org.codehaus.jackson.map.DeserializationConfig;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.map.SerializationConfig;
import org.restlet.data.Status;
import org.restlet.engine.header.HeaderConstants;
import org.restlet.ext.jackson.JacksonRepresentation;
import org.restlet.ext.servlet.ServletUtils;
import org.restlet.representation.Representation;
import org.restlet.resource.ServerResource;
import org.restlet.util.Series;

import com.sibvisions.rad.server.http.HttpContext;
import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.LoggerFactory;

/**
 * The <code>AbstractService</code> is the base class for REST services.
 * 
 * @author Ren Jahn
 */
abstract class AbstractService extends ServerResource
{
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Class members
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /** the logger. */
    private static ILogger logger;
    
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Overwritten methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * {@inheritDoc}
     */
    @Override
    public Representation handle()
    {
        HttpContext ctxt = new HttpContext(ServletUtils.getRequest(getRequest()), ServletUtils.getResponse(getResponse())); 
        
        try
        {
            return super.handle();
        }
        finally
        {
            ctxt.release();
        }
    }
    
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // User-defined methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * Handles an exception. This method logs the excception as debug message and sets
     * the status dependent of the exception.
     * 
     * @param pException the exception
     * @return <code>null</code>
     */
    protected Representation handleException(Throwable pException)
    {
        debug(pException);

        if (pException instanceof SecurityException)
        {
            setStatus(Status.CLIENT_ERROR_FORBIDDEN);
            return null;
        }
        else if (pException instanceof UnknownObjectException)
        {
            setStatus(Status.CLIENT_ERROR_NOT_FOUND);
            return null;
        }
        
        setStatus(Status.CLIENT_ERROR_BAD_REQUEST);
        
        return null;
    }
    
    /**
     * Gets whether the request is a dry-run test. That means, no data manipulation should be done.
     * 
     * @return <code>true</code> if the request is a dry-run request (Request-Header: X-DRYRUN is set to "true"),
     *         <code>false</code> otherwise
     */
    protected boolean isDryRun()
    {
        Series series = (Series)getRequest().getAttributes().get(HeaderConstants.ATTRIBUTE_HEADERS);
        
        return Boolean.parseBoolean(series.getFirstValue("x-dryrun"));
    }
    
    /**
     * Configures a Jackson representation.
     * 
     * @param pObject the object to transfer
     * @return the configured representation
     */
    protected JacksonRepresentation toRepresentation(Object pObject)
    {
        JacksonRepresentation rep = new JacksonRepresentation(pObject);
        
        ObjectMapper mapper = rep.getObjectMapper();
        
        mapper.configure(SerializationConfig.Feature.WRITE_DATES_AS_TIMESTAMPS, false);
        mapper.configure(DeserializationConfig.Feature.USE_BIG_DECIMAL_FOR_FLOATS, true);
        mapper.configure(DeserializationConfig.Feature.USE_BIG_INTEGER_FOR_INTS, true);
        
        return rep;
    }
    
    /**
     * Logs debug information.
     * 
     * @param pInfo the debug information
     */
    public void debug(Object... pInfo)
    {
        if (logger == null)
        {
            logger = LoggerFactory.getInstance(getClass());
        }
        
        logger.debug(pInfo);
    }

    /**
     * Logs information.
     * 
     * @param pInfo the information
     */
    public void info(Object... pInfo)
    {
        if (logger == null)
        {
            logger = LoggerFactory.getInstance(getClass());
        }
        
        logger.info(pInfo);
    }

    
    /**
     * Logs error information.
     * 
     * @param pInfo the error information
     */
    public void error(Object... pInfo)
    {
        if (logger == null)
        {
            logger = LoggerFactory.getInstance(getClass());
        }

        logger.error(pInfo);
    }
    
}   // AbstractService
