/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [RH] - creation
 * 11.10.2008 - [RH] - toString() optimized, (DataBook) checks changed to IDataBook
 * 02.11.2008 - [RH] - convertObjectToStorage and back removed, use IDataBook instead of DataBook
 *                     convertAndCheckToTypeClass add in setValue for correct DataType conversion 
 * 17.11.2008 - [RH] - createEmptyRow() optimized 
 * 09.04.2009 - [RH] - interface review - compareTo uses SortDefinition
 *                                        equals added
 *                                        getValue(int) added
 *                                        getValue(s) throw no ModelException anymore
 *                                        register,unregisterEditingControl removed
 *                                        clone renamed to createDataRow()
 *                                        IChangeableDataRow methods/functionality moved to ChangeableDataRow
 * 16.04.2009 - [RH] - remove/add/getDetailDataBooks moved to IDataBook.   
 *                     constructor optimized.   
 * 18.04.2009 - [RH] - get/set/DataPage/RowIndex moved to ChangeableDataRow                                 
 * 12.06.2009 - [JR] - toString: used StringBuilder [PERFORMANCE]
 * 30.03.2010 - [RH] - #6: getValueAsString in IDataRow   
 * 06.05.2010 - [JR] - getValuesAsString implemented
 * 31.03.2011 - [JR] - #318: add/removeControl forwarded to the IRowDefinition
 * 20.12.2012 - [RH] - setDefaultValues() is add (former createDataRow function in MemDataBook)
 * 03.04.2014 - [RZ] - #2 - added and implemented eventValuesChanged(String pColumnName)
 */
package com.sibvisions.rad.model.mem;

import java.io.Serializable;
import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Map;

import javax.rad.model.ColumnDefinition;
import javax.rad.model.IDataRow;
import javax.rad.model.IRowDefinition;
import javax.rad.model.ModelException;
import javax.rad.model.RowDefinition;
import javax.rad.model.SortDefinition;
import javax.rad.model.datatype.IDataType;
import javax.rad.model.event.DataRowEvent;
import javax.rad.model.event.DataRowHandler;
import javax.rad.model.ui.IControl;

import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.type.CommonUtil;

/**
 * An <code>DataRow</code> is a list of table column's.<br>
 * The <code>DataRow</code> is also a storage independent row.<br>
 * 
 * <br>Example:
 * <pre>
 * <code>
 * // construct a RowDefinition
 * RowDefinition rdRowDefinition = new RowDefinition();
 * 
 * // construct some ColumnDefinitions
 * ColumnDefinition cdId   = new ColumnDefinition("id");
 * ColumnDefinition cdName = new ColumnDefinition("name");
 * 
 * rdRowDefinition.addColumnDefinition(cdId);		
 * rdRowDefinition.addColumnDefinition(cdName);
 *
 * // construct DataRow
 * DataRow drDataRow = new DataRow(rdRowDefinition);
 * 
 * drDataRow.setValue("id", new BigDecimal(1));
 * drDataRow.setValue("name", "The name");  
 * </code>
 * </pre>
 * 
 * @see javax.rad.model.IDataRow
 * @see javax.rad.model.RowDefinition
 * @see com.sibvisions.rad.model.remote.RemoteDataBook
 * @see com.sibvisions.rad.model.mem.MemDataPage
 * 
 * @author Roland Hrmann
 */
public class DataRow implements IDataRow, 
                                Serializable
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The <code>RowDefinition</code> of this <code>DataRow</code>. */
	protected IRowDefinition	rdRowDefinition;

	/** The storage for this <code>DataRow</code>. */
	protected Object[]			oaStorage = null;

	/** The <code>ArrayUtil</code> of all <code>IRepaintListeners</code>. */
	private transient ArrayUtil<WeakReference<IControl>>	auControls;

    /** The map which maps column names to {@link DataRowHandler}s. */
    private transient Map<String, DataRowHandler> hmpEventValuesChanged;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Constructs a <code>DataRow</code> without a new instance of <code>IRowDefinition</code>.
	 */
	public DataRow()
	{		
		this(null, null);
	}
	
	/**
	 * Constructs a <code>DataRow</code> with a given <code>IRowDefinition</code>.
	 * 
	 * @param pRowDefinition the <code>IRowDefinition</code>
	 */
	public DataRow(IRowDefinition pRowDefinition)
	{
		this(pRowDefinition, null);
	}

	/**
	 * Constructs a <code>DataRow</code> with a given <code>IRowDefinition</code> 
	 * and initialize it a copy of the <code>Object[]<></code> data.
	 * 
	 * @param pRowDefinition the <code>IRowDefinition</code>
	 * @param pData the <code>Object[]<></code> with data of the <code>DataRow</code>.
	 */
	protected DataRow(IRowDefinition pRowDefinition, Object[] pData)
	{
		if (pRowDefinition == null)
		{
			pRowDefinition = new RowDefinition();
		}		
		
		if (pData == null)
		{
			oaStorage = new Object[pRowDefinition.getColumnCount()];
		}
		else
		{
			if (pData.length <= pRowDefinition.getColumnCount())
			{
				oaStorage = new Object[pRowDefinition.getColumnCount()];
			}
			else
			{
				oaStorage = new Object[pData.length];			
			}
			System.arraycopy(pData, 0, oaStorage, 0, pData.length);
		}
		rdRowDefinition = pRowDefinition;
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public IRowDefinition getRowDefinition()
	{
		return rdRowDefinition;
	}

	/**
	 * {@inheritDoc}
	 */
	public Object getValue(int pColumnIndex) throws ModelException
	{
		if (pColumnIndex < 0 || pColumnIndex >= rdRowDefinition.getColumnCount())
		{
			throw new ModelException("Column index '" + pColumnIndex + "' doesn't exist!");
		}		
		
		if (pColumnIndex >= oaStorage.length)
		{
			return null;
		}
		
		Object value = oaStorage[pColumnIndex];
		Object preparedValue = rdRowDefinition.getColumnDefinition(pColumnIndex).getDataType().prepareValue(value);
		
		if (preparedValue != value)
		{
			oaStorage[pColumnIndex] = preparedValue;
			return preparedValue;
		}
		
		return value;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public Object getValue(String pColumnName) throws ModelException
	{
		int iCellIndex = rdRowDefinition.getColumnDefinitionIndex(pColumnName);
		if (iCellIndex < 0)
		{
			throw new ModelException("Column name '" + pColumnName + "' doesn't exist!");
		}		
		
		return getValue(iCellIndex);
	}

	/**
	 * {@inheritDoc}
	 */
	public String getValueAsString(String pColumnName) throws ModelException
	{
		Object obj = getValue(pColumnName);
		return rdRowDefinition.getColumnDefinition(pColumnName).getDataType().convertToString(obj);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setValue(String pColumnName, Object pValue) throws ModelException
	{
		IDataRow drOldRow = this.createDataRow(null);

		setValueDRInternal(pColumnName, pValue);
		
		if (hmpEventValuesChanged != null)
		{
			if (hmpEventValuesChanged.containsKey(null))
			{
				hmpEventValuesChanged.get(null).dispatchEvent(new DataRowEvent(this, new String[] {pColumnName}, drOldRow));
			}
			
			if (hmpEventValuesChanged.containsKey(pColumnName))
			{
				hmpEventValuesChanged.get(pColumnName).dispatchEvent(new DataRowEvent(this, new String[] {pColumnName}, drOldRow));
			}
		}
		
		invokeRepaintListeners();		
	}

	/**
	 * {@inheritDoc}
	 */
	public Object[] getValues(String[] pColumnNames) throws ModelException
	{
		if (pColumnNames == null)
		{
			return oaStorage.clone();			
		}
		else
		{
			Object[] pValues = new Object[pColumnNames.length];
			for (int i = 0; i < pColumnNames.length; i++)
			{
				pValues[i] = getValue(pColumnNames[i]);
			}
			
			return pValues;
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public String[] getValuesAsString(String[] pColumnNames) throws ModelException
	{
		String[] sColumnNames;
		
		if (pColumnNames == null)
		{
			sColumnNames = rdRowDefinition.getColumnNames();
		}
		else
		{
			sColumnNames = pColumnNames;
		}

		String[] sResult = new String[sColumnNames.length];
		
		for (int i = 0; i < sColumnNames.length; i++)
		{
			sResult[i] = getValueAsString(sColumnNames[i]);
		}
		
		return sResult;

	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setValues(String[] pColumnNames, Object[] pValues) throws ModelException
	{
		if (pColumnNames == null)
		{
			pColumnNames = rdRowDefinition.getColumnNames();
		}
		else if (pColumnNames.length == 0) // no event, if no column is set!
		{
			return;
		}
		
		IDataRow drOldRow = this.createDataRow(null);
		
		for (int i = 0; i < pColumnNames.length; i++)
		{
			if (pValues == null || i >= pValues.length)
			{
				setValueDRInternal(pColumnNames[i], null);
			}
			else
			{
				setValueDRInternal(pColumnNames[i], pValues[i]);
			}
		}
		
		if (hmpEventValuesChanged != null)
		{
			if (hmpEventValuesChanged.containsKey(null))
			{
				hmpEventValuesChanged.get(null).dispatchEvent(new DataRowEvent(this, pColumnNames, drOldRow));
			}
			
			for (String columnName : pColumnNames)
			{
				if (hmpEventValuesChanged.containsKey(columnName))
				{
					hmpEventValuesChanged.get(columnName).dispatchEvent(new DataRowEvent(this, new String[] {columnName}, drOldRow));
				}
			}
		}
		
		invokeRepaintListeners();		
	}

	/**
	 * {@inheritDoc}
	 */
	public IDataRow createDataRow(String[] pColumnNames) throws ModelException
	{
		return new DataRow(rdRowDefinition.createRowDefinition(pColumnNames), 
				           getValues(pColumnNames));
	}

	/**
	 * {@inheritDoc}
	 */
	public IDataRow createEmptyRow(String[] pColumnNames) throws ModelException
	{
		return new DataRow(rdRowDefinition.createRowDefinition(pColumnNames));
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean equals(Object pObject)
	{
		if (pObject instanceof IDataRow)
		{
			return compareTo((IDataRow)pObject) == 0;
		}
		else
		{
			return super.equals(pObject);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public int compareTo(IDataRow pDataRow)
	{
		return compareTo(pDataRow, null);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public int compareTo(IDataRow pDataRow, SortDefinition pSortDefinition)
	{
		if (pDataRow == null)
		{
			return 1;
		}
		if (pSortDefinition == null)
		{
			pSortDefinition = new SortDefinition(pDataRow.getRowDefinition().getColumnNames());
		}
		String[] saColumnNames = pSortDefinition.getColumns();
		boolean[] bAscending = pSortDefinition.isAscending();
		
		for (int i = 0; i < saColumnNames.length; i++)
		{
			int iColumnIndex = rdRowDefinition.getColumnDefinitionIndex(saColumnNames[i]);

			if (iColumnIndex >= 0)
			{
				try
				{
					int compare = rdRowDefinition.getColumnDefinition(iColumnIndex).getDataType().
					                 compareTo(getValue(iColumnIndex), pDataRow.getValue(saColumnNames[i]));
					
					if (compare != 0)
					{
						if (bAscending != null && i < bAscending.length && !bAscending[i])
						{
							return -compare;
						}
						else
						{
							return compare;
						}
					}
				}
				catch (ModelException modelException)
				{
					return -1;
				}
			}
			else
			{
				throw new IllegalArgumentException("Column with the name '" + saColumnNames[i] + "' does not exist!");
			}
			
		}
		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean equals(IDataRow pDataRow, String[] pColumnNames)
	{
		return compareTo(pDataRow, new SortDefinition(pColumnNames)) == 0;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void addControl(IControl pControl)
	{
		if (auControls == null)
		{
			auControls = new ArrayUtil<WeakReference<IControl>>();
		}
		else
		{
			for (int i = auControls.size() - 1; i >= 0; i--)
			{
				if (auControls.get(i).get() == null)
				{
					auControls.remove(i);
				}
			}
		}
		
		if (auControls.indexOf(pControl) < 0)
		{
			auControls.add(new WeakReference<IControl>(pControl));
		}
		
		//forward the control to the row definition
		rdRowDefinition.addControl(pControl);
	}

	/**
	 * {@inheritDoc}
	 */
	public void removeControl(IControl pControl)
	{
		if (auControls != null)
		{
			for (int i = auControls.size() - 1; i >= 0; i--)
			{
				IControl control = auControls.get(i).get();
				if (control == null || control == pControl)
				{
					auControls.remove(i);
				}
			}
		}
		
		//forward the control to the row definition
		rdRowDefinition.removeControl(pControl);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IControl[] getControls()
	{
		ArrayUtil<IControl> result = new ArrayUtil<IControl>();
		if (auControls != null)
		{
			for (int i = auControls.size() - 1; i >= 0; i--)
			{
				IControl control = auControls.get(i).get();
				if (control == null)
				{
					auControls.remove(i);
				}
				else
				{
					result.add(0, control);
				}
			}
		}
		return result.toArray(new IControl[result.size()]);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataRowHandler eventValuesChanged()
	{
		return eventValuesChanged(null);
	}

	/**
	 * {@inheritDoc}
	 */
	public DataRowHandler eventValuesChanged(String pColumnName)
	{
		if (hmpEventValuesChanged == null)
		{
			hmpEventValuesChanged = new HashMap<String, DataRowHandler>();
		}
		
		DataRowHandler drh = hmpEventValuesChanged.get(pColumnName);
		
        if (drh == null)
        {
            drh = new DataRowHandler();
            
            hmpEventValuesChanged.put(pColumnName, drh);
        }
        
        return drh;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void notifyRepaintControls()
	{		
		invokeRepaintListeners();
	}
		
	/**
	 * {@inheritDoc}
	 */
	public void saveEditingControls() throws ModelException
	{
		invokeSaveEditingControls();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void cancelEditingControls()
	{
		invokeCancelEditingControls();
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString()
	{
		StringBuilder sbResult = new StringBuilder();

		String[] sColumnNames = getRowDefinition().getColumnNames();

		try
		{
			sbResult.append("{");

			for (int i = 0; i < sColumnNames.length; i++)
			{
				String columnName = sColumnNames[i];
				
				if (i > 0)
				{
					sbResult.append(", ");
				}
				sbResult.append(columnName);
				sbResult.append("=");
				sbResult.append(getValue(columnName));
			}
			
			sbResult.append("}");
		}
		catch (ModelException modelException)
		{
			sbResult.append("\n");
			sbResult.append(CommonUtil.dump(modelException, true));
		}
		
		return sbResult.toString();
	}	
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public int hashCode()
	{
		String[] sColumns = rdRowDefinition.getColumnNames();
		int h = 0;
        
		try
		{
			for (int i = 0; i < sColumns.length; i++)
			{
				h = (h * 37);
				Object oValue = getValue(sColumns[i]);
				if (oValue != null)
				{
					h += oValue.hashCode();
				}
			}
		}
		catch (ModelException modelException)
		{
			return h;
		}
		return h;
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Sets the value of the named column in this <code>IDataRow</code>.
	 * 
	 * @param pColumnName
	 *            the column name
	 * @param pValue
	 *            the new value for the column in this <code>IDataRow</code>
	 * @throws ModelException 
	 *            if the column name is not in this <code>IDataRow</code>
	 *            or the pValue is not convertible/too large to/for the <code>IDataType</code> 
	 *            of the column     
	 */
	protected void setValueDRInternal(String pColumnName, Object pValue) throws ModelException
	{
		int iColumnIndex = getRowDefinition().getColumnDefinitionIndex(pColumnName);
		// if ColumnDefinition been added!! -> Extend AbstractStorage
		if (iColumnIndex >= oaStorage.length)
		{
			Object[] alNewStorage = new Object[getRowDefinition().getColumnCount()];
			System.arraycopy(oaStorage, 0, alNewStorage, 0, oaStorage.length);
			oaStorage = alNewStorage;
		}
		else if (iColumnIndex < 0)
		{
			throw new ModelException("Column name doesn't exist! - " + pColumnName);
		}
		IDataType ctDataType = getRowDefinition().getColumnDefinition(iColumnIndex).getDataType();
		oaStorage[iColumnIndex] = ctDataType.convertAndCheckToTypeClass(pValue);		
	}	
	
	/**
	 * It invokes for each <code>IComponent</code> the <code>notifyRepaint()</code> method.<br>
	 */
	protected void invokeRepaintListeners()
	{		
		if (auControls != null)
		{
			IControl[] controls = getControls();
			for (int i = 0; i < controls.length; i++)
			{
				controls[i].notifyRepaint();
			}
		}
	}
		
	/**
	 * It invokes for each <code>IComponent</code>  the <code>saveEditing()</code> method.
	 * @throws ModelException if saving the editors fails.
	 */
	protected void invokeSaveEditingControls() throws ModelException
	{
		if (auControls != null)
		{
			IControl[] controls = getControls();
			for (int i = 0; i < controls.length; i++)
			{
				controls[i].saveEditing();
			}
		}
	}
	
	/**
	 * It invokes for each <code>IComponent</code>  the <code>cancelEditing()</code> method.<br>
	 */
	protected void invokeCancelEditingControls()
	{
		if (auControls != null)
		{
			IControl[] controls = getControls();
			for (int i = 0; i < controls.length; i++)
			{
				controls[i].cancelEditing();
			}
		}
	}
	
	/**
	 * Sets the Default Values from the RowDefinition to the DataRow.
	 * 
	 * @throws ModelException if the set value fails
	 */
	public void setDefaultValues() throws ModelException
	{
		int iCount = rdRowDefinition.getColumnCount();
		
		for (int i = 0; i < iCount; i++)
		{
			ColumnDefinition cd = rdRowDefinition.getColumnDefinition(i);
			setValue(cd.getName(), cd.getDefaultValue());
		}
	}
	
} 	// DataRow
