/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 04.01.2011 - [RH] - creation
 * 06.01.2011 - [JR] - forward POJO handling to AbstractStorage because of name conversions
 * 07.01.2011 - [JR] - isNewModified, isOldModified implemented
 */
package com.sibvisions.rad.persist.event;

import javax.rad.persist.DataSourceException;
import javax.rad.persist.IStorage;
import javax.rad.type.bean.IBean;

import com.sibvisions.rad.persist.AbstractStorage;

/**
 * The <code>StorageEvent</code> contains information about changes in an 
 * {@link AbstractStorage}.
 * 
 * @author Roland Hrmann
 */
public class StorageEvent
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Specifies the type of change.
	 */
	public enum ChangedType 
	{
		/** Calculate row.  This is called after fetch, refetchRow, insert and update method calls. */
		CALCULATE_ROW,
		
		/** before insert. */
		BEFORE_INSERT,
		/** after insert. */
		AFTER_INSERT,

		/** before update. */
		BEFORE_UPDATE,
		/** after update. */
		AFTER_UPDATE,
		
		/** before delete. */
		BEFORE_DELETE,
		/** after delete. */
		AFTER_DELETE,
	}
	
	/** The changed type. */
	private ChangedType type;

	/** The changed storage. */
	private AbstractStorage storage;
	
	/** The old IBean. */
	private IBean ibOld;
	
	/** The new IBean. */
	private IBean ibNew;
	
	/** whether the new object is modified. */
	private boolean bNewModified = false;
	
	/** whether the old object is modified. */
	private boolean bOldModified = false;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Constructs the DataRowEvent.
	 * 
	 * @param pStorage the changed storage.
	 * @param pType    the type of change.
	 * @param pOld     the old bean or <code>null</code> if old bean is not allowed
	 * @param pNew     the new bean or <code>null</code> if new bean is not allowed
	 */
	public StorageEvent(AbstractStorage pStorage, ChangedType pType, IBean pOld, IBean pNew)
	{
		storage  = pStorage;
		type     = pType;
		
		ibOld = pOld;
		ibNew = pNew;
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Gets the IStorage that is changed.
	 * 
	 * @return the IStorage that is changed.
	 */
	public IStorage getStorage()
	{
		return storage;
	}
	
	/**
	 * Gets the type of change.
	 * 
	 * @return the type of change.
	 */
	public ChangedType getType()
	{
		return type;
	}
	
	/**
	 * Sets the old bean properties.
	 * 
	 * @param pOld the current old bean.
	 */
	public void setOld(Object pOld)
	{
		//not allowed?
		if (ibOld != null)
		{
			bOldModified = true;
			
			if (pOld instanceof IBean)
			{
				ibOld = (IBean)pOld;
			}
			else
			{
				try
				{
					storage.updateBean(ibOld, pOld);
				}
				catch (DataSourceException dse)
				{
					throw new RuntimeException(dse);
				}
			}
		}
	}

	/**
	 * Gets the old IBean.
	 * 
	 * @return the old IBean.
	 */
	public IBean getOld()
	{
		return ibOld;
	}

	/**
	 * Gets the old POJO from type <T>.
	 * 
	 * @param <T>		the type of the POJO.
	 * @param pClass	the class of the POJO.
	 * @return the new POJO from type <T>.
	 */
	public <T> T getOld(Class<T> pClass)
	{
		try
		{
			return (T)storage.createPOJO(pClass, ibOld);
		}
		catch (DataSourceException dse)
		{
			throw new RuntimeException(dse);
		}
			
	}	
	
	/**
	 * Gets the new IBean.
	 * 
	 * @return the new IBean.
	 */
	public IBean getNew()
	{
		return ibNew;
	}	

	/**
	 * Gets the new POJO from type <T>.
	 * 
	 * @param <T>		the type of the POJO.
	 * @param pClass	the class of the POJO.
	 * @return the new POJO from type <T>.
	 */
	public <T> T getNew(Class<T> pClass)
	{
		try
		{
			return (T)storage.createPOJO(pClass, ibNew);
		}
		catch (DataSourceException dse)
		{
			throw new RuntimeException(dse);
		}
	}	

	/**
	 * Sets the new bean properties.
	 * 
	 * @param pNew the current new bean.
	 */
	public void setNew(Object pNew)
	{
		//not allowed?
		if (ibNew != null)
		{
			bNewModified = true;
			
			if (pNew instanceof IBean)
			{
				ibNew = (IBean)pNew;
			}
			else
			{
				try
				{
					storage.updateBean(ibNew, pNew);
				}
				catch (DataSourceException dse)
				{
					throw new RuntimeException(dse);
				}
			}
		}
	}
	
	/**
	 * Gets wheter the new object was modified after creation.
	 * 
	 * @return <code>true</code> if the new objectt was modified, <code>false</code> otherwise
	 */
	public boolean isNewModified()
	{
		return bNewModified;
	}

	/**
	 * Gets wheter the old object was modified after creation.
	 * 
	 * @return <code>true</code> if the old objectt was modified, <code>false</code> otherwise
	 */
	public boolean isOldModified()
	{
		return bOldModified;
	}
	
}	// StorageEvent
