/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [RH] - creation
 * 17.11.2008 - [RH] - storeXXX -> saveXXXX; restoreXXXX -> reloadXXX changed
 * 19.04.2009 - [RH] - interface reviewed, write back isolation levels moved from IDataBook
 * 07.05.2009 - [RH] - open/isOpen/close added
 */
package javax.rad.model;

import javax.rad.model.IDataBook.WriteBackIsolationLevel;

/**
 * The <code>IDataSource</code> defines the methods to access any kind of table 
 * oriented IDataSource: e.g. a database, XML file, ...<br>
 * <br>
 * Its used in the <code>IDataPage/IDataBook</code> to read and save data from/to the IDataSource.
 * 
 * @see javax.rad.model.IDataPage
 * @see javax.rad.model.IDataBook
 * 
 * @author Roland Hrmann
 */
public interface IDataSource
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Method definitions
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * It opens the IDataSource. <br>
     *
	 * @throws ModelException if the open dosn't work out.
	 */
	public void open() throws ModelException;
	
	/**
	 * Returns if the IDataSource is open.
	 * 
	 * IDataBooks need to check on open if the used IDataSource is open.
	 * @return true if the IDataSource is open.
	 */
	public boolean isOpen();
	
	/**
	 * It closes the IDataSource and all its IDataBooks.
	 */
	public void close();
	
	/**
	 * It saves all changes from all <code>IDataBook</code>'s to the IDataSource. <br>
	 * The recommended way to do this, is to call on every <code>IDataBook</code>
	 * storeAllRows().
	 * 
     * <br><br>Events:<br>
     * It calls before the store on all registered IComponents the saveEditing() method, to set all
     * changes to the DataBook before the store operation.
     *  
	 * @throws ModelException
	 *             if not all changes could be stored in the <code>IDataBook</code>
	 */
	public void saveAllDataBooks() throws ModelException;
		
	/**
	 * It removes all changes from all <code>IDataBook</code>'s till last storeAllDataBooks() call. 
	 * The recommended way to do this, is to call on every <code>IDataBook</code>
	 * restoreAllRows().
	 * 
     * <br><br>Events:<br>
     * It calls before the restore on all registered IComponents the cancelEditing() method, to 
     * cancel the editing mode of the GUI control.
	 * @throws ModelException
	 *             if not all changes could be restored in the <code>IDataBook</code>
   	 */
	public void reloadAllDataBooks() throws ModelException;	
	
	/**
	 * It removes all changes from all <code>IDataBook</code>'s till last storeAllDataBooks() call.
	 * 
     * <br><br>Events:<br>
     * It calls before the restore on all registered IComponents the cancelEditing() method, to 
     * cancel the editing mode of the GUI control.
	 * @throws ModelException
	 *             if not all changes could be restored in the <code>IDataBook</code>
   	 */
	public void restoreAllDataBooks() throws ModelException;	
	
	/**
	 * Register a <code>IDataBook</code> at the <code>IDataSource</code>.
	 * 
	 * @param pDataBook
	 * 			the new <code>IDataBook</code> to register
	 */
	public void addDataBook(IDataBook pDataBook);	

	/**
	 * Unregister a <code>IDataBook</code> at the <code>IDataSource</code>.
	 * 
	 * @param pDataBook
	 * 			the <code>IDataBook</code> to unregister
	 */
	public void removeDataBook(IDataBook pDataBook);
	
	/**
	 * Returns all registered <code>IDataBook</code>s from the <code>IDataSource</code>.
	 * 
	 * @return all registered <code>IDataBook</code>s from the <code>IDataSource</code>.
	 */
	public IDataBook[] getDataBooks();

	/**
	 * Sets when the <code>IDataSource</code> needs to write back the data to the storage.<br>
	 * The default write back isolation level is <code>DATA_ROW</code>.
	 * 
	 * @see javax.rad.model.IDataBook.WriteBackIsolationLevel
	 * @param pIsolationLevel
	 *            the new write back isolation level.
	 */
	public void setWritebackIsolationLevel(WriteBackIsolationLevel pIsolationLevel);

	/**
	 * Returns the current write back isolation level.
	 * 
	 * @return the current write back isolation level.
	 */	
	public WriteBackIsolationLevel getWritebackIsolationLevel();
	
} // IDataSource
