/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [RH] - creation
 * 09.04.2009 - [RH] - interface review - compareTo uses SortDefinition
 *                                        equals added
 *                                        getValue(int) added
 *                                        getValue(s) throw no ModelException anymore
 *                                        register,unregisterEditingControl removed
 *                                        clone renamed to createDataRow()
 *                                        IChangeableDataRow methods/functionality moved to ChangeableDataRow
 * 16.04.2009 - [RH] - remove/add/getDetailDataBooks moved to IDataBook.
 * 18.04.2009 - [RH] - get/set/DataPage/RowIndex moved to ChangeableDataRow  
 * 30.03.2010 - [RH] - #6: getValueAsString in IDataRow                                                                       
 * 06.05.2010 - [JR] - getValuesAsString defined
 * 03.04.2014 - [RZ] - #2 - added eventValuesChanged(String pColumnName)                                                                 
 */
package javax.rad.model;

import javax.rad.model.event.DataRowHandler;
import javax.rad.model.ui.IControl;

/**
 * An <code>IDataRow</code> is a list of columns.<br>
 * The <code>IDataRow</code> is also storage independent row.<br>
 * 
 * @see javax.rad.model.IRowDefinition
 * @see javax.rad.model.IDataBook
 * @see javax.rad.model.IDataPage
 * 
 * @author Roland Hrmann
 */
public interface IDataRow extends Comparable<IDataRow>
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Method definitions
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Returns the <code>IRowDefinition</code> of the <code>IDataRow</code>.
	 * 
	 * @return the <code>IRowDefinition</code> of the <code>IDataRow</code>.
	 */
	public IRowDefinition getRowDefinition();

	/**
	 * Returns the value of the column by index.
	 * 
	 * @param pColumnIndex
	 *            the column index
	 * @return the value of the column.
	 * @throws ModelException 
	 *            if the column index is not in this <code>IDataRow</code>
	 */
	public Object getValue(int pColumnIndex) throws ModelException;

	/**
	 * Returns the value of the named column.
	 * 
	 * @param pColumnName
	 *            the name of the column
	 * @return the value of the named column.
	 * @throws ModelException 
	 *            if the <code>ColumnDefinition</code> name is not in this <code>IDataRow</code>
	 */
	public Object getValue(String pColumnName) throws ModelException;

	/**
	 * Returns the value of the named column as <code>String</code>.
	 * 
	 * @param pColumnName
	 *            the name of the column
	 * @return the value of the named column as <code>String</code>.
	 * @throws ModelException 
	 *            if the <code>ColumnDefinition</code> name is not in this <code>IDataRow</code>
	 */
	public String getValueAsString(String pColumnName) throws ModelException;
	
	/**
	 * Sets the value of the named column in this <code>IDataRow</code>.
	 * <br><br>
	 * Events:<br>
	 * After the value is changed, the IDataRowListener.changed(this, new String[] { pColumnName }, drOldRow) method is called.<br>
	 * After the value is changed, all registered IControls get called the notifyRepaint() method.
	 * 
	 * @param pColumnName
	 *            the column name
	 * @param pValue
	 *            the new value for the column in this <code>IDataRow</code>
	 * @throws ModelException 
	 *            if the <code>ColumnDefinition</code> name is not in this <code>IDataRow</code>
	 *            or the pValue is not convertible/too large to/for the <code>IDataType</code> 
	 *            of the column     
	 */
	public void setValue(String pColumnName, Object pValue) throws ModelException;

	/**
	 * Returns a Object[] of values from the specified columns in this <code>IDataRow</code>.
	 * 
	 * @param pColumnNames
	 *            a String[] of column names
	 * @return a Object[] of values from the specified columns in this <code>IDataRow</code>
	 * @throws ModelException 
	 *            if the <code>ColumnDefinition</code> name is not in this <code>IDataRow</code>
	 */
	public Object[] getValues(String[] pColumnNames) throws ModelException;

	/**
	 * Returns the values of the named columns as <code>String[]</code>.
	 * 
	 * @param pColumnNames the names of the columns
	 * @return the values of the named columns as <code>String[]</code>.
	 * @throws ModelException if one the <code>ColumnDefinition</code> name is not in this <code>IDataRow</code>
	 */
	public String[] getValuesAsString(String[] pColumnNames) throws ModelException;
	
	/**
	 * Sets the column's values from the pValues <code>Object[]</code>
	 * to the this <code>IDataRow</code>.<br>
	 * It finds the corresponding target column over the column name in the pColumnNames
	 * <code>String[]</code>.
	 * <br><br>
	 * Events:<br>
	 * After the value is changed, the IDataRowListener.changed(this, pColumnNames, drOldRow) method is called.<br>
	 * After the value is changed, all registered IControls get called the notifyRepaint() method.
	 * 
	 * @param pColumnNames
	 *            a String[] of column names
	 * @param pValues
	 *            a Object[] of values to set in the corresponding columns of the <code>IDataRow</code>
	 * @throws ModelException 
	 *            if the <code>ColumnDefinition</code> name is not in this <code>IDataRow</code>
	 *            or the pValue is not convertible/too large to/for the <code>DataType</code>
	 *            of the column     
	 */
	public void setValues(String[] pColumnNames, Object[] pValues) throws ModelException;
		
	/**
	 * It compares the drDataRow with this <code>IDataRow</code> and it uses the columns and
	 * order information (asc, desc) in the SortDefintion. It reacts like a normal compareTo with
	 * specific columns, but if in the SortDefinition the order is descending for a columns it 
	 * multiplies the result with -1. -> invert the result.
	 * 
	 * @param pDataRow
	 *            the <code>DataRow</code> to compare with this <code>IDataRow</code>
	 * @param pSortDefinition
	 *            the SortDefinition to us in the compare
	 * @return a negative integer, zero, or a positive integer as this <code>IDataRow</code>
	 *         is less than, equal to, or greater than the specified <code>DataRow</code>.
	 */
	public int compareTo(IDataRow pDataRow, SortDefinition pSortDefinition);

	/**
	 * Compares the drDataRow with this <code>IDataRow</code>, but it only compares the
	 * specified columns.
	 * 
	 * @param pDataRow
	 *            the <code>DataRow</code> to compare with this <code>IDataRow</code>
	 * @param pColumnNames
	 *            a String[] of column names to compare
	 * @return true if equals.
	 */
	public boolean equals(IDataRow pDataRow, String[] pColumnNames);

	/**
	 * Returns a cloned <code>IDataRow</code> with only a subset of specified 
	 * column names.
	 * 
	 * @param pColumnNames
	 *            a String[] of column names
	 * @return a cloned <code>IDataRow</code> with only a subset of specified 
	 *         column names.
	 * @throws ModelException
	 *             if the <code>IDataRow</code> couldn't constructed
	 */
	public IDataRow createDataRow(String[] pColumnNames) throws ModelException;

	/**
	 * Returns a new empty <code>IDataRow</code> with only a subset of column's,
	 * specified by a String[] of column names.
	 * 
	 * @param pColumnNames
	 *            a String[] of column names
	 * @return a new empty IDataRow with only a subset of columns, specified
	 *         by a String[] of column names.
	 * @throws ModelException
	 *             if the <code>IDataRow</code> couldn't constructed
	 */
	public IDataRow createEmptyRow(String[] pColumnNames) throws ModelException;

	/**
	 * Adds a <code>IControl</code> to the <code>IDataRow</code>.<br>
	 * The registered IControl methods will be called if <code>IDataRow</code> has changed, restored or stored.
	 * 
	 * @param pControl the <code>IControl</code>
	 * @see javax.rad.model.ui.IControl
	 */
	public void addControl(IControl pControl);

	/**
	 * Removes the <code>IControl</code> from the <code>IDataRow</code>.
	 * 
	 * @param pControl the <code>IControl</code>
	 * @see javax.rad.model.ui.IControl
	 */
	public void removeControl(IControl pControl);

	/**
	 * Returns all <code>IControl</code>'s from the <code>IDataRow</code>.
	 * 
	 * @return all <code>IControl</code>'s from the <code>IDataRow</code>.
	 */
	public IControl[] getControls();

    /**
     * Gets the EventHandler for values changed event.
     * 
     * @return the EventHandler for values changed events.
     */
	public DataRowHandler eventValuesChanged();

	/**
	 * Gets the EventHandler for values changed event of the specified column.
	 * 
	 * @param pColumnName the name of the column.
	 * @return the EventHandler for values changed events of the specified column.
	 */
	public DataRowHandler eventValuesChanged(String pColumnName);
	
	/**
	 * Its invokes for each <code>IComponent</code> the <code>notifyRepaint()</code> method.<br>
	 */
	public void notifyRepaintControls();
		
	/**
	 * Its invokes for each <code>IComponent</code>  the <code>saveEditing()</code> method.
	 * @throws ModelException if saving the editors fails.
	 */
	public void saveEditingControls() throws ModelException;
	
	/**
	 * Its invokes for each <code>IComponent</code>  the <code>cancelEditing()</code> method.<br>
	 */
	public void cancelEditingControls();
	
} 	// IDataRow
