/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 16.11.2008 - [HM] - creation
 * 25.11.2008 - [JR] - dispatchEvent: fixed NullPointerException
 * 13.12.2008 - [JR] - dispatchEvent: throwed throwable during method call [BUGFIX]
 * 17.02.2009 - [JR] - setBusy, callIntern implemented
 * 03.04.2009 - [JR] - dispatchCall: always use the eventDispatcher from the component instead
 *                     of the component itself (normally the same)
 * 31.12.2010 - [JR] - translate: removed isNotified() check -> if we have a translation then use it
 * 18.03.2011 - [JR] - #313: component moved/resized implemented
 * 31.03.2011 - [JR] - #161: ITranslatable interface added
 * 14.05.2011 - [JR] - updateTranslation: parent instance check - otherwise StackOverflow
 * 27.05.2011 - [JR] - #372: 
 *                     * updateTranslation checks translation loop
 *                     * setTranslation unsets translation map parent if it was changed from the component
 * 04.10.2011 - [JR] - #477: beforeAddNotify implemented    
 * 02.11.2011 - [JR] - #493: en/disable translation
 *                   - #494: don't create an automatic translation chain
 * 04.04.2014 - [RZ] - #1: added eventKey(Key) which allows listening for a certain key
 * 05.04.2014 - [JR] - #1001: don't change text if translation is disabled                 
 */
package javax.rad.genui;

import java.util.HashMap;
import java.util.Map;

import javax.rad.model.ui.ITranslatable;
import javax.rad.ui.IColor;
import javax.rad.ui.IComponent;
import javax.rad.ui.IContainer;
import javax.rad.ui.ICursor;
import javax.rad.ui.IDimension;
import javax.rad.ui.IFactory;
import javax.rad.ui.IFont;
import javax.rad.ui.IImage;
import javax.rad.ui.IInsets;
import javax.rad.ui.IPoint;
import javax.rad.ui.IRectangle;
import javax.rad.ui.control.ICellFormatter;
import javax.rad.ui.control.INodeFormatter;
import javax.rad.ui.event.ComponentHandler;
import javax.rad.ui.event.FocusHandler;
import javax.rad.ui.event.Key;
import javax.rad.ui.event.KeyHandler;
import javax.rad.ui.event.MouseHandler;
import javax.rad.ui.event.UIKeyEvent;
import javax.rad.util.EventHandler;
import javax.rad.util.TranslationMap;

import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.LoggerFactory;

/**
 * Platform and technology independent component.
 * It is designed for use with AWT, Swing, SWT, JSP, JSF,... .
 * 
 * 
 * @author Martin Handsteiner
 * 
 * @param <C> instance of IComponent
 */
public abstract class UIComponent<C extends IComponent> extends UIResource<C> 
                                                        implements IComponent,
                                                                   ITranslatable
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** This EventHandler is used to create dynamic Runnable interfaces. */
	private static EventHandler<Runnable> runnableEventHandler = new EventHandler<Runnable>(Runnable.class);
	
	/** The cell formatter provider. */
	private static EventHandler<ICellFormatter> cellFormatterProvider = new EventHandler<ICellFormatter>(ICellFormatter.class);

	/** The node formatter provider. */
	private static EventHandler<INodeFormatter> nodeFormatterProvider = new EventHandler<INodeFormatter>(INodeFormatter.class);

	/** the component logger. */
	private ILogger logger = null;

	/** the UIParent of this UICompoennt. */
	protected IContainer parent = null;
	
	/** the expected parent for beforeAddNotify. */
	private IComponent parentExpected = null;

	/** the translation map for this component. */
	private TranslationMap tmpUserdefined = null;
	
	/** the reference to a translation map of a parent. */
	private TranslationMap tmpCurrent = null;
	
    /** the map which maps {@link Key}s to {@link KeyHandler}s. */
    private Map<Key, KeyHandler> mpEventKeys;

    /** the tooltip. */
	private String sToolTip = null;
	
	/** the time of the last translation modification. */
	protected long lLastTranslationModified = -1;
	
	/** the flag indicates whether the component was "before" notified. */
	private boolean bBeforeNotified = false;
	
	/** the flag indicates whether the component was notified. */
	private boolean bNotified = false;
	
	/** whether translate should translate texts. */
	protected boolean bTranslate = true;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	  
    /**
     * Creates a new instance of <code>UIComponent</code>.
     *
     * @param pComponent the Component.
     * @see IComponent
     */
	protected UIComponent(C pComponent)
	{
		super(pComponent);
		
		pComponent.setEventSource(this);
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Interface Implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public String getName()
    {
    	return uiResource.getName();
    }

	/**
	 * {@inheritDoc}
	 */
	public void setName(String pName)
    {
    	uiResource.setName(pName);
    }

	/**
	 * {@inheritDoc}
	 */
	public IFactory getFactory()
    {
    	return uiResource.getFactory();
    }

	/**
	 * {@inheritDoc}
	 */
	public IDimension getPreferredSize()
    {
    	return getComponentUIResource().getPreferredSize();
    }

	/**
	 * {@inheritDoc}
	 */
	public void setPreferredSize(IDimension pPreferredSize)
    {
		// ensure that the factory gets his own resource, to prevent exception on factory internal casts!
		if (pPreferredSize instanceof UIDimension)
		{
			getComponentUIResource().setPreferredSize(((UIDimension)pPreferredSize).uiResource);
		}
		else
		{
			getComponentUIResource().setPreferredSize(pPreferredSize);
		}
    }

	/**
	 * {@inheritDoc}
	 */
	public boolean isPreferredSizeSet()
    {
    	return getComponentUIResource().isPreferredSizeSet();
    }

	/**
	 * {@inheritDoc}
	 */
	public IDimension getMinimumSize()
    {
    	return getComponentUIResource().getMinimumSize();
    }

	/**
	 * {@inheritDoc}
	 */
	public void setMinimumSize(IDimension pMinimumSize)
    {
		// ensure that the factory gets his own resource, to prevent exception on factory internal casts!
		if (pMinimumSize instanceof UIDimension)
		{
			getComponentUIResource().setMinimumSize(((UIDimension)pMinimumSize).uiResource);
		}
		else
		{
			getComponentUIResource().setMinimumSize(pMinimumSize);
		}
    }

	/**
	 * {@inheritDoc}
	 */
	public boolean isMinimumSizeSet()
    {
    	return getComponentUIResource().isMinimumSizeSet();
    }

	/**
	 * {@inheritDoc}
	 */
	public IDimension getMaximumSize()
    {
    	return getComponentUIResource().getMaximumSize();
    }

	/**
	 * {@inheritDoc}
	 */
	public void setMaximumSize(IDimension pMaximumSize)
    {
		// ensure that the factory gets his own resource, to prevent exception on factory internal casts!
		if (pMaximumSize instanceof UIDimension)
		{
			getComponentUIResource().setMaximumSize(((UIDimension)pMaximumSize).uiResource);
		}
		else
		{
			getComponentUIResource().setMaximumSize(pMaximumSize);
		}
    }

	/**
	 * {@inheritDoc}
	 */
	public boolean isMaximumSizeSet()
    {
    	return getComponentUIResource().isMaximumSizeSet();
    }

	/**
	 * {@inheritDoc}
	 */
	public IColor getBackground()
    {
    	return uiResource.getBackground();
    }

	/**
	 * {@inheritDoc}
	 */
	public void setBackground(IColor pBackground)
    {
		// ensure that the factory gets his own resource, to prevent exception on factory internal casts!
		if (pBackground instanceof UIColor)
		{
	    	uiResource.setBackground(((UIColor)pBackground).getUIResource());
		}
		else
		{
	    	uiResource.setBackground(pBackground);
		}
    }

	/**
	 * {@inheritDoc}
	 */
	public boolean isBackgroundSet()
    {
    	return uiResource.isBackgroundSet();
    }

	/**
	 * {@inheritDoc}
	 */
	public IColor getForeground()
    {
    	return uiResource.getForeground();
    }

	/**
	 * {@inheritDoc}
	 */
	public void setForeground(IColor pForeground)
    {
		// ensure that the factory gets his own resource, to prevent exception on factory internal casts!
		if (pForeground instanceof UIColor)
		{
	    	uiResource.setForeground(((UIColor)pForeground).getUIResource());
		}
		else
		{
	    	uiResource.setForeground(pForeground);
		}
    }

	/**
	 * {@inheritDoc}
	 */
	public boolean isForegroundSet()
    {
    	return uiResource.isForegroundSet();
    }

	/**
	 * {@inheritDoc}
	 */
	public ICursor getCursor()
    {
    	return uiResource.getCursor();
    }

	/**
	 * {@inheritDoc}
	 */
	public void setCursor(ICursor pCursor)
    {
		// ensure that the factory gets his own resource, to prevent exception on factory internal casts!
		if (pCursor instanceof UICursor)
		{
	    	uiResource.setCursor(((UICursor)pCursor).uiResource);
		}
		else
		{
	    	uiResource.setCursor(pCursor);
		}
    }

	/**
	 * {@inheritDoc}
	 */
	public boolean isCursorSet()
    {
    	return uiResource.isCursorSet();
    }

	/**
	 * {@inheritDoc}
	 */
	public IFont getFont()
    {
    	return uiResource.getFont();
    }

	/**
	 * {@inheritDoc}
	 */
	public void setFont(IFont pFont)
    {
		// ensure that the factory gets his own resource, to prevent exception on factory internal casts!
		if (pFont instanceof UIFont)
		{
	    	uiResource.setFont(((UIFont)pFont).getUIResource());
		}
		else
		{
	    	uiResource.setFont(pFont);
		}
    }

	/**
	 * {@inheritDoc}
	 */
	public boolean isFontSet()
    {
    	return uiResource.isFontSet();
    }

	/**
	 * {@inheritDoc}
	 */
	public String getToolTipText()
    {
    	return sToolTip;
    }
	
	/**
	 * {@inheritDoc}
	 */
	public void setToolTipText(String pText)
    {
		sToolTip = pText;

		uiResource.setToolTipText(translate(pText));
    }

	/**
	 * {@inheritDoc}
	 */
	public void setFocusable(boolean pFocusable)
    {
    	uiResource.setFocusable(pFocusable);
    }
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isFocusable()
    {
    	return uiResource.isFocusable();
    }

	/**
	 * {@inheritDoc}
	 */
	public void requestFocus()
    {
		// Request focus has to address the uiResource 
		// (for eg. ToolbarPanel with Table inside -> 
		//    Table is uiResource, 
		//    ToolbarPanel is the componentUIResource 
		// The ToolbarPanel is used for add, so visibility, location, bounds, ... should address the componentUIResource
		// The Table is the control, so focus, enable, font, ... should address the uiResource.
    	uiResource.requestFocus();
    }

	/**
	 * {@inheritDoc}
	 */
	public IContainer getParent()
    {
		return parent;
    }
	
	/**
	 * {@inheritDoc}
	 */
	public void setParent(IContainer pParent)
	{
		if (pParent == null)
		{
			if (parent != null && parent.indexOf(this) >= 0)
			{
				throw new IllegalArgumentException("Can't unset parent, because this component is still added!");
			}
		}
		else
		{
			if (pParent.indexOf(this) < 0)
			{
				throw new IllegalArgumentException("Can't set parent, because this component is not added!");
			}
		}
  
		boolean bUpdateTrans = parent != null && parent != pParent;
		
		parent = pParent;
		
		if (bUpdateTrans && isNotified())
		{
			updateTranslation();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isVisible()
    {
    	return getComponentUIResource().isVisible();
    }

	/**
	 * {@inheritDoc}
	 */
	public void setVisible(boolean pVisible)
    {
		getComponentUIResource().setVisible(pVisible);
    }

	/**
	 * {@inheritDoc}
	 */
	public void setEnabled(boolean pEnable)
    {
    	uiResource.setEnabled(pEnable);
    }

	/**
	 * {@inheritDoc}
	 */
	public boolean isEnabled()
    {
    	return uiResource.isEnabled();
    }
	
	/**
	 * {@inheritDoc}
	 */
	public IPoint getLocationRelativeTo(IComponent pComponent)
    {
    	return uiResource.getLocationRelativeTo(pComponent);
    }
	
	/**
	 * {@inheritDoc}
	 */
	public void setLocationRelativeTo(IComponent pComponent, IPoint pLocation)
    {
    	uiResource.setLocationRelativeTo(pComponent, pLocation);
    }
	
	/**
	 * {@inheritDoc}
	 */
	public IPoint getLocation()
    {
		IPoint loc = getComponentUIResource().getLocation();
		
		IInsets insets = getComponentUIResourceInsets();
		if (insets != null)
		{
			loc.setX(loc.getX() + insets.getLeft());
			loc.setY(loc.getY() + insets.getTop());
		}
		
		return loc;
    }

	/**
	 * {@inheritDoc}
	 */
	public void setLocation(IPoint pLocation)
    {
		setLocation(pLocation.getX(), pLocation.getY());
    }

	/**
	 * {@inheritDoc}
	 */
	public IDimension getSize()
    {
		IDimension dim = getComponentUIResource().getSize();
		
		IInsets insets = getComponentUIResourceInsets();
		if (insets != null)
		{
			dim.setWidth(dim.getWidth() - insets.getLeft() - insets.getRight());
			dim.setHeight(dim.getHeight() - insets.getTop() - insets.getBottom());
		}
		
    	return getComponentUIResource().getSize();
    }

	/**
	 * {@inheritDoc}
	 */
	public void setSize(IDimension pSize)
    {
		setSize(pSize.getWidth(), pSize.getHeight());
    }

	/**
	 * {@inheritDoc}
	 */
	public IRectangle getBounds()
    {
		IRectangle rect = getComponentUIResource().getBounds();
		
		IInsets insets = getComponentUIResourceInsets();
		if (insets != null)
		{
			int left = insets.getLeft();
			int top = insets.getTop();
			rect.setX(rect.getX() + left);
			rect.setY(rect.getY() + top);
			rect.setWidth(rect.getWidth() - left - insets.getRight());
			rect.setHeight(rect.getHeight() - top - insets.getBottom());
		}
		
		return rect;
    }

	/**
	 * {@inheritDoc}
	 */
	public void setBounds(IRectangle pBounds)
    {
		setBounds(pBounds.getX(), pBounds.getY(), pBounds.getWidth(), pBounds.getHeight());
    }

	/**
	 * {@inheritDoc}
	 */
	public Object getResource()
    {
		return getComponentUIResource().getResource();
    }

	/**
	 * {@inheritDoc}
	 */
	public IComponent getEventSource()
	{
		return uiResource.getEventSource();
	}

	/**
	 * {@inheritDoc}
	 */
	public void setEventSource(IComponent pEventSource)
	{
		uiResource.setEventSource(pEventSource);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public MouseHandler eventMousePressed()
	{
		return uiResource.eventMousePressed();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public MouseHandler eventMouseReleased()
	{
		return uiResource.eventMouseReleased();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public MouseHandler eventMouseClicked()
	{
		return uiResource.eventMouseClicked();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public MouseHandler eventMouseEntered()
	{
		return uiResource.eventMouseEntered();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public MouseHandler eventMouseExited()
	{
		return uiResource.eventMouseExited();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public KeyHandler eventKeyPressed()
	{
		return uiResource.eventKeyPressed();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public KeyHandler eventKeyReleased()
	{
		return uiResource.eventKeyReleased();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public KeyHandler eventKeyTyped()
	{
		return uiResource.eventKeyTyped();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public ComponentHandler eventComponentResized()
	{
		return uiResource.eventComponentResized();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public ComponentHandler eventComponentMoved()
	{
		return uiResource.eventComponentMoved();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public FocusHandler eventFocusGained()
	{
		return uiResource.eventFocusGained();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public FocusHandler eventFocusLost()
	{
		return uiResource.eventFocusLost();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IImage capture(int iWidth, int iHeight)
	{
		return getComponentUIResource().capture(iWidth, iHeight);
	}
	
	//ITRANSLATABLE
	
    /**
     * {@inheritDoc}
     */
    public void setTranslation(TranslationMap pTranslation)
    {
        if (tmpCurrent != pTranslation)
        {
            lLastTranslationModified = -1;
        }
        
        tmpUserdefined = pTranslation;
        tmpCurrent = pTranslation;
        
        if (isNotified())
        {
            updateTranslation();
        }
    }
    
    /**
     * {@inheritDoc}
     */
    public TranslationMap getTranslation()
    {
        return tmpUserdefined;
    }
    
    /**
     * {@inheritDoc}
     */
    public void setTranslationEnabled(boolean pEnabled)
    {
        bTranslate = pEnabled;
    }

    /**
     * {@inheritDoc}
     */
    public boolean isTranslationEnabled()
    {
        return bTranslate;
    }
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * Set the preferred size with primitive types.
     * 
     * @param pWidth the width.
     * @param pHeight the height.
     */
    public void setPreferredSize(int pWidth, int pHeight)
    {
        getComponentUIResource().setPreferredSize(getFactory().createDimension(pWidth, pHeight));
    }

    /**
     * Set the minimum size with primitive types.
     * 
     * @param pWidth the width.
     * @param pHeight the height.
     */
    public void setMinimumSize(int pWidth, int pHeight)
    {
        getComponentUIResource().setMinimumSize(getFactory().createDimension(pWidth, pHeight));
    }

    /**
     * Set the maximum size with primitive types.
     * 
     * @param pWidth the width.
     * @param pHeight the height.
     */
    public void setMaximumSize(int pWidth, int pHeight)
    {
        getComponentUIResource().setMaximumSize(getFactory().createDimension(pWidth, pHeight));
    }

    /**
     * Set the location with primitive types.
     * 
     * @param pX the width.
     * @param pY the width.
     */
    public void setLocation(int pX, int pY)
    {
        int left = 0;
        int top = 0;
        IInsets insets = getComponentUIResourceInsets();
        if (insets != null)
        {
            left = insets.getLeft();
            top = insets.getTop();
        }
        getComponentUIResource().setLocation(getFactory().createPoint(pX - left, pY - top));
    }

    /**
     * Set the size with primitive types.
     * 
     * @param pWidth the width.
     * @param pHeight the height.
     */
    public void setSize(int pWidth, int pHeight)
    {
        int left = 0;
        int top = 0;
        int right = 0;
        int bottom = 0;
        IInsets insets = getComponentUIResourceInsets();
        if (insets != null)
        {
            left = insets.getLeft();
            top = insets.getTop();
            right = insets.getRight();
            bottom = insets.getBottom();
        }
        getComponentUIResource().setSize(getFactory().createDimension(pWidth + left + right, pHeight + top + bottom));
    }

    /**
     * Set the bounds with primitive types.
     * 
     * @param pX the width.
     * @param pY the width.
     * @param pWidth the width.
     * @param pHeight the height.
     */
    public void setBounds(int pX, int pY, int pWidth, int pHeight)
    {
        int left = 0;
        int top = 0;
        int right = 0;
        int bottom = 0;
        IInsets insets = getComponentUIResourceInsets();
        if (insets != null)
        {
            left = insets.getLeft();
            top = insets.getTop();
            right = insets.getRight();
            bottom = insets.getBottom();
        }
        getComponentUIResource().setBounds(getFactory().createRectangle(pX - left, pY - top, pWidth + left + right, pHeight + top + bottom));
    }

    /**
     * Creates a cell formatter instance with the given object and method name.
     * @param pCellFormatter the object.
     * @param pMethodName the method name.
     * @return the cell formatter.
     */
	public static ICellFormatter createCellFormatter(Object pCellFormatter, String pMethodName)
	{
		return cellFormatterProvider.createListener(pCellFormatter, pMethodName);
	}
	
    /**
     * Creates a node formatter instance with the given object and method name.
     * @param pNodeFormatter the object.
     * @param pMethodName the method name.
     * @return the cell formatter.
     */
	public static INodeFormatter createNodeFormatter(Object pNodeFormatter, String pMethodName)
	{
		return nodeFormatterProvider.createListener(pNodeFormatter, pMethodName);
	}

	/**
	 * Gets the component which will be added/removed to an {@link UIContainer} instead
	 * of this component. Thats needed to create complex Controls.
	 *
	 * Example Control: an extended Table Control with Toolbar. 
	 * So a ToolbarPanel with Table inside -> 
	 *    Table is uiResource, 
	 *    ToolbarPanel is the componentUIResource 
	 * The ToolbarPanel is used for add, so visibility, location, bounds, ... should address the componentUIResource
	 * The Table is the control, so focus, enable, font, ... should address the uiResource.
	 * 
	 * @return the component which should be added/removed
	 */
	protected IComponent getComponentUIResource()
	{
		return getUIResource();
	}
	
	/**
	 * Gets the component which will be added/removed to an {@link UIContainer} instead
	 * of this component. Thats needed to create complex Controls.
	 *
	 * Example Control: an extended Table Control with Toolbar. 
	 * So a ToolbarPanel with Table inside -> 
	 *    Table is uiResource, 
	 *    ToolbarPanel is the componentUIResource 
	 * The ToolbarPanel is used for add, so visibility, location, bounds, ... should address the componentUIResource
	 * The Table is the control, so focus, enable, font, ... should address the uiResource.
	 * 
	 * @return the component which should be added/removed
	 */
	protected IInsets getComponentUIResourceInsets()
	{
		return null;
	}
	
	/**
	 * Logs debug information.
	 * 
	 * @param pInfo the debug information
	 */
	public void debug(Object... pInfo)
	{
		if (logger == null)
		{
			logger = LoggerFactory.getInstance(getClass());
		}
		
		logger.debug(pInfo);
	}

	/**
	 * Logs information.
	 * 
	 * @param pInfo the information
	 */
	public void info(Object... pInfo)
	{
		if (logger == null)
		{
			logger = LoggerFactory.getInstance(getClass());
		}
		
		logger.info(pInfo);
	}

	
	/**
	 * Logs error information.
	 * 
	 * @param pInfo the error information
	 */
	public void error(Object... pInfo)
	{
		if (logger == null)
		{
			logger = LoggerFactory.getInstance(getClass());
		}

		logger.error(pInfo);
	}

	/**
	 * Invoked before this component is added.
	 *  
	 * @param pParent the parent
	 */
	public void beforeAddNotify(IComponent pParent)
	{
		parentExpected = pParent;
		
		bBeforeNotified = true;
		
		try
		{
			updateTranslation();
		}
		finally
		{
			parentExpected = null;
		}
	}

	/**
	 * Gets whether this component is notified before it was added.
	 *  
	 * @return <code>true</code> this component is notified, <code>false</code> otherwise
	 */
	public boolean isBeforeNotified()
	{
		return bBeforeNotified;
	}
	
    /**
     * Makes this <code>UIComponent</code> displayable by adding it to an {@link UIContainer}. 
     * This method is called internally by the genui and should not be called directly.
     * 
     * @see #removeNotify
     * @see #isNotified()
     */
	public void addNotify()
	{
		bNotified = true;

		if (!bBeforeNotified)
		{
			updateTranslation();
		}
	}
	
	/**
     * Determines whether this <code>UIComponent</code> is displayable. A component is 
     * displayable when it is added to an {@link UIContainer}.
     * <p>
     * An <code>UIComponent</code> is made displayable either when it is added to
     * a displayable containment hierarchy or when its containment
     * hierarchy is made displayable.
     * A containment hierarchy is made displayable when its ancestor 
     * {@link javax.rad.genui.container.UIWindow} is either packed or made visible.
     * <p>
     * An <code>UIComponent</code> is made undisplayable either when it is removed from
     * a displayable containment hierarchy or when its containment hierarchy
     * is made undisplayable.  A containment hierarchy is made 
     * undisplayable when its ancestor {@link javax.rad.genui.container.UIWindow} is disposed.
	 *  
	 * @return <code>true</code> if the component is displayable, <code>false</code> otherwise
	 * @see UIContainer#add(IComponent, Object, int)
	 * @see UIContainer#remove(int)
	 * @see javax.rad.genui.container.UIWindow#setVisible(boolean)
	 * @see javax.rad.genui.container.UIWindow#pack()
	 * @see javax.rad.genui.container.UIWindow#dispose()
	 */
	public boolean isNotified()
	{
		return bNotified;
	}
	
    /** 
     * Makes this <code>UIComponent</code> undisplayable by removing it to an {@link UIContainer}.
     * <p>
     * This method is called by the genui internally and should not be called directly. 
     * Code overriding this method should call <code>super.removeNotify</code> as the first line 
     * of the overriding method.
     *
     * @see #addNotify
     * @see #isNotified()
     */
	public void removeNotify()
	{
		bBeforeNotified = false;
		bNotified = false;
	}
	
	/**
	 * Gets the current translation mapping usable for this <code>UIComponent</code>.
	 * 
	 * @return the current translation mapping
	 */
	protected TranslationMap getCurrentTranslation()
	{
		return tmpCurrent;
	}
	
	/**
	 * Translates the given text with the avaliable translation. If this component has no translation,
	 * then the translation from the parent component will be used.
	 * 
	 * @param pText the text to translate
	 * @return the translated text or <code>pText</code> if no translation was found
	 */
	public String translate(String pText)
	{
		if (bTranslate && tmpCurrent != null)
		{
			return tmpCurrent.translate(pText);
		}
		else
		{
			return pText;
		}
	}

	/**
	 * Notification for updating the translation. This method will be called when 
	 * the <code>UIComponent</code> will be added to a displayable containment hierarchy, 
	 * when its containment hierarchy is made displayable or the translation table will
	 * be changed.
	 * 
	 * @see #setTranslation(TranslationMap)
	 * @see #addNotify()
	 */
	public void updateTranslation()
	{
		IComponent comParent;
		
		if (parent == null)
		{
			comParent = parentExpected;
		}
		else
		{
			comParent = parent;
		}
		
		if (comParent != null && tmpUserdefined == null)
		{
			if (tmpCurrent != ((UIComponent)comParent).tmpCurrent)
			{
				lLastTranslationModified = -1;
			}
			
			tmpCurrent = ((UIComponent)comParent).tmpCurrent;
		}
		
		if (isTranslationChanged())
		{
			if (bTranslate && sToolTip != null)
			{
				uiResource.setToolTipText(translate(sToolTip));
			}
			
			if (tmpCurrent != null)
			{
				lLastTranslationModified = tmpCurrent.lastModified();
			}
		}
	}
	
	/**
	 * Dispatches the given {@link UIKeyEvent} to all EventHandlers that have
	 * subscribed to the {@link Key} of the event.
	 * 
	 * @param pKeyEvent the {@link UIKeyEvent} to dispatch
	 */
	public void doEventKey(UIKeyEvent pKeyEvent)
	{
		Key key = Key.getKey(pKeyEvent);
		
		if (mpEventKeys.containsKey(key))
		{
			mpEventKeys.get(key).dispatchEvent(pKeyEvent);
		}
	}
	
	/**
	 * The EventHandler for the given {@link Key}.
	 * 
	 * @param pKey the {@link Key} whose EventHandler is returned
	 * @return the EventHandler for the given {@link Key}
	 */
	public KeyHandler eventKey(Key pKey)
	{
		if (mpEventKeys == null)
		{
			mpEventKeys = new HashMap<Key, KeyHandler>();
			
			eventKeyPressed().addListener(this, "doEventKey");
			eventKeyReleased().addListener(this, "doEventKey");
			eventKeyTyped().addListener(this, "doEventKey");
		}
		
		if (!mpEventKeys.containsKey(pKey))
		{
			String keyEventTypeText = null;
			
			switch (pKey.getKeyEventType())
			{
				case UIKeyEvent.KEY_PRESSED:
					keyEventTypeText = "keyPressed";
					break;
					
				case UIKeyEvent.KEY_RELEASED:
					keyEventTypeText = "keyReleased";
					break;
					
				case UIKeyEvent.KEY_TYPED:
					keyEventTypeText = "keyTyped";
					break;
					
				default:
					// Do nothing, this shouldn't happen.
					
			}

			KeyHandler handler = new KeyHandler(keyEventTypeText);
			
			mpEventKeys.put(pKey, handler);
			
			return handler;
		}
		else
		{
	        return mpEventKeys.get(pKey);
		}
	}
	
	/**
	 * Gets whether the translation was changed.
	 * 
	 * @return <code>true</code> if translation was changed, <code>false</code> otherwise
	 */
	protected boolean isTranslationChanged()
	{
		IComponent comParent;
		
		if (parent == null)
		{
			comParent = parentExpected;
		}
		else
		{
			comParent = parent;
		}
		
		return lLastTranslationModified == -1 
			   //this check is important because it is possible that a nested class calls isTranslationChanged()
			   //before the check in updateTranslation was executed!
			   || (comParent != null && tmpCurrent != ((UIComponent)comParent).tmpCurrent)
			   //good if this method is called after updateTranslation()
		       || (tmpCurrent != null && tmpCurrent.lastModified() != lLastTranslationModified);
	}

    /**
     * Causes <code>pRunnable.run()</code> to be executed asynchronously on the event dispatching thread. 
     * This will happen after all pending events have been processed. This method 
     * should be used when an application thread needs to update the GUI.
     *  
     * @param pRunnable the asynchronous call
     * @see #invokeAndWait(Runnable)
     */
    public static void invokeLater(Runnable pRunnable)
    {
        UIFactoryManager.getFactory().invokeLater(pRunnable);
    }
    
    /**
     * Causes <code>pRunnable.run()</code> to be executed asynchronously on the event dispatching thread. 
     * This will happen after all pending events have been processed. This method 
     * should be used when an application thread needs to update the GUI.
     *  
     * @param pObject the object
     * @param pMethod the asynchronous call
     * @see #invokeAndWait(Runnable)
     */
    public static void invokeLater(Object pObject, String pMethod)
    {
        UIFactoryManager.getFactory().invokeLater(runnableEventHandler.createListener(pObject, pMethod));
    }
    
    /**
     * Causes <code>pRunnable.run()</code> to be executed synchronously on the event dispatching thread. This call blocks 
     * until all pending events have been processed and (then) <code>pRunnable.run()</code> returns. 
     * This method should be used when an application thread needs to update the GUI.
     * 
     * @param pRunnable the call
     * @throws Exception if the call caueses an exception
     */
    public static void invokeAndWait(Runnable pRunnable) throws Exception
    {
        UIFactoryManager.getFactory().invokeAndWait(pRunnable);
    }
    
    /**
     * Causes <code>pRunnable.run()</code> to be executed synchronously on the event dispatching thread. This call blocks 
     * until all pending events have been processed and (then) <code>pRunnable.run()</code> returns. 
     * This method should be used when an application thread needs to update the GUI.
     * 
     * @param pRunnable the object
     * @param pMethod the call
     * @throws Exception if the call caueses an exception
     */
    public static void invokeAndWait(Object pRunnable, String pMethod) throws Exception
    {
        UIFactoryManager.getFactory().invokeAndWait(runnableEventHandler.createListener(pRunnable, pMethod));
    }
    
    /**
     * Causes <code>pRunnable.run()</code> to be executed asynchronously in a new thread. 
     * Action calls and UI Calls in the thread should be synchronized with the event dispatching thread
     * by using invokeLater or invokeAndWait.
     * This gives the IFactory implementation a chance to decide how and when to run the threads.
     * 
     * @param pRunnable the call
     */
    public static void invokeInThread(Runnable pRunnable)
    {
        UIFactoryManager.getFactory().invokeInThread(pRunnable);
    }
    
    /**
     * Causes <code>pRunnable.run()</code> to be executed asynchronously in a new thread. 
     * Action calls and UI Calls in the thread should be synchronized with the event dispatching thread
     * by using invokeLater or invokeAndWait.
     * This gives the IFactory implementation a chance to decide how and when to run the threads.
     * 
     * @param pRunnable the object
     * @param pMethod the call
     */
    public static void invokeInThread(Object pRunnable, String pMethod)
    {
        UIFactoryManager.getFactory().invokeInThread(runnableEventHandler.createListener(pRunnable, pMethod));
    }
	
}	// UIComponent
