/*
 * Copyright 2011 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 11.12.2011 - [JR] - creation
 * 18.12.2011 - [JR] - #525: convertValue implemented
 */
package com.sibvisions.rad.server.http.rest;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.HashMap;
import java.util.List;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentMap;

import javax.rad.model.ModelException;
import javax.rad.model.condition.Equals;
import javax.rad.model.condition.ICondition;
import javax.rad.model.datatype.TimestampDataType;
import javax.rad.persist.ColumnMetaData;
import javax.rad.persist.DataSourceException;
import javax.rad.persist.IStorage;
import javax.rad.persist.MetaData;
import javax.rad.type.bean.IBean;

import org.codehaus.jackson.map.DeserializationConfig;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.map.SerializationConfig;
import org.codehaus.jackson.map.util.StdDateFormat;
import org.restlet.data.Form;
import org.restlet.data.Parameter;
import org.restlet.data.Status;
import org.restlet.engine.header.HeaderConstants;
import org.restlet.ext.jackson.JacksonRepresentation;
import org.restlet.representation.Representation;
import org.restlet.resource.Delete;
import org.restlet.resource.Get;
import org.restlet.resource.Options;
import org.restlet.resource.Post;
import org.restlet.resource.Put;
import org.restlet.resource.ServerResource;
import org.restlet.util.Series;

import com.sibvisions.rad.persist.AbstractStorage;
import com.sibvisions.rad.server.DirectServerSession;
import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.LoggerFactory;

/**
 * The <code>AbstractStorageServerResource</code> lists all available records of an {@link AbstractStorage}.
 * 
 * @author Ren Jahn
 */
public class AbstractStorageServerResource extends ServerResource
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the logger. */
	private static ILogger logger = LoggerFactory.getInstance(AbstractStorageServerResource.class);
	
	/** the standard date format. */
	private static final StdDateFormat DATE_FORMAT = StdDateFormat.instance;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Fetches records from the storage. It is possible to filter the result if query parameters are used.
	 * 
	 * @return the records as JSON representation
	 * @throws Throwable if fetch fails
	 */
	@Get
	public Representation executeFetch() throws Throwable
	{
		try
		{
			DirectServerSession session = ((RESTSession)getRequest().getClientInfo().getUser()).getSession();
			
			ConcurrentMap<String, Object> cmpAttrib = getRequest().getAttributes();
	
			AbstractStorage storage = (AbstractStorage)session.get((String)cmpAttrib.get(RESTAdapter.PARAM_OBJECT_NAME));
	
			List<IBean> liBeans = storage.fetchBean(getCondition(storage), null, 0, -1);
			
			return configure(new JacksonRepresentation(liBeans));
		}
		catch (Throwable th)
		{
			logger.debug(th);
			
			throw th;
		}
	}
	
	/**
	 * Inserts a new record.
	 * 
	 * @param pRepresentation the new record
	 * @return the inserted record
	 * @throws Throwable if insert fails
	 */
	@Post
	public Representation executeInsert(Representation pRepresentation) throws Throwable
	{
		if (pRepresentation == null)
		{
			setStatus(Status.CLIENT_ERROR_BAD_REQUEST);
			return null;
		}
		
		try
		{
			DirectServerSession session = ((RESTSession)getRequest().getClientInfo().getUser()).getSession();
			
			ConcurrentMap<String, Object> cmpAttrib = getRequest().getAttributes();
	
			AbstractStorage storage = (AbstractStorage)session.get((String)cmpAttrib.get(RESTAdapter.PARAM_OBJECT_NAME));
			
			HashMap<String, Object> hmpObject = JSONUtil.getObject(pRepresentation, HashMap.class);
			
			if (hmpObject == null)
			{
				setStatus(Status.CLIENT_ERROR_BAD_REQUEST);
				return null;
			}
			
			IBean bean = storage.createEmptyBean();
			
			copyValues(storage, bean, hmpObject);
			
			bean = storage.insert(bean);
			
			return configure(new JacksonRepresentation(bean));
		}
		catch (Throwable th)
		{
			logger.debug(th);
			
			throw th;
		}
	}
	
	/**
	 * Updates a record. It is not possible to change columns that are primary key columns.
	 * 
	 * @param pRepresentation the new record
	 * @return the updated record
	 * @throws Throwable if update fails
	 */
	@Put
	public Representation executeUpdate(Representation pRepresentation) throws Throwable
	{
		if (pRepresentation == null)
		{
			setStatus(Status.CLIENT_ERROR_BAD_REQUEST);
			return null;
		}
		
		try
		{
			DirectServerSession session = ((RESTSession)getRequest().getClientInfo().getUser()).getSession();
			
			ConcurrentMap<String, Object> cmpAttrib = getRequest().getAttributes();
	
			AbstractStorage storage = (AbstractStorage)session.get((String)cmpAttrib.get(RESTAdapter.PARAM_OBJECT_NAME));
	
			List<IBean> liBeans = storage.fetchBean(getCondition(storage), null, 0, 2);
			
			if (liBeans.size() == 0)
			{
				setStatus(Status.CLIENT_ERROR_NOT_FOUND);
				return null;
			}
			
			if (liBeans.size() > 1)
			{
				setStatus(Status.CLIENT_ERROR_CONFLICT);
				return null;
			}
			
			HashMap<String, Object> hmpObject = JSONUtil.getObject(pRepresentation, HashMap.class);
			
			if (hmpObject == null)
			{
				setStatus(Status.CLIENT_ERROR_BAD_REQUEST);
				return null;
			}
			
			IBean bean = liBeans.get(0);
			
			copyValues(storage, bean, hmpObject);
			
			storage.update(bean);
			
			return configure(new JacksonRepresentation(bean));
		}
		catch (Throwable th)
		{
			logger.debug(th);
			
			throw th;
		}
	}
	
	/**
	 * Deletes records from the storage. It is possible to delete more than one row if query parameters are used. If no
	 * record is found, the NOT_FOUND status is set. If PK is used and more than one records were found, the CONFLICT status 
	 * is set. 
	 * 
	 * @return the number of deleted records
	 * @throws Throwable if delete fails
	 */
	@Delete
	public Representation executeDelete() throws Throwable
	{
		try
		{
			DirectServerSession session = ((RESTSession)getRequest().getClientInfo().getUser()).getSession();
			
			ConcurrentMap<String, Object> cmpAttrib = getRequest().getAttributes();
	
			AbstractStorage storage = (AbstractStorage)session.get((String)cmpAttrib.get(RESTAdapter.PARAM_OBJECT_NAME));
			
			ICondition cond = getCondition(storage);
			
			List<IBean> liBeans = storage.fetchBean(cond, null, 0, 2);
			
			if (liBeans.size() == 0)
			{
				setStatus(Status.CLIENT_ERROR_NOT_FOUND);
				return null;
			}
			
			//PK -> exactly one record
			if (cmpAttrib.get(RESTAdapter.PARAM_PK) != null)
			{
				if (liBeans.size() > 1)
				{
					setStatus(Status.CLIENT_ERROR_CONFLICT);
					return null;
				}
			}
			
			logger.info("Number of records to delete: ", Integer.valueOf(liBeans.size()));

			if (!isDryRun())
			{
				for (IBean bean : liBeans)
				{
					storage.delete(bean);
				}
			}
			
			return configure(new JacksonRepresentation(Integer.valueOf(liBeans.size())));
		}
		catch (Throwable th)
		{
			logger.debug(th);
			
			throw th;
		}
	}
	
	/**
	 * Gets the metadata for the storage.
	 * 
	 * @return the metadata
	 * @throws Throwable if metadata detection fails
	 */
	@Options
	public Representation executeGetMetaData() throws Throwable
	{
		try
		{
			DirectServerSession session = ((RESTSession)getRequest().getClientInfo().getUser()).getSession();
			
			ConcurrentMap<String, Object> cmpAttrib = getRequest().getAttributes();
	
			AbstractStorage storage = (AbstractStorage)session.get((String)cmpAttrib.get(RESTAdapter.PARAM_OBJECT_NAME));
			
			return configure(new JacksonRepresentation(storage.getMetaData()));
		}
		catch (Throwable th)
		{
			logger.debug(th);
			
			throw th;
		}
	}
	
	/**
	 * Gets th condition based on the URL and query parameters.
	 * 
	 * @param pStorage the storage
	 * @return the condition or <code>null</code> if no condition is used
	 * @throws DataSourceException if metadata detection fails
	 */
	private ICondition getCondition(IStorage pStorage) throws DataSourceException
	{
		ICondition cond = null;
		
		MetaData mdat = pStorage.getMetaData();

		String[] sPKColumns = mdat.getPrimaryKeyColumnNames();
		
		Object oPK = getRequest().getAttributes().get(RESTAdapter.PARAM_PK); 

		
		//URL with a PK -> assume that the storage has exactly one PK column
		if (oPK != null)
		{
			if (sPKColumns == null || sPKColumns.length > 1)
			{
				setStatus(Status.CLIENT_ERROR_CONFLICT);
				return null;
			}
			
			cond = new Equals(sPKColumns[0], convertValue(mdat, sPKColumns[0], oPK));
		}
		
		//It is possible to add additional conditions via query parameter
		String[] sValidColumnNames = mdat.getColumnNames();
		
		String sColName;
		String sValue;
		
		Form query = getQuery();
		Parameter param;
		
		for (int i = 0, anz = query.size(); i < anz; i++)
		{
			param = query.get(i);

			sColName = param.getName().toUpperCase();
		
			sValue = param.getValue();
			
			//empty means null
			if (sValue != null && sValue.length() == 0)
			{
				sValue = null;
			}
			
			if (ArrayUtil.contains(sValidColumnNames, sColName))
			{
				if (cond == null)
				{
					cond = new Equals(sColName, convertValue(mdat, sColName, sValue));
				}
				else
				{
					cond = cond.and(new Equals(sColName, convertValue(mdat, sColName, sValue)));
				}
			}
		}
		
		return cond;
	}
	
	/**
	 * Gets whether the request is a dry-run test. That means, no data manipulation should be done.
	 * 
	 * @return <code>true</code> if the request is a dry-run request (Request-Header: X-DRYRUN is set to "true"),
	 *         <code>false</code> otherwise
	 */
	private boolean isDryRun()
	{
		Series series = (Series)getRequest().getAttributes().get(HeaderConstants.ATTRIBUTE_HEADERS);
		
		return Boolean.parseBoolean(series.getFirstValue("x-dryrun"));
	}

	/**
	 * Configures a Jackson representation.
	 * 
	 * @param pRepresentation the representation
	 * @return the configured representation
	 */
	private JacksonRepresentation configure(JacksonRepresentation pRepresentation)
	{
		ObjectMapper mapper = pRepresentation.getObjectMapper();
		
		mapper.configure(SerializationConfig.Feature.WRITE_DATES_AS_TIMESTAMPS, false);
		mapper.configure(DeserializationConfig.Feature.USE_BIG_DECIMAL_FOR_FLOATS, true);
		mapper.configure(DeserializationConfig.Feature.USE_BIG_INTEGER_FOR_INTS, true);
		
		return pRepresentation;
	}
	
	/**
	 * Copies the values from a request to a predefined bean object.
	 * 
	 * @param pStorage the storage to use
	 * @param pBean the predefined bean
	 * @param pData the request data
	 * @throws Exception if copy fails because of metadata access problems or type conversion errors
	 */
	private void copyValues(AbstractStorage pStorage, IBean pBean, HashMap<String, Object> pData) throws Exception
	{
		MetaData mdat = pStorage.getMetaData();
		
		String[] sValidColumnNames = mdat.getColumnNames();
		String[] sPKColumns = mdat.getPrimaryKeyColumnNames();
		
		String sColumnName;
		Object oValue;
		
		for (Entry<String, Object> entry : pData.entrySet())
		{
			sColumnName = entry.getKey().toUpperCase();
			
			//PK update is not allowed
			if (ArrayUtil.contains(sValidColumnNames, sColumnName) 
				&& (sPKColumns == null || !ArrayUtil.contains(sPKColumns, sColumnName)))
			{
				oValue = entry.getValue();
				
				if (mdat.getColumnMetaData(sColumnName).getTypeIdentifier() == TimestampDataType.TYPE_IDENTIFIER)
				{
					//try to convert
					oValue = DATE_FORMAT.parse(oValue.toString());
				}

				//We need BigDecimal
				if (oValue instanceof Number && !(oValue instanceof BigDecimal))
				{
					if (oValue instanceof BigInteger)
					{
						oValue = new BigDecimal((BigInteger)oValue); 
					}
					else
					{
						oValue = new BigDecimal(((Number)oValue).toString());
					}
				}
				
				pBean.put(sColumnName, oValue);
			}
		}
	}
	
	/**
	 * Converts a value to an object type of the column metadata type.
	 * 
	 * @param pMetaData the meta data
	 * @param pColumnName the column name
	 * @param pValue the input value
	 * @return the converted output value
	 */
	private Object convertValue(MetaData pMetaData, String pColumnName, Object pValue)
	{
		try
		{
			return ColumnMetaData.createDataType(pMetaData.getColumnMetaData(pColumnName)).convertToTypeClass(pValue);
		}
		catch (ModelException e)
		{
			throw new RuntimeException(e);
		}
	}
	
}	// AbstractStorageServerResource
