/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [JR] - creation
 * 05.10.2008 - [JR] - new password mechanism implemented
 * 27.10.2008 - [JR] - implemented changed IConnection methods:
 *                     open, openSub
 *                   - always send the properties when the connection
 *                     is not open
 * 28.10.2008 - [JR] - setProperty: send only changed properties immediately
 * 30.10.2008 - [JR] - setProperty: deny client properties only when connected   
 * 01.02.2009 - [JR] - callIntern: compression property checked for stream compression    
 * 04.02.2009 - [JR] - reopen implemented  
 * 04.04.2009 - [JR] - callIntern: used new ChangedHashtable features (get...(String.class))     
 * 05.04.2009 - [JR] - no WeakReference cache for CallBackListener's
 *                   - KeyValueList for Connection-Id/CallBack-Id's (removed when te connection was closed) 
 * 12.05.2009 - [JR] - call: added not open check        
 *                   - getInputStream/getOutputStream throws Throwable (more flexibility for sub classes) 
 * 13.05.2009 - [JR] - doCallBack: SwingUtilities.invokeLater replaced with a generic way
 * 12.08.2009 - [JR] - callIntern: throw CommunicationException when a SocketException occurs
 * 04.10.2009 - [JR] - setNewPassword: old password as parameter 
 * 21.12.2009 - [JR] - BROKEN: refactoring stack creation (ArrayUtil not necessary)
 * 20.01.2010 - [JR] - Properties constructor added (used from HttpConnection)
 *                   - callIntern: used BufferedInputStream for reading response
 * 23.02.2010 - [JR] - #18: callIntern: use IConnectionConstants.PROPERTY_CLASSES for property transfer     
 * 06.03.2010 - [JR] - #72: UniversalSerializer as default serializer    
 * 29.04.2010 - [JR] - #119: open(): serializer.init called   
 * 01.05.2010 - [JR] - #119: open(): removed reset call       
 * 18.05.2010 - [JR] - create an instance of Reflective to ensure the correct UI thread! 
 * 08.10.2010 - [JR] - #138: callIntern: catched Throwable instead of SocketException because NoRoutToHostException
 *                           is an IOException and not a SocketException
 * 12.10.2011 - [JR] - #482: count result objects and don't use the call count returned from the server,
 *                           because it contains callback results and properties 
 * 11.07.2013 - [JR] - #728: isCalling implemented      
 * 04.04.2014 - [RZ] - #997: implemented addPropertyChangedListener and removePropertyChangedListener                                                 
 */
package com.sibvisions.rad.remote;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Hashtable;
import java.util.List;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import javax.rad.remote.CommunicationException;
import javax.rad.remote.ConnectionInfo;
import javax.rad.remote.IConnection;
import javax.rad.remote.IConnectionConstants;
import javax.rad.remote.event.CallBackEvent;
import javax.rad.remote.event.ICallBackListener;
import javax.rad.remote.event.IConnectionPropertyChangedListener;
import javax.rad.remote.event.PropertyEvent;

import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.ChangedHashtable;
import com.sibvisions.util.KeyValueList;
import com.sibvisions.util.Reflective;
import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.LoggerFactory;
import com.sibvisions.util.type.CommonUtil;

/**
 * This is the default <code>IConnection</code> implementation. It's independent
 * of the communication protocol.
 * 
 * @author Ren Jahn
 */
public abstract class AbstractSerializedConnection implements IConnection
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the property name for the serializer. */
	public static final String PROP_SERIALIZER = "serializer";

	/** the reference to the correct UI thread. */
	private Reflective refUIThread = new Reflective();
	
	/** call-back key/listener mapping. */
	private Hashtable<Object, CallBackInfo> htCallBack = null;
	
	/** connection-id to call-back key mapping. */
	private KeyValueList<Object, Object> kvlConCallBack = null;
	
	/** synchronization object for the synchronous communication. */
	private Object oSync = new Object();
	
	/** the used serializer for the communication between client and server. */
	private ISerializer serializer = null;

    /** the collection of {@link IConnectionPropertyChangedListeners}. */
    private ArrayUtil<IConnectionPropertyChangedListener> auPropertyChangedListeners;

    /** sequence for callback id generation. */
	private long lSequence = 0;

	/** whether a call is active. */
	private boolean bCalling = false;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Creates a new instance of <code>AbstractSerializedConnection</code> with 
	 * properties containing relevant information. The supported property
	 * keys are:
	 * <ul>
	 *   <li>AbstractSerializedConnection.PROP_SERIALIZER</li>
	 * </ul>
	 * 
	 * @param pProperties the properties for the connection
	 * @throws ClassNotFoundException if the serializer is defined and could not be created                                  
	 */
	public AbstractSerializedConnection(Properties pProperties) throws ClassNotFoundException
	{
		this(createSerializer(pProperties.getProperty(PROP_SERIALIZER)));
	}

	/**
	 * Creates a new instance of <code>AbstractSerializedConnection</code>.
	 * 
	 * @param pSerializer the serializer for the communication between client and server or 
	 *                    null to use the default serializer
	 * @see ISerializer
	 * @see ByteSerializer
	 */
	public AbstractSerializedConnection(ISerializer pSerializer)
	{
		if (pSerializer == null)
		{
			this.serializer = new UniversalSerializer();
		}
		else
		{
			this.serializer = pSerializer;
		}
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	public void open(ConnectionInfo pConnectionInfo) throws Throwable
	{
		Object oConnectionId = callIntern(pConnectionInfo,
										  new String[] {IConnection.OBJ_SESSION}, 
										  new String[] {IConnection.MET_SESSION_CREATE}, 
										  null, 
										  null)[0];
		
		pConnectionInfo.setConnectionId(oConnectionId);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void openSub(ConnectionInfo pConnectionInfo, ConnectionInfo pConnectionInfoSub) throws Throwable
	{
		//Don't call through the existing session, because the properties will not
		//set into the sub session!
		Object oConnectionId = callIntern(pConnectionInfoSub,
										  new String[] {IConnection.OBJ_SESSION}, 
										  new String[] {IConnection.MET_SESSION_SUBSESSION_CREATE}, 
										  new Object[][] { {pConnectionInfo.getConnectionId()} }, 
										  null)[0];
		
		pConnectionInfoSub.setConnectionId(oConnectionId);
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isOpen(ConnectionInfo pConnectionInfo)
	{
		return pConnectionInfo != null && pConnectionInfo.getConnectionId() != null;
	}

	/**
	 * {@inheritDoc}
	 */
	public void close(ConnectionInfo pConnectionInfo) throws Throwable
	{
		if (isOpen(pConnectionInfo))
		{
			try
			{
				call(pConnectionInfo,
					 new String[] {IConnection.OBJ_SESSION}, 
					 new String[] {IConnection.MET_SESSION_DESTROY}, 
					 null, 
					 null);
			}
			finally
			{
				Object oConId = pConnectionInfo.getConnectionId();
				
				pConnectionInfo.setConnectionId(null);

				if (kvlConCallBack != null)
				{
					//remove the cached call-back information for the connection! 
					List<Object> liCallBackInfo = kvlConCallBack.remove(oConId);
					
					if (liCallBackInfo != null)
					{
						for (int i = 0, anz = liCallBackInfo.size(); i < anz; i++)
						{
							htCallBack.remove(liCallBackInfo.get(i));
						}
					}
				}
			}
		}
		else
		{
			throw new IllegalStateException("Connection not open");
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void reopen(ConnectionInfo pConnectionInfo) throws Throwable
	{
		try
		{
			close(pConnectionInfo);
		}
		catch (Throwable th)
		{
			//not important, because the connection will be opened!
		}
		
		open(pConnectionInfo);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public Object[] call(ConnectionInfo pConnectionInfo, 
						 String[] pObjectName, 
						 String[] pMethod, 
						 Object[][] pParams, 
						 ICallBackListener[] pCallBack) throws Throwable
	{
		if (pConnectionInfo == null)
		{
			throw new IllegalArgumentException("Invalid connection information: null");
		}

		if (pConnectionInfo.getConnectionId() == null)
		{
			throw new IllegalStateException("The connection is not open!");
		}
		
		return callIntern(pConnectionInfo, pObjectName, pMethod, pParams, pCallBack);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public ConnectionInfo[] setAndCheckAlive(ConnectionInfo pConnectionInfo, ConnectionInfo[] pSubConnections) throws Throwable
	{
		Hashtable<Object, ConnectionInfo> htMapping;
		
		Object[] oConIds;
		
		//cache the subconnections for analyzing the result object
		if (pSubConnections != null)
		{
			int anz = pSubConnections.length;
			
			htMapping = new Hashtable<Object, ConnectionInfo>(anz);
			
			oConIds = new Object[anz];
			
			for (int i = 0; i < anz; i++)
			{
				oConIds[i] = pSubConnections[i].getConnectionId();
				
				htMapping.put(oConIds[i], pSubConnections[i]);
			}
		}
		else
		{
			htMapping = null;
			oConIds   = null;
		}
		
		oConIds = (Object[])callIntern(pConnectionInfo,
									   new String[] {IConnection.OBJ_SESSION}, 
									   new String[] {IConnection.MET_SESSION_SETCHECKALIVE}, 
									   new Object[][] { oConIds }, 
									   null)[0];	    
		
		if (oConIds == null)
		{
			return null;
		}
		else
		{
			//use the untouched input-objects as result
			
			int anz = oConIds.length;
			
			ConnectionInfo[] ciResult = new ConnectionInfo[anz];
			
			for (int i = 0; i < anz; i++)
			{
				ciResult[i] = htMapping.get(oConIds[i]);
			}
			
			return ciResult;
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setProperty(ConnectionInfo pConnectionInfo, String pName, Object pValue) throws Throwable
	{
	    if (pName != null)
	    {
    		//client properties can only be changed when the connection is closed!
    		if (pName.startsWith(IConnectionConstants.PREFIX_CLIENT))
    		{
    			if (isOpen(pConnectionInfo))
    			{
    				throw new SecurityException("Client properties are not accessible after the connection was opened!");
    			}
    
    			Object oOldValue = pConnectionInfo.getProperties().put(pName, pValue);
    			
    			firePropertyChanged(pName, oOldValue, pValue);
    		}
    		else
    		{
    			Object oOldValue = pConnectionInfo.getProperties().put(pName, pValue);
    			
                firePropertyChanged(pName, oOldValue, pValue);
    
                //send only changed session parameters immediately to the server
    			if (!CommonUtil.equals(oOldValue, pValue))
    			{
    	            if (pName != null && pName.startsWith(IConnectionConstants.PREFIX_SERVER + IConnectionConstants.PREFIX_SESSION) && isOpen(pConnectionInfo))
    				{
    					callIntern(pConnectionInfo, 
    							   new String[] {}, 
    							   new String[] {}, 
    							   null, 
    							   null);
    				}
    			}
    		}
	    }
	    else
	    {
	        throw new IllegalArgumentException("Property name is undefined!");
	    }
	}
	
	/**
	 * {@inheritDoc}
	 */
	public Object getProperty(ConnectionInfo pConnectionInfo, String pName) throws Throwable
	{
		if (pConnectionInfo != null)
		{
			return pConnectionInfo.getProperties().get(pName);
		}
		
		return null;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public Hashtable<String, Object> getProperties(ConnectionInfo pConnectionInfo) throws Throwable
	{
		if (pConnectionInfo != null)
		{
			return (Hashtable<String, Object>)pConnectionInfo.getProperties().clone();
		}
		
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	public void addPropertyChangedListener(IConnectionPropertyChangedListener pListener)
	{
		if (auPropertyChangedListeners == null)
		{
			auPropertyChangedListeners = new ArrayUtil<IConnectionPropertyChangedListener>();
		}
		
		auPropertyChangedListeners.add(pListener);
	}

	/**
	 * {@inheritDoc}
	 */
	public void removePropertyChangedListener(IConnectionPropertyChangedListener pListener)
	{
		if (auPropertyChangedListeners != null)
		{
			auPropertyChangedListeners.remove(pListener);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setNewPassword(ConnectionInfo pConnectionInfo, String pOldPassword, String pNewPassword) throws Throwable
	{
		callIntern(pConnectionInfo,
				   new String[] {IConnection.OBJ_SESSION}, 
				   new String[] {IConnection.MET_SESSION_SET_NEW_PASSWORD}, 
				   new Object[][] { {pOldPassword, pNewPassword} }, 
				   null);
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Abstract methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Gets the <code>OutputStream</code> for submitting requests to the
	 * server.
	 * 
	 * @param pConnectionInfo the connection information
	 * @return output stream
	 * @throws Throwable if it's not possible to get the output stream
	 */
	public abstract OutputStream getOutputStream(ConnectionInfo pConnectionInfo) throws Throwable;

	/**
	 * Gets the <code>InputStream</code> for reading the response from the 
	 * server.
	 * 
	 * @param pConnectionInfo the connection information
	 * @return input stream
	 * @throws Throwable if it's not possible to get the the input stream
	 */
	public abstract InputStream getInputStream(ConnectionInfo pConnectionInfo) throws Throwable;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void finalize() throws Throwable
	{
		htCallBack = null;
		kvlConCallBack = null;
		
		super.finalize();
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Creates the serializer from a given class name.
	 * 
	 * @param pClassName the {@link ISerializer} implementation class
	 * @return the serializer implementation
	 * @throws ClassNotFoundException if the serializer could not be created
	 */
	protected static ISerializer createSerializer(String pClassName) throws ClassNotFoundException
	{
		if (pClassName != null)
		{
			ISerializer serializer = null;
			
			try
			{
				serializer = (ISerializer)Class.forName(pClassName).newInstance();
				
				return serializer;
			}
			catch (RuntimeException e)
			{
				throw e;
			}
			catch (ClassNotFoundException e)
			{
				throw e;
			}
			catch (Exception e)
			{
				throw new ClassNotFoundException(pClassName);
			}
		}
		else
		{
			return null;
		}
	}
	
	/**
	 * Calls desired methods from a remote server object.
	 *
	 * @param pConnectionInfo the connection information
	 * @param pObjectName list of server object names/aliases
	 * @param pMethod method names which should be called
	 * @param pParams parameters for the method calls
	 * @param pCallBack callback listeners for asynchronous or null for synchronous calls 
	 * @return result list from the remote method calls
	 * @throws Throwable communication error, security checks, invalid method, ...
	 */
	private Object[] callIntern(ConnectionInfo pConnectionInfo,
								String[] pObjectName, 
								String[] pMethod, 
								Object[][] pParams, 
								ICallBackListener[] pCallBack) throws Throwable
	{
	    synchronized (oSync)
	    {
			try
			{
				bCalling = true;

				//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
				// Validation
				//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		    	
		    	//It's allowed to let the object name empty -> action call
		    	//It's allowed to let the method name empty -> doesn't call anything but transfers callback results/properties
		    	
				if (pMethod == null)
				{
					throw new IllegalArgumentException("No remote method specified!");
				}
				
				if (pObjectName != null && pMethod.length != pObjectName.length)
				{
					throw new IllegalArgumentException("More or less objects than methods!");
				}
	
				if (pParams != null && pParams.length != pMethod.length)
				{
					throw new IllegalArgumentException("More or less params than methods!");
				}
				
				Object oConnectionId = pConnectionInfo.getConnectionId();
				
				//Without a connection id, the only allowed call is the session create/open call
				if (oConnectionId == null)
				{
					//this code can only be reached when callIntern was used wrong, because the empty
					//session id will be checked in callIntern
					if (pObjectName == null 
						|| !IConnection.OBJ_SESSION.equals(pObjectName[0]) 
						|| (!IConnection.MET_SESSION_CREATE.equals(pMethod[0]) 
							&& !IConnection.MET_SESSION_SUBSESSION_CREATE.equals(pMethod[0])))
					{
						throw new IOException("Connection is not open!");
					}
				}
		
				if (pCallBack != null && pCallBack.length != pMethod.length)
				{
					throw new IllegalArgumentException("More or less callbacks than methods!");
				}
				
				//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
				// setup call(s)
				//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
				
				Object[] oCallBack;
				
				int iCallCount = pMethod.length;
				
			    //Map callbacks for asynchronous execution
			    if (pCallBack != null)
			    {
			    	oCallBack = new Object[pCallBack.length];
			    	
			    	for (int i = 0, anz = pCallBack.length; i < anz; i++)
			    	{
			    		if (pCallBack[i] != null)
			    		{
			    			oCallBack[i] = createCallBackId();
			    			
			    			if (htCallBack == null)
			    			{
			    				htCallBack = new Hashtable<Object, CallBackInfo>();
			    				
			    				kvlConCallBack = new KeyValueList<Object, Object>();
			    			}
			    			
			    			//Map the callback information for the callback id because we need
			    			//the information for Callback listener notifications!
			    			htCallBack.put(oCallBack[i], 
			    						   new CallBackInfo(oConnectionId, 
			    								            pObjectName != null ? pObjectName[i] : null, 
			    							                pMethod[i], 
			    							                pCallBack[i]));
			    			
			    			//Cache the created callback ids. Otherwise it's not possible to remove the
			    			//callback identifiers when a connection will be closed!
			    			kvlConCallBack.put(oConnectionId, oCallBack[i]);
			    		}
			    	}
			    }
			    else
			    {
			    	oCallBack = null;
			    }
		
			    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
				// REQUEST
				//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		
			    ByteArrayOutputStream baosContent = new ByteArrayOutputStream(IConnection.COMPRESSION_BYTES);
			    DataOutputStream dosContent  = new DataOutputStream(baosContent);
			    
			    if (oConnectionId == null)
			    {
			    	//first request -> send the serializer classname!
				    dosContent.writeUTF(serializer.getClass().getName());
				    //no connection id (= null), causes NullPointerException. Use "" instead!
			    	dosContent.writeUTF("");
			    }
			    else
			    {
			    	dosContent.writeUTF((String)pConnectionInfo.getConnectionId());
			    }
			    
			    List<Entry<String, Object>> liProperties;
	
			    //When the connection is already open -> transfer only property changes. When
			    //establishing the connection, all current properties will be sent to the server!
			    //Important when open/close/open the connection!
			    if (isOpen(pConnectionInfo))
			    {
				    liProperties = pConnectionInfo.getProperties().getChanges(IConnectionConstants.PROPERTY_CLASSES);
			    }
			    else
			    {
				    liProperties = pConnectionInfo.getProperties().getMapping(IConnectionConstants.PROPERTY_CLASSES);
	
				    //with the next call -> only changes will be transfered!
			    	pConnectionInfo.getProperties().clearChanges();
			    }
			    
			    iCallCount += (liProperties != null ? 1 : 0);
			    
			    //Write call count
			    serializer.write(dosContent, Integer.valueOf(iCallCount));
			    
			    //Send properties before sending calls. Thats important that the server/session
			    //can use the properties before accessing the session!
			    if (liProperties != null)
			    {
			    	//use a transferable object-type!
			    	List<Object[]> liTransferProperties = new ArrayUtil<Object[]>(liProperties.size());
			    	
			    	for (Entry<String, Object> entry : liProperties)
			    	{
			    		liTransferProperties.add(new Object[] {entry.getKey(), entry.getValue()});
			    	}
			    	
			    	serializer.write(dosContent,
			    					 new Object[] {IConnection.OBJ_SESSION,
			    					  			   IConnection.MET_SESSION_SET_PROPERTY,
			    					  			   new Object[] {liTransferProperties},
			    					  			   null});
			    }
	
			    //Call(s)
			    for (int i = 0, anz = pMethod.length; i < anz; i++)
			    {
			    	serializer.write(dosContent, 
			    		             new Object[] {pObjectName != null ? pObjectName[i] : null, 
			    					               pMethod[i], 
			    					               pParams != null ? pParams[i] : null,
			    					               oCallBack != null ? oCallBack[i] : null});
			    }
			    
			    baosContent.close();
			    dosContent.close();
	
			    //Request-Header (same as in in Server.java)
			    //
			    //<STREAM-IDENTIFIER>       1Byte  (A = Acknowledge; E = Established) 
			    //<OPTION-FLAG-1>           1Byte  (0x01 = COMPRESSED; 0x02 = UNCOMPRESSED>
			    // - <SERIALIZER-CLASSNAME> xBytes (only with A, via DataOutputStream)
			    // - <SESSION-ID>           xBytes (via DataOutputStream)
			    // - <CALL-COUNT>           xBytes (via ISerializer)
			    // - <CALL-PARAMETER>       xBytes (via ISerializer)
			    //    - [Objectname, Method, Parameter, CallBack-ID]
			    
			    OutputStream osRequest;
			    
			    try
			    {
			    	osRequest = getOutputStream(pConnectionInfo);
			    }
			    catch (Throwable se)
			    {
			    	throw new CommunicationException(se);
			    }
			    
			    dosContent = new DataOutputStream(osRequest);
	
			    try
			    {
				    if (oConnectionId == null)
				    {
				    	osRequest.write(IConnection.FLAG_ACKNOWLEDGE);
				    }
				    else
				    {
				    	osRequest.write(IConnection.FLAG_ESTABLISHED);
				    }
			    
					boolean bCompressionSupported = Boolean.parseBoolean((String)pConnectionInfo.getProperties().get(IConnectionConstants.COMPRESSION));
		
				    //Use compression when the compression mode is enabled and the maximum number of uncompressed bytes are reached 
				    if (bCompressionSupported && baosContent.size() >= IConnection.COMPRESSION_BYTES)
				    {
					    osRequest.write(IConnection.MODE_COMPRESSED);
				    	
					    GZIPOutputStream zosContent = new GZIPOutputStream(osRequest);
					    zosContent.write(baosContent.toByteArray());
					    zosContent.flush();
					    zosContent.finish();
				    }
				    else
				    {
					    osRequest.write(IConnection.MODE_UNCOMPRESSED);
					    osRequest.write(baosContent.toByteArray());
				    }
				    
				    osRequest.flush();
			    }
			    finally
			    {
			    	dosContent.close();
			    }
	
			    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
				// RESPONSE
				//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
			    //store the reference to the original stream, for reading until EOF
			    InputStream isResponseOrig = getInputStream(pConnectionInfo);
			    InputStream isResponse = new BufferedInputStream(isResponseOrig);
	
			    //now it's ok to think that the transfer was successful
			    pConnectionInfo.setLastCallTime(System.currentTimeMillis());
			    
			    //Response-Header (same as in Server.java)
			    //
			    //<STREAM-IDENTIFIER>  1Byte  (E = Established; B = Broken) 
			    //<OPTION-FLAG-1>      1Byte  (0x01 = COMPRESSED; 0x02 = UNCOMPRESSED>
			    // - <RESULT-COUNT>    xBytes (only with E, via ISerializer)
			    // - <RESULT-TYPE>     1Byte  (only with E, via ISerializer)		    
			    // - <RESULT-OBJECT>   xBytes (only with E, via ISerializer)		    
			    // - <RESULT-TEXT>     xBytes (only with B, via DataInputStream)
			    
				//Get stream identifier
				char chStreamID = (char)isResponse.read();
		
				if (chStreamID != IConnection.FLAG_ESTABLISHED 
					&& chStreamID != IConnection.FLAG_BROKEN)
				{
					throw new IOException("Invalid stream identifier '" + chStreamID + "'");
				}
				
				int iMode = isResponse.read();
				
				//Handle compression
				if (iMode == IConnection.MODE_COMPRESSED)
				{
					isResponse = new GZIPInputStream(isResponse);
				}
			    
				DataInputStream disContent = new DataInputStream(isResponse);
	
				try
				{
					if (chStreamID == IConnection.FLAG_BROKEN)
					{
						//Broken streams -> deserialize the exception with DataInputStream!
						
						Throwable thResult = (Throwable)Class.forName(disContent.readUTF()).getConstructor(new Class[] {String.class}).newInstance(disContent.readUTF());
						
						//Get the number of StackTrace elements
						int iCount = disContent.read();
						
						if (iCount > 0)
						{
							//assemble the StackTrace
							StackTraceElement[] stack = new StackTraceElement[iCount];
							
							for (int i = 0; i < iCount; i++)
							{
								stack[i] = new StackTraceElement(disContent.readUTF(), disContent.readUTF(), disContent.readUTF(), disContent.readInt()); 
							}
							
							thResult.setStackTrace(stack);
						}
						
						throw thResult;
					}
					else
					{
						iCallCount = ((Integer)serializer.read(disContent)).intValue();
						
						Object   oCallBackId;
						Object   oReturn;
						
						ArrayUtil<Object> auResult = new ArrayUtil<Object>(iCallCount);
						
						byte byResultType;
						
						
						//Interpret call results
						for (int i = 0; i < iCallCount; i++)
						{
							oCallBackId = null;
							
							oReturn = serializer.read(disContent);
							
							//Check result types
							if (oReturn.getClass() == Byte.class)
							{
								byResultType = ((Byte)oReturn).byteValue();
				
								oReturn = serializer.read(disContent);
								
								if (byResultType == IConnection.TYPE_CALLBACK_ERROR || byResultType == IConnection.TYPE_CALLBACK_RESULT)
								{
									oCallBackId = serializer.read(disContent);
									
									//Exceptions from async calls -> forward via callback listener
									doCallBack(byResultType, oCallBackId, oReturn);
								}
								else if (byResultType == IConnection.TYPE_PROPERTY_RESULT)
								{
									ChangedHashtable<String, Object> chtProperties = pConnectionInfo.getProperties();
									
									if (chtProperties != null)
									{
										//Set the properties received from server as they are!
										ArrayUtil<Object[]> auProperties = (ArrayUtil<Object[]>)oReturn;
									
										Object[] oProps;
										
										for (int k = 0, anz = auProperties.size(); k < anz; k++)
										{
											oProps = (Object[])auProperties.get(k); 
											
											Object oOldValue = chtProperties.put((String)oProps[0], (Object)oProps[1], false);

											firePropertyChanged((String)oProps[0], oOldValue, (Object)oProps[1]);
										}
									}
								}
								else if (byResultType == IConnection.TYPE_CALL_ERROR || byResultType == IConnection.TYPE_CALL_RESULT)
								{
									//Exceptions from sync calls -> throw immediate
									if (byResultType == IConnection.TYPE_CALL_ERROR)
									{
										throw (Throwable)oReturn;
									}
									
									auResult.add(oReturn);
								}
								else
								{
									throw new IOException("Invalid return type");
								}
							}
							else
							{
								throw new IOException("Invalid response type");
							}
						}
						
						return auResult.toArray(new Object[auResult.size()]);
					}
				}
				finally
				{
					//"clear" the rest of the original stream (important for android http implementation)
					byte[] byNirvana = new byte[4096];
					
					while (isResponseOrig.read(byNirvana) >= 0)
					{
						//do nothing
					}
	
					isResponseOrig.close();
					
					if (isResponse != isResponseOrig)
					{
						isResponse.close();
					}
					
					disContent.close();
				}
			}
			finally
			{
				bCalling = false;
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isCalling()
	{
		return bCalling;
	}		
	
    /**
     * Create a new id for a callback interface.
     * 
     * @return unique id
     */
    private synchronized Object createCallBackId()
    {
    	//ascending number + SC (= Serialized Connection -> derived from the classname)
    	return Long.toHexString(++lSequence) + "SC";
    }
    
    /**
     * Notifies an <code>ICallBackListener</code> with the result of an asynchronous
     * method call.
     * 
     * @param pResultType {@link IConnection#TYPE_CALLBACK_RESULT} or {@link IConnection#TYPE_CALLBACK_ERROR}
     * @param pCallBackId identifier for the callback mapping
     * @param pResult result of the remote method call
     */
    private void doCallBack(byte pResultType, Object pCallBackId, Object pResult)
    {
    	CallBackInfo cbiInfo = htCallBack.remove(pCallBackId);

    	
    	if (cbiInfo != null)
    	{
    		//cleanup
    		kvlConCallBack.remove(cbiInfo.getConnectionId(), pCallBackId);
    		
    		if (htCallBack.size() == 0)
    		{
    			htCallBack = null;
    			kvlConCallBack = null;
    		}
    		
    		//Redundant check, because the check will be performed in the Thread, but with this
    		//call we avoid an unnecessary Thread-start!
    		if (cbiInfo.getCallBackListener() != null)
    		{
        		refUIThread.invokeLater(new CallBackWorker(cbiInfo, pResultType, pResult));
    		}    			
    	}
    	else
    	{
    		throw new IllegalArgumentException("CallBack ID not found: '" + pCallBackId + "'");
    	}
    }
    
	/**
	 * Fires the property changed event on all registered {@link IConnectionPropertyChangedListener}s
	 * if the parameters <code>pOldValue</code> and <code>pNewValue</code> are
	 * not the same and both are not null.
	 * 
	 * @param pName the name of the property
	 * @param pOldValue the old value of the property
	 * @param pNewValue the new value of the property
	 */
	private void firePropertyChanged(String pName, Object pOldValue, Object pNewValue)
	{
		if (auPropertyChangedListeners != null)
		{
			if (!CommonUtil.equals(pOldValue, pNewValue))
			{
				PropertyEvent event = new PropertyEvent(pName, pOldValue, pNewValue);
				
				for (IConnectionPropertyChangedListener listener : auPropertyChangedListeners)
				{
					listener.propertyChanged(event);
				}
			}
		}
	}
    
	//****************************************************************
	// Subclass definition
	//****************************************************************

    /**
     * The <code>CallBackInfo</code> is a POJO that holds information about asynchronous
     * remote method calls.
     * 
     * @author Ren Jahn
     */
    private static final class CallBackInfo
    {
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// Class members
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	
    	/** the connection id associated with this call-back info. */
    	private Object oConnectionId;
    	
    	/** callback listener which should get the result of a remote method call. */
    	private ICallBackListener cblListener;
    	
    	/** object name of the remote method call. */
    	private String sObjectName;
    	
    	/** method name of the remote call. */
    	private String sMethodName;

    	/** create time, in millis. */
    	private long lCreated = System.currentTimeMillis();
    	
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// Initialization
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	
    	/**
    	 * Creates a new instance of <code>CallBackInfo</code> which holds the information for
    	 * an asynchronous remote method call.
    	 * 
    	 * @param pConnectionId the connection id associated with this call-back information
    	 * @param pObjectName object name for the remote method call
    	 * @param pMethodName method name for the remote call
    	 * @param pListener callback listener which should get the result of a remote method call
    	 */
    	private CallBackInfo(Object pConnectionId, String pObjectName, String pMethodName, ICallBackListener pListener)
    	{
    		oConnectionId = pConnectionId;
    		sObjectName  = pObjectName;
    		sMethodName  = pMethodName;
    		cblListener = pListener;
    	}
    	
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// User-defined methods
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	
    	/**
    	 * Returns the connection id associated with this information object.
    	 * 
    	 * @return the connection id
    	 */
    	private Object getConnectionId()
    	{
    		return oConnectionId;
    	}
    	
    	/**
    	 * Returns the object name of the remote method call.
    	 * 
    	 * @return object name
    	 */
    	private String getObjectName()
    	{
    		return sObjectName;
    	}
    	
    	/**
    	 * Returns the method name of the remote call.
    	 * 
    	 * @return method name
    	 */
    	private String getMethodName()
    	{
    		return sMethodName;
    	}
    	
    	/**
    	 * Gets the listener which should get the result of a remote method call.
    	 * 
    	 * @return callback listener
    	 */
    	private ICallBackListener getCallBackListener()
    	{
    		return cblListener;
    	}
    	
    	/**
    	 * Gets the creation time of the object.
    	 * 
    	 * @return create time in millis
    	 */
    	private long getCreateTime()
    	{
    		return lCreated;
    	}
    	
    }	// CallBackInfo
    
    /**
     * The <code>CallBackWorker</code> notifies an <code>ICallBackListener</code> with the
     * result of a remote method call from the client to the server.
     * 
     * @author Ren Jahn
     */
    private static final class CallBackWorker implements Runnable
    {
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// Class members
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    	/** callback information to do the callback. */
    	private CallBackInfo cbiInfo;
    	
    	/** result type can be {@link IConnection#TYPE_CALLBACK_ERROR} or {@link IConnection#TYPE_CALLBACK_RESULT}. */
    	private byte byType;
    	
    	/** result of an asynchronous method call. */
    	private Object oResult;
    	
    	/** the logger. */
    	private ILogger log;
    	
    	/** creation time of this object. */
    	private long lCreated = System.currentTimeMillis();
    	
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// Initialization
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    	/**
    	 * Creates a new instance of <code>CallBackWorker</code>.
    	 * 
    	 * @param pInfo <code>CallBackInfo</code> with all important callback information
    	 * @param pResultType one of the following values {@link IConnection#TYPE_CALLBACK_ERROR}, {@link IConnection#TYPE_CALLBACK_RESULT} 
    	 * @param pResult result of a remote method call
    	 */
    	private CallBackWorker(CallBackInfo pInfo, byte pResultType, Object pResult)
    	{
    		this.cbiInfo = pInfo;
    		this.byType = pResultType;
    		this.oResult = pResult;
    	}
    	
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// Interface implementation
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	
    	/**
    	 * {@inheritDoc}
    	 */
    	public void run()
    	{
    		ICallBackListener cblistener = cbiInfo.getCallBackListener();
    		
    		
    		if (cblistener != null)
    		{
	    		try
	    		{
	        		Object object = null;
	        		Throwable throwable = null;

	        		switch (byType)
	    			{
	    				case IConnection.TYPE_CALLBACK_ERROR:
	    					throwable = (Throwable)oResult;
	    					break;
	    				case IConnection.TYPE_CALLBACK_RESULT:
	    				default:
	    					object = oResult;
							break;
	    			}
	    			
	    			cblistener.callBack
	    			(
	    				new CallBackEvent
	    				(
	    					cbiInfo.getObjectName(), 
	    					cbiInfo.getMethodName(), 
	    					object,
	    					throwable,
	    					cbiInfo.getCreateTime(),
	    					lCreated
	    				)
	    			);
	    		}
	    		catch (Throwable th)
	    		{
	    			error(th);
	    		}
    		}
    	}
    	
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// User-defined methods
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    	/**
    	 * Logs an error.
    	 * 
    	 * @param pInfo the log information
    	 */
    	private void error(Object... pInfo)
    	{
    		if (log == null)
    		{
    			log = LoggerFactory.getInstance(getClass()); 
    		}
    		
    		log.error(pInfo);		
    	}
    	
    }	// CallBackWorker
	
}	// AbstractSerializedConnection
