/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 10.02.2009 - [JR] - creation
 * 12.02.2009 - [JR] - get: update time (cache object longer)
 * 16.02.2009 - [JR] - get: removed update time (cache objects for the configured timeout and not longer!)
 * 05.04.2009 - [JR] - put: key parameter
 *                   - Object key instead of String key
 * 14.06.2009 - [JR] - get: null key check [BUGFIX]    
 * 02.01.2010 - [JR] - put: null object removes the key    
 * 17.09.2013 - [JR] - removed Memory.gc           
 */
package com.sibvisions.util;

import java.util.Hashtable;
import java.util.Map;
import java.util.UUID;

/**
 * The <code>ObjectCache</code> is a utility class to cache/store objects for a
 * period of time. The cache handles the object expiration and the access to the
 * cached objects. The cache stores an object with a unique access key. With this
 * access key it's possible to access the object from the store.
 * 
 * @author Ren Jahn
 */
public class ObjectCache
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the current timeout-check interval. */
	private static final long CHECK_DELAY = 30000L;
	
	/** the object cache. */
	private static Hashtable<Object, Element> htStore = new Hashtable<Object, Element>();
	
	/** the timeout-check thread. */
	private static Thread thCheckTimeout = null;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Invisible constructor, because the <code>ObjectCache</code> is a utility class.
	 */
	protected ObjectCache()
	{
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a random acces key.
	 * 
	 * @return the access key                 
	 */
	public static Object createKey()
	{
		return UUID.randomUUID().toString();
	}
	
	/**
	 * Puts an object to the cache, with a timeout for expiration.
	 * 
	 * @param pObject the cachable object
	 * @param pTimeout the timeout (in millis) for the object. After this time, the
	 *                 object will be removed from the cache
	 * @return the access key for the object                 
	 */
	public static Object put(Object pObject, long pTimeout)
	{
		if (pObject == null)
		{
			throw new NullPointerException("Object is null!");
		}
		
		Object sKey = createKey();
		
		put(sKey, pObject, pTimeout);
		
		return sKey;
	}
	
	/**
	 * Puts an object to the cache, with a specific key and timeout for expiration.
	 * 
	 * @param pKey the key for accessing the object
	 * @param pObject the cachable object
	 * @param pTimeout the timeout (in millis) for the object. After this time, the
	 *                 object will be removed from the cache
	 * @return the previous cached object or <code>null</code> if there was no previous object
	 *         or the object is expired               
	 */
	public static Object put(Object pKey, Object pObject, long pTimeout)
	{
		if (pObject == null)
		{
			return htStore.remove(pKey);
		}
		else
		{
			Element elOld = htStore.put(pKey, new Element(pObject, pTimeout));
			
			startTimeoutCheck();
			
			if (elOld != null && elOld.isValid())
			{
				return elOld.object;
			}
			else
			{
				return null;
			}
		}
	}

	/**
	 * Gets an object from the cache.
	 *  
	 * @param pKey the access key of the object
	 * @return the object or <code>null</code> if the object was expired or
	 *         the key was not found
	 */
	public static Object get(Object pKey)
	{
		if (pKey == null)
		{
			return null;
		}
		
		Element element = htStore.get(pKey);
		
		if (element == null)
		{
			return null;
		}
		else 
		{
			if (element.isValid())
			{
				return element.object;
			}
			else
			{
				htStore.remove(pKey);
				
				return null;
			}
		}
	}
	
	/**
	 * Removes an object from the cache.
	 * 
	 * @param pKey the access key of the object
	 * @return the removed object or <code>null</code> if the object was expired or
	 *         the key was not found
	 */
	public static Object remove(Object pKey)
	{
		Element element = htStore.remove(pKey);
		
		if (element != null && element.isValid())
		{
			return element;
		}
		
		return null;
	}
	
	/**
	 * Starts the timeout check.
	 */
	private static void startTimeoutCheck()
	{
		if (ThreadHandler.isStopped(thCheckTimeout))
		{
			thCheckTimeout = ThreadHandler.start(new Check());
		}
	}
	
	//****************************************************************
	// Subclass definition
	//****************************************************************

	/**
	 * The <code>Element</code> encapsulates an exchange object. It includes the
	 * last access time and a timeout. After the timeout, the object is invalid 
	 * and can not be used.
	 * 
	 * @author Ren Jahn
	 */
	private static final class Element
	{
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Class members
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/** the creation time. Needed to calculate the expiration. */
		private long creation = System.currentTimeMillis();
		
		/** element timeout. Needed to calculate the expiration. */
		private long timeout;

		/** the cached object. */
		private Object object;
		
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Initialization
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
		 * Creates a new instance of <code>Element</code> with an object
		 * and the desired timeout.
		 * 
		 * @param pObject the cached object
		 * @param pTimeout the timeout (in millis)
		 */
		private Element(Object pObject, long pTimeout)
		{
			object = pObject;
			timeout = pTimeout;
		}
		
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// User-defined methods
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
		 * Determines if the element and the containing object is valid or expired.
		 * The element is expired if the timeout has reached.
		 * 
		 * @return <code>true</code> if the timeout has not reached or the timeout is endless (<= 0);
		 *         <code>false</code> if the timeout has reached.  
		 *         
		 */
		private boolean isValid()
		{
			return timeout <= 0 || creation + timeout >= System.currentTimeMillis();
		}
		
	}	// Element
	
	/**
	 * The <code>Check</code> class handles the timeout check of cached elements in
	 * the store. If an element is timed out, it will be removed from the store.
	 * 
	 * @author Ren Jahn
	 */
	private static class Check implements Runnable
	{
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Interface implementation
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
		 * Checks the valid state of {@link Element}s.
		 */
		public void run()
		{
			try
			{
				while (!ThreadHandler.isStopped(thCheckTimeout))
				{
					Thread.sleep(CHECK_DELAY);
					
					Hashtable<Object, Element> htClone = new Hashtable<Object, Element>(htStore);

					//remove invalid elements from the store
					for (Map.Entry<Object, Element> entry : htClone.entrySet())
					{
						if (!entry.getValue().isValid())
						{
							htStore.remove(entry.getKey());
						}
					}
					htClone = null;
				}
			}
			catch (InterruptedException ie)
			{
				//not a problem, because the thread will be started with the next put action
			}
		}
		
	}	// Check
	
}	// ObjectCache
