/*
 * Copyright 2014 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 28.01.2014 - [JR] - creation
 */
package com.sibvisions.rad.server.security;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;

import javax.rad.server.IConfiguration;
import javax.rad.server.ISession;

import com.sibvisions.rad.persist.jdbc.DBAccess;
import com.sibvisions.rad.persist.jdbc.DBCredentials;
import com.sibvisions.rad.server.config.Configuration;
import com.sibvisions.rad.server.config.Zone;
import com.sibvisions.rad.server.security.DBSecurityManager.DBAccessController;
import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.log.LoggerFactory;
import com.sibvisions.util.xml.XmlNode;

/**
 * The <code>AbstractDBSecurityManager</code> is the base class for all security managers that use a database for
 * authentication.
 * 
 * @author Ren Jahn
 */
public abstract class AbstractDBSecurityManager extends AbstractSecurityManager
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the database credentials. */
	private DBCredentials credentials = null;
	
	/** database connection. */
	private Connection con = null;

	/** the last modified date of the configuration. */
	private long lConfigModified = -1;
	
	/** the list of registered statements. */
	private List<Statement> liStatements = null;
	
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Abstract methods implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Updates relevant information after configuration was changed.
	 * 
	 * @param pConfig the session configuration
	 * @throws Exception if an exception occurs during statement creation
	 */
	protected abstract void updateConfiguration(IConfiguration pConfig) throws Exception;
	
	/**
	 * Initializes all statements after opening a database connection.
	 * 
	 * @param pConnection the connection to use
	 * @throws Exception if an exception occurs during statement creation
	 */
	protected abstract void initStatements(Connection pConnection) throws Exception;

	/**
	 * Gets the query which should be use for connection check. A simple query like
	 * <code>select 1 from dual</code> is enough.
	 * 
	 * @return the alive check query
	 */
	protected abstract String getAliveQuery();
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void release()
	{
		try
		{
			closeConnection();
		}
		catch (Exception e)
		{
			error(e);
		}
	}	
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void finalize() throws Throwable 
	{
		//save a call and don't invoke release
		if (con != null)
		{
			try
			{
				con.close();
			}
			catch (Throwable th)
			{
				//ignore
			}
		}

		super.finalize();
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Opens a database connection to the database of an application.
	 * 
	 * @param pSession the session for which the connection should be opened
	 * @return a new or reused connection to the database
	 * @throws Exception if the application zone is invalid or the connection can not be opened 
	 * @throws IllegalArgumentException if the database configuration is invalid (parameters are missing, ...) 
	 */
	protected Connection openConnection(ISession pSession) throws Exception
	{
		String sApplication = pSession.getApplicationName();
		
		DBCredentials dbcred = getCredentials(pSession);
		
		
		if (dbcred == null)
		{
			throw new IllegalArgumentException("Database credentials were not found!"); 
		}
		
		//Validation

		if (dbcred.getDriver() == null)
		{
			DBAccess dba = DBAccess.getDBAccess(dbcred.getUrl());
			
			if (dba != null)
			{
				dbcred = new DBCredentials(dba.getDriver(), dbcred.getUrl(), dbcred.getUserName(), dbcred.getPassword());
				
				if (dbcred.getDriver() == null)
				{
					throw new IllegalArgumentException("Parameter 'driver' is missing for application '" + sApplication + "'");
				}
			}
			else
			{
				throw new IllegalArgumentException("Parameter 'url' is missing for application '" + sApplication + "'");
			}
		}

		if (dbcred.getUrl() == null)
		{
			throw new IllegalArgumentException("Parameter 'url' is missing for application '" + sApplication + "'");
		}

		if (dbcred.getUserName() == null)
		{
			throw new IllegalArgumentException("Parameter 'username' is missing for application '" + sApplication + "'");
		}

		if (dbcred.getPassword() == null)
		{
			throw new IllegalArgumentException("Parameter 'password' is missing for application '" + sApplication + "'");
		}

		//Connection check
		boolean bOpenCon = false;
		
		long lModified;
		
		IConfiguration cfgSession = pSession.getConfig();
		
		if (cfgSession instanceof Zone)
		{
			lModified = ((Zone)cfgSession).getFile().lastModified();
		}
		else
		{
			//maybe a special IConfiguration implementation (paranoid mode)
			lModified = Configuration.getApplicationZone(sApplication).getFile().lastModified();
		}
		
		if (con == null || !credentials.equals(dbcred) || lConfigModified != lModified)
		{
			//if settings were changed, the connection need to be re-established
			bOpenCon = true;
		}
		else
		{
			bOpenCon = !isConnectionAlive();
		}
		
		if (bOpenCon)
		{
			closeConnection();
			
			try
			{
				Class.forName(dbcred.getDriver());
			}
			catch (ClassNotFoundException e)
			{
				throw new ClassNotFoundException("JDBC driver '" + dbcred.getDriver() + "' for application '" + sApplication + "' was not found!", e);
			}
			
			try
			{				
			    //Use DBAccess to use JVx default connection settings
				DBAccess dba = DBAccess.getDBAccess(dbcred);
				dba.open();
				
				con = dba.getConnection();
				con.setAutoCommit(false);

				updateConfiguration(cfgSession);
				
				initStatements(con);
			}
			catch (SQLException sqle)
			{
				closeConnection();
				
				throw new Exception("Can not open database connection with '" + dbcred.getUrl() + "' for application '" + sApplication + "'", sqle);
			}

			//Cache values for later validation
			credentials = dbcred;
			
			lConfigModified = lModified;
		}
		
		return con;
	}
	
	/**
	 * Close all statements and the connection.
	 * 
	 * @throws Exception if one statement can not be closed
	 */
	protected void closeConnection() throws Exception
	{
		if (con != null)
		{
			if (liStatements != null)
			{
				for (Statement stmt : liStatements)
				{
					try
					{
						stmt.close();
					}
					catch (Exception e)
					{
						debug(e);
					}
				}
			}
			
			try
			{
				con.close();
			}
			catch (Throwable th)
			{
				debug(th);
			}
			finally
			{
				con = null;
			}
		}
	}
	
	/**
	 * Gets the configured database credentials for the given session.
	 * 
	 * @param pSession the session 
	 * @return the configured credentials
	 * @see #getCredentials(IConfiguration)
	 */
	protected DBCredentials getCredentials(ISession pSession)
	{
		return getCredentials(pSession.getConfig());
	}
	
	/**
	 * Gets the configured database credentials from a given configuration. This method handles
	 * credentials, set in the security manager and credentials configured as datasource.
	 * 
	 * @param pConfig the configuration 
	 * @return the configured credentials
	 */
	public static DBCredentials getCredentials(IConfiguration pConfig)
	{
		try
		{
			XmlNode xmnDb = pConfig.getNode("/application/securitymanager/database");
			
			if (xmnDb == null)
			{
				return DataSourceHandler.createDBCredentials(pConfig, "default");
			}
			else
			{
				XmlNode xmnDs = xmnDb.getNode("/datasource"); 
			
				if (xmnDs == null)
				{
					return DataSourceHandler.createDBCredentials(xmnDb);
				}
				else
				{
					return DataSourceHandler.createDBCredentials(pConfig, xmnDs.getValue());
				}
			}
		}
		catch (Exception e)
		{
			LoggerFactory.getInstance(AbstractDBSecurityManager.class).error(e);
		}
		
		return null;
	}
	
	/**
	 * Gets the current connection to the database. The connection is validated to ensure that it is usable.
	 * 
	 * @return the connection for the security manager or <code>null</code> if the security manager did
	 *         not open a connection
	 * @throws Exception if db access fails
	 */
	public Connection getConnection() throws Exception
	{
		if (con != null)
		{
			if (!isConnectionAlive())
			{
				//re-open the connection with cached credentials 
				try
				{			
					DBAccess dba = DBAccess.getDBAccess(credentials);

					con = dba.getConnection();
					con.setAutoCommit(false);
					
					initStatements(con);
				}
				catch (SQLException sqle)
				{
					closeConnection();
					
					throw new Exception("Can not open database connection with '" + credentials.getUrl() + "'", sqle);
				}
			}
		}
		
		return con;
	}
	
	/**
	 * Gets the connection to the database. The connection is validated to ensure that it is usable.
	 * 
	 * @param pSession the session that wants access to the database
	 * @return the connection
	 * @throws Exception if db access fails
	 */
	public Connection getConnection(ISession pSession) throws Exception
	{
		return openConnection(pSession);
	}	

	/**
	 * Creates an access controller for a {@link ISession}.
	 * 
	 * @param pSession the session which requests the access controller
	 * @return the access controller
	 */
	protected IAccessController createAccessController(ISession pSession)
	{
		String sAccCtrl = pSession.getConfig().getProperty("/application/securitymanager/accesscontroller");
		
		if (sAccCtrl != null && sAccCtrl.trim().length() > 0)
		{
			try
			{
				Class<?> clazz = Class.forName(sAccCtrl);
				
				return (IAccessController)clazz.newInstance();
			}
			catch (ClassNotFoundException cnfe)
			{
				throw new SecurityException("Access controller '" + sAccCtrl + "' was not found!");
			}
			catch (InstantiationException ie)
			{
				throw new SecurityException("Can't instantiate access controller '" + sAccCtrl + "'!");
			}
			catch (IllegalAccessException iae)
			{
				throw new SecurityException("Access controller '" + sAccCtrl + "' not accessible!");
			}
		}
		
		return new DBAccessController();
	}
	
	/**
	 * Commits all changes.
	 * 
	 * @throws SQLException if commit fails
	 */
	protected void commit() throws SQLException
	{
		con.commit();
	}
	
	/**
	 * Reverts all changes.
	 */
	protected void rollback()
	{
		try
		{
			con.rollback();
		}
		catch (Throwable th)
		{
			error(th);
		}
	}

	/**
	 * Checks whether the connection is still alive, means whether the connection can be used.
	 * 
	 * @return <code>true</code> if the connection is alive/valid
	 * @see #getAliveQuery()
	 */
	protected boolean isConnectionAlive()
	{
		String sQuery = getAliveQuery();
		
		//No query means no alive check and we assume that the connection is still alive
		if (sQuery == null)
		{
			return true;
		}
		
		Statement stmt = null;

		ResultSet res = null;
		
		try
		{
			stmt = con.createStatement();
			
			res = stmt.executeQuery(sQuery);
			res.next();
			
			return true;
		}
		catch (Throwable th)
		{
			return false;
		}
		finally
		{
			if (res != null)
			{
				try
				{
					res.close();
				}
				catch (Throwable th)
				{
					//nothing to be done
				}
			}
			
			stmt = close(stmt);
		}
	}
	
	/**
	 * Registers a statment as closable statement. All closable statements will be closed
	 * if connection will be closed.
	 * 
	 * @param pStatement the statement
	 */
	protected void register(Statement pStatement)
	{
		if (liStatements == null)
		{
			liStatements = new ArrayUtil<Statement>();
		}
		
		if (!liStatements.contains(pStatement))
		{
			liStatements.add(pStatement);
		}
	}
	
	/**
	 * Unregisters a statement.
	 * 
	 * @param pStatement the statement
	 * @return <code>true</code> if unregistration was successful, <code>false</code> if statement was not registered
	 *         as closable
	 * @see #register(Statement)         
	 */
	protected boolean unregister(Statement pStatement)
	{
		if (liStatements == null)
		{
			return false;
		}
		
		return liStatements.remove(pStatement);
	}
	
	/**
	 * Creates a new instance of {@link PreparedStatement}. The statement will be registered for automatic close.
	 * 
	 * @param pConnection the database connection
	 * @param pSql the SQL statement, e.g. a query
	 * @return the new statement
	 * @throws SQLException if statement creation fails
	 */
	protected PreparedStatement prepareStatement(Connection pConnection, String pSql) throws SQLException
	{
		PreparedStatement stmt = pConnection.prepareStatement(pSql);
		
		register(stmt);
		
		return stmt;
	}
	
    /**
     * Creates a new instance of {@link CallableStatement}. The statement will be registered for automatic close.
     * 
     * @param pConnection the database connection
     * @param pSql the call statement
     * @return the new statement
     * @throws SQLException if statement creation fails
     */
	protected CallableStatement prepareCall(Connection pConnection, String pSql) throws SQLException
	{
	    CallableStatement stmt = pConnection.prepareCall(pSql);
	    
	    register(stmt);
	    
	    return stmt;
	}
	
	/**
	 * Closes a statement.
	 * 
	 * @param pStatement the statement
	 * @param <T> the statement type
	 * @return <code>null</code>
	 */
	protected <T extends Statement> T close(T pStatement)
	{
		if (pStatement != null)
		{
			try
			{
				pStatement.close();
			}
			catch (Exception e)
			{
				debug(e);
			}
		}
		
		return null;
	}
	
	/**
	 * Closes a {@link ResultSet}.
	 * 
	 * @param pResultSet the ResultSet
	 */
	protected void close(ResultSet pResultSet)
	{
        try
        {
            pResultSet.close();
        }
        catch (Exception e)
        {
            debug(e);
        }
	}
	
}	// AbstractDBSecurityManager
