/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 12.12.2011 - [JR] - creation
 */
package com.sibvisions.rad.server.http.rest;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintStream;

import org.codehaus.jackson.map.DeserializationConfig;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.map.SerializationConfig;
import org.restlet.representation.Representation;

/**
 * The <code>JSONUtil</code> is an internal utility class for REST handling.
 * 
 * @author Ren Jahn
 */
public final class JSONUtil
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the object mapper. */
	private static ObjectMapper mapper = createObjectMapper();
	
	/** the dump stream (default: System.out). */
	private static PrintStream stream = System.out;
	
	/** whether the JSON stream should be dumped. */
	private static boolean bDump = false;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Invisible constructor because <code>BeanUtil</code> is a utility
	 * class.
	 */
	private JSONUtil()
	{
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Creates a new {@link ObjectMapper} with preconfigured settings.
	 * 
	 * @return the object mapper
	 */
	public static ObjectMapper createObjectMapper()
	{
		ObjectMapper omap = new ObjectMapper();
		omap.configure(SerializationConfig.Feature.WRITE_DATES_AS_TIMESTAMPS, false);
		omap.configure(DeserializationConfig.Feature.USE_BIG_DECIMAL_FOR_FLOATS, true);
		omap.configure(DeserializationConfig.Feature.USE_BIG_INTEGER_FOR_INTS, true);
		omap.configure(DeserializationConfig.Feature.FAIL_ON_UNKNOWN_PROPERTIES, false);

		return omap;
	}
	
	/**
	 * Gets the result object from a representation.
	 * 
	 * @param pRepresentation the object representation
	 * @return the object from the representation
	 * @throws IOException if object creation fails
	 */
	public static Object getObject(Representation pRepresentation) throws IOException
	{
		return getObject(pRepresentation, Object.class);
	}
	
	/**
	 * Gets the result object from a representation.
	 * 
	 * @param <T> the result object type
	 * @param pRepresentation the object representation
	 * @param pClass the expected class type
	 * @return the object from the representation
	 * @throws IOException if object creation fails
	 */
	public static <T> T getObject(Representation pRepresentation, Class pClass) throws IOException
	{
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		pRepresentation.write(baos);

		byte[] byData = baos.toByteArray();
		
		if (byData.length > 0)
		{
			T result = (T)mapper.readValue(byData, pClass);
			
			if (bDump)
			{
				stream.println(mapper.writer().withDefaultPrettyPrinter().writeValueAsString(result));
			}
			
			return result;
		}
		else
		{
			return null;
		}
	}
	
	/**
	 * Sets whether the JSON stream should be dumped.
	 * 
	 * @param pDump <code>true</code> to enable dumps, <code>false</code> otherwise
	 */
	public static void setDumpStreamEnabled(boolean pDump)
	{
		bDump = pDump;
	}
	
	/**
	 * Gets whether JSTOM stream dumping is enabled.
	 * 
	 * @return <code>true</code> if stream dumping is enabled, <code>false</code> otherwise
	 */
	public static boolean isDumpStreamEnabled()
	{
		return bDump;
	}

	/**
	 * Sets the dump stream.
	 * 
	 * @param pStream the stream or <code>null</code> to reset the stream to its default value (System.out)
	 */
	public static void setDumpStream(PrintStream pStream)
	{
		if (pStream != null)
		{
			stream = pStream;
		}
		else
		{
			stream = System.out;
		}
	}
	
	/**
	 * Gets the dump stream.
	 * 
	 * @return the stream
	 */
	public static PrintStream getDumpStream()
	{
		return stream;
	}
	
}	// JSONUtil
