/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 11.02.2009 - [JR] - creation
 * 13.02.2009 - [JR] - setProperty: check if the connection is open
 *                   - call: start callback in it's owtn thread
 * 09.04.2009 - [JR] - setAndCheckAlive: always forward to server instead of return null
 *                     (updates the alive time for the server-side session!)      
 * 10.04.2009 - [JR] - setProperty: don't allow to change client properties when connected
 * 13.05.2009 - [JR] - close: checked isOpen() [BUGFIX] 
 * 27.05.2009 - [JR] - used IDirectServer instead of IServer  
 * 11.08.2009 - [JR] - open/openSub: syncProperties [BUGFIX]
 * 04.10.2009 - [JR] - setNewPassword: old password parameter
 * 15.10.2009 - [JR] - setProperty: always set internal property and only call sync when the connection is open [BUGFIX]
 * 16.10.2009 - [JR] - static server instance [BUGFIX]   
 * 14.11.2009 - [JR] - open, openSub, call: connection-id validation [BUGFIX]   
 *                   - set/getProperty: connection info check    
 * 23.02.2010 - [JR] - #18: syncProperties: the server sends only allowed properties - removed class check
 * 06.06.2010 - [JR] - #134: remove missing properties 
 * 28.02.2013 - [JR] - #643: callback validation removed because server checks it      
 * 11.07.2013 - [JR] - #728: isCalling implemented    
 *                   - setLastCallTime calls       
 */
package com.sibvisions.rad.server;

import java.io.IOException;
import java.util.Hashtable;
import java.util.Map.Entry;

import javax.rad.remote.ConnectionInfo;
import javax.rad.remote.IConnection;
import javax.rad.remote.IConnectionConstants;
import javax.rad.remote.event.ICallBackListener;

import com.sibvisions.util.ChangedHashtable;

/**
 * The <code>DirectServerConnection</code> is an {@link IConnection} implementation for 
 * a direct server communication. The calls will be sent to the server without
 * serialization.
 * 
 * @author Ren Jahn
 */
public class DirectServerConnection implements IConnection
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the communication server. */
	private IDirectServer server = null;
	
	/** whether a call is active. */
	private boolean bCalling = false;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of {@link DirectServerConnection} with a new {@link Server} instance.
	 */
	public DirectServerConnection()
	{
		if (server == null)
		{
			server = Server.getInstance();
		}
	}

	/**
	 * Creates a new instance of {@link DirectServerConnection} for a {@link IDirectServer} implementation.
	 * 
	 * @param pServer the server
	 */
	public DirectServerConnection(IDirectServer pServer)
	{
		server = pServer;
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void open(ConnectionInfo pConnectionInfo) throws Throwable
	{
		pConnectionInfo.setLastCallTime(System.currentTimeMillis());		

		if (pConnectionInfo.getConnectionId() != null)
		{
			//Sessions can't be created more than once!
			throw new SecurityException("Session is already open!");
		}
		
		Object oId = server.createSession(pConnectionInfo.getProperties());

		pConnectionInfo.setConnectionId(oId);
		
		syncProperties(pConnectionInfo);
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void openSub(ConnectionInfo pConnectionInfo, ConnectionInfo pConnectionInfoSub) throws Throwable
	{
		pConnectionInfo.setLastCallTime(System.currentTimeMillis());		

		if (pConnectionInfoSub.getConnectionId() != null)
		{
			//Sessions can't be created more than once!
			throw new SecurityException("Session is already open!");
		}

		Object oId = server.createSubSession(pConnectionInfo.getConnectionId(), pConnectionInfoSub.getProperties());
		
		pConnectionInfoSub.setConnectionId(oId);
		
		syncProperties(pConnectionInfoSub);
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isOpen(ConnectionInfo pConnectionInfo)
	{
		return pConnectionInfo != null && pConnectionInfo.getConnectionId() != null;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void close(ConnectionInfo pConnectionInfo) throws Throwable
	{
		if (isOpen(pConnectionInfo))
		{
			pConnectionInfo.setLastCallTime(System.currentTimeMillis());		
	
			try
			{
				server.destroySession(pConnectionInfo.getConnectionId());
			}
			finally
			{
				pConnectionInfo.setConnectionId(null);
			}
		}
		else
		{
			throw new IllegalStateException("Connection not open");
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void reopen(ConnectionInfo pConnectionInfo) throws Throwable
	{
		pConnectionInfo.setLastCallTime(System.currentTimeMillis());		

		try
		{
			close(pConnectionInfo);
		}
		catch (Throwable th)
		{
			//kein Problem
		}
		
		open(pConnectionInfo);
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized Object[] call(ConnectionInfo pConnectionInfo,
						 			  String[] pObjectName, 
						 			  String[] pMethod, 
						 			  Object[][] pParams, 
						 			  ICallBackListener[] pCallBack) throws Throwable
    {
		try
		{
			bCalling = true;

			//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			// Validation 
			// (same checks as in AbstractSerializedConnection.call)
			//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
			if (pConnectionInfo == null)
			{
				throw new IllegalArgumentException("Invalid connection information: null");
			}
	
			if (pConnectionInfo.getConnectionId() == null)
			{
				throw new IllegalStateException("The connection is not open!");
			}
			
			//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			// Validation 
			// (same checks as in AbstractSerializedConnection.callIntern)
			//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
			if (pMethod == null)
			{
				throw new IllegalArgumentException("No remote method specified!");
			}
			
			if (pObjectName != null && pMethod.length != pObjectName.length)
			{
				throw new IllegalArgumentException("More or less objects than methods!");
			}
	
			if (pParams != null && pParams.length != pMethod.length)
			{
				throw new IllegalArgumentException("More or less params than methods!");
			}
			
			Object oConnectionId = pConnectionInfo.getConnectionId();
			
			//without connection-id: only create connection is possible
			if (oConnectionId == null)
			{
				if (pObjectName == null 
					|| !IConnection.OBJ_SESSION.equals(pObjectName[0]) 
					|| (!IConnection.MET_SESSION_CREATE.equals(pMethod[0]) 
						&& !IConnection.MET_SESSION_SUBSESSION_CREATE.equals(pMethod[0])))
				{
					throw new IOException("Connection is not open!");
				}
			}
	
			if (pCallBack != null && pCallBack.length != pMethod.length)
			{
				throw new IllegalArgumentException("More or less callbacks than methods!");
			}
			
			//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			// CALL(s)
			//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			
			Object[] oResult = new Object[pMethod.length];
			
			Object oConId = pConnectionInfo.getConnectionId();
			
			Throwable thError;
			
			for (int i = 0, anz = oResult.length; i < anz; i++)
			{
				thError = null;
				
				try
				{
					if (pCallBack == null || pCallBack[i] == null)
					{
						oResult[i] = server.execute(oConId, 
								                    pObjectName != null ? pObjectName[i] : null, 
								                    pMethod[i], 
								                    pParams != null ? pParams[i] : null);
					}
					else
					{
						server.executeCallBack(oConId,
											   pCallBack[i],
								               pObjectName != null ? pObjectName[i] : null, 
								               pMethod[i], 
								               pParams != null ? pParams[i] : null);
						
						oResult[i] = null;
					}
				}
				catch (Throwable th)
				{
					oResult[i] = null;
					thError = th;
				}
				finally
				{
					pConnectionInfo.setLastCallTime(System.currentTimeMillis());		
				}
				
				//update the properties independent from call errors
				try
				{
					syncProperties(pConnectionInfo);
				}
				catch (Throwable th)
				{
					//don't override previous errors!
					if (thError == null)
					{
						thError = th;
					}
				}		
				finally
				{
					pConnectionInfo.setLastCallTime(System.currentTimeMillis());		
				}
				
				if (thError != null)
				{
					throw thError;
				}
			}
			
			return oResult;
		}
		finally
		{
			bCalling = false;
		}
    }
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isCalling()
	{
		return bCalling;
	}	
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized ConnectionInfo[] setAndCheckAlive(ConnectionInfo pConnectionInfo, 
			                                              ConnectionInfo[] pSubConnections) throws Throwable
	{
		pConnectionInfo.setLastCallTime(System.currentTimeMillis());		

		Hashtable<Object, ConnectionInfo> htMapping;
		
		Object[] oConIds;

		if (pSubConnections != null)
		{
			htMapping = new Hashtable<Object, ConnectionInfo>();
			
			oConIds = new Object[pSubConnections.length];
			
			for (int i = 0, anz = pSubConnections.length; i < anz; i++)
			{
				oConIds[i] = pSubConnections[i].getConnectionId();
				
				htMapping.put(oConIds[i], pSubConnections[i]);
			}
		}
		else
		{
			htMapping = null;
			oConIds = null;
		}
		
		Object[] oValid = server.setAndCheckAlive(pConnectionInfo.getConnectionId(), oConIds);
		
		syncProperties(pConnectionInfo);
		
		if (oValid == null)
		{
			return null;
		}
		else
		{
			ConnectionInfo[] ciResult = new ConnectionInfo[oValid.length];
			
			for (int i = 0, anz = oValid.length; i < anz; i++)
			{
				ciResult[i] = htMapping.get(oValid[i]);
			}
			
			return ciResult;
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void setProperty(ConnectionInfo pConnectionInfo, String pName, Object pValue) throws Throwable
	{
		//client properties can only be changed when the connection is closed!
		if (pName != null && pName.startsWith(IConnectionConstants.PREFIX_CLIENT))
		{
			if (isOpen(pConnectionInfo))
			{
				throw new SecurityException("Client properties are not accessible after the connection was opened!");
			}

			pConnectionInfo.getProperties().put(pName, pValue, false);
		}
		else
		{
			pConnectionInfo.getProperties().put(pName, pValue, false);

			if (isOpen(pConnectionInfo))
			{
				pConnectionInfo.setLastCallTime(System.currentTimeMillis());		
		
				server.setProperty(pConnectionInfo.getConnectionId(), pName, pValue);
				
				syncProperties(pConnectionInfo);
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized Object getProperty(ConnectionInfo pConnectionInfo, String pName) throws Throwable
	{
		if (pConnectionInfo != null)
		{
			pConnectionInfo.setLastCallTime(System.currentTimeMillis());		
	
			if (isOpen(pConnectionInfo))
			{
				syncProperties(pConnectionInfo);
			}
			
			return pConnectionInfo.getProperties().get(pName);
		}
		
		return null;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized Hashtable<String, Object> getProperties(ConnectionInfo pConnectionInfo) throws Throwable
	{
		if (pConnectionInfo != null)
		{
			pConnectionInfo.setLastCallTime(System.currentTimeMillis());
		
			if (isOpen(pConnectionInfo))
			{
				syncProperties(pConnectionInfo);
			}
			
			return (Hashtable<String, Object>)pConnectionInfo.getProperties().clone();
		}
		
		return null;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void setNewPassword(ConnectionInfo pConnectionInfo, String pOldPassword, String pNewPassword) throws Throwable
	{
		pConnectionInfo.setLastCallTime(System.currentTimeMillis());		
		
		server.setNewPassword(pConnectionInfo.getConnectionId(), pOldPassword, pNewPassword);
		
		syncProperties(pConnectionInfo);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Synchronize the client-side properties with the server-side properties.
	 * 
	 * @param pConnectionInfo the connection information
	 * @throws Throwable communication error, security checks, invalid method, ...
	 */
	private void syncProperties(ConnectionInfo pConnectionInfo) throws Throwable
	{
		//with this call we get only accessible properties - NOT all!!!
		ChangedHashtable<String, Object> chtServerProps = server.getProperties(pConnectionInfo.getConnectionId());
		ChangedHashtable<String, Object> chtClientProps = pConnectionInfo.getProperties();
		
		for (Entry<String, Object> entry : chtServerProps.entrySet())
		{
			chtClientProps.put(entry.getKey(), entry.getValue(), false);
		}
		
		Hashtable<String, Object> htKeys = new Hashtable<String, Object>(chtClientProps);
		
		//remove missing properties
		for (String sKey : htKeys.keySet())
		{
			if (!chtServerProps.containsKey(sKey))
			{
				chtClientProps.remove(sKey);
			}
		}
	}
	
}	// DirectServerConnection
