/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 30.05.2009 - [RH] - creation.
 * 02.03.2010 - [RH] - reorganized MetaData -> ServerMetaData, ColumnMetaData -> ServerColumnMetaData
 * 27.03.2010 - [JR] - #92: default value support 
 * 28.12.2010 - [RH] - #230 - quoting of all DB objects like columns, tables, views.   
 * 11.03.2011 - [RH] - #308 - DB specific automatic quoting implemented        
 * 21.07.2011 - [RH] - #436 - OracleDBAccess and PostgresDBAccess should translate JVx quotes in specific insert                      
 * 18.11.2011 - [RH] - #510: All XXDBAccess should provide a SQLException format method 
 *                     #511: Postgres DBManipulation schema & table detection in getColumnMetaData fails
 * 16.12.2011 - [JR] - #498: enum detection supported                      
 *                   - #528: createStorage
 * 22.12.2011 - [RH] - #532: Like/Equals bug in Postgres fixed.
 * 08.05.2012 - [JR] - #575: convert value(s) to database specific value(s) 
 * 08.09.2013 - [RH] - #787: PostgreSQLDBAccess connect error
 */
package com.sibvisions.rad.persist.jdbc;

import java.sql.CallableStatement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Types;
import java.util.List;
import java.util.Map;

import javax.rad.model.ModelException;
import javax.rad.model.condition.CompareCondition;
import javax.rad.model.condition.Equals;
import javax.rad.model.condition.ICondition;
import javax.rad.model.condition.Like;
import javax.rad.model.condition.LikeIgnoreCase;
import javax.rad.model.condition.LikeReverse;
import javax.rad.model.condition.LikeReverseIgnoreCase;
import javax.rad.model.datatype.IDataType;
import javax.rad.model.datatype.StringDataType;
import javax.rad.persist.DataSourceException;

import org.postgresql.util.PGobject;

import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.log.ILogger.LogLevel;
import com.sibvisions.util.type.StringUtil;
import com.sibvisions.util.type.StringUtil.CaseSensitiveType;

/**
 * The <code>PostgreSQLDBAccess</code> is the implementation for Postgres databases.<br>
 *  
 * @see com.sibvisions.rad.persist.jdbc.DBAccess
 * 
 * @author Roland Hrmann
 */
public class PostgreSQLDBAccess extends DBAccess
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the enum datatype. */
	public static final int TYPE_ENUM = -900;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Constructs a new OracleDBAccess Object.
	 */
	public PostgreSQLDBAccess()
	{
		super();		
		
		setDriver("org.postgresql.Driver");	
	}
		
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDatabaseSpecificLockStatement(String pWritebackTable, ServerMetaData pServerMetaData, ICondition pPKFilter) throws DataSourceException
	{
		return super.getDatabaseSpecificLockStatement(pWritebackTable, pServerMetaData, pPKFilter) + " NO WAIT";											
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object[] insertDatabaseSpecific(String pWritebackTable, String pInsertStatement, ServerMetaData pServerMetaData, 
                                           Object[] pNewDataRow, String pDummyColumn)  throws DataSourceException
    {
		return insertPostgres(pWritebackTable, pInsertStatement, pServerMetaData, pNewDataRow, pDummyColumn);
    }
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Map<String, Object> getDefaultValues(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		return super.getDefaultValues(pCatalog, pSchema, pTable.toLowerCase());
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Object translateDefaultValue(String pColumnName, int pDataType, String pDefaultValue) throws Exception
	{
		//PostgreSql JDBC returns 'value'::xxxxx or 1234
		return super.translateDefaultValue(pColumnName, pDataType, StringUtil.removeQuotes(pDefaultValue, "'"));
	}
	
	/** 
	 * {@inheritDoc}
	 */
	@Override
	public SQLException formatSQLException(SQLException pSqlException)
	{
		return formatSQLException(pSqlException, pSqlException.getMessage(), pSqlException.getSQLState());
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Returns the newly inserted row from an Postgres Database. <br>
	 * It uses RETURNING .. INTO to get the primary key values back from the database. 
	 * 
	 * @param pTablename		the table to use for the insert
	 * @param pInsertStatement	the SQL Statement to use for the insert
	 * @param pServerMetaData	the meta data to use.
	 * @param pNewDataRow		the new IDataRow with the values to insert
	 * @param pDummyColumn		true, if all writeable columns are null, but for a correct INSERT it have
	 *                          to be minimum one column to use in the syntax.
	 * @return the newly inserted row from an Postgres Database.
	 * @throws DataSourceException
	 *             if an <code>Exception</code> occur during insert the <code>IDataRow</code> 
	 *             to the storage
	 */	
	private Object[] insertPostgres(String pTablename, String pInsertStatement, ServerMetaData pServerMetaData,
			                      Object[] pNewDataRow, String pDummyColumn)  throws DataSourceException
	{
		StringBuffer sInsertStatement = new StringBuffer(pInsertStatement);
		
		// use RETURNING to get all PK column values filled in in from the trigger
		sInsertStatement.append(" RETURNING ");
		
		int[] pPKColumnIndices = pServerMetaData.getPrimaryKeyColumnIndices();
		for (int i = 0; pPKColumnIndices != null && i < pPKColumnIndices.length; i++)
		{
			if (i > 0)
			{
				sInsertStatement.append(", ");
			}
			
			sInsertStatement.append(pServerMetaData.getServerColumnMetaData(pPKColumnIndices[i]).getColumnName().getQuotedName());
		}		
		
		CallableStatement csInsert = null;
		ResultSet rsResult = null;
		try
		{
			// #436 - OracleDBAccess and PostgresDBAccess should translate JVx quotes in specific insert
			String sSQL = translateQuotes(sInsertStatement.toString());
			debug("executeSQL->", sSQL);
			csInsert = getConnection().prepareCall(sSQL);
		
			ServerColumnMetaData[] cmdServerColumnMetaData = pServerMetaData.getServerColumnMetaData();
			int[] iaWriteables = pServerMetaData.getWritableColumnIndices();
			
			if (pDummyColumn == null)
			{
				setColumnsToStore(csInsert, cmdServerColumnMetaData, iaWriteables, pNewDataRow, null);
			}
			else
			{
				for (int i = 0; i < cmdServerColumnMetaData.length; i++)
				{
					if (cmdServerColumnMetaData[i].getColumnName().getQuotedName().equals(pDummyColumn))
					{
						csInsert.setObject(1, null, cmdServerColumnMetaData[i].getSQLType());
						break;
					}
				}					
			}

			rsResult = csInsert.executeQuery();
		
			// use RETURNING to get the PK column values filled in by the trigger
			// get the out parameters, and set the PK columns
			if (rsResult.next())
			{
				for (int i = 0; pPKColumnIndices != null && i < pPKColumnIndices.length; i++)
				{
					pNewDataRow[pPKColumnIndices[i]] = rsResult.getObject(i + 1);
				}
			}
			
			return pNewDataRow;
		}
		catch (SQLException sqlException)
		{			
			throw new DataSourceException("Insert failed! - " + sInsertStatement, formatSQLException(sqlException));
		}	
		finally
		{
			if (rsResult != null)
			{
				try
				{
					rsResult.close();
				}
				catch (Exception e)
				{
					//nothing to be done
				}
			}
			if (csInsert != null)
			{
				try
				{
					csInsert.close();
				}
				catch (SQLException sqlException)
				{
					// Do nothing!
				}			
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected int setColumnsToStore(PreparedStatement pInsert, ServerColumnMetaData[] pServerColumnMetaData, 
            int[] iaWriteables, Object[] pNew, Object[] pOld) throws DataSourceException
	{
		int i = 1;
		for (int j = 0; j < iaWriteables.length; j++)
		{
			int k = iaWriteables[j];
			if (pOld == null && pNew[k] != null 
					|| pOld != null 
					   && pServerColumnMetaData[k].getDataType().compareTo(pNew[k], pOld[k]) != 0)
			{
				try
				{
					// PostgresSQL jdbc feature, specify always the SQLType. 
					pInsert.setObject(i, convertValueToDatabaseSpecificObject(pNew[k]), pServerColumnMetaData[k].getSQLType());
					i++;
				}
				catch (SQLException sqlException)
				{
					throw new DataSourceException("Set value into PreparedStatement failed!", formatSQLException(sqlException));
				}
			}
		}
		return --i;
	}
	
	/* 
	 * {@inheritDoc}
	 */
	@Override
	public boolean isAutoQuote(String pName)
	{
		CaseSensitiveType type = StringUtil.getCaseSensitiveType(pName);
		
		return type == CaseSensitiveType.LowerCase ? false : true;
	}	
	
	/* 
	 * {@inheritDoc}
	 */
	@Override
	public void setUsername(String pUsername)
	{
		String sUserName = pUsername;
		if (pUsername != null)
		{
			// PostgreSQL has case sensitive Usernames during connect; 
			// without quoting the Username in create USER, the username is default lowercase
			// to prevent Connect FATAL errors, lowercase unquoted Usernames!
			if (sUserName.equals(DBAccess.removeQuotes(pUsername)))
			{
				sUserName = sUserName.toLowerCase();
			}
			else
			{
				sUserName = DBAccess.removeQuotes(pUsername);
			}
		}
		super.setUsername(sUserName);
	}
	
	/* 
	 * {@inheritDoc}
	 */
	@Override
	protected ServerColumnMetaData[] getColumnMetaDataIntern(String pFromClause, 
										     	String[] pQueryColumns,
										     	String pBeforeQueryColumns, 
										     	String pWhereClause, 
										     	String pAfterWhereClause) throws DataSourceException
	{
		ServerColumnMetaData[] scmd = super.getColumnMetaDataIntern(pFromClause, pQueryColumns, pBeforeQueryColumns, pWhereClause, 
				                                                    pAfterWhereClause);
		
		//ENUM detection
		for (int i = 0; i < scmd.length; i++)
		{
			if (scmd[i].getSQLType() == Types.OTHER && isEnum(scmd[i]))
			{
				scmd[i].setDetectedType(TYPE_ENUM);

				List<Object> lLabels;
				try
				{
					lLabels = executeSql("select e.enumlabel from pg_enum e, pg_type t " +
							   		"where e.enumtypid = t.oid and t.typtype = 'e' and t.typname = ?", scmd[i].getSQLTypeName());

					if (lLabels != null)
					{
						scmd[i].setAllowedValues(lLabels.toArray());
					}
				}
				catch (SQLException ex)
				{
					throw new DataSourceException("Create enum link reference to '" + scmd[i].getName() + "' failed!", ex);
				}
			}
		}
		
		return scmd;
	}
	
	/** 
	 * {@inheritDoc}
	 */
	@Override
	protected TableInfo getTableInfoIntern(String pWriteBackTable) throws DataSourceException
	{	
		TableInfo tableInfo = super.getTableInfoIntern(pWriteBackTable);
			
		String sTable = DBAccess.removeQuotes(tableInfo.getTable());
		
		// correct schemas detection, because of jdbc driver bug!
		if (tableInfo.getSchema() != null)
		{
			// so the in the pFromClause was the schema specified, but the jdbc driver just return it as it is, so maybe it has the wrong case letters.
			String sSchema = tableInfo.getSchema();
			String sSchemaNoQuote = DBAccess.removeQuotes(sSchema);
			if (sSchemaNoQuote.equals(sSchema))
			{
				// ok not quoted schema, postgres is default lowerCase, so try that.
				sSchema = sSchema.toLowerCase();
			}
			else
			{
				// its quoted, just use it like it is without the quotes. Have to be correct!
				sSchema = sSchemaNoQuote;
			}
			List<Object[]> tables = checkTablesAndViews(sSchema, sTable);
			if (tables.size() == 0)
			{
				// try it with to lower, the default case for postgres
				tables = checkTablesAndViews(DBAccess.removeQuotes(sSchema), sTable.toLowerCase());
			}
			
			if (tables.size() == 1)
			{
				return new TableInfo(tableInfo.getCatalog(), ((Name)tables.get(0)[0]).getRealName(), ((Name)tables.get(0)[1]).getRealName());				
			}
		}

		// more then one table with the same name exists. (different schema)
		// then use the default schema (search_path) of the user - 99% correct.
		
		PreparedStatement psDefaultSchema = null;
		ResultSet rsDefaultSchema = null;
		long lStart = System.currentTimeMillis();
		try
		{
			psDefaultSchema = getPreparedStatement("SHOW search_path", false);
			rsDefaultSchema = psDefaultSchema.executeQuery();
		
			if (rsDefaultSchema.next())
			{
				String sSearchPath = rsDefaultSchema.getString(1);
						
				ArrayUtil<String> auSchemas = StringUtil.separateList(sSearchPath, ",", true);
				for (int i = 0; i < auSchemas.size(); i++)
				{
					String sSchema = auSchemas.get(i);
					if (sSchema.contains("$user"))
					{
						// use the user name for the schema
						sSchema = getUsername();
					}
					List<Object[]> tables = checkTablesAndViews(DBAccess.removeQuotes(sSchema), sTable);
					if (tables.size() == 0)
					{
						// try it with to lower, the default case for postgres
						tables = checkTablesAndViews(DBAccess.removeQuotes(sSchema), sTable.toLowerCase());
					}
					if (tables.size() == 1)
					{
						// just one table , then use it.
						return new TableInfo(tableInfo.getCatalog(), ((Name)tables.get(0)[0]).getRealName(), ((Name)tables.get(0)[1]).getRealName());				
					}
				}
			}
			
			return tableInfo;
		}
		catch (SQLException e)
		{
			throw new DataSourceException("Jdbc statement close failed", formatSQLException(e));
		}						
		finally
		{
    		if (rsDefaultSchema != null)
    		{
				try
				{
	    			rsDefaultSchema.close();

				}
				catch (SQLException e)
				{
					// Do nothing!
				}						
    		}
    		if (psDefaultSchema != null)
    		{
				try
				{
					psDefaultSchema.close();

				}
				catch (SQLException e)
				{
					// Do nothing!
				}						
    		}
    		
    		if (isLogEnabled(LogLevel.DEBUG))
    		{
    		    debug("getColumnMetaData() - getDefaultSchema in time=", Long.valueOf((System.currentTimeMillis() - lStart)));
    		}
		}		
    }	

	/**
	 * {@inheritDoc} 
	 */
	@Override
	protected Object convertDatabaseSpecificObjectToValue(ServerColumnMetaData pColumnMetaData, Object pValue) throws SQLException
	{
		if (pValue instanceof PGobject)
		{
			return ((PGobject)pValue).getValue();
		}
		
		return pValue;
	}
	
	/**
	 * {@inheritDoc} 
	 */
	@Override
	protected boolean setDatabaseSpecificType(ResultSetMetaData pMetaData, int pColumnIndex, ServerColumnMetaData pColumnMetaData) throws SQLException
	{
		int iType = pMetaData.getColumnType(pColumnIndex);
		
		if (iType == Types.OTHER)
		{
			pColumnMetaData.setDataType(StringDataType.TYPE_IDENTIFIER);
			pColumnMetaData.setPrecision(Integer.MAX_VALUE);
			
			return true;
		}
		
		return super.setDatabaseSpecificType(pMetaData, pColumnIndex, pColumnMetaData);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String createWhereParam(ServerMetaData pServerMetaData, CompareCondition pCompare)
	{
		try
		{
			ServerColumnMetaData scmd = pServerMetaData.getServerColumnMetaData(pCompare.getColumnName()); 
		
			if (scmd.getDetectedType() == TYPE_ENUM)
			{
				return "cast(? as " + scmd.getSQLTypeName() + ")";
			}
			else
			{
				Object oValue = pCompare.getValue();
				
				if (pCompare instanceof Equals && !isTypeEqual(scmd, pCompare) && !(oValue instanceof String)
						|| pCompare instanceof LikeReverse 
						|| pCompare instanceof LikeReverseIgnoreCase 
						|| pCompare instanceof Like 
						|| pCompare instanceof LikeIgnoreCase)					
				{
					return "cast(? as varchar)";
				}
			}
		}
		catch (ModelException me)
		{
			//use default
		}

		return super.createWhereParam(pServerMetaData, pCompare);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String createWhereColumn(ServerMetaData pServerMetaData, CompareCondition pCompare, String pColumnName)
	{
		try
		{
			ServerColumnMetaData scmd     = pServerMetaData.getServerColumnMetaData(pCompare.getColumnName()); 
			IDataType            dataType = scmd.getDataType();
			
			if (pCompare instanceof Equals && !isTypeEqual(scmd, pCompare) && !(dataType instanceof StringDataType)
					|| pCompare instanceof LikeReverse 
					|| pCompare instanceof LikeReverseIgnoreCase 
					|| pCompare instanceof Like 
					|| pCompare instanceof LikeIgnoreCase)					
			{
				return "cast(" + pColumnName + " as varchar)";
			}
		}
		catch (ModelException me)
		{
			//use default
		}

		return super.createWhereColumn(pServerMetaData, pCompare, pColumnName);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Returns the List of tables and views, if existing.
	 * Columns: Catalog, Schema, Name, Type("TABLE", "VIEW"), Comment
	 * 
	 * @param pSchema 		the schema to search for.
	 * @param pTable 		the table to search for.
	 * @return the List of tables.
	 * @throws DataSourceException	if the tables couldn't determined
	 */
	private List<Object[]> checkTablesAndViews(String pSchema, String pTable) throws DataSourceException
	{
		long lStart = System.currentTimeMillis();

		ResultSet rsTables = null;
		try
		{
			rsTables = getConnection().getMetaData().getTables(getConnection().getCatalog(), pSchema, pTable, new String [] { "TABLE", "VIEW" });
			
			List<Object[]> lTables = new ArrayUtil<Object[]>(); 
			
			while (rsTables.next())
			{
				lTables.add(new Object[] { new Name(rsTables.getString("TABLE_SCHEM"), quote(rsTables.getString("TABLE_SCHEM"))),
										   new Name(rsTables.getString("TABLE_NAME"), quote(rsTables.getString("TABLE_NAME"))),
						                   rsTables.getString("TABLE_TYPE"),
						                   rsTables.getString("REMARKS") });
			}
			
			return lTables;
		}
		catch (SQLException e)
		{
			throw new DataSourceException("Jdbc statement close failed", formatSQLException(e));
		}						
		finally
		{
    		if (rsTables != null)
    		{
				try
				{
	    			rsTables.close();
				}
				catch (SQLException e)
				{
					// Do nothing
				}						
    		}

    		if (isLogEnabled(LogLevel.DEBUG))
    		{
    		    debug("getTables() time=", Long.valueOf((System.currentTimeMillis() - lStart)));
    		}
		}
	}
	
	/**
	 * Gets whether the given column has an enum as datatype.
	 * 
	 * @param pColumnMetaData the column metadata
	 * @return <code>true</code> if the column has an enum datatype, <code>false</code> otherwise
	 * @throws DataSourceException if enum detection fails
	 */
	protected boolean isEnum(ServerColumnMetaData pColumnMetaData) throws DataSourceException
	{
		if (pColumnMetaData.getSQLType() == Types.OTHER)
		{ 
			String sTypeName = pColumnMetaData.getSQLTypeName();

			try
			{
				List<Object> liOid = executeSql("select oid from pg_type where typname = ? and typtype = 'e'", sTypeName);
	
				return liOid.size() == 1;
			}
			catch (SQLException sqle)
			{
				throw new DataSourceException("Enum detection failed", formatSQLException(sqle));
			}
		}

		return false;
	}
	
} // PostgreSQLDBAccess
