/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 19.11.2008 - [RH] - creation
 * 20.11.2008 - [RH] - createFilter() for typical default Filters
 * 04.05.2009 - [RH] - createEqualsFilter(Object[],...) added
 * 14.07.2009 - [JR] - createEqualsFilter: meta data added and used for column positions
 * 18.10.2009 - [RH] - createEqualsFilter - if the ColumnNames size == 0 then return null, instead of an index out of bounds exception. [BUGFIX]
 * 11.12.2012 - [JR] - createFullTextFilter: pIncludedColumns added
 * 23.02.2013 - [JR] - toString implemented
 *                   - createCondition with removabel column names implemented
 */
package javax.rad.model.condition;

import javax.rad.model.ColumnDefinition;
import javax.rad.model.IDataBook;
import javax.rad.model.IDataRow;
import javax.rad.model.ModelException;
import javax.rad.model.datatype.BigDecimalDataType;
import javax.rad.model.datatype.BinaryDataType;
import javax.rad.model.datatype.BooleanDataType;
import javax.rad.model.datatype.IDataType;
import javax.rad.model.datatype.StringDataType;
import javax.rad.model.datatype.TimestampDataType;
import javax.rad.persist.ColumnMetaData;
import javax.rad.ui.celleditor.ILinkedCellEditor;

import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.type.CommonUtil;


/**
 * The <code>Filter</code> class provide helper functions to create combined  <code>ICondition</code>s.<br> 
 * 
 * @see javax.rad.model.condition.ICondition
 * @author Roland Hrmann, Martin Handsteiner
 */
public class Filter
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		
	/**
	 * Should not be used.
	 */
	protected Filter() 
	{ 
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a Filter with AND combined LIKE conditions over all filter column names. 
	 * 
	 * @param pFilterDataRow		the IDataRow used for the Filter
	 * @param pFilterColumnNames	the Columns to use in the Filter
	 * @return an Filter with AND combined LIKE conditions over all filter column names.
	 */
	public static ICondition createLikeFilter(IDataRow pFilterDataRow, String[] pFilterColumnNames)
	{
		if (pFilterColumnNames == null || pFilterColumnNames.length == 0)
		{
			return null;
		}
		ICondition cResult = new Like(pFilterDataRow, pFilterColumnNames[0]);
		
		for (int i = 1; i < pFilterColumnNames.length; i++)
		{
			cResult = cResult.and(new Like(pFilterDataRow, pFilterColumnNames[i]));
		}
		return cResult;
	}

	/**
	 * Creates a Filter with AND combined LikeIgnoreCase conditions over all filter column names. 
	 * 
	 * @param pFilterDataRow		the IDataRow used for the Filter
	 * @param pFilterColumnNames	the Columns to use in the Filter
	 * @return an Filter with AND combined LikeIgnoreCase conditions over all filter column names.
	 */
	public static ICondition createLikeIgnoreCaseFilter(IDataRow pFilterDataRow, String[] pFilterColumnNames)
	{
		if (pFilterColumnNames == null || pFilterColumnNames.length == 0)
		{
			return null;
		}
		ICondition cResult = new LikeIgnoreCase(pFilterDataRow, pFilterColumnNames[0]);
		
		for (int i = 1; i < pFilterColumnNames.length; i++)
		{
			cResult = cResult.and(new LikeIgnoreCase(pFilterDataRow, pFilterColumnNames[i]));
		}
		return cResult;
	}
	
	/**
	 * Creates a Filter with AND combined EQUALS conditions over all filter column names. 
	 * 
	 * @param pFilterDataRow		the IDataRow used for the Filter
	 * @param pFilterColumnNames	the Columns to use in the Filter
	 * @return an Filter with AND combined EQUALS conditions over all filter column names.
	 */

	public static ICondition createEqualsFilter(IDataRow pFilterDataRow, String[] pFilterColumnNames)
	{
		if (pFilterColumnNames == null || pFilterColumnNames.length == 0)
		{
			return null;
		}
		ICondition cResult = new Equals(pFilterDataRow, pFilterColumnNames[0]);
		
		for (int i = 1; i < pFilterColumnNames.length; i++)
		{
			cResult = cResult.and(new Equals(pFilterDataRow, pFilterColumnNames[i]));
		}
		return cResult;
	}
	
	/**
	 * Creates a Filter with AND combined EQUALS conditions over all filter column names. 
	 * 
	 * @param pColumnNames		the column names to use
	 * @param pDataRow			the values Object[] of the row to use.
	 * @param pColumnMetaData   the meta data for the columns used by the <code>pDataRow</code>
	 * @return an Filter with AND combined EQUALS conditions over all filter column names.
	 */
	public static ICondition createEqualsFilter(String[] pColumnNames, Object[] pDataRow, ColumnMetaData[] pColumnMetaData)
	{
		if (pColumnNames.length == 0)
		{
			return null;
		}
		int[] iColumnPos = new int[pColumnNames.length];
		
		int iPos;

		//search the positions of the columns in the meta data
		for (int i = 0, anz = pColumnMetaData.length; i < anz; i++)
		{
			iPos = ArrayUtil.indexOf(pColumnNames, pColumnMetaData[i].getName());
			
			if (iPos >= 0)
			{
				iColumnPos[iPos] = i;
			}
		}

		//use the correct column positions
		ICondition cPKFilter = new Equals(pColumnNames[0], pDataRow[iColumnPos[0]]);		
		for (int i = 1; i < pColumnNames.length; i++)
		{
			cPKFilter = cPKFilter.and(new Equals(pColumnNames[i], pDataRow[iColumnPos[i]]));
		}
		
		return cPKFilter;
	}	
	
	/**
	 * Creates an Filter with AND combined and for each <br>
	 * - column with the DataType String with LikeIgnoreCase condition and <br>
	 * - column with the DataType BigDecimal & Boolean with Equals condition and <br>
	 * - for two column in order with the DataType Timestamp the first with GreaterEquals 
	 *   and the second with LessEquals condition and <br>
	 * - one column with the DataType Timestamp with Equals condition and <br>
	 * - column with an ILinkedCellEditor.setValidationEnabled(true) with Equals otherwise with a LikeIgnoreCase condition<br>
	 * over all columns. 
	 * 
	 * @param pFilterDataRow		the IDataRow used for the Filter
	 * @return an Filter with AND combined with a guessed default conditions over all columns.
	 */
	public static ICondition createFilter(IDataRow pFilterDataRow)
	{
		return createFilter(pFilterDataRow, pFilterDataRow.getRowDefinition().getColumnNames());
	}
	
	/**
	 * Creates an Filter with AND combined and for each <br>
	 * - column with the DataType String with LikeIgnoreCase condition and <br>
	 * - column with the DataType BigDecimal & Boolean with Equals condition and <br>
	 * - for two column in order with the DataType Timestamp the first with GreaterEquals 
	 *   and the second with LessEquals condition and <br>
	 * - one column with the DataType Timestamp with Equals condition and <br>
	 * - column with an ILinkedCellEditor.setValidationEnabled(true) with Equals otherwise with a LikeIgnoreCase condition<br>
	 * over all filter column names. 
	 * 
	 * @param pFilterDataRow		the IDataRow used for the Filter
	 * @param pFilterColumnNames	the Columns to use in the Filter
	 * @return an Filter with AND combined with a guessed default conditions over all filter column names.
	 */
	public static ICondition createFilter(IDataRow pFilterDataRow, String[] pFilterColumnNames)
	{
		if (pFilterColumnNames == null || pFilterColumnNames.length == 0)
		{
			return null;
		}
		ICondition cResult  = null;
		ICondition cCurrent = null;
				
		for (int i = 0; i < pFilterColumnNames.length; i++)
		{
			ColumnDefinition cdColumn = pFilterDataRow.getRowDefinition().getColumnDefinition(i);
			IDataType       dtDataType = cdColumn.getDataType();
			
			if (dtDataType instanceof TimestampDataType
			    && i + 1 < pFilterColumnNames.length
			    && pFilterDataRow.getRowDefinition().getColumnDefinition(i + 1).getDataType() instanceof TimestampDataType)
			{
				// - for two column in order with the DataType Timestamp the first with GreaterEquals 
				//   and the second with LessEquals condition
				cCurrent = new GreaterEquals(pFilterDataRow, pFilterColumnNames[i]);
			}
			else if (dtDataType instanceof TimestampDataType
				     && cCurrent instanceof GreaterEquals)
			{
				// - for two column in order with the DataType Timestamp the first with GreaterEquals 
				//   and the second with LessEquals condition
				cCurrent = new LessEquals(pFilterDataRow, pFilterColumnNames[i]);
			}
			else if (dtDataType instanceof BigDecimalDataType
			    || dtDataType instanceof BooleanDataType
			    // - column with an ILinkedCellEditor.setValidationEnabled(true) with Equals
			    || dtDataType.getCellEditor() instanceof ILinkedCellEditor
			       && ((ILinkedCellEditor)dtDataType.getCellEditor()).isValidationEnabled()
			    // - one column with the DataType Timestamp with Equals condition
			    || dtDataType instanceof TimestampDataType)
			{
				cCurrent = new Equals(pFilterDataRow, pFilterColumnNames[i]);
			}
			else if (dtDataType instanceof StringDataType)
			{
				cCurrent = new LikeIgnoreCase(pFilterDataRow, pFilterColumnNames[i]);		
			}
			else
			{
				cCurrent = null;
			}

			if (cCurrent != null)
			{
				if (cResult == null)
				{
					cResult = cCurrent;
				}
				else
				{
					cResult = cResult.and(cCurrent);
				}
			}
		}
		return cResult;
	}
	
	/**
	 * Full text filter.
	 * 
	 * @param pDataBook the databook to search.
	 * @param pSearchString the search string.
	 * @param pIncludedColumns the list of column names for filtering or <code>null</code> to use all column names
	 * @return The full text filter
	 */
	public static ICondition createFullTextFilter(IDataBook pDataBook, String pSearchString, String... pIncludedColumns)
	{
		if (pSearchString == null || pSearchString.trim().length() == 0)
		{
			return null;
		}
		else
		{
			String[] columnNames;
			
			if (pIncludedColumns != null && pIncludedColumns.length > 0)
			{
				columnNames = pIncludedColumns;
			}
			else
			{
				columnNames = pDataBook.getRowDefinition().getColumnNames();
			}
			
			for (int i = columnNames.length - 1; i >= 0; i--)
			{
				try
				{
					if (pDataBook.getRowDefinition().getColumnDefinition(columnNames[i]).getDataType() instanceof BinaryDataType
							|| !pDataBook.getRowDefinition().getColumnDefinition(columnNames[i]).isWritable())
					{
						columnNames = ArrayUtil.remove(columnNames, i);
					}
				}
				catch (ModelException e)
				{
					columnNames = ArrayUtil.remove(columnNames, i);
				}
			}
			
			pSearchString = "*" + pSearchString.trim() + "*";
				
			Or orCondition = new Or();
	
			for (int i = 0; i < columnNames.length; i++)
			{
				orCondition.add(new LikeIgnoreCase(columnNames[i], pSearchString));
			}
			
			return orCondition;
		}
	}

	/**
	 * Creates a string representation of the given condition.
	 * 
	 * @param pCondition the condition
	 * @return the string representation (like SQL code)
	 */
	public static String toString(ICondition pCondition)
	{
		StringBuilder result = new StringBuilder();
		
		if (pCondition instanceof CompareCondition)
		{
			CompareCondition cCompare = (CompareCondition)pCondition;
			Object           oValue   = cCompare.getValue();
			
			if (!cCompare.isIgnoreNull() || oValue != null)
			{
				String sColumnName = cCompare.getColumnName();
				Object oConditionValue = CommonUtil.nvl(cCompare.getValue(), "?");
				
				if ((cCompare instanceof LikeReverse || cCompare instanceof LikeReverseIgnoreCase)
					&& cCompare.getValue() != null)
				{
					if (cCompare instanceof LikeReverse)
					{
						result.append(oConditionValue);
					}
					else if (cCompare instanceof LikeReverseIgnoreCase)
					{
						result.append("UPPER(");
						result.append(oConditionValue);
						result.append(")");
					}						
				}
				else if (cCompare instanceof LikeIgnoreCase)
				{
					result.append("UPPER(");
					result.append(sColumnName);
					result.append(")");
				}						
				else 
				{
					result.append(sColumnName);
				}						
				
				result.append(' ');
				
				if (cCompare.getValue() == null)
				{
					result.append("IS NULL");
				}
				else
				{
					if (cCompare instanceof Equals)
					{
						result.append("= ");
					}
					else if (cCompare instanceof LikeIgnoreCase || cCompare instanceof LikeReverseIgnoreCase)
					{
						result.append("LIKE UPPER(");
					}						
					else if (cCompare instanceof Like
							|| cCompare instanceof LikeReverse)
					{
						result.append("LIKE ");
					}						
					else if (cCompare instanceof Greater)
					{
						result.append("> ");
					}						
					else if (cCompare instanceof GreaterEquals)
					{
						result.append(">= ");
					}						
					else if (cCompare instanceof Less)
					{
						result.append("< ");
					}						
					else if (cCompare instanceof LessEquals)
					{
						result.append("<= ");
					}		
					else
					{
						result.append(' ');
					}

					if (cCompare instanceof LikeReverse || cCompare instanceof LikeReverseIgnoreCase)
					{
						result.append(sColumnName);						
					}
					else
					{
						result.append(oConditionValue);
					}
					
					if (cCompare instanceof LikeIgnoreCase || cCompare instanceof LikeReverseIgnoreCase)
					{
						result.append(")");
					}						
				}
			}
		}
		else if (pCondition instanceof OperatorCondition)
		{			
			OperatorCondition cOperator = (OperatorCondition)pCondition;
						
			ICondition[] caConditions = cOperator.getConditions();
			for (int i = 0; i < caConditions.length; i++)
			{
				String sTempSQL = toString(caConditions[i]);
							
				if (sTempSQL != null && sTempSQL.length() > 0)
				{
					if (i > 0 && result.length() > 0)
					{
						if (cOperator instanceof And)
						{
							result.append(" AND ");
						}
						else if (cOperator instanceof Or)
						{
							result.append(" OR ");
						}						
					}
					if (caConditions[i] instanceof OperatorCondition)
					{
						result.append("(");
						result.append(sTempSQL);
						result.append(")");
					}
					else
					{
						result.append(sTempSQL);
					}
				}
			}
		}
		else if (pCondition instanceof Not)
		{
			ICondition cCond = ((Not)pCondition).getCondition();
			String sTempSQL  = toString(cCond);
					
			result.append("NOT ");			
			if (cCond instanceof OperatorCondition)
			{
				result.append("(");
				result.append(sTempSQL);
				result.append(")");
			}
			else
			{
				result.append(sTempSQL);
			}
		}
		
		return result.toString();    
	}

	/**
	 * Creates a new {@link ICondition} from the given {@link ICondition} and ignores or uses specific columns.
	 * 
	 * @param pCondition the condition
	 * @param pInclude <code>true</code> to include only given column, <code>false</code> to exclude given columns
	 * @param pColumns columns which should be ignored or used
	 * @return the condition without removed columns
	 */
	public static ICondition createCondition(ICondition pCondition, boolean pInclude, String... pColumns)
	{
		ICondition condition = null;
		
		if (pCondition instanceof CompareCondition)
		{
			CompareCondition cCompare = (CompareCondition)pCondition;
			
			String sColumnName = cCompare.getColumnName();
			
			if (pInclude)
			{
				if (ArrayUtil.contains(pColumns, sColumnName))
				{
					condition = cCompare;
				}
			}
			else
			{
				//ignore given columns!
				if (!ArrayUtil.contains(pColumns, sColumnName))
				{
					condition = cCompare;
				}
			}
		}
		else if (pCondition instanceof OperatorCondition)
		{			
			OperatorCondition cOperator = (OperatorCondition)pCondition;
						
			ICondition cond;
			
			ICondition[] caConditions = cOperator.getConditions();
			
			for (int i = 0; i < caConditions.length; i++)
			{
				cond = createCondition(caConditions[i], pInclude, pColumns);
							
				if (cond != null)
				{
					if (i > 0 && condition != null)
					{
						if (cOperator instanceof And)
						{
							condition = condition.and(cond);
						}
						else if (cOperator instanceof Or)
						{
							condition = condition.or(cond);
						}
					}
					else 
					{
						condition = cond;
					}
				}
			}
		}
		else if (pCondition instanceof Not)
		{
			ICondition cCond = ((Not)pCondition).getCondition();
			
			ICondition cond = createCondition(cCond, pInclude, pColumns);
		
			if (cond != null)
			{
				condition = new Not(cond);
			}
		}
		
		return condition;
	}     
	
}	// Filter
