/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 21.11.2008 - [JR] - creation
 * 15.04.2009 - [JR] - log methods implemented
 * 22.04.2009 - [JR] - removed log methods
 * 29.08.2009 - [JR] - added simple showMessage calls  
 * 12.06.2010 - [JR] - show... now returns IContent 
 *                   - show... generic opener
 * 20.09.2013 - [JR] - #798: notifyActivate implemented                   
 */
package javax.rad.application.genui;

import java.util.Hashtable;

import javax.rad.application.IContent;
import javax.rad.application.IWorkScreen;
import javax.rad.application.IWorkScreenApplication;

/**
 * The <code>WorkScreen</code> is a default implementation of {@link IWorkScreen}.
 * 
 * @author Ren Jahn
 */
public class WorkScreen extends Content 
                        implements IWorkScreen
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the parent application. */
	protected IWorkScreenApplication application;

	/** the parameter mapping. */
	private Hashtable<String, Object> htParameter;
	
	/** true, if this workscreen shoul be modal. */
	private boolean modal = false;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of <code>WorkScreen</code> for a parent
	 * application.
	 * 
	 * @param pApplication the parent application
	 */
	public WorkScreen(IWorkScreenApplication pApplication)
	{
		application = pApplication;
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public void notifyActivate()
	{
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IWorkScreenApplication getApplication()
	{
		return application;
	}

	/**
	 * {@inheritDoc}
	 */
	public void save() throws Exception
	{
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void reload() throws Exception
	{
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isModal()
	{
		return modal;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setModal(boolean pModal)
	{
		modal = pModal;
	}

	/**
	 * {@inheritDoc}
	 */
	public Object setParameter(String pName, Object pValue)
	{
		if (htParameter == null)
		{
			htParameter = new Hashtable<String, Object>();
		}
		
		if (pValue == null)
		{
			Object oOld = htParameter.remove(pValue);
			
			if (htParameter.isEmpty())
			{
				htParameter = null;
			}
			
			return oOld;
		}
		else
		{
			return htParameter.put(pName, pValue);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public Object getParameter(String pName)
	{
		if (htParameter == null)
		{
			return null;
		}
		
		return htParameter.get(pName);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Shows a message with the {@link javax.rad.application.IMessageConstants#MESSAGE_ICON_INFO} icon and the
	 * {@link javax.rad.application.IMessageConstants#MESSAGE_BUTTON_OK} button.
	 * 
	 * @param <OP> the opener type
	 * @param pOpener the opener of the information
	 * @param pMessage the message/information to show
	 * @return the message content or <code>null</code> if the message has no content
	 * @throws Throwable if the message could not be initialized 
	 */
	public <OP> IContent showInformation(OP pOpener, String pMessage) throws Throwable
	{
		return getApplication().showMessage(pOpener, MESSAGE_ICON_INFO, MESSAGE_BUTTON_OK, pMessage, null, null);
	}

	/**
	 * Shows a message with the {@link javax.rad.application.IMessageConstants#MESSAGE_ICON_INFO} icon and the
	 * {@link javax.rad.application.IMessageConstants#MESSAGE_BUTTON_OK} button.
	 * 
	 * @param <OP> the opener type
	 * @param pOpener the opener of the information
	 * @param pMessage the message/information to show
	 * @param pOkAction the action to call when OK was pressed
	 * @return the message content or <code>null</code> if the message has no content
	 * @throws Throwable if the message could not be initialized 
	 */
	public <OP> IContent showInformation(OP pOpener, String pMessage, String pOkAction) throws Throwable
	{
		return getApplication().showMessage(pOpener, MESSAGE_ICON_INFO, MESSAGE_BUTTON_OK, pMessage, pOkAction, null);
	}

	/**
	 * Shows a message with the {@link javax.rad.application.IMessageConstants#MESSAGE_ICON_ERROR} icon and the
	 * {@link javax.rad.application.IMessageConstants#MESSAGE_BUTTON_OK} button.
	 * 
	 * @param <OP> the opener type
	 * @param pOpener the opener of the information
	 * @param pMessage the message/error to show
	 * @return the message content or <code>null</code> if the message has no content
	 * @throws Throwable if the message could not be initialized 
	 */
	public <OP> IContent showError(OP pOpener, String pMessage) throws Throwable
	{
		return getApplication().showMessage(pOpener, MESSAGE_ICON_ERROR, MESSAGE_BUTTON_OK, pMessage, null, null);
	}
	
	/**
	 * Shows a message with the {@link javax.rad.application.IMessageConstants#MESSAGE_ICON_ERROR} icon and the
	 * {@link javax.rad.application.IMessageConstants#MESSAGE_BUTTON_OK} button.
	 * 
	 * @param <OP> the opener type
	 * @param pOpener the opener of the information
	 * @param pMessage the message/error to show
	 * @param pOkAction the action to call when OK was pressed
	 * @return the message content or <code>null</code> if the message has no content
	 * @throws Throwable if the message could not be initialized 
	 */
	public <OP> IContent showError(OP pOpener, String pMessage, String pOkAction) throws Throwable
	{
		return getApplication().showMessage(pOpener, MESSAGE_ICON_ERROR, MESSAGE_BUTTON_OK, pMessage, pOkAction, null);
	}

	/**
	 * Shows a message with the {@link javax.rad.application.IMessageConstants#MESSAGE_ICON_QUESTION} icon and the
	 * {@link javax.rad.application.IMessageConstants#MESSAGE_BUTTON_YES_NO} buttons.
	 * 
	 * @param <OP> the opener type
	 * @param pOpener the opener of the information
	 * @param pMessage the message/question to show
	 * @param pOkAction the action to call when yex was pressed
	 * @return the message content or <code>null</code> if the message has no content
	 * @throws Throwable if the message could not be initialized 
	 */
	public <OP> IContent showQuestion(OP pOpener, String pMessage, String pOkAction) throws Throwable
	{
		return getApplication().showMessage(pOpener, MESSAGE_ICON_QUESTION, MESSAGE_BUTTON_YES_NO, pMessage, pOkAction, null);
	}

	/**
	 * Shows a message with the {@link javax.rad.application.IMessageConstants#MESSAGE_ICON_QUESTION} icon and the
	 * {@link javax.rad.application.IMessageConstants#MESSAGE_BUTTON_YES_NO} buttons.
	 * 
	 * @param <OP> the opener type
	 * @param pOpener the opener of the information
	 * @param pMessage the message/question to show
	 * @param pOkAction the action to call when yex was pressed
	 * @param pCancelAction the action to call when no/x was pressed
	 * @return the message content or <code>null</code> if the message has no content
	 * @throws Throwable if the message could not be initialized 
	 */
	public <OP> IContent showQuestion(OP pOpener, String pMessage, String pOkAction, String pCancelAction) throws Throwable
	{
		return getApplication().showMessage(pOpener, MESSAGE_ICON_QUESTION, MESSAGE_BUTTON_YES_NO, pMessage, pOkAction, pCancelAction);
	}
	
}	// WorkScreen
