/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 30.10.2008 - [HM] - creation
 * 24.03.2011 - [JR] - #317: setEnabled performs cancelEditing
 * 08.06.2011 - [JR] - #385: remove dummy editor
 * 10.06.2011 - [JR] - set transparent background
 * 21.02.2013 - [HM] - avoid unnessary cancelEditing.
 */
package com.sibvisions.rad.ui.swing.ext;

import java.awt.Color;
import java.awt.Component;
import java.awt.EventQueue;
import java.awt.Font;
import java.awt.KeyboardFocusManager;

import javax.rad.model.IDataBook;
import javax.rad.model.IDataRow;
import javax.rad.model.ModelException;
import javax.rad.model.datatype.IDataType;
import javax.rad.model.ui.ICellEditor;
import javax.rad.model.ui.ICellEditorHandler;
import javax.rad.model.ui.ICellEditorListener;
import javax.rad.model.ui.IEditorControl;
import javax.rad.ui.IColor;
import javax.rad.util.ExceptionHandler;
import javax.rad.util.TranslationMap;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JTextField;

import com.sibvisions.rad.ui.swing.ext.format.ICellFormatter;
import com.sibvisions.rad.ui.swing.ext.layout.JVxBorderLayout;
import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.LoggerFactory;

/**
 * Editor that implements IControl interface.
 *  
 * @author Martin Handsteiner
 */
public class JVxEditor extends JPanel 
                       implements IEditorControl,
                                  ICellFormatterEditorListener,
                                  Runnable 
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Class members
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the component logger. */
	private static ILogger logger = null;

	/** The DataRow to be edited. */
	private IDataRow dataRow = null;
	
	/** The column to be edited. */
	private String columnName = null;
	
	/** The CellEditor. */
	private ICellEditor cellEditor = null;
	
	/** The cellFormatListener. */
	private ICellFormatter cellFormatter = null;
	
	/** The used CellEditor. */
	private ICellEditorHandler<JComponent> cellEditorHandler = null;
	
	/** The translation mapping. */
	private TranslationMap translation = null;

	/** Cell Editor started editing. */
	private JTextField dummyEditor = new JTextField(10);
	
	/** Tells, if the CellEditor should save immediate. */
	private boolean savingImmediate = false;
	
	/** Tells, if notifyRepaint is called the first time. */
	private boolean firstNotifyRepaintCall = true;
	/** Ignore Cancel call. */
	private boolean isCancelling = false;
	
	
	/** is notified. */
	private boolean isNotified = false;
	
	/** Cell Editor started editing. */
	private boolean editingStarted = false;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Initialization
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** 
	 * Constructs an editor.
	 */
	public JVxEditor()
	{
		super(new JVxBorderLayout());
		
		dummyEditor.setEditable(false);
		dummyEditor.setEnabled(false);
		
		add(dummyEditor, JVxBorderLayout.CENTER);
		
		super.setBackground(null);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setEnabled(boolean pEnabled)
	{
		super.setEnabled(pEnabled);
		
		cancelEditing();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBackground(Color pColor)
	{
		super.setBackground(pColor);
		
		cancelEditing();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setForeground(Color pEnabled)
	{
		super.setForeground(pEnabled);
		
		cancelEditing();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setFont(Font pEnabled)
	{
		super.setFont(pEnabled);
		
		cancelEditing();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addNotify()
	{
		super.addNotify();
		
		isNotified = true;

		if (EventQueue.isDispatchThread())
		{
			run();
		}
		else
		{
			notifyRepaint();
		}
//		run(); // run() instead of notifyRepaint() to ensure immediate paint at first time, necessary for designmode.
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeNotify()
	{
		isNotified = false;

		super.removeNotify();
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Interface implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    //RUNNABLE
    
	/**
	 * The run method is invoked from AWT EventQueue. 
	 * It enables events from the model again. 
	 * Due to performance reasons the events are disabled from the first call of
	 * notifyRepaint until the EventQueue calls the run method. 
	 * This minimizes the repaints of the control. 
	 */
	public void run()
	{
		try
		{
			if (!isCancelling)
			{
				isCancelling = true;
				if (isNotified)
				{
					cancelEditing();
				}
			}
		}
		finally
		{
			isCancelling = false;
			firstNotifyRepaintCall = true;
		}
	}
	
	//ICONTROL
	
	/**
	 * {@inheritDoc}
	 */
	public void notifyRepaint()
	{
		if (firstNotifyRepaintCall && !editingStarted)
		{
			firstNotifyRepaintCall = false;
			
			JVxUtil.invokeLater(this);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void cancelEditing()
	{
		editingStarted = false;

		if (cellEditorHandler != null)
		{
			try
			{
				cellEditorHandler.cancelEditing();
			}
			catch (ModelException pException)
			{
				if (logger == null)
				{
					logger = LoggerFactory.getInstance(getClass());
				}

				logger.error(pException);

				uninstallEditor();
				
				installEditor();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void saveEditing() throws ModelException
	{
		if (editingStarted)
		{
			// Set immediate to false to avoid rekursion, if DataRowListener stores again.
			editingStarted = false;

			cellEditorHandler.saveEditing();

			// This should not be needed, as a saveEditing causes an notifyRepaint event.
//			cancelEditing();
		}
	}

	/**
     * Gets the CellFormatter.
     *
     * @return the CellFormatter.
     * @see #setCellFormatter
     */
    public ICellFormatter getCellFormatter()
    {
    	return cellFormatter;
    }

    /**
     * Sets the CellFormatter.
     * 
	 * @param pCellFormatter the CellFormatter
     * @see #getCellFormatter
     */
    public void setCellFormatter(ICellFormatter pCellFormatter)
    {
    	cellFormatter = pCellFormatter;
    }
	
    /**
     * Sets the translation mapping for this table.
     * 
     * @param pTranslation the translation mapping
     */
    public void setTranslation(TranslationMap pTranslation)
    {
    	if (translation != pTranslation)
    	{
        	translation = pTranslation;
        	
        	if (cellEditorHandler != null)
        	{
        		cellEditorHandler.updateEditor();
        	}
        	notifyRepaint();
    	}
    }
    
    /**
     * Gets the translation mapping for this table.
     * 
     * @return the current translation mapping or <code>null</code> if there is no
     *         translation mapping
     */
    public TranslationMap getTranslation()
    {
    	return translation;
    }
    
	//ICellEditorListener
	
	/**
	 * {@inheritDoc}
	 */
	public void editingStarted() 
	{
		try
		{
			editingStarted = true; // first set editingStarted true, to prevent events on update.

			if (dataRow instanceof IDataBook)
			{
				IDataRow oldDataRow = dataRow.createDataRow(null);
				
				((IDataBook)dataRow).update();
				
				if (!oldDataRow.equals(dataRow, new String[] {columnName})) // Only if value is changed, cancel editing.
				{
					editingStarted = false;
					notifyRepaint();
				}
			}
		}
		catch (ModelException pModelException)
		{
			editingStarted = false;
			notifyRepaint();
			
			ExceptionHandler.raise(pModelException);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void editingComplete(String pCompleteType) 
	{
		if (pCompleteType == ICellEditorListener.ESCAPE_KEY)
		{
			cancelEditing();
		}
		else
		{
			try
			{
				saveEditing();
			}
			catch (ModelException ex)
			{
				// TODO maybe a request focus should occur, but we have to prevent tab change, ... 
				cancelEditing();
				
				ExceptionHandler.raise(ex);
			}
			if (pCompleteType == ICellEditorListener.ENTER_KEY)
			{
				KeyboardFocusManager.getCurrentKeyboardFocusManager().focusNextComponent();
			}
			else if (pCompleteType == ICellEditorListener.SHIFT_ENTER_KEY)
			{
				KeyboardFocusManager.getCurrentKeyboardFocusManager().focusPreviousComponent();
			}
			else if (pCompleteType == ICellEditorListener.TAB_KEY)
			{
				KeyboardFocusManager.getCurrentKeyboardFocusManager().focusNextComponent();
			}
			else if (pCompleteType == ICellEditorListener.SHIFT_TAB_KEY)
			{
				KeyboardFocusManager.getCurrentKeyboardFocusManager().focusPreviousComponent();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isSavingImmediate() 
	{
		return savingImmediate;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void requestFocus()
	{
		Component comp = KeyboardFocusManager.getCurrentKeyboardFocusManager().getDefaultFocusTraversalPolicy().getFirstComponent(this);

		if (comp != null)
		{
			comp.requestFocus();
		}
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // User-defined methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
     * Gets the DataRow edited by this control.
     *
     * @return the DataRow.
     * @see #setDataRow
     */
	public IDataRow getDataRow()
	{
		return dataRow;
	}
	
	/**
     * Sets the DataRow edited by this control.
     *
     * @param pDataRow the DataRow.
     * @throws ModelException if the column name for the row is invalid
     * @see #getDataRow
     * @see #setColumnName(String)
     */
	public void setDataRow(IDataRow pDataRow) throws ModelException
	{
		uninstallEditor();
		
		dataRow = pDataRow;
		
		installEditor();
	}
	
	/**
     * Gets the column name edited by this control.
     *
     * @return the column name.
     * @see #setColumnName
     */
	public String getColumnName()
	{
		return columnName;
	}
	
	/**
     * Sets the column edited by this control.
     *
     * @param pColumnName the column.
     * @throws ModelException if the column name is invalid
     * @see #getColumnName
     */
	public void setColumnName(String pColumnName) throws ModelException
	{
		uninstallEditor();
		
		columnName = pColumnName;
		
		installEditor();
	}

	/**
	 * Sets whether the CellEditor should save immediate.
	 * @param pSavingImmediate true, if the CellEditor should save immediate.
	 */
	public void setSavingImmediate(boolean pSavingImmediate)
	{
		savingImmediate = pSavingImmediate;
	}

	/**
	 * Uninstalls the CellEditor and its CellEditorComponent.
	 */
	private void uninstallEditor()
	{
		if (cellEditorHandler != null)
		{
			// TODO It is unclear, if uninstalling an Editor should call cancel or save.
			// Till it is not clear and no problems occur, nothing is done, what is nearby a cancel.
			// cellEditorHandler.cancelEditing() or cellEditorHandler.saveEditing();
			cellEditorHandler.uninstallEditor();

			dataRow.removeControl(this);
			
			remove(cellEditorHandler.getCellEditorComponent());
			
			cellEditorHandler = null;
		}
		else
		{
			remove(dummyEditor);
		}
	}

	/**
	 * Installs the CellEditor and its CellEditorComponent.
	 */
	private void installEditor()
	{
		dummyEditor.setBackground(null);
		dummyEditor.setEditable(true);
		dummyEditor.setEnabled(true);
		dummyEditor.setEditable(false);
		dummyEditor.setEnabled(false);

		if (dataRow != null && columnName != null)
		{
			try
			{
				IDataType dataType = dataRow.getRowDefinition().getColumnDefinition(columnName).getDataType();
				ICellEditor editor;
				if (cellEditor == null)
				{
					editor = dataType.getCellEditor();
				}
				else
				{
					editor = cellEditor;
				}
				
		    	if (editor == null)
		    	{
		    		editor = JVxUtil.getDefaultCellEditor(dataType.getTypeClass());
		    	}
				
				cellEditorHandler = editor.createCellEditorHandler(this, dataRow, columnName);
				
				add(cellEditorHandler.getCellEditorComponent(), JVxBorderLayout.CENTER);
				dataRow.addControl(this);

				if (isNotified)
				{
					run(); // run() instead of notifyRepaint() to ensure immediate paint at first time, necessary for designmode.
				}
			}
			catch (ModelException ex)
			{
				dummyEditor.setBackground(JVxUtil.getSystemColor(IColor.INVALID_EDITOR_BACKGROUND));
			}
		}
		
		if (cellEditorHandler == null)
		{
			add(dummyEditor, JVxBorderLayout.CENTER);
		}
	}
	
	/**
     * Gets the CellEditor that edits the given column in the given DataRow.
     * If the CellEditor is null, the editor from the columns DataType is used to edit.
     *
     * @return the CellEditor.
     * @see #setCellEditor
     */
	public ICellEditor getCellEditor()
	{
		return cellEditor;
	}
	
	/**
     * Sets the CellEditor that edits the given column in the given DataRow.
     * If the CellEditor is null, the editor from the columns DataType is used to edit.
     *
     * @param pCellEditor the CellEditor.
     * @throws ModelException if the column name of the editor is invalid
     * @see #getCellEditor
     */
	public void setCellEditor(ICellEditor pCellEditor) throws ModelException
	{
		uninstallEditor();
		
		cellEditor = pCellEditor;
		
		installEditor();
	}

	/** 
	 * The current used CellEditor for editing.
	 *  
	 * @return The current used CellEditor for editing.
	 */
	public ICellEditorHandler<JComponent> getCellEditorHandler()
	{
		return cellEditorHandler;
	}
	
}	// JVxEditor
