/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [JR] - creation
 * 29.04.2009 - [JR] - getServerDir implemented
 * 26.08.2009 - [JR] - initDirectories: check the directory structure if the path contains the basedir
 * 26.08.2010 - [JR] - removed classes property because it's not relevant for the general configuration
 * 04.11.2010 - [JR] - isApplication implemented
 * 18.11.2010 - [JR] - #206: changed ApplicationZone creation and isApplication check
 * 11.04.2011 - [JR] - #333: application names are now case sensitive
 * 13.04.2011 - [JR] - ApplicationListOption used
 * 28.07.2011 - [JR] - #445: support virtual filesystems e.g. JBoss
 * 16.06.2013 - [JR] - #673: support external apps folders (see AppSettings)
 * 03.07.2013 - [JR] - #713: getApplicationZone: search config.xml
 */
package com.sibvisions.rad.server.config;

import java.io.File;
import java.net.URL;
import java.util.Hashtable;
import java.util.List;

import com.sibvisions.rad.IPackageSetup;
import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.SecureHash;
import com.sibvisions.util.log.LoggerFactory;
import com.sibvisions.util.type.FileUtil;
import com.sibvisions.util.type.ResourceUtil;
import com.sibvisions.util.xml.XmlNode;
import com.sibvisions.util.xml.XmlWorker;

/**
 * The <code>Configuration</code> enables the access to the server zone and 
 * all available application zones.
 * The zones are organized in following directories:
 * <pre>
 * <b>&lt;CONFIG-DIRECTORY&gt;</b>
 * |- <b>apps</b>                   (applications directory)
 *    |- <b>&lt;application name&gt;</b>  <b>(application zone)</b> (application directory - case sensitive)
 *       |- <b>src</b>              (source files)
 *       |- <b>libs</b>             (additional libraries)
 *       |- <b>classes</b>          (compiled sources)
 *       |- config.xml       (application configuration file)
 * |- <b>server</b>                 <b>(server zone)</b> (server directory)
 *    |- config.xml          (server configuration file)
 * </pre>
 * 
 * @author Ren Jahn
 */
public final class Configuration
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** Application name restriction. */
	public enum ApplicationListOption
	{
		/** All application names are allowed. */
		All,
		/** Only visible application names are allowed. */
		Visible,
		/** Only hidden application names are allowed. */
		Hidden
	}
	
	/** the name of the configuration directory. */
	private static final String NAME_RAD = "rad";

	/** the name of the applications directory. */
	private static final String NAME_APPS = "apps";
	
	/** the name of the server directory. */
	private static final String NAME_SERVER = "server";
	
	
	/** the last value of the system property for the base directory. */
	private static String sOldBaseDir = null;
	
	/** the resource directory. */
	private static File fiBaseDir = null;
	
	/** the configuration directory. */
	private static File fiConfigDir = null;
	
	/** the applications directory. */
	private static File fiAppsDir = null;
	
	/** the server directory. */
	private static File fiServerDir = null;
	
	/** cached application zones. */
	private static Hashtable<String, ApplicationZone> htAppZones = null;
	
	/** the server zone. */
	private static ServerZone zoServer = null;
	
	/** the app settings. */
	private static AppSettings asApps = null;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Invisible constructor, because the <code>Configuration</code> class is a utility class.
	 */
	private Configuration()
	{
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Initialize the directory locations. The locations are dependent of the
	 * system Property {@value IPackageSetup#CONFIG_BASEDIR}. If the system property
	 * is not set, the location of the directories is dependent from the location
	 * of the <code>Configuration</code> class path.
	 */
	private static void initDirectories()
	{
		String sBaseDir = ResourceUtil.getAccessibleProperty(IPackageSetup.CONFIG_BASEDIR, "");
		
		//Only the system property can be changed, the location of the Configuration.class is
		//always in the same directory!
		if (sOldBaseDir == null || !sBaseDir.equals(sOldBaseDir))
		{
			String sConfigClassName = ResourceUtil.getFqClassName(Configuration.class); 
			
			//Start searching at the place where classes or jar files are contained
			String sPath = ResourceUtil.getLocationForClass(sConfigClassName);
			
			if (sPath != null)
			{
				boolean bDeepCheck = true;
				
				//If the system property is set, then the directory must exist!
				if (sBaseDir.trim().length() == 0 || !new File(sBaseDir).exists())
				{
					File fiPath = new File(sPath);
					
					if (!fiPath.isDirectory())
					{
						//The classes are included in a jar file? We assume that class files are contained
						//in a classes directory of the parent directory
						fiPath = new File(fiPath.getParentFile().getParentFile(), "classes");
					}
					
					//Check if we are a sub directory of the application, e.g. during development:
					//<Directory>/rad/apps/<name>/libs/server/<jarfile>
					fiBaseDir = fiPath.getParentFile();
					
					if (!fiBaseDir.exists())
					{
						URL url = ResourceUtil.getResource(ResourceUtil.getFqClassName(Configuration.class));
						
						String sUrl = url.toExternalForm();
						sUrl = sUrl.substring(0, sUrl.indexOf(sConfigClassName));
						
						try
						{
							int iPos = sUrl.lastIndexOf('/');
							
							iPos = sUrl.lastIndexOf('/', iPos - 1);
							
							sUrl = sUrl.substring(0, iPos) + "/rad.zip";
							
							URL urlZip = new URL(sUrl);
							
							String sMD5 = SecureHash.getHash(SecureHash.MD5, urlZip.openStream());

							File fiTemp = new File(System.getProperty("java.io.tmpdir"), "rad_" + sMD5);

							//unzip only when the archive was changed. It is allowed to change
							//the configuration of already installed applications, until it is installed
							//again
							if (!fiTemp.exists())
							{
								FileUtil.unzip(urlZip.openStream(), fiTemp);
							}
							
							fiBaseDir = fiTemp;
							
							//don't check directory structure again!
							bDeepCheck = false;
						}
						catch (Exception e)
						{
							LoggerFactory.getInstance(Configuration.class).error("Base directory is not available!", e);
						}
					}
				}
				else
				{
					//Use the system property
					fiBaseDir = new File(sBaseDir);
				}
				
				if (bDeepCheck)
				{
					//Check if the current basedir is a sub directory of the expected basedir
					File fiSub = fiBaseDir;
					String sSubName;
					
					while (fiSub != null)
					{						
						sSubName = fiSub.getName();
						
						if (sSubName.equalsIgnoreCase(NAME_APPS) || sSubName.equalsIgnoreCase(NAME_SERVER))
						{
							fiSub = fiSub.getParentFile();
	
							//parent directory must be the RAD directory!
							if (fiSub != null && fiSub.getName().equals(NAME_RAD))
							{
								fiBaseDir = fiSub.getParentFile();
								
								fiSub = null;
							}
						}
						else
						{
							fiSub = fiSub.getParentFile();
						}
					}
				}
				
				fiConfigDir = new File(fiBaseDir, NAME_RAD);
				fiAppsDir   = new File(fiConfigDir, NAME_APPS);
				fiServerDir = new File(fiConfigDir, NAME_SERVER);
				
				sOldBaseDir = sBaseDir;
			}
		}
	}
	
	/**
	 * Returns the directory with the configuration files.
	 * 
	 * @return the configuration directory
	 */
	public static File getConfigurationDir()
	{
		initDirectories();
		
		return fiConfigDir;
	}
	
	/**
	 * Returns the directory with the applications.
	 * 
	 * @return the applications directory
	 */
	public static File getApplicationsDir()
	{
		initDirectories();
		
		return fiAppsDir;
	}
	
	/**
	 * Returns the server directory.
	 * 
	 * @return the server directory
	 */
	public static File getServerDir()
	{
		initDirectories();
		
		return fiServerDir;
	}

	/**
	 * Returns the current zone for an application.
	 * 
	 * @param sApplicationName the desired application
	 * @return the zone for the application of null if the application is not available
	 * @throws Exception if the application zone has errors
	 */
	public static ApplicationZone getApplicationZone(String sApplicationName) throws Exception
	{
		ApplicationZone app = null;
		
		
		if (sApplicationName == null || sApplicationName.trim().length() == 0)
		{
			app = new ApplicationZone((File)null);
		}
		else
		{
			if (htAppZones != null)
			{
				app = htAppZones.get(sApplicationName);
			}
			
			//It's possible that the application will be deleted. In that case, the
			//already loaded zone will be invalid!
			//It's better to clean the cache and try loading the configuration again,
			//to throw the expected exceptions
			if (app != null && !app.isValid())
			{
				htAppZones.remove(sApplicationName);
				
				app = null;
			}
			
			//Try to load the application zone from the applications directory
			if (app == null)
			{
				File fiApp = new File(getApplicationsDir(), sApplicationName);
				
				if (!ApplicationZone.isValid(fiApp))
				{
					AppSettings aset = getAppSettings();
					
					List<String> liDirs = aset.getAppLocations();
					
					if (liDirs != null)
					{
						File fiExternalApp;
						
						for (int i = 0, anz = liDirs.size(); i < anz && app == null; i++)
						{
							fiExternalApp = new File(liDirs.get(i), sApplicationName); 
							
							if (ApplicationZone.isValid(fiExternalApp))
							{
								app = new ApplicationZone(fiExternalApp);
							}
						}
					}
				}
				else
				{
					//valid application found
					app = new ApplicationZone(fiApp);
				}
				
				if (app == null)
				{
					//try to find config.xml "upwards" (but config.xml must be an application config.xml!
					
					File fiParent = getApplicationsDir();
					
					File fiConfig;
					
					XmlWorker xmw = new XmlWorker();
					XmlNode xmn;
					
					List<XmlNode> liSubNodes;
					
					while (fiParent != null 
						   && app == null)
					{
						fiConfig = new File(fiParent, Zone.NAME_CONFIG);
						
						if (fiConfig.exists() && fiConfig.canRead())
						{
							//found directory that equals the application name -> config.xml is ok
							if (sApplicationName.equals(fiParent.getName()))
							{
								app = new ApplicationZone(fiParent);
							}
							else
							{
								try
								{
									xmn = xmw.read(fiConfig);
									
									liSubNodes = xmn.getSubNodes();
	
									//an empty xml is ok, and if not empty, it must contain /application
									if (xmn == null 
										|| liSubNodes == null 
										|| xmn.getNode("/application") != null)
									{
										app = new ApplicationZone(fiParent);
									}
								}
								catch (Exception e)
								{
									LoggerFactory.getInstance(Configuration.class).debug("Invalid config.xml '", fiConfig, "'", e);
								}
							}
						}
						
						fiParent = fiParent.getParentFile();
					}
				}				
				
				//Throw an exception if not found!
				if (app == null)
				{
					app = new ApplicationZone(fiApp);
				}
				
				if (htAppZones == null)
				{
					htAppZones = new Hashtable<String, ApplicationZone>();
				}
				
				htAppZones.put(sApplicationName, app);
			}
		}
		
		return app;
	}
	
	/**
	 * Returns the server zone.
	 * 
	 * @return the server zone
	 * @throws Exception if the server zone has errors
	 */
	public static ServerZone getServerZone() throws Exception
	{
		if (zoServer == null)
		{
			zoServer = new ServerZone(getServerDir());
		}
		else
		{
			//if the server zone is invalid, there must be a general problem -> null
			//causes an error in the accesing component 
			if (!zoServer.isValid())
			{
				return null;
			}
		}
		
		return zoServer;
	}
	
	/**
	 * Gets a list of all available application names. This method doesn't check if applications
	 * are valid.
	 * 
	 * @param pListOption an option to ignore/list specific applications
	 * @return a list of available application names
	 */
	public static List<String> listApplicationNames(ApplicationListOption pListOption)
	{
		ArrayUtil<String> auApps = new ArrayUtil<String>();
		
		File[] fiApps = getApplicationsDir().listFiles();
		
		AppSettings aset = getAppSettings();
		
		List<String> liDirs = aset.getAppLocations();
		
		if (liDirs != null)
		{
			File[] fiExtApps;
			
			File fiDir;
			
			for (String sDir : liDirs)
			{
				fiDir = new File(sDir);
				
				if (fiDir.exists())
				{
					fiExtApps = new File(sDir).listFiles();
					
					if (fiExtApps != null && fiExtApps.length > 0)
					{
						fiApps = ArrayUtil.addAll(fiApps, fiExtApps);
					}
				}
			}
		}
		
		if (fiApps != null)
		{
			if (pListOption == null)
			{
				pListOption = ApplicationListOption.Visible;
			}
			
			for (File file : fiApps)
			{
				if (file.isDirectory())
				{
					switch (pListOption)
					{
						case Hidden:
							if (file.isHidden() || file.getName().charAt(0) == '.')
							{
								auApps.add(file.getName());
							}
							break;
						case Visible:
							if (!file.isHidden() && file.getName().charAt(0) != '.')
							{
								auApps.add(file.getName());
							}
							break;
						default:
							auApps.add(file.getName());
					}
				}
			}
				
		}
		
		return auApps;
	}
	
	/**
	 * Checks if a given application name exists as application.
	 * 
	 * @param pApplicationName an application name
	 * @return <code>true</code> if the application is available and is valid (configuration is available),
	 *         <code>false</code> otherwise
	 */
	public static boolean isApplication(String pApplicationName)
	{
		if (pApplicationName != null)
		{
			if (!ApplicationZone.isValid(new File(getApplicationsDir(), pApplicationName)))
			{
				AppSettings aset = getAppSettings();
				
				List<String> liDirs = aset.getAppLocations();
				
				if (liDirs != null)
				{
					File fiExternalApp;
					
					for (int i = 0, anz = liDirs.size(); i < anz; i++)
					{
						fiExternalApp = new File(liDirs.get(i), pApplicationName); 
						
						if (ApplicationZone.isValid(fiExternalApp))
						{
							return true;
						}
					}
				}
				
				return false;
			}
			else
			{
				return true;
			}
		}
		
		return false;
	}

	/**
	 * Gets the application settings.
	 * 
	 * @return the application settings
	 */
	public static AppSettings getAppSettings()
	{
		if (asApps == null)
		{
			initDirectories();

			try
			{
				asApps = new AppSettings(fiConfigDir);
			}
			catch (Exception e)
			{
				//shouldn't happen because AppSettings are optional
				throw new RuntimeException(e);
			}
		}
		
		return asApps;
	}
	
}	// Configuration
