/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [JR] - creation
 * 11.05.2011 - [JR] - closeConnections implemented
 */
package com.sibvisions.rad.server;

import java.util.List;

import javax.rad.server.ISession;
import javax.rad.server.event.ISessionListener;

import com.sibvisions.rad.server.security.ISecurityManager;
import com.sibvisions.util.ArrayUtil;

/**
 * The <code>Monitoring</code> class will be used as server-side object for
 * administer the communication server.<p>
 * The object must be injected to a life-cycle object before it's usable.<br>
 * You can inject the object with the following definition<p>
 * 
 * <pre>
 * &#64;Inject(origin = "server", name = "monitoring")
 * private Monitoring monitoring = null;
 * </pre>
 *  
 * @author Ren Jahn
 */
public final class Monitoring implements ISessionListener
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** instance of the communication server.*/
	private Server server = null;
	
	/** the session-id of the last destroyed session. */
	private Object oLastDestroyedSessionId = null;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates an instance of <code>Monitoring</code> for a special
	 * communication server.
	 * 
	 * @param pServer the communication server
	 */
	public Monitoring(Server pServer)
	{
		this.server = pServer;
		
		server.getSessionManager().addSessionListener(this);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public void sessionCreated(ISession pSession)
	{
	}

	/**
	 * {@inheritDoc}
	 */
	public void sessionDestroyed(ISession pSession)
	{
		if (pSession != null)
		{
			oLastDestroyedSessionId = pSession.getId();
		}
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Gets the number of opened sessions.
	 * 
	 * @return session count
	 */
	public int getSessionCount()
	{
		return server.getSessionManager().getSessionCount();
	}
	
	/**
	 * Gets the id of the last destroyed session.
	 * 
	 * @return the session identifier or null if no session was destroyed
	 */
	public Object getLastDestroyedSessionId()
	{
		return oLastDestroyedSessionId;
	}
	
	/**
	 * Gets the session identifiers of all currently opened sessions.
	 * 
	 * @return a list with session identifiers
	 */
	public ArrayUtil<Object> getSessionIds()
	{
		return server.getSessionManager().getSessionIds();
	}
	
	/**
	 * Close all connections for a specific application.
	 * 
	 * @param pApplicationName the application name
	 */
	public void closeConnections(String pApplicationName)
	{
		DefaultSessionManager sessman = server.getSessionManager();
		
		List<Object> liIds = sessman.getSessionIds();
		
		ISession sess;
		
		for (Object oId : liIds)
		{
			try
			{
				sess = sessman.get(oId);
			
				if (pApplicationName.equals(sess.getApplicationName()))
				{
					server.destroySession(oId);
				}
			}
			catch (Throwable th)
			{
				//maybe already expired
			}
		}
		
		ISecurityManager secman = sessman.getSecurityManagerFromCache(pApplicationName);
		
		if (secman != null)
		{
			secman.release();
		}
	}
	
	/**
	 * Releases the security manger for a specific application.
	 * 
	 * @param pApplicationName the application name.
	 */
	public void releaseSecurityManager(String pApplicationName)
	{
		ISecurityManager secman = server.getSessionManager().getSecurityManagerFromCache(pApplicationName);
		
		if (secman != null)
		{
			secman.release();
		}
	}
	
}	// Monitoring
