/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [JR] - creation
 * 05.10.2008 - [JR] - createSession: moved authentication to MasterSession
 * 27.04.2009 - [JR] - replaced logging
 * 06.05.2009 - [JR] - implemented AbstractSessionManager
 * 10.05.2009 - [JR] - getSecurityManager: used AbstractSession as parameter instead of the Application name
 * 17.08.2009 - [JR] - CONTROLLER_DELAY: 10000L instead of 5000L
 * 07.10.2009 - [JR] - called logout from the SecurityManager when a session will be destroyed
 * 14.11.2009 - [JR] - destroy: don't throw invalid session id - throw SessionExpired (user-friendly) and of
 *                     course, it's possible!
 * 25.12.2010 - [JR] - removed final from class   
 * 29.12.2010 - [JR] - #231: createSecurityManager implemented     
 * 14.02.2011 - [JR] - #285: getSecurityManager checks the configuration   
 * 11.05.2011 - [JR] - getSecurityManagerFromCache implemented
 *                   - cached securitymanager class name (fix for sub classes)   
 * 25.05.2011 - [JR] - #362: destroy sub sessions with destroy()
 *                   - #364: destroy: call logout() of security manager only for master sessions
 * 23.08.2013 - [JR] - #774: isAvailable implemented    
 * 17.09.2013 - [JR] - removed Memory.gc                                                          
 */
package com.sibvisions.rad.server;

import java.net.InetAddress;
import java.util.Hashtable;
import java.util.Map;

import javax.rad.remote.IConnectionConstants;
import javax.rad.remote.SessionExpiredException;
import javax.rad.server.AbstractSessionManager;
import javax.rad.server.ISession;
import javax.rad.server.event.ISessionListener;

import com.sibvisions.rad.remote.ISerializer;
import com.sibvisions.rad.server.security.AbstractSecurityManager;
import com.sibvisions.rad.server.security.ISecurityManager;
import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.ChangedHashtable;
import com.sibvisions.util.ThreadHandler;
import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.LoggerFactory;

/**
 * The <code>DefaultSessionManager</code> handles the access to all sessions
 * created through client connections. 
 * 
 * @author Ren Jahn
 */
public class DefaultSessionManager extends AbstractSessionManager
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** delay for the <code>Controller</code> between two checks. */
	private static final long CONTROLLER_DELAY = 10000L;

	/** the relevant system properties. */
	private static final String[] USED_SYSPROPS = {"user.timezone", "user.name", "os.name", "os.version", 
												   "os.arch", "java.vendor", "java.version", 
												   "java.class.version", "java.vm.name", "file.encoding", 
												   "file.separator", "path.separator", "line.separator"};
	
	
	/** list of sessions. */
	private Hashtable<Object, AbstractSession> htSessions = new Hashtable<Object, AbstractSession>();
	
	/** list of used security managers. */
	private Hashtable<String, ISecurityManager> htSecManager = null;
	
	/** list of used security manager classes. */
	private Hashtable<String, String> htSecManagerClass = null;

	/** list of all registered event listeners. */
	private ArrayUtil<ISessionListener> auEvents = null;
	
	/** object for the synchronized access to session objects. */
	private Object oSync = new Object();
	
	/** controller thread for continuous session verifying. */
	private Thread thController = null;
	
	/** the logger. */
	private ILogger log = LoggerFactory.getInstance(getClass());
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates an instance of <code>DefaultSessionManager</code> for a special
	 * communication server.
	 * 
	 * @param pServer communication server
	 */
	protected DefaultSessionManager(Server pServer)
	{
		super(pServer);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public final AbstractSession get(Object pSessionId)
	{
		AbstractSession session = null;
		
		if (pSessionId != null)
		{
			synchronized (oSync)
			{
				session = htSessions.get(pSessionId);
			}

			if (session != null)
			{
				long lNow = System.currentTimeMillis();

				if (session.isInactive(lNow) || !session.isAlive(lNow))
				{
					destroy(pSessionId);
					
					throw new SessionExpiredException("Session expired '" + pSessionId + "'");
				}
			}
			else
			{
				//If the Controller kills a session, then the session won't be found, though
				//the session-id is vaild! It's a better behaviour to send the client a session expired
				//message instead of invalid session-id. It's user friendly.
				//
				//If the session-id is a fake id (hack attempt), then the message is not correct, but
				//in that case the hacker should believe everything is fine.
				throw new SessionExpiredException("Session expired '" + pSessionId + "'");
			}
		}
		else
		{
			throw new SecurityException("Invalid session id '" + pSessionId + "'");
		}
		
		return session;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public final void addSessionListener(ISessionListener pListener)
	{
		if (auEvents == null)
		{
			auEvents = new ArrayUtil<ISessionListener>();			
		}
		
		auEvents.add(pListener);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public final void removeSessionListener(ISessionListener pListener)
	{
		if (auEvents != null)
		{
			auEvents.remove(pListener);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public final ISessionListener[] getSessionListeners()
	{
		if (auEvents == null)
		{
			return new ISessionListener[0];
		}
		else
		{
			ISessionListener[] islResult = new ISessionListener[auEvents.size()];
			
			return auEvents.toArray(islResult);
		}
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Creates an authenticated session for an application.
	 * 
	 * @param pRequest the request which creates the session
	 * @param pSerializer the serializer for the session
	 * @param pProperties the initial session properties
	 * @return session identifier of newly created <code>Session</code>
	 * @throws Throwable if the security manager detects a problem or the session properties can not be set
	 */
	public final Object createSession(IRequest pRequest, 
									  ISerializer pSerializer, 
									  ChangedHashtable<String, Object> pProperties) throws Throwable
	{
		//Start/Create a NEW session
		MasterSession session = new MasterSession
		(
			this, 
			getInitialProperties(pProperties, pRequest)
		);
		
		session.setSerializer(pSerializer);			
		
		addIntern(session);
		
		postCreateSession(session);
		
		return session.getId();
	}

	/**
	 * Creates a new <code>SubSession</code> for an already authenticated main session.
	 * This method doesn't check if the session is valid.
	 * 
	 * @param pRequest the request which creates the sub session
	 * @param pSession a valid session
	 * @param pProperties the initial session properties
	 * @return the new <code>SubSession</code>
	 * @throws Throwable the session properties can not be set
	 */
	public final Object createSubSession(IRequest pRequest, 
			              				 AbstractSession pSession, 
			              				 ChangedHashtable<String, Object> pProperties) throws Throwable
	{
		MasterSession sessMaster;
		
		//A Sub Session can only be created through a Master Session
		if (pSession instanceof SubSession)
		{
			sessMaster = ((SubSession)pSession).getMasterSession();
		}
		else
		{
			sessMaster = (MasterSession)pSession;
		}
		
		SubSession sub = new SubSession(sessMaster, getInitialProperties(pProperties, pRequest));
		
		addIntern(sub);
		
		postCreateSubSession(sessMaster, sub);
		
		return sub.getId();
	}
	
	/**
	 * Destroyes a session and dependent sub sessions.
	 * 
	 * @param pSessionId session identifier
	 * @throws SecurityException if the session identifier is unknown
	 */
	public final void destroy(Object pSessionId)
	{
		if (pSessionId != null)
		{
			ISession session;

			synchronized (oSync)
			{
				session = htSessions.remove(pSessionId);
			}

			if (session == null)
			{
				//same reason as described in get(Object)
				throw new SessionExpiredException("Session expired '" + pSessionId + "'");
			}

			try
			{
				if (session instanceof MasterSession)
				{
					//it's not possible to authenticate without security manager -> a securitymanager must be present!
					ISecurityManager secman = htSecManager.get(session.getApplicationName());
					
					try
					{
						secman.logout(session);
					}
					catch (Throwable th)
					{
						//should not happen!
						log.error(th);
					}
					
					//Discard all dependent sessions
					ArrayUtil<SubSession> auSubSessions = ((MasterSession)session).removeSubSessions();
					
					if (auSubSessions != null)
					{
						for (int i = 0, anz = auSubSessions.size(); i < anz; i++)
						{
							try
							{
								destroy(auSubSessions.get(i).getId());
							}
							catch (Throwable se)
							{
								log.info(se);
								//nothing to be done
							}
						}
					}
				}
			}
			finally
			{
				//notify listeners
				fireSessionDestroyed(session);
			}
		}
	}

	/**
	 * Adds a session to the internal store of known sessions.
	 * 
	 * @param pSession the session
	 */
	private void addIntern(AbstractSession pSession)
	{
		synchronized (oSync)
		{
			htSessions.put(pSession.getId(), pSession);
		}
		
		//notify listeners
		fireSessionCreated(pSession);
		
		startController();
	}
	
	/**
	 * Gets the security manager for an application.
	 * 
	 * @param pSession the accessing session
	 * @return the security manager or null if the application has no security manager
	 * @throws Exception if it's not possible to initialize a defined security manager
	 */
	final ISecurityManager getSecurityManager(ISession pSession) throws Exception
	{
		String sApplicationName = pSession.getApplicationName();
		
		ISecurityManager ismSecurity = null;
		
		
		//Ideally the SecurityManager will be reused
		if (htSecManager != null)
		{
			ismSecurity = htSecManager.get(sApplicationName);
		}
		
		String sSecMan = pSession.getConfig().getProperty("/application/securitymanager/class");

		if (ismSecurity != null)
		{
			//check if the security manager was changed
			
			if (sSecMan == null || !sSecMan.equals(htSecManagerClass.get(sApplicationName)))
			{
				htSecManager.remove(sApplicationName);
				htSecManagerClass.remove(sApplicationName);
				
				ismSecurity.release();
				ismSecurity = null;
			}
		}
		
		if (ismSecurity == null)
		{
			try
			{
				ismSecurity = createSecurityManager(pSession);

				if (htSecManager == null)
				{
					htSecManager = new Hashtable<String, ISecurityManager>();
					htSecManagerClass = new Hashtable<String, String>();
				}
				
				htSecManager.put(sApplicationName, ismSecurity);
				//it is important to cache the configured class because it is possible that createSecurityManager returns a different class!
				htSecManagerClass.put(sApplicationName, sSecMan);
			}
			catch (Throwable th)
			{
				throw new Exception("Error during instantiation of security manager", th);
			}
		}
		
		return ismSecurity;
	}
	
	/**
	 * Gets the security manager from the cache.
	 * 
	 * @param pApplicationName the name of the application
	 * @return the security manager or <code>null</code> if no security manager was created for the given application
	 */
	final ISecurityManager getSecurityManagerFromCache(String pApplicationName)
	{
		if (htSecManager != null)
		{
			return htSecManager.get(pApplicationName);
		}
		
		return null;
	}
	
	/**
	 * Creates a new security manager instance for the given session.
	 * 
	 * @param pSession the session which needs a security manager
	 * @return the security manager
	 * @throws Exception if the class or default constructor was not found
	 */
	protected ISecurityManager createSecurityManager(ISession pSession) throws Exception
	{
		//forward this call because we need a separate method for subclasses, to over-write
		//the creation
		return AbstractSecurityManager.createSecurityManager(pSession);
	}
	
	/**
	 * Gets the number of opened sessions.
	 * 
	 * @return session count
	 */
	final int getSessionCount()
	{
		synchronized (oSync)
		{
			return htSessions.size();
		}
	}
	
	/**
	 * Returns a copy of the session ids of all opened sessions.
	 * 
	 * @return list of opened sessions, guaranteed not null
	 */
	final ArrayUtil<Object> getSessionIds()
	{
		synchronized (oSync)
		{
			return new ArrayUtil<Object>(htSessions.keySet());
		}
	}
	
	/**
	 * Notify all listeners that have registered for notification on
	 * session events, that a session was created.
	 *  
	 * @param pSession newly created session 
	 */
	private void fireSessionCreated(ISession pSession)
	{
		if (pSession == null || auEvents == null)
		{
			return;
		}
		
		for (int i = 0, anz = auEvents.size(); i < anz; i++)
		{
			auEvents.get(i).sessionCreated(pSession);
		}
	}
	
	/**
	 * Notify all listeners that have registered for notification on
	 * session events, that a session was destroyed.
	 *  
	 * @param pSession newly created session 
	 */
	private void fireSessionDestroyed(ISession pSession)
	{		
		if (pSession == null || auEvents == null)
		{
			return;
		}
		
		for (int i = 0, anz = auEvents.size(); i < anz; i++)
		{			
			auEvents.get(i).sessionDestroyed(pSession);
		}
	}
	
	/**
	 * Starts the session controller if it's not already started.
	 */
	private void startController()
	{
		if (ThreadHandler.isStopped(thController))
		{
			thController = ThreadHandler.start(new Controller());

			log.debug("Start Controller: ", thController.getName());
		}
	}
	
	/**
	 * Validate the activity of all opened sessions. If a session exceeds the
	 * maximum inactive time, it will be destroyed.
	 */
	private void validateSessions()
	{
		ISession session;
		
		//Get a clone of session ids to be independent of create and destroy calls
		ArrayUtil<Object> alSessionIds = getSessionIds();
		
		long lNow = System.currentTimeMillis();
		
		Object oId;
		
		
		for (int i = 0, anz = alSessionIds.size(); i < anz; i++)
		{
			oId = alSessionIds.get(i);
			
			//Thread safety of the Hashtable is sufficient, because we work with
			//a clone of session ids
			session = htSessions.get(oId);
			
			//maybe already destroyed?
			if (session != null)
			{
				if (session.isInactive(lNow) || !session.isAlive(lNow))
				{
					log.debug("Destroy invalid session: ", oId);
					
					destroy(oId);
				}
			}
		}
	}
	
	/**
	 * Sets the default properties for new sessions.
	 * 
	 * @param pProperties the initial client properties
	 * @param pRequest the request which started the session
	 * @return the initial session properties
	 * @throws Exception if an error occurs while setting properties
	 */
	private ChangedHashtable<String, Object> getInitialProperties(ChangedHashtable<String, Object> pProperties, 
			                                                      IRequest pRequest) throws Exception
	{
		ChangedHashtable<String, Object> chtProperties;
		
		//-----------------------------------------------------------
		// Take client properties
		//-----------------------------------------------------------

		if (pProperties != null)
		{
			chtProperties = pProperties;
		}
		else
		{
			chtProperties = new ChangedHashtable<String, Object>();
		}
		
		//-----------------------------------------------------------
		// Take server information
		//-----------------------------------------------------------
		
		InetAddress iaLocal = InetAddress.getLocalHost();
		
		chtProperties.put(IConnectionConstants.PREFIX_SERVER + "server_version", com.sibvisions.rad.IPackageSetup.SERVER_VERSION);
		chtProperties.put(IConnectionConstants.PREFIX_SERVER + "spec_version", javax.rad.IPackageSetup.SPEC_VERSION);
		chtProperties.put(IConnectionConstants.PREFIX_SERVER + "hostname", iaLocal.getHostName());
		chtProperties.put(IConnectionConstants.PREFIX_SERVER + "address", iaLocal.getHostAddress());

		//-----------------------------------------------------------
		// System Properties
		//-----------------------------------------------------------

		String sValue;
		String sPrefix = IConnectionConstants.PREFIX_SERVER + IConnectionConstants.PREFIX_SYSPROP;
		
		for (int i = 0, anz = DefaultSessionManager.USED_SYSPROPS.length; i < anz; i++)
		{
			sValue = System.getProperty(DefaultSessionManager.USED_SYSPROPS[i]);
			
			if (sValue != null)
			{
				chtProperties.put(sPrefix + USED_SYSPROPS[i], sValue);
			}
		}

		//-----------------------------------------------------------
		// Request information
		//-----------------------------------------------------------

		if (pRequest != null)
		{
			Hashtable<String, Object> htAccessProp = pRequest.getProperties();
			
			if (htAccessProp != null)
			{
				sPrefix = IConnectionConstants.PREFIX_SERVER + IConnectionConstants.PREFIX_REQUEST;
				
				for (Map.Entry<String, Object> entry : htAccessProp.entrySet())
				{
					if (entry.getValue() instanceof String)
					{
						chtProperties.put(sPrefix + entry.getKey(), (String)entry.getValue());
					}
				}
			}
		}
		
		
		return chtProperties;
	}
	
	/**
	 * Configures a session after it is created.
	 * 
	 * @param pSession the session
	 */
	protected void postCreateSession(ISession pSession)
	{
	}

	/**
	 * Configures a sub session after it is created.
	 * 
	 * @param pMaster the master session
	 * @param pSession the session
	 */
	protected void postCreateSubSession(ISession pMaster, ISession pSession)
	{
	}

	/**
	 * Gets whether a session is known from this session manager. This means whether the session is
	 * in the list of currently opened sessions. 
	 * 
	 * @param pSession the session to check
	 * @return <code>true</code> if the session is available, <code>false</code> otherwise
	 */
	public boolean isAvailable(ISession pSession)
	{
		if (pSession != null)
		{
			if (htSessions == null || htSessions.get(pSession.getId()) == null)
			{
				return false;
			}

			//don't check inactivity or alive of session, because this method only checks if the session
			//is "known"
			
			return true;
		}
		
		return false;
	}
	
	//****************************************************************
	// Subclass definition
	//****************************************************************
	
	/**
	 * The <code>Controller</code> checks the session activity
	 * and kills zombies.
	 * 
	 * @author Ren Jahn
	 */
	private final class Controller extends Thread
	{
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Overwritten methods
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		
		/**
		 * {@inheritDoc}
		 */
		@Override
		public void run()
		{
			try
			{
				while (!ThreadHandler.isStopped())
				{
					validateSessions();
					
					Thread.sleep(DefaultSessionManager.CONTROLLER_DELAY);
				}
			}
			catch (Throwable th)
			{
				//doesn't matter, because the controller will be started when a new session will
				//be created
				log.debug(th);
			}
			
			log.error("Controller stopped: ", getName());
		}
		
	}	// Controller
	
}	// DefaultSessionManager
