/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 02.03.2010 - [RH] - creation
 * 11.03.2011 - [RH] - #308 - DB specific automatic quoting implemented
 * 19.08.2011 - [JR] - #458: clone(): Name[] cloned     
 * 11.07.2013 - [JR] - #727: PrimaryKeyType introduced
 */
package com.sibvisions.rad.persist.jdbc;

import javax.rad.model.ModelException;
import javax.rad.persist.ColumnMetaData;
import javax.rad.persist.MetaData;

import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.type.BeanUtil;

/**
 * The <code>ServerMetaData</code> is a description of all columns as <code>ServerColumnMetaData</code>,
 * the Primary Key columns, Representation columns and auto increment columns. 
 * It also includes the server relevant infos, in addition to the <code>MetaData</code> just for the client.
 *  
 * @see com.sibvisions.rad.persist.jdbc.ServerColumnMetaData
 * 
 * @author Roland Hrmann
 */
public class ServerMetaData implements Cloneable
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the possible types for the pk columns. */
	public enum PrimaryKeyType
	{
		/** primary key is used. */
		PrimaryKeyColumns,
		/** unique key is used. */
		UniqueKeyColumns,
		/** all columns are used. */
		AllColumns
	};

	/** The MetaData for the client. */
	private MetaData metaData;

	/** The array of all <code>ServerColumnMetaData</code>. */
	private ArrayUtil<ServerColumnMetaData>	auServerColumnMetaData = new ArrayUtil<ServerColumnMetaData>();

	/** Caches the PK column indices for fast access. */
	private int[] 	iaPKColumnIndices;
	
	/** Caches the auto increment column indices for fast access. */
	private int[] 	iaAutoIncrementColumnIndices;

	/** Caches the writable columns indices for fast access. */
	private int[] 	iaWritableColumnIndices;
	
	/** the from clause, for metadata caching. */
	private String 	sFromClause;
	
	/** the query columns, for metadata caching. */
	private String[]	saQueryColumns;

	/** the primary key columns names, for metadata caching. */
	private Name[]		naPrimaryKeyColumnNames;

	/** the quoted primary key columns names, for metadata caching. */
	private String[]	saPrimaryKeyQuotedColumnNames;

	/** the primary key columns names, for metadata caching. */
	private Name[]		naRepresentationColumnNames;

	/** the quoted primary key columns names, for metadata caching. */
	private String[]	saRepresentationQuotedColumnNames;

	/** the primary key columns names, for metadata caching. */
	private Name[]		naAutoIncrementColumnNames;

	/** the quoted primary key columns names, for metadata caching. */
	private String[]	saAutoIncrementQuotedColumnNames;
	
	/** the primary key type. */
	private PrimaryKeyType pktype = PrimaryKeyType.PrimaryKeyColumns;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Creates a new instance of <code>ServerMetaData</code> with new {@link MetaData}.
	 */
	public ServerMetaData()
	{
		metaData = new MetaData();
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ServerMetaData clone()
	{
		try
		{
			ServerMetaData smd = (ServerMetaData)super.clone();
		
			if (auServerColumnMetaData != null)
			{
				smd.auServerColumnMetaData = new ArrayUtil<ServerColumnMetaData>(auServerColumnMetaData.size());

				for (int i = 0, anz = auServerColumnMetaData.size(); i < anz; i++)
				{
					smd.auServerColumnMetaData.add(auServerColumnMetaData.get(i).clone());
				}
			}
			
			if (iaPKColumnIndices != null)
			{
				smd.iaPKColumnIndices = iaPKColumnIndices.clone();
			}
			
			if (iaAutoIncrementColumnIndices != null)
			{
				smd.iaAutoIncrementColumnIndices = iaAutoIncrementColumnIndices.clone();
			}
			
			if (iaWritableColumnIndices != null)
			{
				smd.iaWritableColumnIndices = iaWritableColumnIndices.clone();
			}
			
			if (saQueryColumns != null)
			{
				smd.saQueryColumns = saQueryColumns.clone();
			}
			
			if (naPrimaryKeyColumnNames != null)
			{
				smd.naPrimaryKeyColumnNames = naPrimaryKeyColumnNames.clone();
			}
			
			if (naRepresentationColumnNames != null)
			{
				smd.naRepresentationColumnNames = naRepresentationColumnNames.clone();
			}
			
			if (naAutoIncrementColumnNames != null)
			{
				smd.naAutoIncrementColumnNames = naAutoIncrementColumnNames.clone();
			}
			
			//the quoted column names are created on demand - no need for clone
			
			return smd;
		}
		catch (CloneNotSupportedException cnse)
		{
		    // this shouldn't happen, since we are Cloneable
		    throw new InternalError();
		}
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Returns the MetaData client infos.
	 * 
	 * @return the MetaData client infos.
	 */
	public MetaData getMetaData()
	{
		return metaData;
	}	
	
	/**
	 * Adds an new ServerColumnMetaData column.
	 * 
	 * @param pServerColumnMetaData	the column meta data to add
	 */
	public void addServerColumnMetaData(ServerColumnMetaData pServerColumnMetaData)
	{
		metaData.addColumnMetaData(pServerColumnMetaData.getColumnMetaData());
		auServerColumnMetaData.add(pServerColumnMetaData);

		iaWritableColumnIndices = null;
		iaPKColumnIndices = null;
		iaAutoIncrementColumnIndices = null;
	}
	
	/**
	 * Sets all ServerMetaData columns.
	 * 
	 * @param pServerColumnMetaData	the column meta data to add
	 */
	public void setServerColumnMetaData(ServerColumnMetaData[] pServerColumnMetaData)
	{
		auServerColumnMetaData.clear();
		auServerColumnMetaData.addAll(pServerColumnMetaData);
		
		ColumnMetaData[] cmd = new ColumnMetaData[pServerColumnMetaData.length];
		for (int i = 0; i < cmd.length; i++)
		{
			cmd[i] = pServerColumnMetaData[i].getColumnMetaData();
		}
		metaData.setColumnMetaData(cmd);
		
		iaWritableColumnIndices = null;
		iaPKColumnIndices = null;
		iaAutoIncrementColumnIndices = null;
	}

	/**
	 * Returns all ServerColumnMetaData columns.
	 * 
	 * @return all ServerColumnMetaData columns.
	 */
	public ServerColumnMetaData[] getServerColumnMetaData()
	{
		return auServerColumnMetaData.toArray(new ServerColumnMetaData[auServerColumnMetaData.size()]);
	}

	/**
	 * Returns the specific ServerColumnMetaData column.
	 * 
	 * @param pServerColumnName		the column name to use.
	 * @return the specific ServerColumnMetaData column.
	 * @throws ModelException	if the column name doesn't exists
	 */
	public ServerColumnMetaData getServerColumnMetaData(String pServerColumnName) throws ModelException
	{
		int i = getServerColumnMetaDataIndex(pServerColumnName);
		
		if (i >= 0)
		{
			return auServerColumnMetaData.get(i);
		}
		throw new ModelException("Column '" + pServerColumnName + "' doesn't exists in MetaData!");				
	}

	/**
	 * Returns the specific ServerColumnMetaData column index. -1 if the Column name doesn't exist.
	 * 
	 * @param pServerColumnName	the column name to use.
	 * @return the specific ServerColumnMetaData column index.
	 */
	public int getServerColumnMetaDataIndex(String pServerColumnName)
	{
		for (int i = 0; i < auServerColumnMetaData.size(); i++)
		{
			if (auServerColumnMetaData.get(i).getName().equals(pServerColumnName))
			{
				return i;
			}
		}
		return -1;
	}
	
	/**
	 * Returns all writable column name indices.
	 *
	 * @return all writable column name indices.
	 */
	public int[] getWritableColumnIndices()
	{
		if (iaWritableColumnIndices == null)
		{
			iaWritableColumnIndices = new int[auServerColumnMetaData.size()]; 
			int i = 0;
			for (int j = 0; j < auServerColumnMetaData.size(); j++)
			{
				if (auServerColumnMetaData.get(j).isWritable())
				{
					iaWritableColumnIndices[i] = j;
					i++;
				}
			}
			iaWritableColumnIndices = ArrayUtil.truncate(iaWritableColumnIndices, i);
		}
		return iaWritableColumnIndices;
	}
	
	/**
	 * Returns the Primary Key column name indices.
	 *
	 * @return the Primary Key column name indices.
	 */
	public int[] getPrimaryKeyColumnIndices()
	{
		String[] saPKNames = metaData.getPrimaryKeyColumnNames();
		if (saPKNames == null)
		{
			return null;
		}
		else if (iaPKColumnIndices == null)
		{
			iaPKColumnIndices = new int[saPKNames.length]; 
			
			for (int i = 0; i < saPKNames.length; i++)
			{
				iaPKColumnIndices[i] = getServerColumnMetaDataIndex(saPKNames[i]);
			}
		}
		return iaPKColumnIndices;
	}
	/**
	 * Returns the auto increment column name indices.
	 *
	 * @return the auto increment column name indices.
	 */
	public int[] getAutoIncrementColumnIndices()
	{
		String[] saAutIncNames = metaData.getAutoIncrementColumnNames();
		if (saAutIncNames == null)
		{
			return null;
		}
		else if (iaAutoIncrementColumnIndices == null)
		{
			iaAutoIncrementColumnIndices = new int[saAutIncNames.length]; 
			
			for (int i = 0; i < saAutIncNames.length; i++)
			{
				iaAutoIncrementColumnIndices[i] = getServerColumnMetaDataIndex(saAutIncNames[i]);
			}
		}
		return iaAutoIncrementColumnIndices;
	}

    /**
	 * Returns the Primary Key column names.
	 *
	 * @return the Primary Key column names.
	 */
	public String[] getPrimaryKeyColumnNames()
	{
		return metaData.getPrimaryKeyColumnNames();
	}
	
    /**
	 * Returns the Primary Key column names.
	 *
	 * @return the Primary Key column names.
	 */
	public String[] getPrimaryKeyQuotedColumnNames()
	{
		if (saPrimaryKeyQuotedColumnNames == null)
		{
			saPrimaryKeyQuotedColumnNames = BeanUtil.toArray(naPrimaryKeyColumnNames, new String [naPrimaryKeyColumnNames.length], "quotedName");
		}
		return saPrimaryKeyQuotedColumnNames;
	}

	/**
	 * Sets the Primary Key column names.
	 *
	 * @param pPrimaryKeyColumnNames 
	 *			the Primary Key column names to set
	 */
	public void setPrimaryKeyColumnNames(Name[] pPrimaryKeyColumnNames)
	{
		naPrimaryKeyColumnNames = pPrimaryKeyColumnNames;
		metaData.setPrimaryKeyColumnNames(BeanUtil.toArray(naPrimaryKeyColumnNames, new String [naPrimaryKeyColumnNames.length], "name"));
		iaPKColumnIndices = null;
		saPrimaryKeyQuotedColumnNames = null;
	}

	/**
	 * Returns the Representation column names. Thats all Unique Key columns as default behavior.
	 *
	 * @return the Representation column names.
	 */
	public String[] getRepresentationColumnNames()
	{
		return metaData.getRepresentationColumnNames();
	}

    /**
	 * Returns the quoted Representation column names.
	 *
	 * @return the quoted Representation column names.
	 */
	public String[] getRepresentationQuotedColumnNames()
	{
		if (saRepresentationQuotedColumnNames == null)
		{
			saRepresentationQuotedColumnNames = BeanUtil.toArray(naRepresentationColumnNames, new String [naRepresentationColumnNames.length], "quotedName");
		}
		return saRepresentationQuotedColumnNames;
	}
	
	/**
	 * Sets the Representation column names. 
	 *
	 * @param pRepresentationColumnNames 
	 *			the Representation column names to set
	 */
	public void setRepresentationColumnNames(Name[] pRepresentationColumnNames)
	{
		naRepresentationColumnNames = pRepresentationColumnNames;
		metaData.setRepresentationColumnNames(BeanUtil.toArray(naRepresentationColumnNames, new String [naRepresentationColumnNames.length], "name"));
		saRepresentationQuotedColumnNames = null;
	}
	
	/**
	 * Returns the auto increment column names.
	 *
	 * @return the auto increment column names.
	 */
	public String[] getAutoIncrementColumnNames()
	{
		return metaData.getAutoIncrementColumnNames();
	}

    /**
	 * Returns the quoted auto increment column names.
	 *
	 * @return the quoted auto increment column names.
	 */
	public String[] getAutoIncrementQuotedColumnNames()
	{
		if (saAutoIncrementQuotedColumnNames == null)
		{
			saAutoIncrementQuotedColumnNames = BeanUtil.toArray(naAutoIncrementColumnNames, new String [naAutoIncrementColumnNames.length], "quotedName");
		}
		return saAutoIncrementQuotedColumnNames;
	}
		
	/**
	 * Sets the auto increment column names.
	 *
	 * @param pAutoIncrementColumnNames 
	 *			the auto increment column names. to set
	 */
	public void setAutoIncrementColumnNames(Name[] pAutoIncrementColumnNames)
	{
		naAutoIncrementColumnNames = pAutoIncrementColumnNames;
		metaData.setAutoIncrementColumnNames(BeanUtil.toArray(naAutoIncrementColumnNames, new String [naAutoIncrementColumnNames.length], "name"));
		saAutoIncrementQuotedColumnNames = null;
		iaAutoIncrementColumnIndices = null;
	}

	/**
	 * Returns all column names of the ColumnMetaData list.
	 * 
	 * @return all column names of the ColumnMetaData list.
	 */
	public String[] getColumnNames()
	{
		return metaData.getColumnNames();
	}

	/**
	 * Sets the from clause.
	 * 
	 * @param pFrom the from clause
	 */
	public void setFromClause(String pFrom)
	{
		sFromClause = pFrom;
	}
	
	/**
	 * Gets the from clause.
	 * 
	 * @return the from clause
	 */
	public String getFromClause()
	{
		return sFromClause;
	}
	
	/**
	 * Sets the query columns.
	 * 
	 * @param pColumns the query columns
	 */
	public void setQueryColumns(String[] pColumns)
	{
		saQueryColumns = pColumns;
	}
	
	/**
	 * Gets the query columns.
	 * 
	 * @return the query columns
	 */
	public String[] getQueryColumns()
	{
		return saQueryColumns;
	}
	
	/**
	 * Gets the type of the Primary key columns.
	 * 
	 * @return the pk type
	 * @see PrimaryKeyType
	 */
	public PrimaryKeyType getPrimaryKeyType()
	{
		return pktype;
	}
	
	/**
	 * Gets the type of the Primary key columns.
	 * 
	 * @param pType the type
	 */
	public void setPrimaryKeyType(PrimaryKeyType pType)
	{
		pktype = pType;
	}
	
} 	// ServerMetaData
