/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 30.05.2009 - [RH] - creation.
 * 01.12.2009 - [RH] - #35: datetime (0000-00-00 00:00:00) workaround
 * 09.12.2009 - [JR] - open overwritten, call setDBProperty instead of setUrl
 * 10.03.2010 - [RH] - Alias detection in MySQL 5.1 broken - workaround added.
 *                     Ticket #82 fixed
 * 28.12.2010 - [RH] - #230 - quoting of all DB objects like columns, tables, views. 
 * 11.03.2011 - [RH] - #308 - DB specific automatic quoting implemented  
 * 25.03.2011 - [RH] - setUsername remove quotes if exists.      
 * 07.07.2011 - [RH] - #418 - getPK, getUKs, getFKs, getDefaultValues() fails, because catalog and schema is mixed up in mysql   
 * 27.07.2011 - [RH] - #442 - PK not found in DBStorage for MySQL DBs if the writeback table use schema prefix   
 * 20.11.2011 - [RH] - #512 - MySQL return the SQLType wrong for TEXT columns
 * 17.12.2011 - [JR] - #526 - enum and set datatype support           
 *                   - #528: createStorage
 */
package com.sibvisions.rad.persist.jdbc;

import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.Hashtable;
import java.util.List;

import javax.rad.persist.DataSourceException;

import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.type.StringUtil;

/**
 * The <code>MySQLDBAccess</code> is the implementation for MySQL databases.<br>
 *  
 * @see com.sibvisions.rad.persist.jdbc.DBAccess
 * 
 * @author Roland Hrmann
 */
public class MySQLDBAccess extends DBAccess
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the enum datatype. */
	public static final int TYPE_ENUM = -900;

	/** the set datatype. */
	public static final int TYPE_SET = -901;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Constructs a new MySQLDBAccess Object.
	 */
	public MySQLDBAccess()
	{
		super();		
		
		setDriver("com.mysql.jdbc.Driver");
		setQuoteCharacters("", ""); //no quoting in MySQL, just write it case sensitive correctly - setQuoteCharacters("`", "`");
	}
		
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void open() throws DataSourceException
	{
		//#35
		//JDBC driver workaround for 0000-00-00 00:00:00
		setDBProperty("zeroDateTimeBehavior", "convertToNull");
		setDBProperty("useOldAliasMetadataBehavior", "true");
		
		super.open();
	}	
	
	/* 
	 * {@inheritDoc}
	 */
	@Override
	public boolean isAutoQuote(String pName)
	{
		return false;
	}	
	
	/* 
	 * {@inheritDoc}
	 */
	@Override
	public ColumnMetaDataInfo getColumnMetaData(String pFromClause, 
										     	String[] pQueryColumns,
										     	String pBeforeQueryColumns, 
										     	String pWhereClause, 
										     	String pAfterWhereClause,
										     	String pWritebackTable,
										     	String[] pWritebackColumns) throws DataSourceException
	{		
		ColumnMetaDataInfo cmdi = super.getColumnMetaData(pFromClause, pQueryColumns, pBeforeQueryColumns, pWhereClause, pAfterWhereClause, pWritebackTable, pWritebackColumns);
		
		cmdi.setSchema(cmdi.getCatalog());
		cmdi.setCatalog(null);
		
		// #442 - PK not found in DBStorage for MySQL DBs if the writeback table use schema prefix 
		String sTable = cmdi.getTable();
		
		if (sTable.contains("."))
		{
			sTable = sTable.substring(sTable.lastIndexOf(".") + 1);
			
			cmdi.setTable(sTable);
		}
		
		ServerColumnMetaData[] scmd = cmdi.getColumnMetaData();
		
		int iPrecision;
		
		Hashtable<String, ColumnInfo> htColInf = null;
		ColumnInfo cinf;
		
		int iType;
		
		for (int i = 0; i < scmd.length; i++)
		{
			iType = scmd[i].getSQLType();
			
			if (iType == Types.LONGVARCHAR)
			{
				// #512 - SqlType TEXT is returned as VARCHAR.
				cinf = null;

				//get detailed table information, if possible
				if (sTable != null && htColInf == null)
				{
					htColInf = getColumnInfo(cmdi.getTable());
				}
				
				if (htColInf != null)
				{
					cinf = htColInf.get(scmd[i].getColumnName().getRealName());
					
					if (cinf != null)
					{
						scmd[i].setSQLTypeName(cinf.type);
					}
				}

				//Use precision to detect type 
				if (cinf == null)
				{
					iPrecision = scmd[i].getPrecision();
					
					if (iPrecision > 16777215 && iPrecision <= 2147483647)
					{
						scmd[i].setSQLTypeName("LONGTEXT");
					}
					else if (iPrecision > 65535 && iPrecision <= 16777215)
					{
						scmd[i].setSQLTypeName("MEDIUMTEXT");
					}
					else if (iPrecision > 255 && iPrecision <= 65535)
					{
						scmd[i].setSQLTypeName("TEXT");
					}
					else
					{
						scmd[i].setSQLTypeName("TINYTEXT");
					}
				}
			}
			else if (iType == Types.CHAR)
			{
				//check ENUM or SET 
				
				cinf = null;

				//get detailed table information, if possible
				if (sTable != null && htColInf == null)
				{
					htColInf = getColumnInfo(cmdi.getTable());
				}
				
				if (htColInf != null)
				{
					cinf = htColInf.get(scmd[i].getColumnName().getRealName());
					
					if (cinf != null)
					{
						String sType = cinf.type.toLowerCase();
						
						if (sType.startsWith("enum"))
						{
							scmd[i].setDetectedType(TYPE_ENUM);
							scmd[i].setSQLTypeName(cinf.type);
						}
						else if (sType.startsWith("set"))
						{
							scmd[i].setDetectedType(TYPE_SET);
							scmd[i].setSQLTypeName(cinf.type);
						}
						if (scmd[i].getDetectedType() == TYPE_ENUM || scmd[i].getDetectedType() == TYPE_SET)
						{
							scmd[i].setAllowedValues(extractValues(scmd[i].getSQLTypeName()));
						}					
					}
				}
			}
		}
		
		return cmdi;
    }
	
	/** 
	 * {@inheritDoc}
	 */
	@Override
	public Key getPK(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		return super.getPK(pSchema, null, pTable);
	}
	
	/** 
	 * {@inheritDoc}
	 */
	@Override
	public List<ForeignKey> getFKs(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		List<ForeignKey> lFKs = super.getFKs(pSchema, null, pTable);
		
		for (int i = 0; i < lFKs.size(); i++)
		{
			ForeignKey fk = lFKs.get(i);
			fk.setPKSchema(fk.getPKCatalog());
			fk.setPKCatalog(null);
		}
			
		return lFKs;
	}

	/** 
	 * {@inheritDoc}
	 */
	@Override
	/**
	 * It gets all columns for each Unique Key and return it.
	 * 
	 * @param pCatalog				the catalog to use
	 * @param pSchema				the schema to use
	 * @param pTable				the table to use
	 * @return all columns for each Unique Key. 
	 * @throws DataSourceException	if an error occur during UK search process.
	 */
	public List<Key> getUKs(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		ResultSet rsResultSet = null;
		try
		{
			ArrayUtil<Key>  auResult           = new ArrayUtil<Key>();
			ArrayUtil<Name> auUniqueKeyColumns = new ArrayUtil<Name>();
			
			long lMillis = System.currentTimeMillis();
			DatabaseMetaData dbMetaData = getConnection().getMetaData();
			
			rsResultSet = dbMetaData.getIndexInfo(pSchema, null, pTable, true, false);
			
			if (!rsResultSet.next())
			{
				rsResultSet.close();
				return auResult;
			}
			
			String sUKName = null;
			do
			{
				if (rsResultSet.getString("COLUMN_NAME") != null)
				{
					if (sUKName != null && !rsResultSet.getString("INDEX_NAME").equals(sUKName))
					{
						Key uk = new Key(sUKName);
						uk.setColumns(auUniqueKeyColumns.toArray(new Name[auUniqueKeyColumns.size()]));
						auResult.add(uk);
						auUniqueKeyColumns.clear();
					}
					sUKName = rsResultSet.getString("INDEX_NAME");
				
					auUniqueKeyColumns.add(new Name(rsResultSet.getString("COLUMN_NAME"), quote(rsResultSet.getString("COLUMN_NAME"))));
				}
			}
			while (rsResultSet.next());
			
			//[JR] #188
			if (auUniqueKeyColumns.size() > 0)
			{
				Key uk = new Key(sUKName);
				uk.setColumns(auUniqueKeyColumns.toArray(new Name[auUniqueKeyColumns.size()]));
				auResult.add(uk);
			}
			
			if (auResult.size() > 0)
			{
				// remove PKs, because a PK is also a index, but we don't wanna return them too.
				Key pk = getPK(pCatalog, pSchema, pTable);
				if (pk != null)
				{
					for (int i = auResult.size() - 1; i >= 0; i--)
					{
						Name[] ukCols = auResult.get(i).getColumns();
						if (ArrayUtil.containsAll(ukCols, pk.getColumns()) && ukCols.length == pk.getColumns().length)
						{
							auResult.remove(i);
						}
					}
				}
			}
			
			logger.debug("getUKs(", pTable, ") in ", Long.valueOf(System.currentTimeMillis() - lMillis), "ms");							

			return auResult;
		}
		catch (SQLException sqlException)
		{
  			logger.error("Unique Keys couldn't determined from database! - ", pTable, sqlException);	
  			return null;
		}		
		finally
		{
			try
			{
	    		if (rsResultSet != null)
	    		{
	    			rsResultSet.close();
	    		}
			}
			catch (SQLException e)
			{
				throw new DataSourceException("Jdbc statement close failed", formatSQLException(e));
			}						
		}			
	}
	
	/** 
	 * {@inheritDoc}
	 */
	@Override
	public Hashtable<String, Object> getDefaultValues(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		return super.getDefaultValues(pSchema, null, pTable);
	}

	/* 
	 * {@inheritDoc}
	 */
	@Override
	public void setUsername(String pUsername)
	{
		super.setUsername(DBAccess.removeQuotes(pUsername));
	}	
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Get detailed column information for the given table. The column details contains the
	 * real data type and not the JDBC data type.
	 * 
	 * @param pTableName the name of the table
	 * @return the information per column
	 */
	protected Hashtable <String, ColumnInfo> getColumnInfo(String pTableName)
	{
		Statement stmt = null;
		
		ResultSet res = null;
		
		
		try
		{
			stmt = getConnection().createStatement();
		
			res = stmt.executeQuery("show columns from " + pTableName);
		
			ColumnInfo cinf;
	
			Hashtable<String, ColumnInfo> htColumns = new Hashtable<String, ColumnInfo>();
			
			while (res.next())
			{
				cinf = new ColumnInfo();
				cinf.type  = res.getString("Type");
				cinf.nullable  = res.getString("Null");
				cinf.key       = res.getString("Key");
				cinf.defaultvalue = res.getString("Default");
				
				htColumns.put(res.getString("Field"), cinf);
			}
			
			return htColumns;
		}
		catch (SQLException sqle)
		{
			logger.debug(sqle);
			
			//empty!
			return new Hashtable<String, ColumnInfo>();
		}
		finally
		{
			if (res != null)
			{
				try
				{
					res.close();
				}
				catch (Exception e)
				{
					//nothing to be done
				}
			}
			
			if (stmt != null)
			{
				try
				{
					stmt.close();
				}
				catch (Exception e)
				{
					//nothing to be done
				}
			}
		}
	}

	/**
	 * Extracts the values from a valuelist, separated with <code>,</code>.
	 * 
	 * @param pTypeDefinition the data type definition e.g. enum ('y', 'n')
	 * @return an array with values
	 */
	protected String[] extractValues(String pTypeDefinition)
	{
		int iPos = pTypeDefinition.indexOf('(');
		
		String sValueList = pTypeDefinition.substring(iPos + 1, pTypeDefinition.lastIndexOf(')'));
		
		String[] sValues = sValueList.split(",");
		
		//clean values
		for (int i = 0; i < sValues.length; i++)
		{
			if (sValues[i].equalsIgnoreCase("null"))
			{
				sValues[i] = null;
			}
			else
			{
				sValues[i] = StringUtil.removeQuotes(sValues[i].trim(), "'");
				
				if (sValues[i].length() == 0)
				{
					sValues[i] = null;
				}
			}
		}
		
		return sValues;
	}
	
	//****************************************************************
	// Subclass definition
	//****************************************************************

	/**
	 * The <code>ColumnInfo</code> stores detailed table column information.
	 * 
	 * @author Ren Jahn
	 */
	private static class ColumnInfo
	{
		/** the field name. */
		@SuppressWarnings("unused")
		private String field;

		/** the type. */
		private String type;
		
		/** mandatory or nullable. */
		@SuppressWarnings("unused")
		private String nullable;
		
		/** key definition. */
		@SuppressWarnings("unused")
		private String key;

		/** the default value. */
		@SuppressWarnings("unused")
		private String defaultvalue;
		
	}	// ColumnInfo
	
} 	// MySQLDBAccess
