/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [RH] - creation
 * 19.11.2008 - [RH] - Parameter Array at select() and k() removed
 * 25.11.2008 - [RH] - lockAndRefetch() added
 * 27.11.2008 - [RH] - writeCSV to stream added
 * 10.02.2009 - [JR] - writeCSVToTempFileStorage: first char upper, other characters lower case [BUGFIX]
 * 20.03.2009 - [JR] - writeCSV: last (null) row excluded [BUGFIX]
 * 29.04.2009 - [RH] - renamed to DBStorage, NLS removed, javadoc reviewed
 * 04.05.2009 - [RH] - interface review
 * 14.07.2009 - [JR] - initMetaData implemented [BUGFIX]
 *                   - open: called initMetaData [BUGFIX]
 *                   - initMetaData: called setPrimaryKeyColumns (single point of contact) 
 *                   - insertBean, updateBean, deleteBean, fetchBean, fetchRowBean implemented
 * 29.07.2009 - [JR] - createCSVFile without filename parameter
 * 31.07.2009 - [JR] - createCSVFile build filename without extension...     
 * 03.08.2009 - [JR] - added get/setMetaDataCache methods    
 * 13.10.2009 - [JR] - fetchBean: isOpen check [BUGFIX]    
 * 14.10.2009 - [RH] - refetch bug with more then one row for the primary key fixed.     
 * 15.10.2009 - [RH] - if writebackTable is null, no insert, update, delete is performed. 
 *                     refetch works over getFromClause() [BUGFIX]
 *            - [JR] - fetchBean(ICondition pFilter) implemented    
 * 18.10.2009 - [RH] - refetchRow returns null if no writebackTable is set -> no pk! [BUGFIXED]
 * 22.10.2009 - [JR] - updateBean needs only one IBean as property [FEATURE]
 * 23.11.2009 - [RH] - ColumnMetaData && PrimaryKey Column is replaced with MetaData class
 * 28.11.2009 - [RH] - is/Set[Default]AutoLinkReference added - enable/disable the automatic mechanism.
 * 04.12.2009 - [RH] - QueryColumns with alias bug fixed
 *                     Primary Key column bug (use now writeback cols instead query cols.)
 * 28.12.2009 - [JR] - getMetaData: checked from.equals(writeback) [FEATURE]
 * 29.12.2009 - [JR] - getMetaData: 
 *                     * checked if UK == PK columns and ignore the UK as representation columns if that is the case
 *                     * code review
 *                     * use first non PK column as link column (if available)
 *                     * checked and ignored if there is no link column        
 * 22.02.2010 - [JR] - #67: 
 *                     * getSubStorages implemented
 *                     * open: MetaData caching only when SessionContext is available
 *                     * getMetaData: create substorage reference definition without object name     
 * 02.03.2010 - [RH] - #75 - createAutomaticLinkstorage(), removeLinkReference(), getFK() functions added
 * 06.03.2010 - [JR] - #71: changed bean handling  
 * 13.03.2010 - [JR] - #88: getMetaData: used TYPE_IDENTIFIER      
 * 23.03.2010 - [RH] - automatic added AutoLinkCellEditors now sort over the first visible column (setDefaultSort)
 *                   - missing setMetaData() call for sub DBStorage created with createAutomaticLinkReference(...) added     
 *                     -> cache bug fixed.
 * 27.03.2010 - [JR] - #92: getDefaultValues integrated   
 *                   - #103: ICachedStorage instead of IStorage                  
 * 28.03.2010 - [JR] - #47: openIntern: allowed values integrated
 * 30.03.2010 - [JR] - #110: set/getAllowedValues, set/getDefaultAllowedValuesm set/getDefaultValue, set/getDefaultDefaultValue
 * 16.06.2010 - [JR] - getFK dropped (not used)
 * 29.09.2010 - [RH] - countRows renamed to getEstimatedRowCount()
 * 04.10.2010 - [RH] - #164: Nullable detection fixed for AutoLinkReferences.
 * 06.10.2010 - [JR] - putMetaDataToCache, getMetaData: group null check [BUGFIX]
 * 09.10.2010 - [JR] - getMetaData: use allowed values only if at least one value is set
 * 30.10.2010 - [JR] - extends AbstractCachedStorage
 * 09.11.2010 - [JR] - used log mechanism from parent class
 * 19.11.2010 - [RH] - getUKs, getPKs return Type changed to a <code>Key</code>, so the usage is adapted.          
 * 23.12.2010 - [RH] - getUKs doesn't return PKs ->check removed
 *                     #225: setDefaultSort for AutoLinkReference Storages changed to the AutoLinkReference column!      
 *                     #226: No duplicate columns anymore in representation columns, if it will be build from intersecting UKs  
 *                     #224: wrong AutomaticLinkColumnName fixed - unused ForeignKeys will be not used in Joins.
 * 03.01.2010 - [RH] - #136 - Mysql PK refetch not working -> extract the plan table without schema. 
 * 05.01.2011 - [RH] - #233 - XXXbean() methods moved to AbstractStorage.
 * 06.01.2011 - [JR] - #234: used ColumnMetaDataInfo 
 * 29.01.2011 - [JR] - #211: getMetaData now calls getDefaultAllowedValues if needed
 * 16.02.2011 - [JR] - #287: metadata cache implemented
 * 11.03.2011 - [RH] - #308 - DB specific automatic quoting implemented         
 *                     #309 - isNullable will be determined from the writebackTable instead of the fromClause
 * 12.04.2011 - [JR] - getMetaData: added url and username to cache key                          
 * 19.08.2011 - [JR] - #459: check if metadata cache is enabled instead of global metadata cache
 * 22.08.2011 - [JR] - #462: refreshMetaData implemented
 * 23.11.2011 - [JR] - cleanup internal metadata cache if global cache is cleared
 * 12.10.2012 - [JR] - #597: ignored BinaryDataType when setting PK with all columns 
 * 19.11.2012 - [JR] - code review: @Override added to abstract methods
 * 18.03.2013 - [RH] - #632: DBStorage: Update on Synonym (Oracle) doesn't work - Synonym Support implemented
 * 03.04.2013 - [JR] - #654: writeCSV changed to iterate through records
 * 04.06.2013 - [JR] - #669: remove metadata from cache 
 * 11.07.2013 - [JR] - #727: PrimaryKeyType set
 * 19.07.2013 - [RH] - #733: AfterWhereClause in DBStorage makes SQL Exception
 */
package com.sibvisions.rad.persist.jdbc;

import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.math.BigDecimal;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.rad.model.ModelException;
import javax.rad.model.SortDefinition;
import javax.rad.model.condition.And;
import javax.rad.model.condition.Filter;
import javax.rad.model.condition.ICondition;
import javax.rad.model.datatype.BinaryDataType;
import javax.rad.model.datatype.IDataType;
import javax.rad.model.datatype.StringDataType;
import javax.rad.model.reference.StorageReferenceDefinition;
import javax.rad.persist.ColumnMetaData;
import javax.rad.persist.DataSourceException;
import javax.rad.persist.IStorage;
import javax.rad.persist.MetaData;
import javax.rad.persist.MetaData.Feature;

import com.sibvisions.rad.model.DataBookUtil;
import com.sibvisions.rad.persist.AbstractCachedStorage;
import com.sibvisions.rad.persist.AbstractStorage;
import com.sibvisions.rad.persist.jdbc.DBAccess.ColumnMetaDataInfo;
import com.sibvisions.rad.persist.jdbc.ServerMetaData.PrimaryKeyType;
import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.KeyValueList;
import com.sibvisions.util.SecureHash;
import com.sibvisions.util.type.BeanUtil;
import com.sibvisions.util.type.CommonUtil;
import com.sibvisions.util.type.StringUtil;

/**
 * The <code>DBStorage</code> is a IStorage for SQL database specific features.<br>
 * <br>
 * The DBStorage allows to influence the creation of the SELECT statement to get the data.<br>
 * The following template shows how the SELECT will be constructed out of the specified properties.<br>
 * <pre>
 * <code>
 * SELECT getBeforeQueryColumns() getQueryColumns()
 * FROM   getFromClause()
 * WHERE  getFilter()
 * AND    getMasterReference()
 * AND    getWhereClause()
 * getAfterWhereClause()
 * ORDER BY getSort()
 * </code>
 * </pre>
 * Example:
 * <pre>
 * <code> 
 * SELECT DISTINCT // ++ comment, optimizer hints 
 *        a.COL1 C, b.COL2 D, a.FK_ID FK_ID, ...
 * FROM   TABLE1 a,
 *        TABLE2 b,
 *        ...
 * WHERE  C LIKE 'a%' AND D IS NOT NULL ... // getFilter().getSQL
 * AND    FK_ID = 23                        // getMasterReference() get all detail rows to a 
 *                                             specific master row in an other DataBook
 * AND    a.ID = b.FK_ID ...                // getLastWhereCondition() * GROUP BY C, D                            // getAfterWhereClause()
 * ORDER BY C DESC							// getSort()
 * </code>
 * </pre>
 * 
 * @see javax.rad.persist.IStorage
 * @see javax.rad.persist.ColumnMetaData
 * @see com.sibvisions.rad.model.remote.RemoteDataBook
 * @see com.sibvisions.rad.persist.jdbc.DBAccess
 * @author Roland Hrmann
 */
public class DBStorage extends AbstractCachedStorage
{	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the meta data cache by cache key. */
	private static Hashtable<Object, ServerMetaData> htServerMetaDataCache = null;
	
	/** the link between group, name and cache key. */
	private static KeyValueList<Object, CacheLink> kvlCacheLink = null;

	
	/** the server storage meta data. */
	private ServerMetaData mdServerMetaData;
	
	/** the list of sub storages. */
	private HashMap<String, IStorage> hmpSubStorages;
		
	/** The open state of this DBStorage. */
	private boolean	 	bIsOpen  = false;
	
	/** Determines if after insert and update the row will be refetched. */
	private boolean	 	bRefetch = true;

	/** The DBAccess of this Storage. */
	private IDBAccess   dbAccess;
	
	/** The FROM clause with the query tables and LEFT/RIGHT INNER OUTER JOIN. */
	private String		sFromClause;

	/** The FROM clause before open(). */
	private String		sFromClauseBeforeOpen;
	
	/** The list of query columns to use in the SELECT statement to get the data from the storage. */
	private String[]	saQueryColumns;
	
	/** The list of query columns before open(). */
	private String[]	saQueryColumnsBeforeOpen;
	
	/** The string to place in the SELECT statement between the SELECT and the first query column. */
	private String      sBeforeQueryColumns;

	/** The string to place in the SELECT statement after the last WHERE condition from the
	 *  Filter or MasterReference (Master-Detail Condition). */
	private String      sWhereClause;

	/** The string to place in the SELECT statement after the WHERE clause and before the ORDER BY clause. */
	private String      sAfterWhereClause;

	/** The list of write back columns to use in the INSERT, UPDATE statements.  */
	private String[]	saWritebackColumns;
	
	/** The write table to use in the INSERT, UPDATE statements.  */
	private String		sWritebackTable;

	/** the schema name of the writeback table. */
	private String		sWritebackTableSchema;
	
	/** the catalog name of the writeback table. */
	private String		sWritebackTableCatalog;
	
	/** the cache key. */
	private Object			oCacheKey = null;
	
	/** The default sort is always used, if no sort is specified.  */
	private SortDefinition	sdDefaultSort;

	/** The restrict condition is always used to fetch data.  */
	private ICondition		cRestrictCondition;

	/** Determines the automatic link reference mode.  */
	private Boolean			bAutoLinkReference;

	/** Determines the default automatic link reference mode. */
	private static boolean	bDefaultAutoLinkReference = true;
	
	/** Determines the automatic default value detection. */
	private Boolean			bDefaultValue;
	
	/** Determines the default automatic default value detection. */
	private static boolean 	bDefaultDefaultValue = true;
	
	/** Determines the allowed values detection. */
	private Boolean			bAllowedValues;
	
	/** Determines the default allowed values detection. */
	private static boolean	bDefaultAllowedValues = true;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization 
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Constructs a <code>DBStorage</code>.
	 */
	public DBStorage()
	{
		super();
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Abstract methods implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void writeCSV(OutputStream pOutputStream, 
			             String[] pColumnNames, 
			             String[] pLabels, 
			             ICondition pFilter, 
			             SortDefinition pSort, 
			             String pSeparator) throws Exception
	{
		OutputStreamWriter out = new OutputStreamWriter(pOutputStream, "ISO-8859-1");

		try
		{
			if (cRestrictCondition != null)
			{
				if (pFilter == null)
				{
					pFilter = cRestrictCondition;
				}
				else
				{
					pFilter = new And(cRestrictCondition, pFilter);
				}
			}
	
			if (pSort == null)
			{
				pSort = sdDefaultSort;
			}
			
			if (pColumnNames == null)
			{
				pColumnNames = mdServerMetaData.getColumnNames();
			}
			
			if (pLabels == null)
			{
				pLabels = new String[pColumnNames.length];
				
				for (int i = 0; i < pColumnNames.length; i++)
				{
					pLabels[i] = mdServerMetaData.getServerColumnMetaData(pColumnNames[i]).getLabel();
					
					if (pLabels[i] == null)
					{
						pLabels[i] = ColumnMetaData.getDefaultLabel(pColumnNames[i]);
					}
				}
			}
			
			// write column headers with the defined label
			for (int i = 0; i < pLabels.length; i++)
			{
				if (i > 0)
				{
					out.write(pSeparator);
				}
				out.write(StringUtil.quote(pLabels[i], '"'));
			}
			out.write("\n");
			
			//cache the column datatypes
			IDataType[] dataTypes = new IDataType[pColumnNames.length];
			//cache the column index
			int[] iColIndex = new int[pColumnNames.length];
			
			for (int i = 0, anz = pColumnNames.length; i < anz; i++)
			{
				dataTypes[i] = ColumnMetaData.createDataType(mdServerMetaData.getServerColumnMetaData(pColumnNames[i]).getColumnMetaData());
				
				iColIndex[i] = mdServerMetaData.getServerColumnMetaDataIndex(pColumnNames[i]);
			}
			
			Object[] oData;
			
			List<Object[]> lResult;
			
			int iStart = 0;
			
			boolean bAllFetched = false;
			
			String sFromClauseIntern = getFromClauseIntern();
			
			while (!bAllFetched)
			{
				lResult = dbAccess.fetch(mdServerMetaData, sBeforeQueryColumns, saQueryColumns, sFromClauseIntern,
				            			 pFilter, sWhereClause, sAfterWhereClause, 
				            			 pSort, iStart, 1000);				

				//continue fetching
				iStart += lResult.size();
				
				//write rows
				for (int i = 0, anz = lResult.size(); i < anz; i++)
				{
					oData = lResult.get(i);
					
					if (oData != null)
					{
						for (int j = 0; j < pColumnNames.length; j++)
						{
							if (j > 0)
							{
								out.write(pSeparator);
							}
							
							DataBookUtil.writeQuoted(out, dataTypes[j], oData[iColIndex[j]], pSeparator);
						}		
						
						out.write("\n");
					}
					else
					{
						bAllFetched = true;
					}
				}
			}
			
			out.flush();
		}
		finally
		{
			try
			{
				out.close();
			}
			catch (Exception e)
			{
				// nothing to be done
			}
		}
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public MetaData executeGetMetaData() throws DataSourceException
	{
		if (!isOpen())
		{
			throw new DataSourceException("DBStorage isn't open!");
		}

		return mdServerMetaData.getMetaData();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public int getEstimatedRowCount(ICondition pFilter) throws DataSourceException
	{
		if (!isOpen())
		{
			throw new DataSourceException("DBStorage isn't open!");			
		}
		
		if (cRestrictCondition != null)
		{
			if (pFilter == null)
			{
				pFilter = cRestrictCondition;
			}
			else
			{
				pFilter = new And(cRestrictCondition, pFilter);
			}
		}

		List<Object[]> olResult = dbAccess.fetch(mdServerMetaData, sBeforeQueryColumns, 
												 new String[] { "COUNT(*)"} , getFromClauseIntern(),
								                 pFilter, sWhereClause, sAfterWhereClause, 
									             null, 0, 1);
		
		if (olResult != null && olResult.size() >= 1 && olResult.get(0).length > 0)
		{
			Object oResult = olResult.get(0)[0];
			if (oResult instanceof Integer)
			{
				return ((Integer)oResult).intValue();
			}
			else if (oResult instanceof BigDecimal)
			{
				return ((BigDecimal)oResult).intValue();
			}
			throw new DataSourceException("countRows() result data typ not supported! - " + oResult.getClass().getName());
		}
		return 0;		
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<Object[]> executeFetch(ICondition pFilter, SortDefinition pSort, int pFromRow, int pMinimumRowCount) throws DataSourceException
	{
		if (!isOpen())
		{
			throw new DataSourceException("DBStorage isn't open!");			
		}
		
		if (cRestrictCondition != null)
		{
			if (pFilter == null)
			{
				pFilter = cRestrictCondition;
			}
			else
			{
				pFilter = new And(cRestrictCondition, pFilter);
			}
		}

		if (pSort == null)
		{
			pSort = sdDefaultSort;
		}
		
		return dbAccess.fetch(mdServerMetaData, sBeforeQueryColumns, saQueryColumns, getFromClauseIntern(),
				              pFilter, sWhereClause, sAfterWhereClause, 
				              pSort, pFromRow, pMinimumRowCount);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object[] executeRefetchRow(Object[] pDataRow) throws DataSourceException
	{
		if (!isOpen())
		{
			throw new DataSourceException("DBStorage isn't open!");			
		}

		return refetchRow(pDataRow, true);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object[] executeInsert(Object[] pDataRow) throws DataSourceException
	{
		if (!isOpen())
		{
			throw new DataSourceException("DBStorage isn't open!");			
		}
		if (getWritebackTable() == null)
		{
			return null;
		}

		Object[] oResult = dbAccess.insert(getWritebackTable(), mdServerMetaData, pDataRow);		
		
		if (isRefetch())
		{
			Object[] result = refetchRow(oResult, false);
			if (result != null)
			{
				return result;
			}
		}
		return oResult;		
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object[] executeUpdate(Object[] pOldDataRow, Object[] pNewDataRow) throws DataSourceException
	{
		if (!isOpen())
		{
			throw new DataSourceException("DBStorage isn't open!");			
		}
		if (getWritebackTable() == null)
		{
			return null;
		}

		Object[] oResult = dbAccess.update(getWritebackTable(),	mdServerMetaData, pOldDataRow, pNewDataRow);
		
		if (isRefetch())
		{
			return refetchRow(oResult, false);
		}
		return oResult;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void executeDelete(Object[] pDeleteDataRow) throws DataSourceException
	{
		if (!isOpen())
		{
			throw new DataSourceException("DBStorage isn't open!");			
		}

		if (getWritebackTable() == null)
		{
			return;
		}
		dbAccess.delete(getWritebackTable(), mdServerMetaData, pDeleteDataRow);
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString()
	{
        StringBuffer sbResult = new StringBuffer();
        
        sbResult.append("IsOpen()=");
        sbResult.append(bIsOpen);
        sbResult.append("\n");
        sbResult.append(dbAccess);
        sbResult.append("\n");

        sbResult.append(sBeforeQueryColumns);
        sbResult.append(" ");
        for (int i = 0; saQueryColumns != null && i < saQueryColumns.length; i++)
        {
        	if (i > 0)
        	{
        		sbResult.append(",");
        	}
            sbResult.append(saQueryColumns[i]);        	
        }
        sbResult.append("\n");
        sbResult.append(sFromClause);        	
        sbResult.append("\n");
        sbResult.append(sWhereClause);
        sbResult.append("\n");
        sbResult.append(sAfterWhereClause);
        sbResult.append("\n");
        
        sbResult.append("Writeback:\n");
        sbResult.append(getWritebackTable());
        sbResult.append("\n");

        for (int i = 0; saWritebackColumns != null && i < saWritebackColumns.length; i++)
        {
        	if (i > 0)
        	{
        		sbResult.append(",");
        	}
            sbResult.append(saWritebackColumns[i]);        	
        }

        return sbResult.toString();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public MetaData getMetaData(String pGroup, String pName) throws DataSourceException
	{
		//link the group and name with our cache key
		if (kvlCacheLink == null)
		{
			kvlCacheLink = new KeyValueList<Object, CacheLink>();
		}
		
		CacheLink link = new CacheLink(pGroup, pName);

		//no duplicates
		kvlCacheLink.put(oCacheKey, link, true);

		//updates the cache!
		return super.getMetaData(pGroup, pName);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Opens the DBStorage and checks if the DBAssess is != null.
	 * If automatic link celleditors are used, then the implicit link DBStorge objects are put with "detailtablename"
	 * as sub storage.
	 * 
	 * @throws DataSourceException if the DBAccess is null.
	 */
	public void open() throws DataSourceException
	{
		sFromClauseBeforeOpen = getFromClause();
		
		String[] saQCols = getQueryColumns();
		
		if (saQCols != null)
		{
			//no clone needed, because setQueryColumns changes the instance
			saQueryColumnsBeforeOpen = saQCols;
		}
		else
		{
			saQueryColumnsBeforeOpen = null;
		}
		
		openInternal(false);
	}
	
	/**
	 * Opens the DBStorage and checks if the DBAccess is != null. It doesn't cache the MetaData!
	 * 
	 * @param pUseRepresentationColumnsAsQueryColumns	<code>yes</code> if the QueryColumns are set with 
	 *                                                  all representation columns including the Primary Key columns.
	 * @throws DataSourceException if the DBAccess is null.
	 */
	protected void openInternal(boolean pUseRepresentationColumnsAsQueryColumns) throws DataSourceException
	{
		if (dbAccess == null)
		{
			throw new DataSourceException("DBAccess isn't set!");
		}
		
		mdServerMetaData = getMetaData(getBeforeQueryColumns(),
		                               getQueryColumns(),
		                               getFromClause(),
		                               getWhereClause(),
		                               getAfterWhereClause(),
		                               getWritebackTable(),
		                               getWritebackColumns(),
		                               isAutoLinkReference(),
		                               pUseRepresentationColumnsAsQueryColumns);
		
		bIsOpen = true;
	}
	
	/**
	 * Refreshs the MetaData with initial settings.
	 * 
	 * @throws DataSourceException if the DBAccess is null.
	 */
	protected void refreshMetaData() throws DataSourceException
	{
		if (hmpSubStorages != null)
		{
			hmpSubStorages.clear();
		}
		
		setFromClause(sFromClauseBeforeOpen);
		setQueryColumns(saQueryColumnsBeforeOpen);
		
		openInternal(false);
	}
	
	/**
	 * Adds a sub storage to the internal list of all sub storages.
	 * 
	 * @param pSubStorage	the sub storage to use.
	 * @param pStorageName	the storage name to use.
	 * @return previous sub storage which was associated with the storage name  
	 */
	private IStorage putSubStorage(String pStorageName, IStorage pSubStorage)
	{
		if (hmpSubStorages == null)
		{
			hmpSubStorages = new HashMap<String, IStorage>();
		}		
		
		return hmpSubStorages.put(pStorageName, pSubStorage);
	}
	
	/**
	 * Creates and sets a new <code>StorageReferenceDefinition</code> with the specified FromClause and
	 * columns and reference columns on all <code>pColumns</code>. 
	 * e.g. its used to make an automatic linked celleditor from a custom written view and set it on all <code>pColumns</code>. 
	 * 
	 * @param pColumns			the columns to use.
	 * @param pFromClause		the FromClause. e.g. VIEW name.
	 * @param pReferenceColumns	the reference columns to use.
	 * @throws ModelException	if the <code>StorageReferenceDefinition</code> couldn't created.
	 */	
	public void createAutomaticLinkReference(String[] pColumns, String pFromClause, String[] pReferenceColumns) throws ModelException
	{
		createAutomaticLinkReference(mdServerMetaData, pColumns, createAutomaticLinkStorage(pFromClause), pReferenceColumns);
	}

	/**
	 * Creates and sets a new <code>StorageReferenceDefinition</code> with the specified <code>DBStorage</code> and
	 * columns and reference columns on all <code>pColumns</code>. 
	 * 
	 * @param pColumns			the columns to use.
	 * @param pDBStorage		the <code>DBStorage</code> to use.
	 * @param pReferenceColumns	the reference columns to use.
	 * @throws ModelException	if the <code>StorageReferenceDefinition</code> couldn't created.
	 */
	public void createAutomaticLinkReference(String[] pColumns, DBStorage pDBStorage, String[] pReferenceColumns) throws ModelException
	{
		createAutomaticLinkReference(mdServerMetaData, pColumns, pDBStorage, pReferenceColumns);
	}
	
	/**
	 * Creates and sets a new <code>StorageReferenceDefinition</code> with the specified <code>DBStorage</code> and
	 * columns and reference columns on all <code>pColumns</code>. 
	 * 
	 * @param pMetaData         the metadata to use.
	 * @param pColumns			the columns to use.
	 * @param pDBStorage		the <code>DBStorage</code> to use.
	 * @param pReferenceColumns	the reference columns to use.
	 * @throws ModelException	if the <code>StorageReferenceDefinition</code> couldn't created.
	 */
	protected void createAutomaticLinkReference(ServerMetaData pMetaData, String[] pColumns, AbstractStorage pDBStorage, String[] pReferenceColumns) throws ModelException
	{
		String sStorageName;
		
		if (pDBStorage.getName() != null)
		{
			sStorageName = pDBStorage.getName();
		}
		else if (pDBStorage instanceof DBStorage)
		{
			DBStorage storage = (DBStorage)pDBStorage;
			
			if (storage.getWritebackTable() != null)
			{
				sStorageName = storage.getWritebackTable().toLowerCase();
			}
			else
			{
				sStorageName = storage.getFromClause().toLowerCase();
			}
		}
		else
		{
			StringBuilder sbStorageName = new StringBuilder();
			
			for (int i = 0; i < pColumns.length; i++)
			{
				if (i > 0)
				{
					sbStorageName.append("_");
				}
				
				sbStorageName.append(pColumns[i]);
			}
			
			sStorageName = sbStorageName.toString();
		}
		
		putSubStorage(sStorageName, pDBStorage);
       						
		// The storage has to be referenced by the getSubStorages Method from outside.
		StorageReferenceDefinition srdLink = new StorageReferenceDefinition(pColumns, ".subStorages." + sStorageName, pReferenceColumns);
   		
		for (int i = 0; i < pColumns.length; i++)
		{
			pMetaData.getServerColumnMetaData(pColumns[i]).setLinkReference(srdLink);
		}
	}
	
	/**
	 * Removes the link reference for an server side Dropdown list (automatic linked celleditor).
	 * If its automatic created, its sub DBStorage is also removed.
	 * 
	 * @param pColumnName the column name to use, where the link reference will be removed 
	 * @throws ModelException if the column doesn't exist.
	 */
	public void removeLinkReference(String pColumnName) throws ModelException
	{
		mdServerMetaData.getServerColumnMetaData(pColumnName).setLinkReference(null);
	}		
	
	/**
	 * Gets whether a column has a link reference set.
	 * 
	 * @param pColumnName the column name
	 * @return <code>true</code> if a link reference is set, <code>false</code> otherwise
	 * @throws ModelException if the column doesn't exist.
	 */
	public boolean hasLinkReference(String pColumnName) throws ModelException
	{
		return hasLinkReference(mdServerMetaData, pColumnName);
	}
	
	/**
	 * Gets whether a column has a link reference set.
	 * 
	 * @param pMetaData the metadata to use
	 * @param pColumnName the column name
	 * @return <code>true</code> if a link reference is set, <code>false</code> otherwise
	 * @throws ModelException if the column doesn't exist.
	 */
	protected boolean hasLinkReference(ServerMetaData pMetaData, String pColumnName) throws ModelException
	{
		return pMetaData.getServerColumnMetaData(pColumnName).getLinkReference() != null;
	}
	
	/**
	 * Returns the meta data for the specified DBStorage with all its parameters.
	 * 
	 * @param pBeforeQueryColumns	the before query columns
	 * @param pQueryColumns			the query columns	
	 * @param pFromClause			the from clause with query tables and join definitions
	 * @param pWhereClause			the last where condition in query
	 * @param pAfterWhereClause		the after where clause in query
	 * @param pWritebackTable		the write back table to use.
	 * @param pWritebackColumns		the write back columns to use.
	 * @param pAutoLinkReference	<code>yes</code> if the <code>LinkReferences</code> are automatic set.
	 * @param pUseRepresentationColumnsAsQueryColumns	<code>yes</code> if the QueryColumns are set with 
	 *                                                  all representation columns including the Primary Key columns.
	 * @return the meta data for the specified DBStorage with all its parameters.
	 * @throws DataSourceException	if the meta data couldn't determined.
	 */
	ServerMetaData getMetaData(String pBeforeQueryColumns,	
					           String[] pQueryColumns,
					           String pFromClause,
							   String pWhereClause,
							   String pAfterWhereClause,
							   String pWritebackTable,
							   String[] pWritebackColumns,
							   boolean pAutoLinkReference,
							   boolean pUseRepresentationColumnsAsQueryColumns) throws DataSourceException
	{	
		//-------------------------------------------------------------------------
		// Calculate checksum as cache key
		//-------------------------------------------------------------------------
		
		oCacheKey = createCheckSum(pBeforeQueryColumns, 
				                   pQueryColumns, 
				                   pFromClause, 
				                   pWhereClause, 
				                   pAfterWhereClause,
				                   pWritebackTable, 
				                   pWritebackColumns, 
				                   Boolean.valueOf(pAutoLinkReference), 
				                   Boolean.valueOf(pUseRepresentationColumnsAsQueryColumns),
				                   ((DBAccess)dbAccess).getUrl(),
				                   ((DBAccess)dbAccess).getUsername());
		
		//-------------------------------------------------------------------------
		// MetaData detection
		//-------------------------------------------------------------------------

		try
		{
			ServerMetaData smdResult;
			
			if (isMetaDataCacheEnabled())
			{
				if (oCacheKey != null && htServerMetaDataCache != null)
				{
					validateMetaDataCache();
					
					smdResult = htServerMetaDataCache.get(oCacheKey);
				}
				else
				{
					smdResult = null;
				}
			}
			else
			{
				//clear internal cache if cache option is changed
				if (htServerMetaDataCache != null)
				{
					htServerMetaDataCache = null;
					kvlCacheLink = null;
				}
				
				smdResult = null;
			}
			
			if (smdResult != null)
			{
				//clone and create sub storages
				smdResult = smdResult.clone();
	
				//re-create sub storages
				
				ServerColumnMetaData[] scmd = smdResult.getServerColumnMetaData();
				
				DBStorage dbsLink;
				
				ForeignKey fk;
				
				for (int i = 0; i < scmd.length; i++)
				{
					fk = scmd[i].getLinkForeignKey();
					
					if (fk != null)
					{
						dbsLink = createAutomaticLinkStorage(fk.getPKTable().getQuotedName());
						dbsLink.setDefaultSort(new SortDefinition(fk.getLinkReferenceColumn().getName()));
						
						putSubStorage(fk.getPKTable().getName().toLowerCase(), dbsLink);
					}
				}
				
				//re-use from clause and query columns
				
				String sFrom = smdResult.getFromClause();
				
				if (sFrom != null)
				{
					setFromClause(sFrom);
				}
				
				String[] sColumns = smdResult.getQueryColumns();
				
				if (sColumns != null)
				{
					setQueryColumns(sColumns);
				}
				
				return smdResult;
			}
			else
			{
				Key              pk                  = null;
				List<Key>        auUniqueKeyColumns  = null;
				List<ForeignKey> auForeignKeys       = null;
				ArrayUtil<DBStorage> auSubStorages   = null;
				DBAccess dba = (DBAccess)dbAccess;
		
				if (pWhereClause == null)
				{
					pWhereClause = "1=2";
				}		
				else
				{
					pWhereClause += " AND 1=2";
				}
		
				Hashtable<String, Object> htDefaults = null;
				Hashtable<String, Object[]> htAllowed = null;
		
				ColumnMetaDataInfo mdInfoWritebackTable = null;
				if (pWritebackTable != null)
				{
					// check if the WritebackTable is a Synonym, if so use the real table instead.
					String sRealWritebackTable = dba.getTableForSynonym(pWritebackTable);
					
					// no Where Clause for determining meta data just use 1=2
					mdInfoWritebackTable = dba.getColumnMetaData(sRealWritebackTable, null, null, "1=2", null, null, null);
					
					String sCatalog = mdInfoWritebackTable.getCatalog();
					String sSchema  = mdInfoWritebackTable.getSchema();
	
					//cache information
					sWritebackTableCatalog = sCatalog;
					sWritebackTableSchema  = sSchema;
					
					debug("WritebackTable Schema=" + sSchema + ",Catalog=" + sCatalog);
					
					// #136 - Mysql PK refetch not working -> extract the plan table without schema. 
					String sTableWithoutSchema = mdInfoWritebackTable.getTable();
		
					if (isDefaultValue())
					{
						htDefaults = dba.getDefaultValues(sCatalog, sSchema, sTableWithoutSchema);
					}
					
					ServerColumnMetaData[] cmdWritebackColumns = mdInfoWritebackTable.getColumnMetaData();
					
					if (isAllowedValues())
					{
						htAllowed = dba.getAllowedValues(sCatalog, sSchema, sTableWithoutSchema);
		
						//#211
						//Set default allowed values, if available, to columnw which does not have default values 
						Object[] oAllowed;
						
						for (int i = 0, anz = cmdWritebackColumns.length; i < anz; i++)
						{
							if (htAllowed == null || htAllowed.get(cmdWritebackColumns[i].getName()) == null)
							{
								oAllowed = dba.getDefaultAllowedValues(sCatalog, sSchema, sTableWithoutSchema, cmdWritebackColumns[i]);
								
								if (oAllowed != null)
								{
									if (htAllowed == null)
									{
										htAllowed = new Hashtable<String, Object[]>();
									}
									
									htAllowed.put(cmdWritebackColumns[i].getName(), oAllowed);
								}
							}
						}
					}
					
					pk = dba.getPK(sCatalog, sSchema, sTableWithoutSchema);
					auUniqueKeyColumns  = dba.getUKs(sCatalog, sSchema, sTableWithoutSchema);
					
					String[] saInternalWritebackColumns = new String[cmdWritebackColumns.length];
					BeanUtil.toArray(cmdWritebackColumns, saInternalWritebackColumns, "quotedName");
					
					if (pWritebackColumns == null)
					{
						pWritebackColumns = saInternalWritebackColumns;			
					}
					
					if (pAutoLinkReference)
					{
						auForeignKeys = dba.getFKs(sCatalog, sSchema, sTableWithoutSchema);
						auSubStorages = new ArrayUtil<DBStorage>(auForeignKeys.size());
						
						if (auForeignKeys != null)
						{
							ForeignKey fkForeignKey;
							
							for (Iterator<ForeignKey> it = auForeignKeys.iterator(); it.hasNext();)
							{
								fkForeignKey = it.next();
								
								// setNullable of Foreign Key
								Name[] saFKColumns = fkForeignKey.getFKColumns();
								boolean bFound = false;
								for (int j = 0, k; j < saFKColumns.length && !bFound; j++)
								{
									k = ArrayUtil.indexOf(saInternalWritebackColumns, saFKColumns[j].getQuotedName());
									if (k >= 0 && cmdWritebackColumns[k].isNullable())
									{
										fkForeignKey.setNullable(true);
										bFound = true;
									}
								}
								
								// create sub storage.
								DBStorage dbsFKLink = createAutomaticLinkStorage(fkForeignKey.getPKTable().getQuotedName());
								
								//-------------------------------------------------------------------------
								// USE the representation columns from the link table for detecting
								// information columns which we can ADD to the master table
								//-------------------------------------------------------------------------
			
								// fill in RepresentationColumns
								
								ServerMetaData mdb = dbsFKLink.mdServerMetaData;
								String[] saPKTableRepresentationColumns = mdb.getRepresentationColumnNames();	
								
								Name   nFirstNonPKColumn = null;
								String sColName;
								
								// USE the first TEXT column from the link table 
								ServerColumnMetaData[] cmda = mdb.getServerColumnMetaData();
								String[] saForeignKeyPKColumns = BeanUtil.toArray(fkForeignKey.getPKColumns(), 
										new String[fkForeignKey.getPKColumns().length], "name");
								bFound = false;
								for (int j = 0; j < cmda.length && !bFound; j++)
								{
									sColName = cmda[j].getName();
									
									if (cmda[j].getColumnMetaData().getDataType() == StringDataType.TYPE_IDENTIFIER
										&& ArrayUtil.indexOf(saForeignKeyPKColumns, sColName) < 0 // not in PK!!!!
										&& ArrayUtil.indexOf(saPKTableRepresentationColumns, sColName) >= 0)
									{
				                		fkForeignKey.setLinkReferenceColumn(cmda[j].getColumnName());
				                		
				                		bFound = true;
									}
									else if (nFirstNonPKColumn == null && ArrayUtil.indexOf(saForeignKeyPKColumns, sColName) < 0)
									{
										//detect the first non PK column
										nFirstNonPKColumn = cmda[j].getColumnName();
									}
								}
								
								// We could not find at least one TEXT column, we will use the first NON PK column, if available
								// No column available -> no column will be added to the master table (see later)
								
								if (!bFound && nFirstNonPKColumn != null)
								{
									fkForeignKey.setLinkReferenceColumn(nFirstNonPKColumn);
								}
								
								if (fkForeignKey.getLinkReferenceColumn() == null)
								{
									// no relevant AutoLink column found, so remove the ForeignKey from the list and 
									// don't use it for the AutoLinkReference feature!
	
									it.remove();
								}
								else
								{
									// #225: setDefaultSort for AutoLinkReference Storages changed to the AutoLinkReference column!      
									dbsFKLink.setDefaultSort(new SortDefinition(fkForeignKey.getLinkReferenceColumn().getName()));
									
									auSubStorages.add(dbsFKLink);
								}
							}
						}
						
						debug("auForeignKeys = ", auForeignKeys);				
						
						//-------------------------------------------------------------------------
						// Update from clause and query columns with the link table information
						// (if the user didn't set this properties)
						//-------------------------------------------------------------------------
		
						if (pFromClause == null || pFromClause.equals(pWritebackTable))
						{				
							if (getQueryColumns() == null)
							{
								pQueryColumns = dba.getQueryColumns(pWritebackTable, auForeignKeys, pWritebackColumns);
								setQueryColumns(pQueryColumns);
							}					
		
							pFromClause = dba.getFromClause(pWritebackTable, auForeignKeys);
							setFromClause(pFromClause);		
						}
					}
					
					if (pFromClause == null)
					{
						pFromClause = pWritebackTable;
					}
				}		
		
				smdResult = new ServerMetaData(); 
				
				//No writeback table -> No writeback feature!
				if (pWritebackTable == null)
				{
					smdResult.getMetaData().removeFeature(Feature.WriteBack);
				}
				
				// Use Where Clause for determining meta data
				// This ensures that in case of wrong definition an error will occur on open.
				// Defining complex selects, queries will be executed faster.
				ColumnMetaDataInfo mdInfo = dba.getColumnMetaData(pFromClause, pQueryColumns, 
					                                              pBeforeQueryColumns, pWhereClause, pAfterWhereClause, 
					                                              pWritebackTable, pWritebackColumns);
		
				// set Auto increment array, default and allowed values
				
				ServerColumnMetaData[] cmda = mdInfo.getColumnMetaData();
				
				ServerColumnMetaData[] scmdWritebackCols = null;
				if (mdInfoWritebackTable != null)
				{
					scmdWritebackCols = mdInfoWritebackTable.getColumnMetaData();
				}
				
			    smdResult.setServerColumnMetaData(cmda);
			    ArrayUtil<Name> saAutoIncrementCols = new ArrayUtil<Name>(); 
			    
				for (int i = 0; i < cmda.length; i++)
				{
					//collect auto increment columns
					if (cmda[i].isAutoIncrement())
					{
						saAutoIncrementCols.add(cmda[i].getColumnName());
					}
					
					//set default values
					if (htDefaults != null)
					{
						cmda[i].setDefaultValue(htDefaults.get(cmda[i].getName()));
					}
					
					//set allowed values
					if (htAllowed != null)
					{
						Object[] oAllowed = htAllowed.get(cmda[i].getName());
						
						//use only if values are set!
						if (oAllowed != null && oAllowed.length > 0)
						{
							cmda[i].setAllowedValues(oAllowed);
						}
					}
					
					// #309 - isNullable will be determined from the writebackTable instead of the fromClause 
					if (scmdWritebackCols != null)
					{
						for (int j = 0; j < scmdWritebackCols.length; j++)
						{
							if (scmdWritebackCols[j].getName().equals(cmda[i].getName()))
							{
								cmda[i].setNullable(scmdWritebackCols[j].isNullable());
							}
						}					
					}
				}
				
				if (!saAutoIncrementCols.isEmpty())
				{
					smdResult.setAutoIncrementColumnNames(saAutoIncrementCols.toArray(new Name[saAutoIncrementCols.size()]));
				}
				
				if (pWritebackTable != null)
				{
					//----------------------------------------------------------------
					// PRIMARY KEY detection
					//----------------------------------------------------------------
					
					if (pk != null 
						&& pk.getColumns() != null 
						&& pk.getColumns().length > 0)
					{
						smdResult.setPrimaryKeyColumnNames(pk.getColumns());
						smdResult.setPrimaryKeyType(PrimaryKeyType.PrimaryKeyColumns);
					}
					else
					{
						if (auUniqueKeyColumns != null && auUniqueKeyColumns.size() > 0)
						{
							Name[] ukColumns = auUniqueKeyColumns.get(0).getColumns(); 
							smdResult.setPrimaryKeyColumnNames(ukColumns);
							
							smdResult.setPrimaryKeyType(PrimaryKeyType.UniqueKeyColumns);
						}
						else
						{
							// found no PK -> use all columns as PK
	
							ArrayUtil<Name> auColumns = new ArrayUtil<Name>();
							
							Name nWriteback;
							
							boolean bFound;
							
							for (int i = 0; i < pWritebackColumns.length; i++)
							{
								bFound = false;
								
								nWriteback = new Name(pWritebackColumns[i], dba.quote(pWritebackColumns[i]));
								
								for (int j = 0; j < cmda.length && !bFound; j++)
								{
									bFound = cmda[j].getColumnName().equals(nWriteback)
											 && cmda[j].getColumnMetaData().getDataType() == BinaryDataType.TYPE_IDENTIFIER; 
								}
								
								if (!bFound)
								{
									auColumns.add(nWriteback);
								}
							}
							
							if (!auColumns.isEmpty())
							{
								smdResult.setPrimaryKeyColumnNames(auColumns.toArray(new Name[auColumns.size()]));
								smdResult.setPrimaryKeyType(PrimaryKeyType.AllColumns);
							}
						} 
					}
					
					//----------------------------------------------------------------
					// UNIQUE KEY columns as representation columns
					//----------------------------------------------------------------
					
					if (auUniqueKeyColumns != null && auUniqueKeyColumns.size() > 0)
					{
						ArrayUtil<Name> auColumns = new ArrayUtil<Name>();
						
						for (int i = 0; i < auUniqueKeyColumns.size(); i++)
						{
							Name[]    ukColumns   = auUniqueKeyColumns.get(i).getColumns(); 
							
							for (int j = 0; j < ukColumns.length; j++)
							{
								// #226: No duplicate columns anymore in representation columns, if it will be build from intersecting UKs  					
								if (auColumns.indexOf(ukColumns[j]) < 0)
								{
									auColumns.add(ukColumns[j]);
								}
							}
						}
		
						if (auColumns.size() > 0)
						{
							Name[] saColumns = new Name[auColumns.size()];
							auColumns.toArray(saColumns);
						
							smdResult.setRepresentationColumnNames(saColumns);
						}
					}
									
					//----------------------------------------------------------------
					// ADD automatic link table reference definition
					//----------------------------------------------------------------
		
					if (pAutoLinkReference)
					{
						String sLinkReferenceColumn;
						
						if (auForeignKeys != null)
						{
							// check if the column name is in one of the automatic link tables, then set a LinkReference to it		
							for (int i = 0, anz = auForeignKeys.size(); i < anz; i++)
							{
								ForeignKey fkForeignKey = auForeignKeys.get(i);
								
								try
								{
									sLinkReferenceColumn = fkForeignKey.getLinkReferenceColumn().getName();
									
									//configure the link-column, if available
									String sRenamedLinkReferenceColumn = dba.getAutomaticLinkColumnName(fkForeignKey);
									
									//TODO [RH] Link auch auf alle FK Columns hngen!!!
									//fkForeignKey.getFKColumns()
									
									ServerColumnMetaData cmd = smdResult.getServerColumnMetaData(sRenamedLinkReferenceColumn);
									cmd.setLabel(ColumnMetaData.getDefaultLabel(sLinkReferenceColumn));
									
									String[] saColumns = ArrayUtil.add(
											BeanUtil.toArray(fkForeignKey.getFKColumns(), new String[fkForeignKey.getFKColumns().length], "name"),
											sRenamedLinkReferenceColumn);
									String[] saRefColumns = ArrayUtil.add(
											BeanUtil.toArray(fkForeignKey.getPKColumns(), new String[fkForeignKey.getPKColumns().length], "name"), 
											sLinkReferenceColumn);
		
									String sSubName = fkForeignKey.getPKTable().getName().toLowerCase();
									
									//Be careful: subStorages is a method name!!!
									StorageReferenceDefinition srdLink = new StorageReferenceDefinition(saColumns, ".subStorages." + sSubName, saRefColumns);
			
									IStorage isSub = auSubStorages.get(i);
									
									cmd.setLinkReference(srdLink);
									cmd.setNullable(fkForeignKey.isNullable());
									cmd.setLinkForeignKey(fkForeignKey);
									
									putSubStorage(sSubName, isSub);
								}
								catch (ModelException modelException)
								{
									// column not found or StorageReferenceDefinition couldn't created
									String[] cols = new String[smdResult.getServerColumnMetaData().length];
									debug(Arrays.toString(BeanUtil.toArray(smdResult.getServerColumnMetaData(), cols, "name")));
									debug(modelException);
								}
							}
						}
					}
				}
				
				//no representation columns available -> use all available columns as representation columns
				if (smdResult.getRepresentationColumnNames() == null)
				{
					Name[] nRepCols = new Name[smdResult.getServerColumnMetaData().length]; 
					for (int k = 0, iSize = nRepCols.length; k < iSize; k++)
					{
						nRepCols[k] = smdResult.getServerColumnMetaData()[k].getColumnName();
					}
					smdResult.setRepresentationColumnNames(nRepCols);
				}
				
				if (pUseRepresentationColumnsAsQueryColumns)
				{
					//use representation columns
					
					String[] saUnsortedQueryColumns = smdResult.getRepresentationColumnNames(); 
					String[] sPKColNames = smdResult.getPrimaryKeyColumnNames();
					
					if (sPKColNames != null)
					{
						//add all PK columns, if not already included
		
						for (int j = 0; j < sPKColNames.length; j++)
						{
							if (ArrayUtil.indexOf(saUnsortedQueryColumns, sPKColNames[j]) < 0)
							{
								saUnsortedQueryColumns = ArrayUtil.add(saUnsortedQueryColumns, sPKColNames[j]);
							}
						}
					}
					
					// remove all other columns from the ColumnMetaData array and sort the query columns to fit the 
					// MetaData order
					
					ServerColumnMetaData[] cmd = smdResult.getServerColumnMetaData();
					for (int i = cmd.length - 1; i >= 0; i--)
					{
						if (ArrayUtil.indexOf(saUnsortedQueryColumns, cmd[i].getName()) < 0)
						{
							cmd = ArrayUtil.remove(cmd, i);
						}
					}
					
					//change the column order of the query columns to fit the metadata column order
					//otherwise we have problems on the client
	
					saQueryColumns = new String[cmd.length];
					
					for (int i = 0; i < cmd.length; i++)
					{
						saQueryColumns[i] = cmd[i].getColumnName().getQuotedName();
					}
					
					smdResult.setServerColumnMetaData(cmd);
				}
	
				//IMPORTANT for cache!
				smdResult.setFromClause(getFromClause());
				smdResult.setQueryColumns(getQueryColumns());
				
				if (htServerMetaDataCache == null)
				{
					htServerMetaDataCache = new Hashtable<Object, ServerMetaData>();
				}
				
				htServerMetaDataCache.put(oCacheKey, smdResult);
				
				updateMetaDataCache(smdResult);
				
				return smdResult;
			}
		}
		catch (DataSourceException dse)
		{
			updateMetaDataCache(null);
			
			if (htServerMetaDataCache != null)
			{
				htServerMetaDataCache.remove(oCacheKey);
			}

			throw dse;
		}
		catch (RuntimeException re)
		{
			updateMetaDataCache(null);

			if (htServerMetaDataCache != null)
			{
				htServerMetaDataCache.remove(oCacheKey);
			}
			
			throw re;
		}
	}	
		
	/**
	 * Updates the meta data cache with current meta data. The cache link will be used to find
	 * updatable entities.
	 * 
	 * @param pMetaData the meta data
	 */
	private void updateMetaDataCache(ServerMetaData pMetaData)
	{
		if (kvlCacheLink != null)
		{
			List<CacheLink> liLinks = kvlCacheLink.get(oCacheKey);
			
			if (liLinks != null)
			{
				for (CacheLink link : liLinks)
				{
					putMetaDataToCache(link.sGroup, link.sName, (pMetaData != null ? pMetaData.getMetaData() : null));
				}
			}
		}
	}
	
	/**
	 * Checks if the internal cached metadata are also available in the global cache. If that is not the case,
	 * the cache was cleared and we should cleanup our internal cache too.<p/>
	 * It is possible that the global cache is cleared, because it is a static cache. In that case we don't have
	 * the info that the cache is cleared, but we have the chance to check if our cached object is in the global
	 * cache.
	 */
	private void validateMetaDataCache()
	{
		if (kvlCacheLink != null)
		{
			List<CacheLink> liLinks = kvlCacheLink.get(oCacheKey);
			
			if (liLinks != null)
			{
				for (CacheLink link : liLinks)
				{
					if (getMetaDataFromCache(link.sGroup, link.sName) == null)
					{
						kvlCacheLink.remove(oCacheKey);
						htServerMetaDataCache.remove(oCacheKey);
					}
				}
			}
		}
	}
	
	/**
	 * Returns if the DBStorage is open.
	 * 
	 * @return true if the DBStorage is open.
	 */
	public boolean isOpen()
	{
		return bIsOpen;
	}
	
	/**
	 * It close the DBStorage.
	 */
	public void close()
	{
		oCacheKey = null;
		
		bIsOpen = false;
	}
	
	/**
	 * Set the <code>IDBAccess</code> of this <code>DBStorage</code> .
	 * 
	 * @param pDBAccess
	 *            the <code>IDBAccess</code> of this <code>DBStorage</code> .
	 */
	public void setDBAccess(IDBAccess pDBAccess)
	{
		dbAccess = pDBAccess;
	}

	/**
	 * Returns the <code>IDBAccess</code> of this <code>DBStorage</code> .
	 * 
	 * @return the <code>IDBAccess</code> of this <code>DBStorage</code> .
	 */
	public IDBAccess getDBAccess()
	{
		return dbAccess;
	}
	
	/**
	 * Returns the query tables to use in the SELECT statement to get the data from the storage.
	 *
	 * @return the query tables to use in the SELECT statement to get the data from the storage.
	 */
	public String getFromClause()
	{
		return sFromClause;
	}

	/**
	 * Returns the query tables to use in the SELECT statement to get the data from the storage.
	 *
	 * @return the query tables to use in the SELECT statement to get the data from the storage.
	 */
	private String getFromClauseIntern()
	{
		if (sFromClause == null)
		{
			return sWritebackTable;
		}
		return sFromClause;
	}
	
	/**
	 * Sets the query tables to use in the SELECT statement to get the data from the storage.
	 *
	 * @param pFromClause 
	 *			the queryTables to set.
	 */
	public void setFromClause(String pFromClause)
	{
		sFromClause = pFromClause;
	}

	/**
	 * Returns the query columns of the SELECT statement.
	 *
	 * @return the query columns of the SELECT statement.
	 */
	public String[] getQueryColumns()
	{
		return saQueryColumns;
	}

	/**
	 * Sets the query columns of the SELECT statement.
	 *
	 * @param pQueryColumns 
	 *			the queryColumns to set
	 */
	public void setQueryColumns(String[] pQueryColumns)
	{
		saQueryColumns = pQueryColumns;
	}

	/**
	 * Returns the string to place in the SELECT statement between the SELECT and the first query column.
	 *
	 * @return the string to place in the SELECT statement between the SELECT and the first query column.
	 */
	public String getBeforeQueryColumns()
	{
		return sBeforeQueryColumns;
	}

	/**
	 * Sets the string to place in the SELECT statement between the SELECT and the first query column.
	 *
	 * @param pBeforeQueryColumns 
	 *			the beforeQueryColumns to set
	 */
	public void setBeforeQueryColumns(String pBeforeQueryColumns)
	{
		sBeforeQueryColumns = pBeforeQueryColumns;
	}

	/**
	 * Returns the string to place in the SELECT statement after the last WHERE condition from the
	 * Filter or MasterReference (Master-Detail Condition).
	 *
	 * @return the string to place in the SELECT statement after the last WHERE condition from the
	 *  	   Filter or MasterReference (Master-Detail Condition).
	 */
	public String getWhereClause()
	{
		return sWhereClause;
	}

	/**
	 * Sets the string to place in the SELECT statement after the last WHERE condition from the
	 * Filter or MasterReference (Master-Detail Condition).
	 *
	 * @param pWhereClause 
	 *			the last Where Condition to set
	 */
	public void setWhereClause(String pWhereClause)
	{
		sWhereClause = pWhereClause;
	}

	/**
	 * Returns the string to place in the SELECT statement after the WHERE clause and before the ORDER BY clause.
	 *
	 * @return the string to place in the SELECT statement after the WHERE clause and before the ORDER BY clause.
	 */
	public String getAfterWhereClause()
	{
		return sAfterWhereClause;
	}

	/**
	 * Sets the string to place in the SELECT statement after the WHERE clause and before the ORDER BY clause.
	 * e.g. GROUP BY a, HAVING ...
	 *
	 * @param pAfterWhereClause 
	 *			the afterWhereClause to set
	 */
	public void setAfterWhereClause(String pAfterWhereClause)
	{
		sAfterWhereClause = pAfterWhereClause;
	}

	/**
	 * Returns the list of write back columns to use in the INSERT, UPDATE statements.
	 *
	 * @return the saWritebackColumns.
	 */
	public String[] getWritebackColumns()
	{
		return saWritebackColumns;
	}

	/**
	 * Sets the list of write back columns to use in the INSERT, UPDATE statements.
	 *
	 * @param pWritebackColumns 
	 *			the pWritebackColumns to set
	 */
	public void setWritebackColumns(String[] pWritebackColumns)
	{
		this.saWritebackColumns = pWritebackColumns;
	}

	/**
	 * Returns the WritebackTable, which is used for the INSERT, UPDATE, DELETE calls.
	 *
	 * @return the sWritebackTable.
	 */
	public String getWritebackTable()
	{
		return sWritebackTable;
	}

	/**
	 * Sets the WritebackTable, which is used for the INSERT, UPDATE, DELETE calls.
	 *
	 * @param pWritebackTable 
	 *			the pWritebackTable to set
	 */
	public void setWritebackTable(String pWritebackTable)
	{
		sWritebackTable = pWritebackTable;
	}	
	
	/**
	 * Returns the default sort. It is used, if no other sort is defined.
	 *
	 * @return the default sort.
	 */
	public SortDefinition getDefaultSort()
	{
		return sdDefaultSort;
	}

	/**
	 * Sets the default sort. It is used, if no other sort is defined.
	 *
	 * @param pDefaultSort the default sort.
	 */
	public void setDefaultSort(SortDefinition pDefaultSort)
	{
		sdDefaultSort = pDefaultSort;
	}	
	
	/**
	 * Returns the restrict condition. It is always added with And to any given condition.
	 *
	 * @return the restrict condition.
	 */
	public ICondition getRestrictCondition()
	{
		return cRestrictCondition;
	}

	/**
	 * Sets the restrict condition. It is always added with And to any given condition.
	 *
	 * @param pRestrictCondition the restrict condition.
	 */
	public void setRestrictCondition(ICondition pRestrictCondition)
	{
		cRestrictCondition = pRestrictCondition;
	}	
	
	/**
	 * Returns if this DBStorage refetchs after insert and update.
	 *
	 * @return if this DBStorage refetchs after insert and update.
	 */
	public boolean isRefetch()
	{
		return bRefetch;
	}

	/**
	 * Sets if this DBStorage refetchs after insert and update.
	 *
	 * @param pRefetch <code>true</code> if this DBStorage refetchs after insert and update.
	 */
	public void setRefetch(boolean pRefetch)
	{
		bRefetch = pRefetch;
	}		
	
	/**
	 * Sets if the automatic link reference detection is en- or disabled. The automatic link reference is defined
	 * through a foreign key reference in the database.
	 *
	 * @param pAutoLinkReference true if the automatic LinkReference mode is on, <code>false</code> to disable 
	 *                           auto link reference mode
	 */
	public void setAutoLinkReference(boolean pAutoLinkReference)
	{
		bAutoLinkReference = Boolean.valueOf(pAutoLinkReference);
	}	
	
	/**
	 * Returns if the automatic LinkReference mode is on or off.	
	 *
	 * @return <code>true</code>if the automatic LinkReference mode is on, otherwise <code>false</code>
	 * @see #setAutoLinkReference(boolean)
	 */
	public boolean isAutoLinkReference()
	{
		if (bAutoLinkReference == null)
		{
			return bDefaultAutoLinkReference;
		}
		
		return bAutoLinkReference.booleanValue();
	}
	
	/**
	 * Sets the default automatic link reference detection mode. The automatic link reference is defined
	 * through a foreign key reference in the database.
	 * 
	 * @param pDefaultAutoLinkReference the default automatic LinkReference mode to use.
	 */
	public static void setDefaultAutoLinkReference(boolean pDefaultAutoLinkReference)
	{
		bDefaultAutoLinkReference = pDefaultAutoLinkReference;
	}

	/**
	 * Returns the default automatic LinkReference mode.
	 * 
	 * @return the default automatic LinkReference mode.
	 * @see #setDefaultAutoLinkReference(boolean)
	 */
	public static boolean isDefaultAutoLinkReference()
	{
		return bDefaultAutoLinkReference;
	}
	
	/**
	 * Sets the default value detection en- or disabled. The default value is a default value
	 * for a column in the database
	 * 
	 * @param pDefaultValue <code>true</code> to enable the default value detection
	 */
	public void setDefaultValue(boolean pDefaultValue)
	{
		bDefaultValue = Boolean.valueOf(pDefaultValue);
	}
	
	/**
	 * Returns the default value detection mode.
	 * 
	 * @return <code>true</code> if the default value detection is enabled, <code>false</code> otherwise
	 * @see #setDefaultValue(boolean)
	 */
	public boolean isDefaultValue()
	{
		if (bDefaultValue == null)
		{
			return bDefaultDefaultValue;
		}
		
		return bDefaultValue.booleanValue();
	}

	/**
	 * Sets the default - default value detection en- or disabled. The default value is a default value
	 * for a column in the database.
	 * 
	 * @param pDefaultValue <code>true</code> to enable the default - default value detection
	 */
	public static void setDefaultDefaultValue(boolean pDefaultValue)
	{
		bDefaultDefaultValue = pDefaultValue;
	}
	
	/**
	 * Returns the default - default value detection mode.
	 * 
	 * @return <code>true</code> if the default - default value detection is enabled, <code>false</code> otherwise
	 * @see #setDefaultDefaultValue(boolean)
	 */
	public static boolean isDefaultDefaultValue()
	{
		return bDefaultDefaultValue;
	}

	/**
	 * Sets the allowed value detection en- or disabled. An allowed value is defined in the database
	 * through check constraints.
	 * 
	 * @param pAllowedValues <code>true</code> to enable the allowed value detection, <code>false</code> otherwise
	 */
	public void setAllowedValues(boolean pAllowedValues)
	{
		bAllowedValues = Boolean.valueOf(pAllowedValues);
	}
	
	/**
	 * Returns the allowed value detection mode.
	 * 
	 * @return <code>true</code> if the allowed value detection is enabled, <code>false</code> otherwise
	 * @see #setAllowedValues(boolean)
	 */
	public boolean isAllowedValues()
	{
		if (bAllowedValues == null)
		{
			return bDefaultAllowedValues;
		}
		
		return bAllowedValues.booleanValue();
	}
	
	/**
	 * Sets the default allowed value detection en- or disabled. An allowed value is defined in the database
	 * through check constraints.
	 * 
	 * @param pAllowedValues <code>true</code> to enable the default allowed value detection, <code>false</code> otherwise
	 */
	public static void setDefaultAllowedValues(boolean pAllowedValues)
	{
		bDefaultAllowedValues = pAllowedValues;
	}
	
	/**
	 * Returns the default allowed value detection mode.
	 * 
	 * @return <code>true</code> if the default allowed value detection is enabled, <code>false</code> otherwise
	 * @see #setDefaultAllowedValues(boolean)
	 */
	public boolean isDefaultAllowedValues()
	{
		return bDefaultAllowedValues;
	}
	
	/**
	 * Gets the catalog name of the writeback table. 
	 * 
	 * @return the detected catalog name or <code>null</code> if no writeback table is set
	 */
	protected String getWritebackTableCatalog()
	{
		return sWritebackTableCatalog;
	}
	
	/**
	 * Gets the schema name of the writeback table. 
	 * 
	 * @return the detected schema name or <code>null</code> if no writeback table is set
	 */
	protected String getWritebackTableSchema()
	{
		return sWritebackTableSchema;
	}
	/**
	 * Refetch and optional locks the specified DataRow via PK.
	 * 
	 * @param pDataRow	the DataRow to refetch
	 * @param pLock		true if the row is locked before. 
	 * @return the refetched DataRow
	 * @throws DataSourceException if the refetch fails
	 */
	protected Object[] refetchRow(Object[] pDataRow, boolean pLock) throws DataSourceException 
	{
		if (pDataRow == null)
		{
			return null;
		}
		if (getWritebackTable() == null)
		{
			return null;
		}

		// construct Filter over PK cols...
		ICondition cFilter = Filter.createEqualsFilter(mdServerMetaData.getPrimaryKeyColumnNames(), 
				pDataRow, mdServerMetaData.getMetaData().getColumnMetaData());
		
		// lock row in storage
		if (pLock)
		{
			dbAccess.lockRow(getFromClauseIntern(), mdServerMetaData, cFilter);
			
			//TODO [RH] Optimize return also the values in same step!!! Attention with AutoCommit
		}
		
		// refetch data
		List<Object[]> olResult = dbAccess.fetch(mdServerMetaData, sBeforeQueryColumns, saQueryColumns, getFromClauseIntern(),
									             cFilter, sWhereClause, sAfterWhereClause, 
									             null, 0, 2);

		if (olResult != null && olResult.size() == 2 && olResult.get(1) == null)
		{
			return olResult.get(0);
		}
		return null;	
	}

	/**
	 * Gets an array of column names as defined in the metadata.
	 * 
	 * @return the column names as array
	 * @see #getMetaData()
	 */
	public String[] getColumnNames()
	{
		return mdServerMetaData.getColumnNames();
	}
	
	/**
	 * Gets all known sub storages as key / value pair.
	 * 
	 * @return the key / value pair with subtablename and substorage or <code>null</code> if no substorages are
	 *         known
	 */
	public Map<String, IStorage> getSubStorages()
	{
		return hmpSubStorages;
	}
	
	/**
	 * Creates a new <code>DBStorage</code> which is configured for automatic link cell editors. The auto link
	 * reference feature is disabled for this storage.
	 * 
	 * @param pFrom The from clause
	 * @return the newly created storage
	 * @throws DataSourceException if the from clause causes errors or the metadata are not available
	 */
	protected DBStorage createAutomaticLinkStorage(String pFrom) throws DataSourceException
	{
		DBStorage dbs = new DBStorage();
		dbs.setDBAccess(dbAccess);
		dbs.setWritebackTable(pFrom);
		dbs.setAutoLinkReference(false);
		dbs.openInternal(true);		
				
		return dbs;
	}
	
	/**
	 * Creates a MD5 checksum with given objects.
	 * 
	 * @param pObjects the objects for the checksum calculation
	 * @return the MD5 checksum
	 */
	protected Object createCheckSum(Object... pObjects)
	{
		try
		{
			SecureHash sh = new SecureHash("MD5");
			
			for (int i = 0; i < pObjects.length; i++)
			{
				if (pObjects[i] != null)
				{
					if (pObjects[i] instanceof String)
					{
						sh.add(((String)pObjects[i]).getBytes());
					}
					else if (pObjects[i] instanceof String[])
					{
						for (String sElement : (String[])pObjects[i])
						{
							if (sElement != null)
							{
								sh.add(sElement.getBytes());
							}
						}
					}
					else if (pObjects[i] instanceof Boolean)
					{
						if (Boolean.TRUE == pObjects[i])
						{
							sh.add(new byte[] {0x1});
						}
						else
						{
							sh.add(new byte[] {0x0});
						}
					}
				}
				else
				{
					sh.add(new byte[] {0x0});
				}
			}
			
			return sh.getHash();
		}
		catch (NoSuchAlgorithmException nsae)
		{
			return null;
		}
	}
	
	//****************************************************************
	// Subclass definition
	//****************************************************************
	
	/**
	 * It stores all relevant information of the cached ResultSet for the fetch(...).
	 *  
	 * @author Ren Jahn
	 */
	private static final class CacheLink
	{
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Class members
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		
		/** the group identifier (lifecycle name). */
		private String sGroup;
		
		/** the storage name. */
		private String sName;
		
		/** the cached hash code. */
		private int iHashCode = -1;
		
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Initialization 
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
		 * Creates a new instance of <code>CacheLink</code>.
		 * 
		 * @param pGroup the group
		 * @param pName the name
		 */
		private CacheLink(String pGroup, String pName)
		{
			sGroup = pGroup;
			sName  = pName;
		}
		
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Overwritten methods
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
		 * {@inheritDoc}
		 */
		@Override
		public boolean equals(Object pObject)
		{
			if (pObject instanceof CacheLink)
			{
				return CommonUtil.equals(sGroup, ((CacheLink)pObject).sGroup)
				       && CommonUtil.equals(sName, ((CacheLink)pObject).sName);
			}
			
			return false;
		}
		
		/**
		 * {@inheritDoc}
		 */
		@Override
		public int hashCode()
		{
			if (iHashCode == -1)
			{
				iHashCode = (sGroup != null ? sGroup.hashCode() : 7) + (sName != null ? sName.hashCode() : 31);
			}
			
			return iHashCode;   
		}
		
	}	// CacheLink	
	
} 	// DBStorage
