/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [RH] - creation
 * 11.10.2008 - [RH] - bug fixes, toString() optimized, NO_SELECTED_ROW optimized
 * 12.10.2008 - [RH] - bug fixes, sync()
 *                     writebackIsolationsLevel support added
 * 02.11.2008 - [RH] - InsertEnabled, UpdateEnabled, DeleteEnabled, ReadOnly added
 *                     getRowCount, SelectedDataRow support added
 *                     comments added
 *                     bugs fixed
 * 11.11.2008 - [RH] - use instead SetSelectedRow -> SetSelectRowInternal with 
 *                     BEFORE & AFTER SELECTED Event-> bugs fixed
 *                     DeleteCascade added, notifyMasterReferenceColumnsChanged(IDataRow) add
 *                     MasterRow set in MemDataPage   
 * 17.11.2008 - [RH] - get/setSelectionMode added; DefaultDeselected removed   
 *                     delete bug with IDataBook.DATAROW WritebackIsolationLevel fixed  
 * 16.04.2009 - [RH] - remove/add/getDetailDataBooks moved to IDataBook.          
 *                     checks in open for PK columns and MasterReferenceDefinition added   
 * 27.04.2009 - [RH] - interface review changes tested and fixed
 * 28.04.2009 - [RH] - javadoc added, NLS removed         
 * 13.05.2009 - [RH] - sort/filter in memory added                           
 * 12.06.2009 - [JR] - toString: used StringBuilder [PERFORMANCE]
 * 02.07.2009 - [JR] - invokeDataBookListeners: finally block [BUGFIX]
 * 13.10.2009 - [RH] - WriteBackIsolationLevel.DATASOURCE is considered at saveAllDataRows() [BUGFIX]
 * 14.10.2009 - [RH] - In AfterDeleting event is the selected row corrected in WriteBackIsolationLevel.DATASOURCE [BUGFIX]
 * 15.10.2009 - [RH] - Insert over Master - Detail reference in WriteBackIsolationLevel.DATASOURCE fixed (UID missed) [BUGFIX]
 *            - [JR] - clearFilterSortInMemDataPages: isOutOfSync called
 *                   - clearFilterSortinMemDataPages renamed to clearFilterSortInMemDataPages
 * 19.03.2010 - [RH] - lock before delete only happens, if isLockAndRefetchEnabled() == true.
 * 25.03.2010 - [JR] - #92: createNewRow implemented
 * 21.07.2010 - [RH] - setTreePath = null if emptyPage is set.
 * 29.09.2010 - [RH] - countRows renamed to getEstimatedRowCount()
 * 15.10.2010 - [RH] - detailChanged renamed to notifyDetailChanged        
 *                     #186: notifyDetailedChanged changed, to set the DetailedChanged state correct. 
 *                           store(), restore() call now setDetailsChangedInMasterBook()       
 * 28.10.2010 - [RH] - #197: insert bug with DATASOURCE level with master/detail. reference columns doesn't copy to details.
 *                           rehash is moved for getDataPageIntern to notifyMasterChanged. UID is correct reset to null.
 *                           updates on null PK (reference columns) not supported anymore. Won't work correctly. 
 * 28.10.2010 - [HM] - open() sets the open state now correctly -> at the end of the open, and not before
 *                     setValues (null, xxx) uses now all columns as default.
 * 15.02.2011 - [RH] - #286: no DataPage to master row exists, then return null -> no changes can exits - performance tuning in getChangedDataRows()  
 * 			           #288: master DataRow rehash not working in DATASOUCE Level over more then 1 hierarchy ! - bug fixed.                  
 * 24.02.2011 - [RH] - #290: if the master DataBook is in DATASOURCE level && Inserting() no fetch is necessary, because no rows can exists!
 * 06.03.2011 - [JR] - #115: 
 *                     * open: check allowed values
 *                     * createNewRow: use default value    
 * 21.03.2011 - [RH] - #315: saveSelectedRow() bug with more then one leave with isInsertung state in DATASOURCE level
 * 31.03.2011 - [JR] - #318: setRowDefinition() clear/sets all controls to the row definition 
 * 31.03.2011 - [RH] - #163 - IChangeableDataRow should support isWritableColumnChanged 
 * 13.04.2011 - [RH] - #336 - First check if a DataPage with the UID exists. -> Reuse new DataPages in DATASOURCE level.
 * 12.05.2011 - [RH] - #347 - DataBook in DATASOURCE level should in notifyMasterChanged always check if rehash is required !
 *                     #348 - sync() fails if a seljoined MemDataBook has the TreePath wrong - null check added.
 * 13.05.2011 - [RH] - #350 - MemDataBook should remove all details in saveSelectedRow(), if a row is deleted 
 * 23.05.2011 - [RH] - #357 - setSort on MemDataBook with setMemSort(true), doesn't setSelectedRow correct after sort
 * 30.05.2011 - [HM] - #374 - If an insert happens, after that more rows have to be fetched, then wrong rows will be fetched
 * 30.05.2011 - [HM] - #375 - delete, cancelEditing missing 
 * 30.05.2011 - [HM] - #369 - delete all rows from a databook, Master changed has to be set.
 * 30.05.2011 - [HM] - #376 - restore row fails in DATASOURCE level with more then one detail level with isInserting rows 
 * 31.05.2011 - [RH] - #377 - MemDataBook has invalid javadoc 
 * 01.06.2011 - [RH] - #378 - Exception in AfterDeleted Event, if an insert() is called in the user event
 * 30.06.2011 - [RH] - #405 - NullPointer Exception in notifyMasterChanged(), if parent IDataBook out of sync
 * 14.07.2011 - [RH] - #424 - MemDataBook.setRowDefintion(null) should addControls
 *                     #425 - MemDataBook close should not close the detail DataBooks.
 * 31.10.2011 - [RH] - #492 - MemDataBook.restoreAllRows set wrong SelectedRow :: save the current row after the restore e.g is Inserting....
 * 02.11.2011 - [RH] - #495 - MemDataBook removeDataPage should setTreePath(null) 
 * 13.11.2011 - [RH] - #505 - delete should should check the WritebackIsolation level
 * 15.11.2011 - [RH] - #507 - restore in MemDataBook delete() has be done after the BEFORE_DELETING event
 * 17.11.2011 - [JR] - #509 - setReadOnlyWithoutSave implemented 
 * 10.12.2012 - [JR] - #615 - setFilter should not save in mem filter mode and datasource writeback isolation level
 * 19.12.2012 - [RH] - Code Review - setUpdating(), setDeleting(), setInserting(), setUID(), store(), restore(), setDetailChanged() moved to ChangeableDataRow
 *                                 - setValueInternal removed, functionality is moved to update()
 *                                 - Changes management in insert, update, delete, store, restore, setDetailChanged is moved to MemDataPage
 * 10.04.2013 - [RH] - #617 - saveAllDataBooks doesn't save all rows in DATASOURCE level - fixed
 * 10.04.2013 - [RH] - #617 - restoreAllRows fails with ArrayIndexOutOfBoundsException - fixed
 * 10.04.2013 - [RH] - #618 - restoreAllRows throws an Exception - fixed
 * 12.04.2013 - [RH] - #361 - Need isUpdate/Delete/InsertAllowed and isUpdate/Delete/InsertEnabled should only get what is set - realized!
 * 13.04.2013 - [RH] - #155 - Reload with SelectionMode==CURRENT and selfjoined tree's - fixed.  
 * 31.07.2013 - [RH] - #746 - setFilter/reload on a master databook, doesn't sync details
 * 03.09.2013 - [RH] - #783 - MemDataBook rehash causes Exception
 * 06.09.2013 - [RH] - #770 - MemDataBook: rehash in an inserting row does not work
 * 24.09.2013 - [RH] - #800 - MemDataBook ArrayIndexOutOfBoundsException during insert
 * 27.09.2013 - [RH] - #804 - MemDataBook for UITree with self-joined data
 */
package com.sibvisions.rad.model.mem;

import java.lang.ref.WeakReference;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.Hashtable;

import javax.rad.genui.UIFactoryManager;
import javax.rad.genui.celleditor.UIChoiceCellEditor;
import javax.rad.genui.celleditor.UILinkedCellEditor;
import javax.rad.model.ColumnDefinition;
import javax.rad.model.IChangeableDataRow;
import javax.rad.model.IDataBook;
import javax.rad.model.IDataPage;
import javax.rad.model.IDataRow;
import javax.rad.model.IDataSource;
import javax.rad.model.IRowDefinition;
import javax.rad.model.ModelException;
import javax.rad.model.RowDefinition;
import javax.rad.model.SortDefinition;
import javax.rad.model.TreePath;
import javax.rad.model.condition.Filter;
import javax.rad.model.condition.ICondition;
import javax.rad.model.event.DataBookEvent;
import javax.rad.model.event.DataBookHandler;
import javax.rad.model.event.DataBookEvent.ChangedType;
import javax.rad.model.reference.ReferenceDefinition;
import javax.rad.model.ui.IControl;
import javax.rad.ui.celleditor.IChoiceCellEditor;

import com.sibvisions.util.ArrayUtil;

/**
 * The <code>MemDataBook</code> is a storage independent table, and handles all operations
 * to load, save and manipulate table oriented data. <br>
 * An <code>MemDataBook</code> has at least one <code>MemDataPage</code> to store all 
 * <code>DataRow</code>'s of this <code>MemDataBook</code>. If an <code>MemDataBook</code> 
 * has detail <code>IDataBook</code>'s, it handles for each detail <code>IDataBook</code> 
 * the <code>IDataPage</code>'s. Thats necessary because every change of the selected
 * <code>IDataRow</code> needs to initiate the change of the corresponding details. 
 * So the <code>IDataBook</code> adds the appropriate <code>IDataPage</code>'s into the 
 * detail <code>IDataBook</code>'s.<br>
 * The <code>MemDataBook</code> is also a <code>IChangeableDataRow</code>, the selected row 
 * of the <code>MemDataBook</code>.
 * 
 * @see com.sibvisions.rad.model.mem.MemDataPage
 * @see javax.rad.model.IDataPage
 * @see javax.rad.model.IDataBook
 * @see javax.rad.model.IRowDefinition
 * @see javax.rad.model.IChangeableDataRow
 * @see javax.rad.model.IDataSource
 * 
 * @author Roland Hrmann
 */
public class MemDataBook extends ChangeableDataRow 
                         implements IDataBook 
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** 
	 * The internal unique for the MemDataBook. Its used for new DataRow's as identifier 
	 * till the get a unique Primary Key value. 
	 */
	private static int iUID = 0;
	
	
	/** The TreePath inside self joined IDataBook's. */
	private TreePath		treePath = new TreePath();
	
	/** It holds the current SelectionMode of the MemDataBook. Default is SelectionMode.CURRENT_ROW. */
	private SelectionMode   iSelectionMode = SelectionMode.CURRENT_ROW;
	
	/** The Filter of the MemDataBook. */
	private ICondition		cFilter;
	/** The Sort of the MemDataBook. */
	private SortDefinition	sSort;
	
	/** The ReferenceDefinition to the master IDataBook. */
	private ReferenceDefinition rdMasterReference;
	
	/** The ReferenceDefinition to the master IDataBook. */
	private ReferenceDefinition rdTreeRootReference;

	/** The ReferenceDefinition to the master IDataBook. */
	private String[] saTreeRootMasterColumnNames;

	/** The write back isolation level. */
	private WriteBackIsolationLevel	iWritebackIsolationLevel;
	
	/** The <code>ArrayUtil</code> of all detail <code>IDataBooks</code>. */
	private ArrayUtil<WeakReference<IDataBook>>	auDetailDataBooks;
	
	/** The DataSource of the MemDataBook. */
	private IDataSource	dsDataSource;

	/** It stores for the lazy reload, the last selected Row PK column values. */
	private ArrayUtil<IDataRow> auStoredSelection = new ArrayUtil<IDataRow>();;
	
	/** 
	 * The current MemDataPage of the MemDataBook. 
	 * It gets synchronized at the first use. (lazy load) 
	 */
	private MemDataPage	dpCurrentDataPage;
	
	/** The Empty MemDataPage, which is used as long no DataRows are available. */
	private MemDataPage	dpEmptyDataPage;	

	/** 
	 * The Hashtable of all MemDataPages of the MemDataBook. It holds for each master DataRow
	 * a MemDataPage. If the master DataRow is Inserting (new), the MemDataPage is stored
	 * under the UID. 
	 */
	private Hashtable<Object, MemDataPage>   htDataPagesCache   = new Hashtable<Object, MemDataPage>();
	
	/** The <code>EventHandler</code> for before selected event. */
	private DataBookHandler	eventBeforeRowSelected;
	/** The <code>EventHandler</code> for after selected event. */
	private DataBookHandler	eventAfterRowSelected;
	/** The <code>EventHandler</code> for before inserting event. */
	private DataBookHandler	eventBeforeInserting;
	/** The <code>EventHandler</code> for after inserting event. */
	private DataBookHandler	eventAfterInserting;
	/** The <code>EventHandler</code> for before inserted event. */
	private DataBookHandler	eventBeforeInserted;
	/** The <code>EventHandler</code> for after inserted event. */
	private DataBookHandler	eventAfterInserted;
	/** The <code>EventHandler</code> for before updating event. */
	private DataBookHandler	eventBeforeUpdating;
	/** The <code>EventHandler</code> for after updating event. */
	private DataBookHandler	eventAfterUpdating;
	/** The <code>EventHandler</code> for before updated event. */
	private DataBookHandler	eventBeforeUpdated;
	/** The <code>EventHandler</code> for after updated event. */
	private DataBookHandler	eventAfterUpdated;
	/** The <code>EventHandler</code> for before deleting event. */
	private DataBookHandler	eventBeforeDeleting;
	/** The <code>EventHandler</code> for after deleting event. */
	private DataBookHandler	eventAfterDeleting;
	/** The <code>EventHandler</code> for before deleted event. */
	private DataBookHandler	eventBeforeDeleted;
	/** The <code>EventHandler</code> for after deleted event. */
	private DataBookHandler	eventAfterDeleted;
	/** The <code>EventHandler</code> for before restore event. */
	private DataBookHandler	eventBeforeRestore;
	/** The <code>EventHandler</code> for after restore event. */
	private DataBookHandler	eventAfterRestore;
	/** The <code>EventHandler</code> for before reload event. */
	private DataBookHandler	eventBeforeReload;
	/** The <code>EventHandler</code> for after reload event. */
	private DataBookHandler	eventAfterReload;
	/** The <code>EventHandler</code> for before selected event. */
	private DataBookHandler	eventBeforeColumnSelected;
	/** The <code>EventHandler</code> for after selected event. */
	private DataBookHandler	eventAfterColumnSelected;
	
	/** Name of the MemDataBook. */
	private String			sName;
	/** The current selected row index. */
	private String			sSelectedColumn	= null;
	
	/**
	 * The read a head row count of the <code>RemoteDataBook</code>. This means how many rows are
	 * read ahead in the <code>MemDataPage</code>.
	 */
	private int	iReadAheadRowCount	= 10;
	/** The current selected row index. */
	private int	iSelectedRowIndex	= -1;

	/** 
	 * It determines that the master has changed and this IDataBoolk needs to synchronize
	 * to the current MemDataPage and selected row depending on the Selection Mode.
	 */
	private boolean 		bMasterChanged 	= false;
	/** It determines that insert()'s are allowed. */
	private boolean			bInsertEnabled 	= true;
	/** It determines that update()'s (setValueXXX) are allowed. */
	private boolean			bUpdateEnabled 	= true;
	/** It determines that delete()'s are allowed. */
	private boolean			bDeleteEnabled 	= true;
	/** It determines that lock on refetch is enabled. */
	private boolean			bLockAndRefetchEnabled = false;
	/** It determines if the MemDataBook is read only. ->all change operations are allowed. */
	private boolean			bReadOnly 		= false;
	/** It defines if all details will be before delete cascade. */
	private boolean         bDeleteCascade 	= true;
	/** Determines if the sort should be made in memory. */
	private boolean			bMemSort 		= true;
	/** Determines if the filter should be made in memory. */
	private boolean			bMemFilter 		= true;
	/** Determines if the RemoteDataBook should write back the changes to the AbstractStorage. */
	private boolean 		bWritebackEnabled = false;
	/** It determines if the MemDataBook is open. */
	private boolean			bIsOpen			= false;
	/** It determines if saveAllRows already is called for this DataBook. Prevent endless loops. */
	private boolean			bSaveAllRows 	= false;
	/** It determines if restoreAllRows already is called for this DataBook. Prevent endless loops. */
	private boolean			bRestoreAllRows = false;

	/** Ignore recursiv reload events. */
	private boolean ignoreReload = false;

	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Contructs a new <code>MemDataBook</code> with an empty {@link RowDefinition}.
	 */
	public MemDataBook()
	{
	}

	/**
	 * Constructs a new <code>MemDataBook</code> with the given {@link RowDefinition}.
	 * 
	 * @param pRowDefinition the row definition for the data book
	 */
	public MemDataBook(RowDefinition pRowDefinition)
	{
		super(pRowDefinition);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	public IDataBook getDataBook()
	{
		return this;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public String getName()
	{
		return sName;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setName(String pName) throws ModelException
	{
		if (isOpen())
		{
			throw new ModelException("It's not allowed on open DataBooks! - " + getName());			
		}				
		sName = pName;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setWritebackIsolationLevel(WriteBackIsolationLevel pIsolationLevel)
	{
		iWritebackIsolationLevel = pIsolationLevel;
	}

	/**
	 * {@inheritDoc}
	 */
	public WriteBackIsolationLevel getWritebackIsolationLevel()
	{	
		if (iWritebackIsolationLevel == null)
		{
			if (getDataSource() != null)
			{
				return getDataSource().getWritebackIsolationLevel();
			}
			return WriteBackIsolationLevel.DATA_ROW;
		}
		return iWritebackIsolationLevel;
	}	
		
	/**
	 * {@inheritDoc}
	 */
	public void setRowDefinition(IRowDefinition pRowDefinition) throws ModelException
	{
		if (isOpen())
		{
			throw new ModelException("It's not allowed on open DataBooks!");			
		}
		
		IControl[] controls = null;
		
		//cleanup old row definition
		if (rdRowDefinition != null)
		{
			controls = getControls();
			
			for (int i = 0; i < controls.length; i++)
			{
				rdRowDefinition.removeControl(controls[i]);
			}
		}
		
		rdRowDefinition = pRowDefinition;
		
		// #424 - MemDataBook.setRowDefintion(null) should addControls
		if (rdRowDefinition == null)
		{
			rdRowDefinition = new RowDefinition();
		}
		
		//configure new row definition
		if (rdRowDefinition != null)
		{
			if (controls == null)
			{
				controls = getControls();
			}
			
			for (int i = 0; i < controls.length; i++)
			{
				rdRowDefinition.addControl(controls[i]);
			}
		}
	}	
	
	/**
	 * {@inheritDoc}
	 */
	public IDataSource getDataSource()
	{
		return dsDataSource;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setDataSource(IDataSource pDataSource) throws ModelException
	{
		if (isOpen())
		{
			throw new ModelException("It's not allowed on open DataBooks!");
		}
		dsDataSource = pDataSource;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public ReferenceDefinition getMasterReference()
	{
		return rdMasterReference;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setMasterReference(ReferenceDefinition pMasterReference) throws ModelException
	{
		if (isOpen())
		{
			throw new ModelException("It's not allowed on open DataBooks!");			
		}
		rdMasterReference = pMasterReference;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isSelfJoined()
	{
		return rdMasterReference != null && rdMasterReference.getReferencedDataBook() == this;
	}

	/**
	 * {@inheritDoc}
	 */
	public IDataPage getDataPage(TreePath pTreePath) throws ModelException
	{
		if (rdTreeRootReference == null)
		{
			return getDataPage(null, pTreePath);
		}
		else
		{
			return getDataPage(rdTreeRootReference.getReferencedDataBook(), pTreePath);
		}
	}

	/**
	 * Returns the master row only with the primary key columns - like defined in
	 * the getMasterReference().getReferencedColumns() created form a root row.
	 * 
	 * @param pRootDataRow	the full/complete master IDataRow to use.
	 * @return internal the master row only with the primary key columns - like defined in
	 * the getMasterReference().getReferencedColumns().
	 * @throws ModelException	if the master row couldn't created.
	 */
	private IDataRow getMasterDataRowFromRootDataRow(IDataRow pRootDataRow) throws ModelException
	{
		IDataRow masterDataRow = rdMasterReference.getReferencedDataBook().createEmptyRow(rdMasterReference.getReferencedColumnNames());
		
		if (rdTreeRootReference != null && pRootDataRow != null)
		{
			masterDataRow.setValues(saTreeRootMasterColumnNames, pRootDataRow.getValues(rdTreeRootReference.getReferencedColumnNames()));
		}
		
		return masterDataRow;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IDataPage getDataPage(IDataRow pRootRow, TreePath pTreePath) throws ModelException
	{
		IDataPage dpDataPage = getDataPageWithRootRow(pRootRow);
		
		if (pTreePath != null)
		{
			for (int i = 0; i < pTreePath.length(); i++)
			{
				// #348 - sync() fails if a seljoined MemDataBook has the TreePath wrong 
				IDataRow masterRow = dpDataPage.getDataRow(pTreePath.get(i));
				if (masterRow == null)
				{
					return null;
				}
				else
				{
					dpDataPage = getDataPageIntern(masterRow);
				}
			}
		}

		return dpDataPage;
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized IDataPage getDataPageWithRootRow(IDataRow pRootDataRow) throws ModelException
	{
		// if no master reference exits, then, don't call getDataPageIntern!!!!
		if (getMasterReference() == null)
		{
			sync();
			return dpCurrentDataPage;
		}		
		return getDataPageIntern(getMasterDataRowFromRootDataRow(pRootDataRow));
	}
	
	/**
	 * {@inheritDoc}
	 */
	public ReferenceDefinition getRootReference()
	{
		return rdTreeRootReference;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setRootReference(ReferenceDefinition pTreeRootReference) throws ModelException
	{
		if (isOpen())
		{
			throw new ModelException("It's not allowed on open DataBooks!");			
		}
		rdTreeRootReference = pTreeRootReference;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public TreePath getTreePath()
	{
		try
		{
			sync();
		}
		catch (ModelException e)
		{
			// Ignore it!
		}
		return treePath;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setTreePath(TreePath pTreePath) throws ModelException
	{
		if (!treePath.equals(pTreePath) || bMasterChanged)
		{
			invokeSaveEditingControls();
			
			// sync has to be after invokeSaveEditingControls, because events on editors could cause an empty dbCurrentDataPage
			// therefore an sync is needed!
			sync();
			
			IDataRow drOld = getDataRow(iSelectedRowIndex);
			
			saveDataRowLevel(null);

			if (eventBeforeRowSelected != null)
			{
				eventBeforeRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_ROW_SELECTED, drOld));
			}

			if (pTreePath == null)
			{
				treePath = new TreePath();
			}
			else
			{
				treePath = pTreePath;
			}
			
			dpCurrentDataPage = (MemDataPage)getDataPage(treePath);
			
			if (getSelectionMode() == SelectionMode.DESELECTED)
			{
				setSelectedRowInternal(-1);
			}
			else
			{
				setSelectedRowInternal(0);
			}
			
			if (eventAfterRowSelected != null)
			{
				eventAfterRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_ROW_SELECTED, drOld));
			}
		}
	}

	/**
	 * Sets the TreePath internal. see setTreePath(...)
	 * It doesn't call saveDataRowLevel(null), invokeSaveEditingControls()!
	 * 
	 * @param pTreePath			the tree path to set.
	 * @throws ModelException	if the TreePath is invalid
	 */
	private void setTreePathInternal(TreePath pTreePath) throws ModelException
	{
		if (!treePath.equals(pTreePath) || bMasterChanged)
		{
			// sync has to be after invokeSaveEditingControls, because events on editors could cause an empty dbCurrentDataPage
			// therefore an sync is needed!
			sync();
			
			IDataRow drOld = getDataRow(iSelectedRowIndex);
			
			if (eventBeforeRowSelected != null)
			{
				eventBeforeRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_ROW_SELECTED, drOld));
			}

			if (pTreePath == null)
			{
				treePath = new TreePath();
			}
			else
			{
				treePath = pTreePath;
			}
			
			dpCurrentDataPage = (MemDataPage)getDataPage(treePath);
			
			if (getSelectionMode() == SelectionMode.DESELECTED)
			{
				setSelectedRowInternal(-1);
			}
			else
			{
				setSelectedRowInternal(0);
			}
			
			if (eventAfterRowSelected != null)
			{
				eventAfterRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_ROW_SELECTED, drOld));
			}
		}
	}
	
	/**
	 * This function is used for rehashing data pages after insert of the master row.
	 * If it is true, the page is dropped, and therefore refetched, if needed.
	 * @return true, the page is dropped, and therefore refetched, if needed.
	 */
	protected boolean isDataPageRefetchPossible()
	{
		return false;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void notifyMasterChanged()
	{
		try
		{				
			// #347 - DataBook in DATASOURCE level should in notifyMasterChanged always check if rehash is required !
			// if master row is inserted and saved, then rehash!
			if (rdMasterReference != null 
					&& rdMasterReference.getReferencedDataBook().getUID() != null
					&& !rdMasterReference.getReferencedDataBook().isInserting())
			{
				// rehash current DataPage, because PK changed after insert in Master
				// #405 - NullPointer Exception in notifyMasterChanged(), if parent IDataBook out of sync
				if (dpCurrentDataPage != null && rdMasterReference.getReferencedDataBook().getUID() != null)
				{
					MemDataPage mdpRehash = htDataPagesCache.remove(rdMasterReference.getReferencedDataBook().getUID());
	
					if (mdpRehash != null)
					{
						// [HM] Force refetch of the Detail DataPage, if it is possible and not changed
						// otherwise rehash the page.
						if (!isDataPageRefetchPossible() || mdpRehash.getChangedDataRows().length > 0)
						{
							IDataRow drMaster = getMasterDataRow(rdMasterReference.getReferencedDataBook());					
							mdpRehash.setMasterDataRow(drMaster);
							htDataPagesCache.put(drMaster, mdpRehash);
						}
						else if (mdpRehash == dpCurrentDataPage) // drop current data page, if the rehashed data page is the current
						{
							// [HM] enable selecting current datarow!
							handleStoreSelection(iSelectionMode);
							
							dpCurrentDataPage = null;
							
							bMasterChanged = true;
						}
					}
				}					
			}
		}
		catch (ModelException e)
		{
			throw new RuntimeException(e);
		}
		
		if (!bMasterChanged)
		{
			try
			{				
				// master changed is only allowed to be set after rehashing, because we have to
				// prevent changing the currentPage in sync.			
				bMasterChanged = true;					

				if (eventAfterReload != null)
				{
					eventAfterReload.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_RELOAD, null));
				}				

				invokeMasterChangedDetailsListeners();
			}
			catch (ModelException e)
			{
				throw new RuntimeException(e);
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void removeDataPage(IDataRow pMasterDataRow, TreePath pTreePath) throws ModelException
	{
		IDataRow  drMaster = null;
		IDataPage dpCurrent = null;
		
		if (isSelfJoined())
		{
			if (pTreePath == null)
			{
				pTreePath = new TreePath();
			}
			IDataRow drRoot = null;
			if (rdTreeRootReference != null)
			{
				drRoot = rdTreeRootReference.getReferencedDataBook();
			}
			
			if (hasDataPage(drRoot, pTreePath))
			{				
				dpCurrent = getDataPage(drRoot, pTreePath);
				drMaster  = dpCurrent.getMasterDataRow();
				
				for (int i = 0, iSize = dpCurrent.getRowCount(); i < iSize; i++)
				{
					removeDataPage(null, pTreePath.getChildPath(i));				
				}				
			}
			else
			{
				return;
			}
		}
		else
		{
			if (hasDataPage(pMasterDataRow))
			{			
				dpCurrent = getDataPage(pMasterDataRow);
				drMaster = dpCurrent.getMasterDataRow();
			}
			else
			{
				return;
			}
		}	
		
		for (int i = 0, iSize = dpCurrent.getRowCount(); i < iSize; i++)
		{
			IDataRow drCurrent = dpCurrent.getDataRow(i);
			
			// delete details first
			if (auDetailDataBooks != null)
			{
				for (int j = 0, size = auDetailDataBooks.size(); j < size; j++)
				{
					IDataBook dataBook = auDetailDataBooks.get(j).get();
					if (dataBook != null)
					{
		        		dataBook.removeDataPage(drCurrent, null);
					}
		        }					
			}
		}
				
		if (drMaster instanceof IChangeableDataRow && ((IChangeableDataRow)drMaster).getUID() != null)
		{
			htDataPagesCache.remove(((IChangeableDataRow)drMaster).getUID());
		}
		else 
		{
			htDataPagesCache.remove(drMaster);
		}
		
		// #495 - MemDataBook removeDataPage should setTreePath(null) 
		if (isSelfJoined() 
				&& treePath != null && treePath.length() > 0)
		{
			// if current TreePath is a detail from the deleted one, then set the treepath to null
			if (treePath.length() >= pTreePath.length())
			{
				boolean bEquals = true;
				for (int i = 0; bEquals && i < treePath.length() && i < pTreePath.length(); i++)
				{
					if (treePath.get(i) != pTreePath.get(i))
					{
						bEquals = false;
					}
				}
				if (bEquals)
				{
					setTreePath(new TreePath());
				}
			}
		}
		
		if (dpCurrent == dpCurrentDataPage)
		{
			dpCurrentDataPage = null;
			
			// #369 - delete all rows from a databook
			// Master change has to be set, to sync the page!!!
			bMasterChanged = true;
			
			setDetailsChangedInMasterBook();
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public SelectionMode getSelectionMode()
	{
		return iSelectionMode;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setSelectionMode(SelectionMode pSelectionMode)
	{
		iSelectionMode = pSelectionMode;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isInsertAllowed()
	{
		return bInsertEnabled && !isReadonly() && dpCurrentDataPage != dpEmptyDataPage;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isInsertEnabled()
	{
		return bInsertEnabled;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setInsertEnabled(boolean pInsertEnabled)
	{
		bInsertEnabled = pInsertEnabled;
		
		invokeRepaintListeners();
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isUpdateAllowed() throws ModelException
	{
		return bUpdateEnabled && !isReadonly() && getSelectedRow() >= 0 && !isDeleting();
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isUpdateEnabled() throws ModelException
	{
		return bUpdateEnabled;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setUpdateEnabled(boolean pUpdateEnabled)
	{
		bUpdateEnabled = pUpdateEnabled;

		invokeRepaintListeners();
	}

	/**
	 * Returns true if lock and refetch is enabled.
	 * 
	 * @return true if lock and refetch is enabled.
	 */
	public boolean isLockAndRefetchEnabled()
	{
		return bLockAndRefetchEnabled;
	}

	/**
	 * Sets whether if lock and refetch is enabled.
	 * 
	 * @param pLockAndRefetchEnabled true if lock and refetch is enabled.
	 */
	public void setLockAndRefetchEnabled(boolean pLockAndRefetchEnabled)
	{
		bLockAndRefetchEnabled = pLockAndRefetchEnabled;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isDeleteAllowed() throws ModelException
	{
		return bDeleteEnabled && !isReadonly() && getSelectedRow() >= 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isDeleteEnabled() throws ModelException
	{
		return bDeleteEnabled;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setDeleteEnabled(boolean pDeleteEnabled)
	{
		bDeleteEnabled = pDeleteEnabled;

		invokeRepaintListeners();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isReadonly()
	{
		return bReadOnly;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setReadOnly(boolean pReadOnly) throws ModelException
	{
		if (isOpen() && !bReadOnly && pReadOnly)
		{
			saveAllRows();
		}
		
		setReadOnlyWithoutSave(pReadOnly);
	}
	
	/**
	 * Returns the count of rows which are read ahead from the IDBAccess.
	 * 
	 * @return the count of rows which are read ahead from the IDBAccess.
	 */
	public int getReadAhead()
	{
		return iReadAheadRowCount;
	}

	/**
	 * Sets the count of rows which are read ahead from the IDBAccess.
	 * 
	 * @param pReadAheadRowCount
	 *            the row count to read ahead from the IDBAccess
	 */
	public void setReadAhead(int pReadAheadRowCount)
	{
		if (pReadAheadRowCount >= 0)
		{
			iReadAheadRowCount = pReadAheadRowCount;
		}
	}			
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isOpen()
	{
		return bIsOpen;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void open() throws ModelException
	{
		if (!isOpen())
		{
			if (getName() == null)
			{
				throw new ModelException("DataBook Name is null!");				
			}

			if (rdRowDefinition.getColumnCount() == 0)
			{
				throw new ModelException("RowDefintion in DataBook contains no columns!");				
			}
			
			//use default values for choice cell editors
			if (UIFactoryManager.getFactory() != null)
			{
				ColumnDefinition coldef;
				
				for (int i = 0, anz = rdRowDefinition.getColumnCount(); i < anz; i++)
				{
					coldef = rdRowDefinition.getColumnDefinition(i);
					Object[] allowedValues = coldef.getAllowedValues();
					
					if (coldef.getDataType().getCellEditor() == null && allowedValues != null && allowedValues.length > 0)
					{
						IChoiceCellEditor editor = UIChoiceCellEditor.getDefaultChoiceCellEditor(allowedValues);
						
						if (editor != null)
						{
							coldef.getDataType().setCellEditor(editor);
						}
						else
						{
							ColumnDefinition cdEnum = coldef.clone();
							cdEnum.setAllowedValues(null);
							String columnName = cdEnum.getName();
							
							MemDataBook mdbEnum = new MemDataBook();
							mdbEnum.setName("enum_" + columnName);
							mdbEnum.getRowDefinition().addColumnDefinition(cdEnum);
							mdbEnum.open();
							
							for (int j = 0; j < allowedValues.length; j++)
							{
								mdbEnum.insert(false);
								mdbEnum.setValue(columnName, allowedValues[j]);
							}
							mdbEnum.saveAllRows();
							
							coldef.getDataType().setCellEditor(new UILinkedCellEditor(
									new ReferenceDefinition(new String[] {columnName}, mdbEnum, new String[] {columnName})));
						}
					}
				}
			}
			
			// check PK Columns in RowDefinition
			String[] saPKColumnNames = rdRowDefinition.getPrimaryKeyColumnNames();		
			for (int i = 0; saPKColumnNames != null && i < saPKColumnNames.length; i++)
			{
				if (rdRowDefinition.getColumnDefinitionIndex(saPKColumnNames[i]) < 0)
				{
					throw new ModelException("Primary key column '" + saPKColumnNames[i] + 
							                 "' doesn't exist in RowDefinition of this DataBook!");
				}
			}

			ReferenceDefinition refdefMaster = getMasterReference();
			if (refdefMaster != null)
			{
				IDataBook dbMaster = refdefMaster.getReferencedDataBook();
				if (!isSelfJoined() && !dbMaster.isOpen())
				{
					throw new ModelException("The master DataBook has to be opened first! - " + dbMaster.getName());
				}
				// check if Master Reference Definition is correct.
				String[] saColumns = refdefMaster.getColumnNames();
				for (int i = 0; i < saColumns.length; i++)
				{
					if (rdRowDefinition.getColumnDefinitionIndex(saColumns[i]) < 0)
					{
						throw new ModelException("Column '" + saColumns[i] + 
								                 "' doesn't exist in detail DataBook '" + getName() + "' !");
					}
				}
				String[] saRefColumns = refdefMaster.getReferencedColumnNames();
				for (int i = 0; i < saRefColumns.length; i++)
				{
					if (dbMaster.getRowDefinition().getColumnDefinitionIndex(saRefColumns[i]) < 0)
					{
						throw new ModelException("Column '" + saRefColumns[i] + 
								                 "' doesn't exist in master DataBook '" + 
								                 dbMaster.getName() + "' !");
					}
				}
				if (isSelfJoined())
				{
					if (getRootReference() != null)
					{
						if (getRootReference().getReferencedDataBook() == dbMaster)
						{
							throw new ModelException("The RootReference DataBook '"
									+ getRootReference().getReferencedDataBook().getName()
									+ "' must be != MasterReference DataBook '" 
									+ dbMaster.getName() + "' !");
						}
						
						String[] columnNames = getRootReference().getColumnNames();
						saTreeRootMasterColumnNames = new String[columnNames.length]; 
						
						for (int i = 0; i < columnNames.length; i++)
						{
							int index = ArrayUtil.indexOf(getMasterReference().getColumnNames(), columnNames[i]);
							
							if (index < 0)
							{
								throw new ModelException("Column name " + columnNames[i] + " of root reference is not in master reference!");
							}
							saTreeRootMasterColumnNames[i] = getMasterReference().getReferencedColumnNames()[index];
						}
						
						getRootReference().getReferencedDataBook().addDetailDataBook(this);
					}
					else
					{
						saTreeRootMasterColumnNames = null;
					}
				}
				else
				{
					refdefMaster.getReferencedDataBook().addDetailDataBook(this);
					refdefMaster.setConnected();
				}
			}
			
			if (getDataSource() != null)
			{
				if (!getDataSource().isOpen())
				{
					throw new ModelException("DataSource is not open!");								
				}						
				getDataSource().addDataBook(this);
			}
			
			rdRowDefinition.addDataBook(this);
			
			iSelectedRowIndex = -1;
			bMasterChanged = true;
			bIsOpen = true;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void close()
	{
		if (isOpen())
		{
			bIsOpen 		          = false;
			dpCurrentDataPage	      = null;
			htDataPagesCache.clear();							
			iSelectedRowIndex 	      = -1;
			oaStorage				  = null;

			if (getMasterReference() != null)
			{
				getMasterReference().getReferencedDataBook().removeDetailDataBook(this);
			}
			
			if (dsDataSource != null)
			{
				dsDataSource.removeDataBook(this);
			}
			if (rdRowDefinition != null)
			{
				rdRowDefinition.removeDataBook(this);
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized int getRowCount() throws ModelException
	{
		if (isOpen())
		{
			sync();

			return dpCurrentDataPage.getRowCount();
		}
		return 0;
	}


	/**
	 * {@inheritDoc}
	 */
	public synchronized int getSelectedRow() throws ModelException
	{
		if (!isOpen())
		{
			return -1;			
		}
		sync();
		return iSelectedRowIndex;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void setSelectedRow(int pDataRowIndex) throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}

		sync();
		
		if (iSelectedRowIndex != pDataRowIndex)
		{		
			invokeSaveEditingControls();
			
			// sync has to be after invokeSaveEditingControls, because events on editors could cause an empty dbCurrentDataPage
			// therefore an sync is needed!
			sync();
			
			IDataRow drOld = getDataRow(iSelectedRowIndex);
			
			if (getWritebackIsolationLevel() == WriteBackIsolationLevel.DATA_ROW)
			{
				saveDataRowLevelDetails(null);
			}

			if (eventBeforeRowSelected != null)
			{
				eventBeforeRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_ROW_SELECTED, drOld));
			}

			setSelectedRowInternal(pDataRowIndex);
			
			if (eventAfterRowSelected != null)
			{
				eventAfterRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_ROW_SELECTED, drOld));
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized String getSelectedColumn() throws ModelException
	{
		// sync() not mandatory !
		return sSelectedColumn;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void setSelectedColumn(String pColumnName) throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}

		sync();
		
		if (pColumnName != null)
		{
			rdRowDefinition.getColumnDefinition(pColumnName); // Check if the column exists.
		}
		
		if ((sSelectedColumn == null && pColumnName != null)
				|| (sSelectedColumn != null && !sSelectedColumn.equals(pColumnName)))
		{
			if (eventBeforeColumnSelected != null)
			{
				eventBeforeColumnSelected.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_COLUMN_SELECTED, null));
			}
			
			sSelectedColumn = pColumnName;
			
			invokeRepaintListeners();
			
			if (eventAfterColumnSelected != null)
			{
				eventAfterColumnSelected.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_COLUMN_SELECTED, null));
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isDeleteCascade()
	{
		return bDeleteCascade;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setDeleteCascade(boolean pDeleteCascade)
	{
		bDeleteCascade = pDeleteCascade;
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized int insert(boolean pBeforeRow) throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}		
		sync();
		if (dpCurrentDataPage == dpEmptyDataPage)
		{
			throw new ModelException("Master DataBook has no selected row! - " + 
					                 getMasterReference().getReferencedDataBook().getName());
		}
		if (!isInsertAllowed())
		{
			throw new ModelException("Insert isn't allowed!");			
		}		
		
		invokeSaveEditingControls();

		saveDataRowLevel(null);

		// #374 - If an insert happens, after that more rows have to be fetched, then wrong rows will be fetched
		// saveDataRowLevel can set the dpCurrentDataPage to null, because notifyMasterChange rehash the details, and force lazy refetch!
		sync();

		// #374 - If an insert happens, after that more rows have to be fetched, then wrong rows will be fetched
		// if first insert without anything fetched, begin fetch, to prevent that a 
		// cursor deliveres the new inserted row, what happens, if the fetch is started after inserting a row.
		if (!dpCurrentDataPage.isAllFetched() && dpCurrentDataPage.getRowCount() == 0)
		{
			getDataRow(0);
		}
		
		IDataRow drOld = getDataRow(iSelectedRowIndex);			
		if (eventBeforeInserting != null)
		{
			eventBeforeInserting.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_INSERTING, null));
		}
		
		int iDataRowIndex = insertInternal(pBeforeRow);
		
		setSelectedRowInternal(iDataRowIndex);
		copyMasterColumnsToCurrentDetail(false);

		if (eventAfterInserting != null)
		{
			eventAfterInserting.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_INSERTING, null));
		}
		if (eventAfterRowSelected != null)
		{
			eventAfterRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_ROW_SELECTED, drOld));
		}
		
		return iDataRowIndex;
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void update() throws ModelException
	{
		if (!isUpdating() && !isInserting() && !isDeleting())
		{
			if (!isOpen())
			{
				throw new ModelException("DataBook isn't open! - " + sName);			
			}
			sync();
			if (!isUpdateAllowed())
			{
				throw new ModelException("Update isn't allowed!");			
			}				
			
			saveDataRowLevel(this);

			IDataRow drOld = getDataRow(iSelectedRowIndex); 
			if (eventBeforeUpdating != null)
			{
				eventBeforeUpdating.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_UPDATING, drOld));
			}
			
			if (isLockAndRefetchEnabled())
			{
				executeLockAndRefetch();
			}
			
			// set current row in DataPage updating, and set the new updating row to the current row.
			// Thats enough! setUpdating, setValueInternal removed from source! HM will love it
			dpCurrentDataPage.update(iSelectedRowIndex);		
			oaStorage = dpCurrentDataPage.getDataRowStorage(iSelectedRowIndex);

			setDetailsChangedInMasterBook(); 

			if (eventAfterUpdating != null)
			{
				eventAfterUpdating.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_UPDATING, drOld));
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void delete() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		sync();
		if (iSelectedRowIndex < 0)
		{
			return;			
		}		
		if (!isDeleteAllowed())
		{
			throw new ModelException("Delete isn't allowed!");
		}				
		
		IDataRow drOld = getDataRow(iSelectedRowIndex);
		if (eventBeforeDeleting != null)
		{
			eventBeforeDeleting.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_DELETING, drOld));
		}
		boolean bIsInserting = isInserting();
		if (bIsInserting)
		{
			// #215 - Wrong event order in delete on an inserting row 
			// AFTER_DELETING event is thrown before BEFORE_RESTORE, to get the old Row!
			if (eventAfterDeleting != null)
			{
				eventAfterDeleting.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_DELETING, drOld));
			}
			if (eventBeforeRestore != null)
			{
				eventBeforeRestore.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_RESTORE, drOld));
			}
		}
		
		// #507 move after events above.
		// restore details, if cascade delete. 
		// Otherwise try to save the details, if WriteBackIsolationLevel == DATA_ROW 	
		restoreSaveAllDetails();		
		if (!isInserting())
		{
			restoreSelectedRow();
		}
		else 
		{
			// #375 - delete, cancelEditing missing 
			// -> cancel editing has to be called, to ensure, that the editors are not saving after delete.
			invokeCancelEditingControls();
		}		

		if (!isInserting())
		{
			// if this is an exiting row, then try to lock it first, that no other can modify it.
			if (isLockAndRefetchEnabled())
			{			
				executeLockAndRefetch();
			}
		}
		boolean bRowRemoved = deleteInternal();		

		if (bRowRemoved)
		{
			// correct selected row, if the last row got deleted
			correctSelectedRowAfterDelete();
			
			if (eventAfterRestore != null)
			{
				eventAfterRestore.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_RESTORE, drOld));
			}

			if (eventAfterRowSelected != null)
			{
				eventAfterRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_ROW_SELECTED, drOld));
			}
		}		

		if (!bIsInserting && eventAfterDeleting != null)
		{
			eventAfterDeleting.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_DELETING, drOld));
		}
		
		// correct selected row
		// #505 - delete should should check the WritebackIsolation level
		if (!bRowRemoved && WriteBackIsolationLevel.DATA_ROW.equals(getWritebackIsolationLevel())) 
		{
			// store depended DataBooks, if the WritebackIsolationLevel == IDataSource.DATA_ROW
			try
			{
				saveDataRowLevel(null);
			}
			catch (ModelException modelException)
			{
				if (isInserting() || isUpdating() || isDeleting())
				{		
					restoreRow();
				}
		        throw modelException;
			}
		}		
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void deleteAllDataRows() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		sync();

		fetchAll();
		for (int i = getRowCount() - 1; i >= 0; i--)
		{
			setSelectedRow(i);
			delete();
		}
	}	
	
	/**
	 * {@inheritDoc}
	 */
	public void notifyDetailChanged()
	{
		try
		{
			// set the Detail changed only, if the row still exists. -> Ticket #330
			if (iSelectedRowIndex < dpCurrentDataPage.getRowCount()) 
			{	
				// check if one detail is changed.
				boolean bDetailChanged = false;
				for (int i = auDetailDataBooks.size() - 1; i >= 0; i--)
				{
					if (auDetailDataBooks.get(i).get() != null)
					{
						IDataBook detail   = auDetailDataBooks.get(i).get();
						int []    iChanges = detail.getChangedDataRows();
						
						if (iChanges != null && iChanges.length > 0)
						{
							bDetailChanged = true;
							break;
						}
						if (detail.isSelfJoined())
						{
							// check if changes in the root exists, then changes are there!
							bDetailChanged = detail.getDataPage(new TreePath()).getChangedDataRows().length > 0;
						}
					}
				}
				
				// set the detail change state correct.
				dpCurrentDataPage.setDetailChanged(iSelectedRowIndex, bDetailChanged);
				oaStorage = dpCurrentDataPage.getDataRowStorage(iSelectedRowIndex);
			}

			setDetailsChangedInMasterBook();
		}
		catch (ModelException e)
		{
			throw new RuntimeException(e);
		}
	}
		
	/**
	 * {@inheritDoc}
	 */
	public synchronized IChangeableDataRow getDataRow(int pDataRowIndex) throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		if (pDataRowIndex < 0)
		{
			return null;
		}
		sync();
		
		return getDataRowInternal(pDataRowIndex);		
	}

	/** 
	 * {@inheritDoc}
	 */
	public IDataRow getOriginalRow() throws ModelException
	{
		if (!isOpen())
		{
			return null;			
		}
		sync();
		return super.getOriginalRow();		
	}

	/** 
	 * {@inheritDoc}
	 */
	public boolean isWritableColumnChanged() throws ModelException
	{
		if (!isOpen())
		{
			return false;			
		}
		sync();
		return super.isWritableColumnChanged();	
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized boolean isDeleting() throws ModelException
	{
		if (!isOpen())
		{
			return false;			
		}
		// getSelectedRow does sync();
		if (getSelectedRow() < 0)
		{
			return false;
		}		
		return super.isDeleting();
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized boolean isUpdating() throws ModelException
	{
		if (!isOpen())
		{
			return false;			
		}
		// getSelectedRow does sync();
		if (getSelectedRow() < 0)
		{
			return false;
		}		
		return super.isUpdating();
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized boolean isInserting() throws ModelException
	{
		if (!isOpen())
		{
			return false;			
		}
		// getSelectedRow does sync();
		if (getSelectedRow() < 0)
		{
			return false;
		}		
		return super.isInserting();
	}
	
	/**
	 * {@inheritDoc} 
	 */
	public synchronized boolean isDetailChanged() throws ModelException
	{
		if (!isOpen())
		{
			return false;			
		}
		sync();
		return super.isDetailChanged();
	}

	/** 
	 * {@inheritDoc}
	 */
	public Object getUID()  throws ModelException
	{
		if (!isOpen())
		{
			return null;			
		}
		sync();
		return super.getUID();
	}

	/**
	 * {@inheritDoc}
	 */
	public ICondition getFilter()
	{
		return cFilter;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void setFilter(ICondition pFilter) throws ModelException
	{
		if (isOpen())
		{
			//#615
			//memfilter means: all rows fetched and if datasource level is set, don't save
			//on filter change.
			//If you have a pseudo master-link view (no master reference between databooks, but master sets filter to
			//filter details). In this case, don't save on "pseudo master" change.
			
			if (!isMemFilter() || getWritebackIsolationLevel() == WriteBackIsolationLevel.DATA_ROW)
			{
				// it saves always if not MemFilter (Data come from remote Storage, a filter would destroy all changes) or
				// if the IsolationLevel DATAROW, save also always.
				saveAllRows();
			}
		}
		// copy Filter, to preserve, User changes during filtering.
		if (pFilter != null)
		{
			cFilter = pFilter.clone();	
		}
		else
		{
			cFilter = null;
		}
		
		if (isOpen())
		{
			if (isMemFilter())
			{
				clearFilterSortInMemDataPages();
				invokeRepaintListeners();		
			}
			else
			{			
				if (getSelectionMode() == SelectionMode.DESELECTED)
				{
					reload();
				}
				else
				{
					reload(SelectionMode.FIRST_ROW);
				}
			}
		}			
	}

	/**
	 * {@inheritDoc}
	 */
	public SortDefinition getSort()
	{
		return sSort;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void setSort(SortDefinition pSort) throws ModelException
	{
		if (isOpen())
		{
			if (!isMemSort() || getWritebackIsolationLevel() == WriteBackIsolationLevel.DATA_ROW)
			{
				// it saves always if not MemFilter (Data come from remote Storage, a filter would destroy all changes) or
				// if the IsolationLevel DATAROW, save also always.
				saveAllRows();
			}			
		}
		// copy Filter, to preserve, User changes during filtering.
		if (pSort != null)
		{
			sSort = pSort.clone();		
		}
		else
		{
			sSort = null;		
		}
		
		if (isOpen())
		{
			if (isMemSort())
			{
				// #357 - setSort on MemDataBook with setMemSort(true), doesn't setSelectedRow correct after sort 
				handleStoreSelection(iSelectionMode);
				
				clearFilterSortInMemDataPages();				
				invokeRepaintListeners();
			}
			else
			{
				reload();
			}
		}			
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void reload() throws ModelException
	{
		reload(getSelectionMode());
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void reload(SelectionMode pSelectionMode) throws ModelException
	{
		if (isOpen() && !ignoreReload)
		{
			handleStoreSelection(pSelectionMode);
			
			if (eventBeforeReload != null)
			{
				eventBeforeReload.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_RELOAD, null));
			}
			
			if (!htDataPagesCache.isEmpty() || dpCurrentDataPage != null)
			{
				restoreAllRows();
			} 
			
			executeRefresh();		

			// Notify Master changed has to be before AFTER_RELOAD event, to prevent undefined states in event.
			// #746 - setFilter/reload on a master databook, doesn't sync details - use invokeMasterChangedDetailsListeners(); instead of notifyMasterChanged()
			invokeMasterChangedDetailsListeners();	
			
			if (eventAfterReload != null)
			{
				eventAfterReload.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_RELOAD, null));
			}
		}
	}
	
	/**
	 * Reload the current DataPage with the pSelectionMode.
	 * 
	 * @see #reload(javax.rad.model.IDataBook.SelectionMode)
	 * @param pSelectionMode	pSelectionMode the Selection mode to use
	 * @throws ModelException	if an DataSourceException happens during get the requested row.
	 */
	public synchronized void reloadDataPage(SelectionMode pSelectionMode) throws ModelException
	{
		if (isOpen())
		{
			handleStoreSelection(pSelectionMode);
			
			if (eventBeforeReload != null)
			{
				eventBeforeReload.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_RELOAD, null));
			}
			
			if (!htDataPagesCache.isEmpty() || dpCurrentDataPage != null)
			{
				restoreDataPageIntern();
			}
			
			if (!isOutOfSync())
			{
				dpCurrentDataPage.clear();
			}	

			// correctSelectedRow has to be before AFTER_RELOAD event, to prevent undefined states in event.
			if (!isOutOfSync())
			{
				correctSelectedRow();
			}		
			
			invokeRepaintListeners();
			
			// Notify Master changed has to be before AFTER_RELOAD event, to prevent undefined states in event.
			notifyMasterChanged();			
			
			if (eventAfterReload != null)
			{
				eventAfterReload.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_RELOAD, null));
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void restoreSelectedRow() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		sync();	
		if (getSelectedRow() < 0)
		{
			return;
		}	
		
		invokeCancelEditingControls();

		if (isInserting() || isUpdating() || isDeleting())
		{		
			restoreRow();
		}

		invokeRepaintListeners();			
	}

	/**
	 * Restore all changes from top -> down to all detail Databooks. It just restires the changes in the specified pDataBook.
	 * 
	 * @param pCurrentDataBook	the Current DataBook in the Levels from Root DataBook to the last detail DataBook.
	 * @param pDataBook		  	the DataBook in which the changes will be restored.
	 * @throws ModelException 	if an Error occur during save to storage.
	 */
	private synchronized void restoreTopDownAllRows(IDataBook pCurrentDataBook, IDataBook pDataBook) throws ModelException
	{
		if (pCurrentDataBook == pDataBook)
		{
			pCurrentDataBook.restoreAllRows();
		}
		else
		{
			int         iOldRowIndex     = pCurrentDataBook.getSelectedRow();
			IDataBook[] auDetailDataBook = pCurrentDataBook.getDetailDataBooks();
			ArrayUtil<TreePath> auOldTreePath = new ArrayUtil<TreePath>();

			if (auDetailDataBook != null)
			{
				for (int i = 0, iSize = auDetailDataBook.length; i < iSize; i++)
				{
					if (auDetailDataBook[i].isSelfJoined())
					{
						auOldTreePath.add(auDetailDataBook[i].getTreePath());
					}
				}
			}			
			
			int[] iaChangedRows = pCurrentDataBook.getChangedDataRows();
			
			for (int i = 0, iSize = iaChangedRows.length; i < iSize; i++)
			{
				pCurrentDataBook.setSelectedRow(iaChangedRows[i]);
				if (pCurrentDataBook.isDetailChanged())
				{
					if (auDetailDataBook != null)
					{
						//save all details, till the detail is not changed anymore!
						for (int j = 0, jSize = auDetailDataBook.length; j < jSize && pCurrentDataBook.isDetailChanged(); j++)
						{
							if (auDetailDataBook[j].isSelfJoined())
							{
								// start in Detail at the root node of the hierarchy
								auDetailDataBook[j].setTreePath(null);
							}
							restoreTopDownAllRows(auDetailDataBook[j], pDataBook);
						}
					}	
					
					// if still changes exists, and self joined, then save the self joined hierarchy down too.
					if (pCurrentDataBook.isSelfJoined() 
							&& pCurrentDataBook.isDetailChanged()
							&& pCurrentDataBook.hasDataPage(pCurrentDataBook))
					{
						// get Master.
						TreePath tpOld = pCurrentDataBook.getTreePath();
						
						// use the current row as master
						pCurrentDataBook.setTreePath(tpOld.getChildPath(pCurrentDataBook.getSelectedRow()));
						
						restoreTopDownAllRows(pCurrentDataBook, pDataBook);
						
						//select back the treepath, selected row will set back later in this method.
						pCurrentDataBook.setTreePath(tpOld);
					}					
				}
			}
			
			pCurrentDataBook.setSelectedRow(iOldRowIndex);
			
			if (auDetailDataBook != null)
			{
				int j = 0;
				for (int i = 0, iSize = auDetailDataBook.length, jSize = auOldTreePath.size(); 
				     i < iSize && j < jSize; i++)
				{
					if (auDetailDataBook[i].isSelfJoined())
					{
						auDetailDataBook[i].setTreePath(auOldTreePath.get(j));
						j++;
					}
				}
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void restoreAllRows() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		if (dpCurrentDataPage == null)
		{
			return;
		}
		sync();	

		invokeCancelEditingControls();

		if (getSelectedRow() >= 0 && (isInserting() || isUpdating() || isDeleting()))
		{
			restoreRow();
		}

		// #492 - MemDataBook.restoreAllRows set wrong SelectedRow :: save the current row after the restore e.g is Inserting....
        int iOldSelectedRow = iSelectedRowIndex;        
		ArrayUtil<TreePath> auOldTreePath = getAllDetailsTreePath();

		ICondition oldFilter = null;
		
		try
		{
			// setFilter null, to see all changed rows!
			if (!(!isMemFilter() || getWritebackIsolationLevel() == WriteBackIsolationLevel.DATA_ROW)
					&& getFilter() != null)
			{
				oldFilter = getFilter();
				setFilter(null);
			}		
			// Restore all rows in current DataPage, then in all others.
			restoreDataPageIntern();
			
			// then restore all changes in all loaded DataPages
			if (!bRestoreAllRows && getMasterReference() != null)
			{
				// check if still changes exists
				Enumeration<MemDataPage> pages = htDataPagesCache.elements();
				MemDataPage              mdp;
				boolean                  bChangeFound = false;
				while (pages.hasMoreElements() && !bChangeFound)
				{
					mdp = pages.nextElement();
					if (mdp.getChangedDataRows().length > 0)
					{
						bChangeFound = true;
					}
				}
	
				// Changes exits, then go to the root DataBook, and go through all changed Rows and save all in "this" DataBook.
				if (bChangeFound)
				{
					IDataBook           dbRoot = this;
					ReferenceDefinition dbParentReference = dbRoot.isSelfJoined() ? dbRoot.getRootReference() : dbRoot.getMasterReference();
					while (dbParentReference != null)
					{
						dbRoot            = dbParentReference.getReferencedDataBook();
				        dbParentReference = dbRoot.isSelfJoined() ? dbRoot.getRootReference() : dbRoot.getMasterReference();
					}
					
					TreePath tpOld = null;
					if (dbRoot.isSelfJoined())
					{
						// start at the root node of the hierarchy
						tpOld = dbRoot.getTreePath();
						dbRoot.setTreePath(null);
					}
					
					bRestoreAllRows = true;
					try
					{
						restoreTopDownAllRows(dbRoot, this);
					}
					finally
					{
						bRestoreAllRows = false;
					}					
					
					if (dbRoot.isSelfJoined() && tpOld != null)
					{
						// restore Root TreePath
						dbRoot.setTreePath(tpOld);
					}					
				}
			}		
		}
		finally
		{
			if (oldFilter != null)
			{
				setFilter(oldFilter);
			}				
		}
		
		if (iSelectedRowIndex != iOldSelectedRow)
		{
			setSelectedRow(iOldSelectedRow);					
		}
		setAllDetailsTreePath(auOldTreePath);
		
		invokeRepaintListeners();		
	}

	/**
	 * Restores all rows in the current DataPage.
	 * 
	 * @see #restoreAllRows()
	 * @throws ModelException - if an ModelException happens during undo the changes.
	 */
	public synchronized void restoreDataPage() throws ModelException
	{
		// store Selection & TreePath
		int iOldSelectedRow = iSelectedRowIndex;
		ArrayUtil<TreePath> auOldTreePath = getAllDetailsTreePath();
		
		restoreDataPageIntern();
		
		// restore selection and TreePath
		if (iOldSelectedRow >= getRowCount())
		{
			iSelectedRowIndex = iOldSelectedRow;
			correctSelectedRowAfterDelete();		
		}
		else
		{
			if (iSelectedRowIndex != iOldSelectedRow)
			{
				setSelectedRow(iOldSelectedRow);					
			}
			setAllDetailsTreePath(auOldTreePath);
		}		
	}
	
	/**
	 * Restores all rows in the current DataPage.
	 * It stores/restores no selectedRow&TreePath
	 * 
	 * @throws ModelException if restoreSelected() fails
	 */
	private synchronized void restoreDataPageIntern() throws ModelException
	{
		int[] iaChangedRows = dpCurrentDataPage.getChangedDataRows();
		
		if (iaChangedRows.length > 0) 
		{
			ICondition oldFilter = null;

			try
			{
				// setFilter null, to see all changed rows!
				if (!(!isMemFilter() || getWritebackIsolationLevel() == WriteBackIsolationLevel.DATA_ROW)
						&& getFilter() != null)
				{
					oldFilter = getFilter();
					setFilter(null);
				}
				
				Arrays.sort(iaChangedRows);
                for (int i = iaChangedRows.length - 1; i >= 0; i--)
                {
                   setSelectedRow(iaChangedRows[i]);
                   restoreRow();
                }						
			}
			finally
			{
				if (oldFilter != null)
				{
					setFilter(oldFilter);
				}				
			}
		}
	}	
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void saveSelectedRow() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		if (getSelectedRow() < 0)
		{
			return;
		}	

        invokeSaveEditingControls();
        
        // sync has to be after invokeSaveEditingControls, because events on editors could cause an empty dbCurrentDataPage
		// therefore an sync is needed!
		sync();
		
		if (isDeleting() || isUpdating() || isInserting())
		{
			// store current TreePaths in Detail DataBooks.
			ArrayUtil<TreePath> auOldTreePath = getAllDetailsTreePath();
        
			saveSelectedRowInternIncludeDetails();
			
			// restore Detail DataBook TreePaths
			setAllDetailsTreePath(auOldTreePath);			
		}
	}

	/**
	 * It saves the selectedRow intern and saves all detail rows under it.
	 * 
	 * @throws ModelException if the save fails.
	 */
	private synchronized void saveSelectedRowInternIncludeDetails() throws ModelException
	{
		saveSelectedRowIntern();
		
		// save all Details -> selfjoined and detail DataBooks.
		if (isSelfJoined())
		{
    		saveSelfJoinedDetailRows();
		}
		if (auDetailDataBooks != null)
		{
			for (int i = 0, size = auDetailDataBooks.size(); i < size; i++)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook != null)
				{
	        		dataBook.saveAllRows();
				}
	        }					
		}
	}
	
	/**
	 * It repositions the current DataRow depending on the current Filter and Sort.
	 * 
	 * @throws ModelException if the repositioning fails.
	 */
	private synchronized void repositionCurrentDataRow() throws ModelException
	{	
		// if sort set, the find correct position. Otherwise use selectedRow
        if (getSort() != null)
        {
    		ChangeableDataRow drNew = new ChangeableDataRow(getRowDefinition(),
    														oaStorage.clone(),
    														dpCurrentDataPage,
    														iSelectedRowIndex);		
    		
			// delete row from current page
    		if (!dpCurrentDataPage.delete(iSelectedRowIndex))
    		{
    			if (!dpCurrentDataPage.store(iSelectedRowIndex))
    			{
    				oaStorage = dpCurrentDataPage.getDataRowStorage(iSelectedRowIndex);				
    			}					
    		}
    		
    		// add row at right position. (sort)
    		int iRow = iSelectedRowIndex;
    		
        	for (int iSize = dpCurrentDataPage.getRowCount(); 
        	         iRow < iSize && dpCurrentDataPage.getDataRow(iRow).compareTo(drNew, sSort) < 0; iRow++)
        	{
        		// nothing todo, it add +1 to iRow for the correct index to sort into.
        	}
        	        	
    		// insert always
    		dpCurrentDataPage.insert(iRow); 							
    		dpCurrentDataPage.setDataRow(iRow, drNew);
    		dpCurrentDataPage.store(iRow);        	

    		setSelectedRowInternal(iRow);
    		setDetailsChangedInMasterBook();
        }
        else
        {
        	store();
        }
	}
	
	/**
	 * It add pNewDataRow to the right position in the pDataPage.
	 * 
	 * @param pNewDataRow	the new DataRow to add.
	 * @param pDataPage		the DataPage to use to add the DataRow.
	 * @param pRow			the row index to use if DataBook isn't sorted
	 * @return the new index where the pNewDataRow is inserted in pDataPage.
	 * @throws ModelException if the add of the DataRow fails.
	 */
	private synchronized int addDataRowAtRightPosition(ChangeableDataRow pNewDataRow, MemDataPage pDataPage, int pRow) throws ModelException
	{
		boolean bInFilter = true;
		
		// if its refetchable(RemoteDataBook), then check filter, then sort it in.
		if (isDataPageRefetchPossible())
		{
			// if server filter or sort, try to find the correct position position in memory. 
	        if (getFilter() != null && !isMemFilter())
	        {
	        	if (!getFilter().isFulfilled(pNewDataRow))
	        	{
	        		bInFilter = false;
	        	}
	        }
	        if (bInFilter && getSort() != null && !isMemSort())
	        {
	        	for (int iSize = pDataPage.getRowCount(); 
	        	         pRow < iSize && pDataPage.getDataRow(pRow).compareTo(pNewDataRow, sSort) <= 0; pRow++)
	        	{
	        		// nothing todo, it add +1 to iRow for the correct index to sort into.
	        	}
	        }
		}
		// if in Filter then insert it. 
		// if its MemDataBook (=!isDataPageRefetchPossible()), then insert always. After that the filter/sort is cleared to refilter/resort!
		if (bInFilter)
		{
			// insert in new page
			pDataPage.insert(pRow); 							
			pDataPage.setDataRow(pRow, pNewDataRow);
			pDataPage.store(pRow); 
		}
		if (!isDataPageRefetchPossible())
		{				
			// clear filter & sort; should be renewed if MemDataBook!
			pDataPage.clear();			

			// find the same row by PrimaryKey
			String[] asPKColumns = rdRowDefinition.getPrimaryKeyColumnNames();
			if (asPKColumns == null)
			{
				asPKColumns = rdRowDefinition.getColumnNames();
			}			
			pRow = pDataPage.searchNext(Filter.createEqualsFilter(pNewDataRow, asPKColumns));			
		}
		return pRow;
	}
	
	/**
	 * It moves the current DataRow to the master column values corresponding DataPage.
	 * 
	 * @throws ModelException move of the DataRow fails.
	 */
	private synchronized void moveDataRowToCorrectMaster() throws ModelException
	{	
		String[] masterCols = rdMasterReference.getColumnNames();

		// create new MasterRow.
		IDataRow drNewMasterRow = rdMasterReference.getReferencedDataBook().createEmptyRow(
										getMasterReference().getReferencedColumnNames());
		drNewMasterRow.setValues(getMasterReference().getReferencedColumnNames(), this.getValues(masterCols));
		
		// is new one already existing?
		MemDataPage dpNewDataPage = null;
		int iRow = 0;
		ChangeableDataRow drNew = new ChangeableDataRow(getRowDefinition(),
														oaStorage.clone(),
														dpNewDataPage,
														iRow);	
		if (hasDataPage(drNewMasterRow))
		{
			dpNewDataPage = (MemDataPage)getDataPage(drNewMasterRow);
			addDataRowAtRightPosition(drNew, dpNewDataPage, 0);
		}
		else if (!isDataPageRefetchPossible())
		{
			// getNew DataPage and add the Updated DataRow.
			dpNewDataPage = (MemDataPage)getDataPage(drNewMasterRow);
			drNew = new ChangeableDataRow(getRowDefinition(),
											oaStorage.clone(),
											dpNewDataPage,
											iRow);			
			dpNewDataPage.insert(iRow); 						
			dpNewDataPage.setDataRow(iRow, drNew);
			dpNewDataPage.store(iRow);							
		}
		
        // delete row from old page
		if (!dpCurrentDataPage.delete(iSelectedRowIndex))
		{
			if (!dpCurrentDataPage.store(iSelectedRowIndex))
			{
				oaStorage = dpCurrentDataPage.getDataRowStorage(iSelectedRowIndex);
			}			
		}	
		setDetailsChangedInMasterBook();

		correctSelectedRowAfterDelete();	
	}

	/**
	 * Saves the selectedRow internal. See also saveSelectedRow()
	 * 
	 * @throws ModelException if the save fails.
	 */
	private synchronized void saveSelectedRowIntern() throws ModelException
	{
        // sync has to be after invokeSaveEditingControls, because events on editors could cause an empty dbCurrentDataPage
		// therefore an sync is needed!
		sync();
		
		if (isDeleting() || isUpdating() || isInserting())
		{
			if (isDeleting())
			{
				IDataRow drOld = getDataRow(iSelectedRowIndex);
				if (eventBeforeDeleted != null)
				{
					eventBeforeDeleted.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_DELETED, drOld));
				}
				
				// delete details first
	    		if (isSelfJoined())
	    		{
		        	if (isWritebackEnabled())
		        	{
		        		saveSelfJoinedDetailRows();
		        	}
		        	else
		        	{
		        		// no Writeback enabled, just remove the DataPage from Storage to remove all details
		    			if (treePath != null)
		    			{
		    				removeDataPage(null, treePath.getChildPath(getSelectedRow()));
		    			}
		    			else
		    			{
		    				removeDataPage(null, null);
		    			}
		        	}
	    		}
				if (auDetailDataBooks != null)
				{
					for (int i = 0, size = auDetailDataBooks.size(); i < size; i++)
					{
						IDataBook dataBook = auDetailDataBooks.get(i).get();
						if (dataBook != null)
						{
				        	if (dataBook.isDeleteCascade())
				        	{
				        		dataBook.saveAllRows();
				        	}
				        	else
				        	{
				        		dataBook.restoreAllRows();
			        			dataBook.removeDataPage(this, null);
				        	}
						}
			        }					
				}
	
		        executeDelete();
		        store();
	
		        // #378 - Exception in AfterDeleted Event, if an insert() is called in the user event 
		        // correct selected row before the AfterDeleted Event, if the last row got deleted
				correctSelectedRowAfterDelete();
				
				if (eventAfterDeleted != null)
				{
					eventAfterDeleted.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_DELETED, drOld));
				}
	
				if (eventAfterRowSelected != null)
				{
					eventAfterRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_ROW_SELECTED, drOld));
				}
			}
			else if (isUpdating())
			{
				IDataRow drOld = getOriginalRow();
				if (eventBeforeUpdated != null)
				{
					eventBeforeUpdated.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_UPDATED, drOld));
				}
				
				// #163 - if no writeable column is affected, don't update to the server! 
				// -> isWriteable() columns just came from server over the getMetaData() call in RemoteDataBook.
				if (isWritableColumnChanged())
				{
					executeUpdate();				
				}
		        
		        // reparent DataRow if it should moved, because of an Master-Link/Selfjoined column update!
				if (getMasterReference() != null)
				{
					String[] masterCols = rdMasterReference.getColumnNames();

					if (!drOld.equals(this, masterCols))
					{
						moveDataRowToCorrectMaster();						
					}
					else
					{
						//store();
						repositionCurrentDataRow();	
					}
				}
				else
				{
					//store();
					repositionCurrentDataRow();	
				}
				
				invokeMasterChangedDetailsListeners();
				
				if (eventAfterUpdated != null)
				{
					eventAfterUpdated.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_UPDATED, drOld));
				}
			}
			else if (isInserting())
			{
				if (eventBeforeInserted != null)
				{
					eventBeforeInserted.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_INSERTED, null));
				}
				// if insert Master first
				if (getMasterReference() != null)
				{
			        ReferenceDefinition dbParentReference = isSelfJoined() ? getRootReference() : getMasterReference();
					if (dbParentReference != null && dbParentReference.getReferencedDataBook().isInserting())
					{
						IDataBook dbReferenced = dbParentReference.getReferencedDataBook();
						if (dbReferenced instanceof MemDataBook)
						{
							((MemDataBook)dbParentReference.getReferencedDataBook()).saveSelectedRowIntern();
						}
						else
						{
							((MemDataBook)dbParentReference.getReferencedDataBook()).saveSelectedRowIntern();
						}
						// #315 - saveSelectedRow() bug with more then one leave with isInsertung state in DATASOURCE level
						// -> sync all details , before we save, so that bMaster changed is false again, and the changes of this level can be 
						// propagated to the details!
						syncDetails();
					}
				}
	
				// save Parent in Tree first.
	    		if (isSelfJoined())
	    		{
	    			// get Master.
	    			TreePath tpOld        = getTreePath();
	    			int      iOldRowIndex = iSelectedRowIndex;
	    			
	    			// if I am not the master
	    			if (tpOld != null && tpOld.length() > 0)
	    			{
		    			// save the master
	    				setTreePathInternal(tpOld.getParentPath());
	    				setSelectedRow(tpOld.getLast());

		    			saveSelectedRowIntern();
		       			
		    			//select back to the current row.
		    			setTreePathInternal(tpOld);
	    				setSelectedRow(iOldRowIndex);
	    			}
	    		}
	    		
				copyMasterColumnsToCurrentDetail(true); 
	
				executeInsert();
				
		        // reparent DataRow if it should moved, because of an Master-Link/Selfjoined column update!
				if (getMasterReference() != null)
				{
					String [] saColumns       = rdMasterReference.getColumnNames();
					IDataRow drCurrentDataRow = createDataRow(saColumns);
					IDataRow drMasterDataRow  = createDataRow(saColumns);
					drMasterDataRow.setValues(saColumns, 
							dpCurrentDataPage.getMasterDataRow().getValues(rdMasterReference.getReferencedColumnNames()));
					
					// compare Master col values with corresponding columns in current row.
					boolean bEqual = drMasterDataRow.compareTo(drCurrentDataRow) == 0;
					
					// if !=, then move the row!
					if (!bEqual)    
					{ 
						moveDataRowToCorrectMaster();
					}
					else
					{
						//store();
						repositionCurrentDataRow();	
					}
				}
				else
				{
					//store();
					repositionCurrentDataRow();	
				}		
				
				if (isSelfJoined() && hasDataPage(this))
				{
					notifyMasterChanged();
				}
				invokeMasterChangedDetailsListeners();
				
				// #288 - master DataRow rehash not working in DATASOUCE Level over more then 1 hierarchy ! - bug fixed.
				setUID(null);
				
				if (eventAfterInserted != null)
				{
					eventAfterInserted.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_INSERTED, null));
				}
			}			
		}
	}
	
	/**
	 * Save all Changed rows in the selfjoined details. It saves hierarchy down to the last leeve/row.
	 * 
	 * @throws ModelException if an Error occur during save to storage.
	 */ 
	private synchronized void saveSelfJoinedDetailRows() throws ModelException
	{ 
		if (isSelfJoined() && hasDataPage(this) && iSelectedRowIndex >= 0)
		{
			TreePath tpOld        = getTreePath();
			int      iOldRowIndex = iSelectedRowIndex;
			
			// use the current row as master
			setTreePathInternal(tpOld.getChildPath(iSelectedRowIndex));
			saveDataPageIntern();			
			
			//select back to the current row.
			setTreePathInternal(tpOld);
			if (iOldRowIndex != iSelectedRowIndex)
			{
				setSelectedRow(iOldRowIndex);
			}
		}	
	}

	/** 
	 * {@inheritDoc}
	 */
	public synchronized void saveDataPage() throws ModelException
	{
		// store Selection & TreePath
		int iOldSelectedRow = iSelectedRowIndex;
		ArrayUtil<TreePath> auOldTreePath = getAllDetailsTreePath();
		
		saveDataPageIntern();
		
		// restore selection and TreePath
		if (iOldSelectedRow >= getRowCount())
		{
			iSelectedRowIndex = iOldSelectedRow;
			correctSelectedRowAfterDelete();		
		}
		else
		{
			if (iSelectedRowIndex != iOldSelectedRow)
			{
				setSelectedRow(iOldSelectedRow);					
			}
			setAllDetailsTreePath(auOldTreePath);
		}		
	}

	/**
	 * Saves all rows in the current DataPage.
	 * It stores/restores no selectedRow&TreePath
	 * 
	 * @throws ModelException if saveSelected() fails
	 */
	private synchronized void saveDataPageIntern() throws ModelException
	{
		int[] iaChangedRows = dpCurrentDataPage.getChangedDataRows();
		
		if (iaChangedRows.length > 0) 
		{
			ICondition oldFilter = null;

			try
			{
				// setFilter null, to see all changed rows!
				if (!(!isMemFilter() || getWritebackIsolationLevel() == WriteBackIsolationLevel.DATA_ROW)
						&& getFilter() != null)
				{
					oldFilter = getFilter();
					setFilter(null);
				}
				
				// #391 - Wrong execution order in DataSource level 
				// no sort, do/redo all changes in the following order
				// 1) all deletes
				// 2) all updates
				// 3) all inserts
				
				// process delete() changes - sorted. -> Performance optimization. 
				Arrays.sort(iaChangedRows);
                for (int i = iaChangedRows.length - 1; i >= 0; i--)
                {
                     if (getDataRow(iaChangedRows[i]).isDeleting())
                     {
                           setSelectedRow(iaChangedRows[i]);
                           saveSelectedRowInternIncludeDetails();
                     }
                }
				iaChangedRows = dpCurrentDataPage.getChangedDataRows();
				for (int i = 0; i < iaChangedRows.length; i++)
				{
					if (getDataRow(iaChangedRows[i]).isUpdating())
					{
						setSelectedRow(iaChangedRows[i]);
						saveSelectedRowInternIncludeDetails();
					}
				}			
				iaChangedRows = dpCurrentDataPage.getChangedDataRows();
				for (int i = 0; i < iaChangedRows.length; i++)
				{
					if (getDataRow(iaChangedRows[i]).isInserting())
					{
						setSelectedRow(iaChangedRows[i]);
						saveSelectedRowInternIncludeDetails();
					}
				}		
			}
			finally
			{
				if (oldFilter != null)
				{
					setFilter(oldFilter);
				}				
			}
		}
	}
	
	/**
	 * Save all changes from top -> down to all detail Databooks. It just save the changes in the specified pDataBook.
	 * 
	 * @param pCurrentDataBook	the Current DataBook in the Levels from Root DataBook to the last detail DataBook.
	 * @param pDataBook		  	the DataBook in which the changes will be saved.
	 * @throws ModelException 	if an Error occur during save to storage.
	 */
	private synchronized void saveTopDownAllRows(IDataBook pCurrentDataBook, IDataBook pDataBook) throws ModelException
	{
		if (pCurrentDataBook == pDataBook)
		{
			pCurrentDataBook.saveAllRows();
		}
		else
		{
			int         iOldRowIndex     = pCurrentDataBook.getSelectedRow();
			IDataBook[] auDetailDataBook = pCurrentDataBook.getDetailDataBooks();
			ArrayUtil<TreePath> auOldTreePath = new ArrayUtil<TreePath>();

			if (auDetailDataBook != null)
			{
				for (int i = 0, iSize = auDetailDataBook.length; i < iSize; i++)
				{
					if (auDetailDataBook[i].isSelfJoined())
					{
						auOldTreePath.add(auDetailDataBook[i].getTreePath());
					}
				}
			}			
			
			int[] iaChangedRows = pCurrentDataBook.getChangedDataRows();
			
			for (int i = 0, iSize = iaChangedRows.length; i < iSize; i++)
			{
				pCurrentDataBook.setSelectedRow(iaChangedRows[i]);
				if (pCurrentDataBook.isDetailChanged())
				{
					if (auDetailDataBook != null)
					{
						//save all details, till the detail is not changed anymore!
						for (int j = 0, jSize = auDetailDataBook.length; j < jSize && pCurrentDataBook.isDetailChanged(); j++)
						{
							if (auDetailDataBook[j].isSelfJoined())
							{
								// start in Detail at the root node of the hierarchy
								auDetailDataBook[j].setTreePath(null);
							}
							saveTopDownAllRows(auDetailDataBook[j], pDataBook);
						}
					}	
					
					// if still changes exists, and self joined, then save the self joined hierarchy down too.
					if (pCurrentDataBook.isSelfJoined() 
							&& pCurrentDataBook.isDetailChanged()
							&& pCurrentDataBook.hasDataPage(pCurrentDataBook))
					{
						// get Master.
						TreePath tpOld = pCurrentDataBook.getTreePath();
						
						// use the current row as master
						pCurrentDataBook.setTreePath(tpOld.getChildPath(pCurrentDataBook.getSelectedRow()));
						
						saveTopDownAllRows(pCurrentDataBook, pDataBook);
						
						//select back the treepath, selected row will set back later in this method.
						pCurrentDataBook.setTreePath(tpOld);
					}					
				}
			}
			
			pCurrentDataBook.setSelectedRow(iOldRowIndex);
			
			if (auDetailDataBook != null)
			{
				int j = 0;
				for (int i = 0, iSize = auDetailDataBook.length, jSize = auOldTreePath.size(); 
				     i < iSize && j < jSize; i++)
				{
					if (auDetailDataBook[i].isSelfJoined())
					{
						auDetailDataBook[i].setTreePath(auOldTreePath.get(j));
						j++;
					}
				}
			}				
		}
	}
	
	/**
	 * Returns all current TreePaths of all Detail DataBooks in a List.
	 * 
	 * @return all current TreePaths of all Detail DataBooks in a List.
	 */
	private ArrayUtil<TreePath> getAllDetailsTreePath()
	{
		ArrayUtil<TreePath> auOldTreePath = new ArrayUtil<TreePath>();

		if (auDetailDataBooks != null)
		{
			for (int i = 0, iSize = auDetailDataBooks.size(); i < iSize; i++)
			{
				if (auDetailDataBooks.get(i).get().isSelfJoined())
				{
					auOldTreePath.add(auDetailDataBooks.get(i).get().getTreePath());
				}
			}
		}
		return auOldTreePath;
	}
	
	/**
	 * Sets all TreePaths of all Detail DataBooks from the List.
	 * 
	 * @param pOldTreePath 		List of TreePath (only one for each self joined Detail DataBook)
	 * @throws ModelException	If the TreePath couldn't set.
	 */
	private void setAllDetailsTreePath(ArrayUtil<TreePath> pOldTreePath) throws ModelException
	{
		if (auDetailDataBooks != null)
		{
			int j = 0;
			for (int i = 0, iSize = auDetailDataBooks.size(), jSize = pOldTreePath.size(); 
			     i < iSize && j < jSize; i++)
			{
				if (auDetailDataBooks.get(i).get().isSelfJoined())
				{
					auDetailDataBooks.get(i).get().setTreePath(pOldTreePath.get(j));
					j++;
				}
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void saveAllRows() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		if (htDataPagesCache.isEmpty() && dpCurrentDataPage == null)
		{
			//no pages, no changes return
			return;
		}
		
		// Editors has to save first to recognize changed Rows!
		invokeSaveEditingControls();

		// sync has to be after invokeSaveEditingControls, because events on editors could cause an empty dbCurrentDataPage
		// therefore an sync is needed!
		sync();	 
		
		if (getWritebackIsolationLevel() == WriteBackIsolationLevel.DATA_ROW)
		{
			saveSelectedRow();
		}
		else
		{
			int iOldSelectedRow = iSelectedRowIndex;
			ArrayUtil<TreePath> auOldTreePath = getAllDetailsTreePath();
	
			ICondition oldFilter = null;
			
			try
			{
				// setFilter null, to see all changed rows!
				if (!(!isMemFilter() || getWritebackIsolationLevel() == WriteBackIsolationLevel.DATA_ROW)
						&& getFilter() != null)
				{
					oldFilter = getFilter();
					setFilter(null);
				}
				
				// Save all rows in current DataPage, then in all others.
				saveDataPageIntern();
					
				// save all changes in the Selfjoined hierarchy
				if (isSelfJoined())
				{
					// select Root Node and go through all loaded detail Pages.
					TreePath tpOld = getTreePath();
					
					// if I am not the Root Node, then select it
					if (tpOld != null && tpOld.length() > 0)
					{		
						setTreePathInternal(null);
					}
					
					for (int i = 0, iRowCount = getRowCount(); i < iRowCount; i++)
					{
						setSelectedRow(i);
						saveSelfJoinedDetailRows();
					}		
		
					// if I am not the Root Node, then restore TreePath
					if (tpOld != null && tpOld.length() > 0)
					{		
						//select back to the current row.
						setTreePathInternal(tpOld);
					}
				}	
				
				// then save changes in all loaded DataPages
				if (!bSaveAllRows && getMasterReference() != null)
				{
					// check if still changes exists
					Enumeration<MemDataPage> pages = htDataPagesCache.elements();
					MemDataPage              mdp;
					boolean                  bChangeFound = false;
					while (pages.hasMoreElements() && !bChangeFound)
					{
						mdp = pages.nextElement();
						if (mdp.getChangedDataRows().length > 0)
						{
							bChangeFound = true;
						}
					}
		
					// Changes exits, then go to the root DataBook, and go through all changed Rows and save all in "this" DataBook.
					if (bChangeFound)
					{
						IDataBook           dbRoot = this;
						ReferenceDefinition dbParentReference = dbRoot.isSelfJoined() ? dbRoot.getRootReference() : dbRoot.getMasterReference();
						while (dbParentReference != null)
						{
							dbRoot            = dbParentReference.getReferencedDataBook();
					        dbParentReference = dbRoot.isSelfJoined() ? dbRoot.getRootReference() : dbRoot.getMasterReference();
						}
						
						TreePath tpOld = null;
						if (dbRoot.isSelfJoined())
						{
							// start at the root node of the hierarchy
							tpOld = dbRoot.getTreePath();
							dbRoot.setTreePath(null);
						}
						
						bSaveAllRows = true;
						try
						{
							saveTopDownAllRows(dbRoot, this);
						}
						finally
						{
							bSaveAllRows = false;
						}
						
						if (dbRoot.isSelfJoined() && tpOld != null)
						{
							// restore Root TreePath
							dbRoot.setTreePath(tpOld);
						}
					}
				}
			}
			finally
			{
				if (oldFilter != null)
				{
					setFilter(oldFilter);
				}				
			}			
	
			// restore selection and TreePath
			if (iOldSelectedRow >= getRowCount())
			{
				iSelectedRowIndex = iOldSelectedRow;
				correctSelectedRowAfterDelete();		
			}
			else
			{
				if (iSelectedRowIndex != iOldSelectedRow)
				{
					setSelectedRow(iOldSelectedRow);					
				}
				setAllDetailsTreePath(auOldTreePath);
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized boolean isOutOfSync()
	{
		return dpCurrentDataPage == null || (rdMasterReference != null && bMasterChanged);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized boolean isAllFetched() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		sync();
		return dpCurrentDataPage.isAllFetched();
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void fetchAll() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		sync();
		if (dpCurrentDataPage == dpEmptyDataPage)
		{
			return;
		}
		dpCurrentDataPage.fetchAll();
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized IDataRow getMasterDataRow() throws ModelException
	{
		if (!isOpen())
		{
			return null;	
		}
		sync();
		return dpCurrentDataPage.getMasterDataRow();
	}	
	
	/**
	 * {@inheritDoc}
	 */
	public void addDetailDataBook(IDataBook pDetailDataBook)
	{
		if (auDetailDataBooks == null)
		{
			auDetailDataBooks = new ArrayUtil<WeakReference<IDataBook>>();
		}
		else
		{
			for (int i = auDetailDataBooks.size() - 1; i >= 0; i--)
			{
				if (auDetailDataBooks.get(i).get() == null)
				{
					auDetailDataBooks.remove(i);
				}
			}			
		}
		auDetailDataBooks.add(new WeakReference(pDetailDataBook));
	}

	/**
	 * {@inheritDoc}
	 */
	public void removeDetailDataBook(IDataBook pDetailDataBook)
	{
		if (auDetailDataBooks != null)
		{
			for (int i = auDetailDataBooks.size() - 1; i >= 0; i--)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook == null || dataBook == pDetailDataBook)
				{
					auDetailDataBooks.remove(i);
				}
			}
		}
	}	
	
	/**
	 * {@inheritDoc}
	 */
	public IDataBook[] getDetailDataBooks()
	{
		ArrayUtil<IDataBook> result = new ArrayUtil<IDataBook>();
		if (auDetailDataBooks != null)
		{
			for (int i = auDetailDataBooks.size() - 1; i >= 0; i--)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook == null)
				{
					auDetailDataBooks.remove(i);
				}
				else
				{
					result.add(0, dataBook);
				}
			}
		}
		return result.toArray(new IDataBook[result.size()]);
	}
	
	/**
	 * Returns only all Changed DataRows with the states isInserting(), isUpdating(), isDeleting(). 
	 * Rows with isDetailChanged() will not be returned.
	 * 
	 * @return only all Changed DataRows with the states isInserting(), isUpdating(), isDeleting(). 
	 * 
	 * @throws ModelException if the changed rows couldn't determined.
	 */
	public synchronized int[] getOnlyIUDChangedDataRows() throws ModelException
	{
		int[] auChanges = getChangedDataRows();
		
		for (int i = 0; auChanges != null && i < auChanges.length;)
		{
			IChangeableDataRow row = getDataRow(auChanges[i]);
			if (!(row.isDeleting() || row.isUpdating() || row.isInserting()))
			{
				auChanges = ArrayUtil.remove(auChanges, i);
			}
			else
			{
				i++;
			}
		}
		
		if (auChanges == null)
		{
			auChanges = new int[] {};
		}
		return auChanges;
	}	
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized int[] getChangedDataRows() throws ModelException
	{
		if (!isOpen())
		{
			return null;			
		} 
		if (dpCurrentDataPage == null)
		{
			return null;
		}
		
		// #286 - no DataPage to master row exists, then return null -> no changes can exits - performance tuning!
		if (getMasterReference() != null && !isSelfJoined())
		{
			IDataBook dbMaster = getMasterReference().getReferencedDataBook();
			IDataRow drMaster = getMasterDataRow(dbMaster);
			if (htDataPagesCache.get(drMaster) == null)
			{
				if (dbMaster.getUID() != null)
				{
					if (htDataPagesCache.get(dbMaster.getUID()) == null)
					{
						return null;
					}
				}
				else
				{
					return null;
				}
			}			
		}
		
		sync();
		
		return dpCurrentDataPage.getChangedDataRows();
	}	
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized IDataPage getDataPage(IDataRow pMasterDataRow) throws ModelException
	{
		if (getMasterReference() == null)
		{
			sync();
			return dpCurrentDataPage;
		}
		else
		{
			return getDataPageIntern(pMasterDataRow);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized boolean hasDataPage(IDataRow pMasterDataRow) throws ModelException
	{
		if (getMasterReference() == null)
		{
			if (pMasterDataRow == null)
			{
				return true;
			}
			return false;
		}
		else
		{
			IDataRow drMaster = getMasterDataRow(pMasterDataRow);
			MemDataPage dpNewCurrentDataPage = htDataPagesCache.get(drMaster);			

			if (dpNewCurrentDataPage == null && pMasterDataRow instanceof IChangeableDataRow)
			{
				if (((IChangeableDataRow)pMasterDataRow).getUID() != null)
				{
					dpNewCurrentDataPage = htDataPagesCache.get(((IChangeableDataRow)pMasterDataRow).getUID());
				}
			}
		
			return dpNewCurrentDataPage != null;
		}
	}

	/**
	 * Returns true if an <code>IDataPage</code> to specified root row and TreePath
	 * from the master <code>DataBook</code> exists.
	 * 
	 * @param pRootDataRow	the root <code>IDataRow</code> of the <code>DataBook</code> to check.
	 * @param pTreePath		the <code>TreePath</code> to use.
	 * @throws ModelException 
	 * 			  if the pRootDataRow don't contains the root columns from the DataBook.
	 * @return true if an <code>IDataPage</code> to specified root row and TreePath
	 * 		   from the master <code>DataBook</code> exists.
	 */
	public synchronized boolean hasDataPage(IDataRow pRootDataRow, TreePath pTreePath) throws ModelException
	{
		IDataRow drMaster = getMasterDataRowFromRootDataRow(pRootDataRow);
		MemDataPage dpNewCurrentDataPage = htDataPagesCache.get(drMaster);			
		
		if (dpNewCurrentDataPage == null && pRootDataRow instanceof IChangeableDataRow)
		{
			if (((IChangeableDataRow)pRootDataRow).getUID() != null)
			{
				dpNewCurrentDataPage = htDataPagesCache.get(((IChangeableDataRow)pRootDataRow).getUID());
			}
		}
		
		if (pTreePath == null)
		{
			return dpNewCurrentDataPage != null;
		}		

		for (int i = 0; i < pTreePath.length(); i++)
		{
			int j = pTreePath.get(i);
			if (j < dpNewCurrentDataPage.getRowCount())
			{
				drMaster = getMasterDataRow(dpNewCurrentDataPage.getDataRow(j));
				dpNewCurrentDataPage = htDataPagesCache.get(drMaster);			
				
				if (dpNewCurrentDataPage == null && drMaster instanceof IChangeableDataRow)
				{
					if (((IChangeableDataRow)drMaster).getUID() != null)
					{
						dpNewCurrentDataPage = htDataPagesCache.get(((IChangeableDataRow)drMaster).getUID());
					}
				}			
				
				if (dpNewCurrentDataPage == null)
				{
					return false;
				}
			}
			else
			{
				return false;
			}
		}
		return true;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeRowSelected()
	{
		if (eventBeforeRowSelected == null)
		{
			eventBeforeRowSelected = new DataBookHandler();
		}
		return eventBeforeRowSelected;
	}

	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterRowSelected()
	{
		if (eventAfterRowSelected == null)
		{
			eventAfterRowSelected = new DataBookHandler();
		}
		return eventAfterRowSelected;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeInserting()
	{
		if (eventBeforeInserting == null)
		{
			eventBeforeInserting = new DataBookHandler();
		}
		return eventBeforeInserting;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterInserting()
	{
		if (eventAfterInserting == null)
		{
			eventAfterInserting = new DataBookHandler();
		}
		return eventAfterInserting;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeInserted()
	{
		if (eventBeforeInserted == null)
		{
			eventBeforeInserted = new DataBookHandler();
		}
		return eventBeforeInserted;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterInserted()
	{
		if (eventAfterInserted == null)
		{
			eventAfterInserted = new DataBookHandler();
		}
		return eventAfterInserted;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeUpdating()
	{
		if (eventBeforeUpdating == null)
		{
			eventBeforeUpdating = new DataBookHandler();
		}
		return eventBeforeUpdating;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterUpdating()
	{
		if (eventAfterUpdating == null)
		{
			eventAfterUpdating = new DataBookHandler();
		}
		return eventAfterUpdating;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeUpdated()
	{
		if (eventBeforeUpdated == null)
		{
			eventBeforeUpdated = new DataBookHandler();
		}
		return eventBeforeUpdated;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterUpdated()
	{
		if (eventAfterUpdated == null)
		{
			eventAfterUpdated = new DataBookHandler();
		}
		return eventAfterUpdated;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeDeleting()
	{
		if (eventBeforeDeleting == null)
		{
			eventBeforeDeleting = new DataBookHandler();
		}
		return eventBeforeDeleting;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterDeleting()
	{
		if (eventAfterDeleting == null)
		{
			eventAfterDeleting = new DataBookHandler();
		}
		return eventAfterDeleting;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeDeleted()
	{
		if (eventBeforeDeleted == null)
		{
			eventBeforeDeleted = new DataBookHandler();
		}
		return eventBeforeDeleted;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterDeleted()
	{
		if (eventAfterDeleted == null)
		{
			eventAfterDeleted = new DataBookHandler();
		}
		return eventAfterDeleted;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeRestore()
	{
		if (eventBeforeRestore == null)
		{
			eventBeforeRestore = new DataBookHandler();
		}
		return eventBeforeRestore;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterRestore()
	{
		if (eventAfterRestore == null)
		{
			eventAfterRestore = new DataBookHandler();
		}
		return eventAfterRestore;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeReload()
	{
		if (eventBeforeReload == null)
		{
			eventBeforeReload = new DataBookHandler();
		}
		return eventBeforeReload;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterReload()
	{
		if (eventAfterReload == null)
		{
			eventAfterReload = new DataBookHandler();
		}
		return eventAfterReload;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeColumnSelected()
	{
		if (eventBeforeColumnSelected == null)
		{
			eventBeforeColumnSelected = new DataBookHandler();
		}
		return eventBeforeColumnSelected;
	}

	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterColumnSelected()
	{
		if (eventAfterColumnSelected == null)
		{
			eventAfterColumnSelected = new DataBookHandler();
		}
		return eventAfterColumnSelected;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public int searchNext(ICondition pCondition) throws ModelException
	{
		sync();
		
		return dpCurrentDataPage.searchNext(pCondition);
	}

	/**
	 * {@inheritDoc}
	 */
	public int searchNext(ICondition pCondition, int pRowNum) throws ModelException
	{
		sync();
		
		return dpCurrentDataPage.searchNext(pCondition, pRowNum);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public int searchPrevious(ICondition pCondition) throws ModelException
	{
		sync();
		
		return dpCurrentDataPage.searchPrevious(pCondition);
	}

	/**
	 * {@inheritDoc}
	 */
	public int searchPrevious(ICondition pCondition, int pRowNum) throws ModelException
	{
		sync();
		
		return dpCurrentDataPage.searchPrevious(pCondition, pRowNum);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getRowIndex()
	{
		try
		{
			sync();
			return iSelectedRowIndex;
		}
		catch (ModelException e)
		{
			return -1;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IDataPage getDataPage()
	{
		try
		{
			sync();
			return dpCurrentDataPage;
		}
		catch (ModelException e)
		{
			return null;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void finalize() throws Throwable
	{
		try
		{
			close();
		}
		catch (Throwable pThrowable)
		{
			// Silent close();
		}
		super.finalize();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized Object getValue(String pColumnName) throws ModelException
	{
		if (!isOpen())
		{
			return null;
		}
		sync();
		if (iSelectedRowIndex < 0)
		{
			return null;
		}
		return super.getValue(pColumnName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void setValue(String pColumnName, Object pValue) throws ModelException
	{				
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		sync();
		if (iSelectedRowIndex < 0)
		{
			throw new ModelException("No selected row!");			
		}		
		if (isDeleting())
		{
			throw new ModelException("Row is already deleted!");			
		}		
		update();

		super.setValue(pColumnName, pValue);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized Object[] getValues(String[] pColumnNames) throws ModelException
	{
		if (!isOpen())
		{
			return null;	
		}
		sync();			

		if (iSelectedRowIndex < 0)
		{
			return null;
		}		
		return super.getValues(pColumnNames);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void setValues(String[] pColumnNames, Object[] pValues) throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		sync();
		if (iSelectedRowIndex < 0)
		{
			throw new ModelException("No selected row!");			
		}		
		if (isDeleting())
		{
			throw new ModelException("Row is already deleted!");			
		}				
		update();

		if (pColumnNames == null)
		{
			pColumnNames = rdRowDefinition.getColumnNames();
		}
		
		super.setValues(pColumnNames, pValues);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized String toString()
	{
		StringBuilder sbResult = new StringBuilder();
		
		sbResult.append(getClass().getName());
		sbResult.append(" ::");
		sbResult.append(sName);
		sbResult.append(":: \n  isOpen         = ");
		sbResult.append(bIsOpen);
		sbResult.append("\n");
		
		if (rdRowDefinition != null)
		{
			sbResult.append("  ");
			sbResult.append(rdRowDefinition);
		}
		if (rdMasterReference != null)
		{
			sbResult.append("  ");
			sbResult.append(rdMasterReference);
			sbResult.append("\n");			
		}
		if (cFilter != null)
		{
			sbResult.append("  Filter         = ");
			sbResult.append(cFilter);
			sbResult.append("\n");			
		}	
		if (sSort != null)
		{
			sbResult.append("  Sort           = ");
			sbResult.append(sSort);
			sbResult.append("\n");			
		}	
		if (auDetailDataBooks != null && auDetailDataBooks.size() > 0)
		{
			sbResult.append("  DetailBooks    =");
			
			for (int i = 0; i < auDetailDataBooks.size(); i++)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook != null)
				{
					sbResult.append(" ");
					sbResult.append(dataBook.getName());
				}
			}		
			sbResult.append("\n");
		}
		
		if (isOpen())
		{
			try
			{
				sync();
			}
			catch (ModelException modelException)
			{
				return sbResult.toString() + " :: " + modelException.getMessage();
			}
				
			if (treePath != null && treePath.length() > 0)
			{
				sbResult.append("  TreePath       = ");
				sbResult.append(treePath);
				sbResult.append("\n");			
			}
			sbResult.append("  SelectedRow    = ");
			sbResult.append(iSelectedRowIndex);
			sbResult.append("\n\nCurrent DataPage:\n  ");
			sbResult.append(dpCurrentDataPage);
			
			Enumeration<MemDataPage> dpCacheElements = htDataPagesCache.elements();
			Enumeration<Object> dpCacheKeys = htDataPagesCache.keys();
			if (dpCacheElements.hasMoreElements())
			{
				sbResult.append("\nDataPagesCache:\n");
				
				while (dpCacheElements.hasMoreElements())
				{
					MemDataPage dpElement = dpCacheElements.nextElement();
					Object drKey     = dpCacheKeys.nextElement();
					if (dpElement != dpCurrentDataPage)
					{
						sbResult.append("  Key=[");
						sbResult.append(drKey);
						sbResult.append("]=");
						sbResult.append(dpElement);
						sbResult.append("\n");
					}
					else
					{
						sbResult.append("  Key=[");
						sbResult.append(drKey);
						sbResult.append("] = Current DataPage\n");
					}
				}
			}
		}
		
        return sbResult.toString();
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	
	
	/**
	 * It constructs a new MemDataPage. Its every time used, when a new MemDataPage is 
	 * necessary.
	 * Derived classed should override this and create the needed IDataPage.
	 *  
	 * @param pMasterDataRow	the master IDataRow for this IDataPage.
	 * @return the new MemDataPage to use.
	 */
	protected MemDataPage createDataPage(IDataRow pMasterDataRow)
	{
		return new MemDataPage(this, pMasterDataRow);
	}

	/**
	 * Returns true if the sort is handled in memory.
	 *
	 * @return true if the sort is handled in memory.
	 */
	protected boolean isMemSort()
	{
		return bMemSort;
	}

	/**
	 * Sets if the sort is handled in memory.
	 *
	 * @param pMemSort true if the sort is handled in memory. 
	 */
	protected void setMemSort(boolean pMemSort)
	{
		bMemSort = pMemSort;
	}

	/**
	 * Returns true if the filter is handled in memory.
	 *
	 * @return true if the filter is handled in memory.
	 */
	protected boolean isMemFilter()
	{
		return bMemFilter;
	}

	/**
	 * Sets if the sort is handled in memory.
	 *
	 * @param pMemFilter true if the filter is handled in memory. 
	 */
	protected void setMemFilter(boolean pMemFilter)
	{
		bMemFilter = pMemFilter;
	}	
	
	/**
	 * Returns true if this data book should write its changes to the storage.
	 * 
	 * @return true to write back all changes, false otherwise.
	 */
	protected boolean isWritebackEnabled()
	{
		return bWritebackEnabled;
	}
	
	/**
	 * Sets if this data book should write back its changes to the storage.
	 * 
	 * @param pWritebackEnabled determines if changes should be written to the store.
	 */
	protected void setWritebackEnabled(boolean pWritebackEnabled)
	{
		bWritebackEnabled = pWritebackEnabled;
	}	
	
	/**
	 * Returns the selection that is stored at the last reload.
	 * 
	 * @return the selection that is stored at the last reload.
	 */
	private ArrayUtil<IDataRow> getStoredSelection()
	{
		return auStoredSelection;
	}
	
	/**
	 * Sets the the current selection before the reload.
	 * 
	 * @param pStoredSelection	the current selection as IDataRow (PK columns) to store 
	 * @throws ModelException   if in selfjoined DataBooks the parent row couldn't determined.
	 */
	private void setStoredSelection(IDataRow pStoredSelection) throws ModelException
	{
		if (pStoredSelection == null)
		{
			auStoredSelection.clear();
			return;
		}
		auStoredSelection.add(pStoredSelection);

		if (isSelfJoined() && treePath != null)
		{
			String[] asPKColumns = rdRowDefinition.getPrimaryKeyColumnNames();
			if (asPKColumns == null)
			{
				asPKColumns = rdRowDefinition.getColumnNames();
			}
			TreePath currTreePath = treePath;
			for (int i = currTreePath.length() - 1; i >= 0; i--)
			{
				setTreePath(currTreePath.getParentPath());
				if (asPKColumns != null && asPKColumns.length > 0)
				{
					auStoredSelection.add(getDataRow(currTreePath.get(i)).createDataRow(asPKColumns));
				}				
			}
		}
	}
		
	/**
	 * Returns the row count in <code>IDBAccess</code> for this RemoteDataBook.
	 * 
	 * @return the row count of the DataBook.
	 * @throws ModelException
	 * 				the the detail DataBook couldn't synchronized with the master DataBook
	 */
	public synchronized int getEstimatedRowCount() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		sync();
		
		return dpCurrentDataPage.getEstimatedRowCount();
	}	
	
	/**
	 * Its called when the inserting DataRow (new row) should be inserted.
	 * Derived classed should override this to implemented the needed functionality.
	 * e.g. store the new row in the storage.
	 * 
	 * @throws ModelException	if an ModelException occur during insert.
	 */
	protected void executeInsert() throws ModelException
	{
	}
	
	/**
	 * Its called when the updating DataRow (changed row) should be updated
	 * Derived classed should override this to implemented the needed functionality.
	 * e.g. update the old row with the new row in the storage.
	 * 
	 * @throws ModelException	if an ModelException occur during update.
	 */
	protected void executeUpdate() throws ModelException
	{
	}
	
	/**
	 * Its called when the deleting DataRow (delete row) should be deleted.
	 * Derived classed should override this to implemented the needed functionality.
	 * e.g. delete the row in the storage.
	 * 
	 * @throws ModelException	if an ModelException occur during delete.
	 */
	protected void executeDelete() throws ModelException
	{
	}

	/**
	 * Its called before the DataRow (existing row) will be changed.
	 * Derived classed should override this to implemented the needed functionality.
	 * e.g. lock and refetch the actual values of the row from storage
	 * 
	 * @throws ModelException	if an ModelException occur during insert.
	 */
	protected void executeLockAndRefetch() throws ModelException
	{
	}
	
	/**
	 * It invokes for all registered IConrols the notifyRepaint() method and calls for
	 * all detail IDataBooks the the notifyMasterChanged method to determine that the 
	 * master has changed.
	 */
	private void invokeMasterChangedDetailsListeners()
	{	
		invokeRepaintListeners();		
		
		if (auDetailDataBooks != null)
		{
			for (int i = 0, size = auDetailDataBooks.size(); i < size; i++)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook != null)
				{
					dataBook.notifyMasterChanged();
				}
			}
		}
	}	
	
	/**
	 * Sync the DataBook an all it detail DataBooks. Its necessary in DATASOURCE level for saving cascades over varies levels. 
	 *
	 * @throws ModelException if the sync isn't possible.
	 */
	private void syncDetails() throws ModelException
	{	
		sync();
		
		if (auDetailDataBooks != null)
		{
			for (int i = 0, size = auDetailDataBooks.size(); i < size; i++)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook != null)
				{
					dataBook.getSelectedRow(); // calls a sync!!!
				}
			}
		}
	}	

	/**
	 * Returns the master row only with the primary key columns - like defined in
	 * the getMasterReference().getReferencedColumns().
	 * 
	 * @param pMasterDataRow	the full/complete master IDataRow to use.
	 * @return internal the master row only with the primary key columns - like defined in
	 * the getMasterReference().getReferencedColumns().
	 * @throws ModelException	if the master row couldn't created.
	 */
	private IDataRow getMasterDataRow(IDataRow pMasterDataRow) throws ModelException
	{
		return pMasterDataRow.createDataRow(getMasterReference().getReferencedColumnNames());
	}
	
	/**
	 * Returns the MemDataPage to the specified master IDataRow. If the none existing,
	 * it will be a new one created and in the MemDataPage Hashtable stored.
	 *   
	 * @param pMasterDataRow	the master IDataRow to use.
	 * @return the MemDataPage to the specified master IDataRow.
	 * @throws ModelException	if master row or the MemDataPage couldn't created.
	 */
	private MemDataPage getDataPageIntern(IDataRow pMasterDataRow) throws ModelException
	{
		// #336 - First check if a DataPage with the UID exists.
		MemDataPage dpNewCurrentDataPage = null;
		if (pMasterDataRow instanceof ChangeableDataRow)
		{
			Object oUID = ((ChangeableDataRow)pMasterDataRow).getUID();
			if (oUID != null)
			{
				dpNewCurrentDataPage = htDataPagesCache.get(oUID);			
				if (dpNewCurrentDataPage != null)
				{
					return dpNewCurrentDataPage;
				}
			}
		}
		IDataRow drMaster = getMasterDataRow(pMasterDataRow);
		dpNewCurrentDataPage = htDataPagesCache.get(drMaster);			

		// init DataPage, if for this master row doesn't exists a DataPage in Cache
		if (dpNewCurrentDataPage == null)
		{		
			IChangeableDataRow cdrMasterDataRow = null;
			if (pMasterDataRow instanceof IChangeableDataRow)
			{
				if (((IChangeableDataRow)pMasterDataRow).getUID() != null)
				{
					cdrMasterDataRow = (IChangeableDataRow)pMasterDataRow;
					dpNewCurrentDataPage = htDataPagesCache.get(cdrMasterDataRow.getUID());					
				}
			}			

			if (dpNewCurrentDataPage == null)
			{
				dpNewCurrentDataPage = createDataPage(drMaster);
				if (cdrMasterDataRow != null && cdrMasterDataRow.getUID() != null)
				{
					// #290- if the master DataBook is in DATASOURCE level && Inserting() no fetch is necessary, because no rows can exists!
					ReferenceDefinition masterRef = getMasterReference();
					if (masterRef != null 
							&& masterRef.getReferencedDataBook().getWritebackIsolationLevel() == IDataBook.WriteBackIsolationLevel.DATASOURCE)
					{
						dpNewCurrentDataPage.setAllFetched(true);
					}
					
					htDataPagesCache.put(cdrMasterDataRow.getUID(), dpNewCurrentDataPage);	
					   
				}
				else
				{
					htDataPagesCache.put(drMaster, dpNewCurrentDataPage);
				}
			}
		}
		return dpNewCurrentDataPage;
	}

	/**
	 * Returns the the IChangeableDataRow synchronized for the specified row index.
	 * 
	 * @param pDataRowIndex	the row index to use.
	 * @return the the IChangeableDataRow synchronized for the specified row index
	 * @throws ModelException if the IChangeableDataRow couldn't get from the MemDataPage.
	 */
	private synchronized IChangeableDataRow getDataRowInternal(int pDataRowIndex) throws ModelException
	{
		if (pDataRowIndex < 0 || dpCurrentDataPage == dpEmptyDataPage)
		{
			return null;
		}
		
		return dpCurrentDataPage.getDataRow(pDataRowIndex);
	}

	/**
	 * It correct after the sync() call the selected row under consideration of the 
	 * Selection Mode and the the stored selection of the last reload.
	 * 
	 * @throws ModelException	if the row couldn't selected.
	 */
	private void correctSelectedRow() throws ModelException
	{
		ArrayUtil<IDataRow> auCurrentDataRows = getStoredSelection(); 
		
		if (getSelectionMode() == SelectionMode.DESELECTED)
		{
			setSelectedRowInternal(-1);						
		}			
		else if (getSelectionMode() == SelectionMode.CURRENT_ROW && auCurrentDataRows != null && auCurrentDataRows.size() > 0)
		{
			if (auCurrentDataRows.size() > 0 && auCurrentDataRows.get(0) == this) 
			{
				// if the databook was deselected the reload function stores this to identify, that it was deselected.
				setSelectedRowInternal(-1);	
			}
			else
			{
				// find the same row by StoredSelection
				String[] asPKColumns = rdRowDefinition.getPrimaryKeyColumnNames();
				if (asPKColumns == null)
				{
					asPKColumns = rdRowDefinition.getColumnNames();
				}
				
				if (isSelfJoined() && auCurrentDataRows.size() > 0)
				{
					setTreePath(null);
				}
				int i = 0;
				IDataRow drCompare = getDataRow(i);
				IDataRow drCurrent;
				for (int j = auCurrentDataRows.size() - 1; j >= 0; j--)
				{
					drCurrent = auCurrentDataRows.get(j);
					while (drCompare != null 
							&& !drCurrent.equals(drCompare, asPKColumns))
					{
						i++;
						drCompare = getDataRow(i);
					}
					if (drCompare == null)
					{
						i = 0;
					}
					setSelectedRowInternal(i);
					if (isSelfJoined() && j - 1 >= 0)
					{
						setTreePath(getTreePath().getChildPath(i));
					}
					i = 0;
					drCompare = getDataRow(i);
				}
			}
		}
		else
		{
			setSelectedRowInternal(0);						
		}
		setStoredSelection(null);
		if (eventAfterRowSelected != null)
		{
			eventAfterRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_ROW_SELECTED, null));
		}
	}

	/**
	 * Corrects the selected row after delete on the same or one row index before. 
	 * if the DESELECTED and the last row is deleted, it deselects.
	 * 
	 * @throws ModelException if selected row couldn't changed.
	 */
	private void correctSelectedRowAfterDelete() throws ModelException
	{
		int iNewRowIndex = iSelectedRowIndex;
		if (iSelectedRowIndex >= getRowCount())
		{
			if (getSelectionMode() == SelectionMode.DESELECTED)
			{
				iNewRowIndex = -1;		
			}
			else
			{
				iNewRowIndex = getRowCount() - 1;						
			}
		}		
		setSelectedRowInternal(iNewRowIndex);
	}
		
	/**
	 * It synchronize the MemDataBook, at the first access after the (master) change.
	 * It gets the corresponding MemDataPage from the Hashtable. If not possible it will 
	 * be created, or an empty MemDataPage will be filled in when the master has no
	 * selected row. It also does the lazy load after the open()/reload call.  
	 * 
	 * @throws ModelException	if the new MemDataPage couldn't determined.
	 */
	private void sync() throws ModelException
	{		
		// init the CurrentDataPage for a Master DataBook
	    if (dpEmptyDataPage == null)
	    {
	    	dpEmptyDataPage = new MemDataPage(null, null);
	    }
	   
		if (rdMasterReference == null)
		{
			if (dpCurrentDataPage == null)
			{
				ignoreReload = true; // only ignore reload when really syncing
				try
				{
					dpCurrentDataPage = dpEmptyDataPage;
					iSelectedRowIndex = -1;
	
					dpCurrentDataPage  = createDataPage(null);
					correctSelectedRow();
				}
				finally
				{
					ignoreReload = false;
				}
			}
		}
		else if (bMasterChanged)
		{
			bMasterChanged = false;
			
			ignoreReload = true; // only ignore reload when really syncing
			try 
			{
				if (dpCurrentDataPage == null)
				{
					dpCurrentDataPage = dpEmptyDataPage;
					iSelectedRowIndex = -1;
					treePath = new TreePath();
				}
	
				ReferenceDefinition dbParentReference = isSelfJoined() ? getRootReference() : getMasterReference();
				if (dbParentReference != null && dbParentReference.getReferencedDataBook().getSelectedRow() < 0)
				{
					if (dpCurrentDataPage != dpEmptyDataPage) // eventAfterSelected should only occur, if the data page changes.
					{
						dpCurrentDataPage = dpEmptyDataPage;
						iSelectedRowIndex = -1;
						treePath = new TreePath();

						setSelectedRowInternal(-1);
					}
				}
				else
				{
					if (isSelfJoined())
					{
						if (dpCurrentDataPage != dpEmptyDataPage
								&& dbParentReference != null
								&& !getMasterDataRowFromRootDataRow(dbParentReference.getReferencedDataBook()).equals(
										dpCurrentDataPage.getMasterDataRow(), saTreeRootMasterColumnNames))
						{
							treePath = new TreePath();
						}
						
						IDataPage dpNewCurrentDataPage = getDataPage(treePath);
	
						if (dpNewCurrentDataPage != dpCurrentDataPage)
						{
							dpCurrentDataPage = (MemDataPage)getDataPage(treePath);
							treePath = new TreePath();
							iSelectedRowIndex = -1;
		
							if (getSelectionMode() == SelectionMode.DESELECTED)
							{
								setSelectedRowInternal(-1);
							}
							else
							{
								setSelectedRowInternal(0);
							}
						}
					}
					else
					{
						MemDataPage dpNewCurrentDataPage = getDataPageIntern(getMasterReference().getReferencedDataBook());
						
						// different master row, so change the current DataPage
						if (dpNewCurrentDataPage != dpCurrentDataPage)
						{
							dpCurrentDataPage = dpNewCurrentDataPage;
							iSelectedRowIndex = -1;
							treePath = new TreePath();
	
							correctSelectedRow();
						}
					}
				}
			}
			finally
			{
				ignoreReload = false;
			}
		}
	}	

	/**
	 * It will be called, when the MemDataBook will be completely refreshed. 
	 * In the MemDataBook, it does nothing, because all is in memory and not restoreable.
	 * Derived classes maybe want to clear the memory.
	 * 
	 * @throws ModelException	if an ModelExcpetion happen during refresh
	 */
	protected void executeRefresh()  throws ModelException
	{
		clearFilterSortInMemDataPages();
		invokeRepaintListeners();		
	}
	
	/**
	 * It clears the AbstractStorage of the MemDataBook. Derived classed maybe need it.
	 */
	protected void clear()
	{
		dpCurrentDataPage	      = null;
		bMasterChanged = true;
		htDataPagesCache.clear();							
		iSelectedRowIndex 	      = -1;
		oaStorage 				  = null;
	}
	
	/**
	 * Sets the selected row with throwing any changed events.
	 * 
	 * @param pDataRowIndex		the new row index to use.
	 * @throws ModelException	if new row couldn't determined or selected.
	 */
	protected void setSelectedRowInternal(int pDataRowIndex) throws ModelException
	{
		IDataRow drNewCurrent = null;
		
		try
		{
			drNewCurrent = getDataRow(pDataRowIndex);
		}
		finally
		{
			if (drNewCurrent != null)
			{
				oaStorage = dpCurrentDataPage.getDataRowStorage(pDataRowIndex);
				iSelectedRowIndex = pDataRowIndex;
			}
			else
			{
				oaStorage = new Object[rdRowDefinition.getColumnCount()];
				iSelectedRowIndex = -1;
			}
		}
		
		invokeMasterChangedDetailsListeners();				
	}
	
	/**
	 * Returns the next unique ID over all DataBooks.
	 * 
	 * @return the next unique ID over all DataBooks.
	 */
	static int getNextUID()
	{
		iUID++;
		return iUID;
	}
	
	/**
	 * It inserts a new before or after the selected row and returns the row index where it is
	 * inserted. 
	 * 
	 * @param pBeforeRow	determines if the row is inserted before (true) or after (false) the current row.
	 * @return the row index, where the row is inserted.
	 * @throws ModelException	if a new row couldn't inserted.
	 */
	private int insertInternal(boolean pBeforeRow) throws ModelException
	{
		int iDataRowIndex = getSelectedRow();
		if (!pBeforeRow && getRowCount() > 0)
		{
			iDataRowIndex++;
		}		
		if (iDataRowIndex < 0)
		{
			iDataRowIndex = 0;
		}
	
		// insert new Row in current DataPage
		dpCurrentDataPage.insert(iDataRowIndex); 
    	// then store the Object[] in MemDataBook
    	oaStorage = dpCurrentDataPage.getDataRowStorage(iDataRowIndex);
    	
		// set master in master, that the details changed.
		setDetailsChangedInMasterBook();
								
		invokeMasterChangedDetailsListeners();				
		
		return iDataRowIndex;
	}
	
	/**
	 * It deletes the current selected row and returns true if the row is removed from
	 * the memory or false if the row is only marked as deleting. The first case happens if
	 * the row is inserting and not stored, then it will be removed.
	 * 
	 * @return true if the row is removed from
	 * the memory or false if the row is only marked as deleting.
	 * @throws ModelException	if the delete operation fails.
	 */
	private boolean deleteInternal() throws ModelException
	{
		// delete details
		if (isSelfJoined() && isDeleteCascade())
		{
			// if it has details
			if (isSelfJoined() && hasDataPage(this))
			{
				// get Master.
				TreePath tpOld = getTreePath();
				
				// use the current row as master
				setTreePath(tpOld.getChildPath(iSelectedRowIndex));
				deleteAllDataRows();
					
				//select back to the current row.
				setTreePath(tpOld);
			}			
		}
		if (auDetailDataBooks != null)
		{
			for (int i = 0, size = auDetailDataBooks.size(); i < size; i++)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook != null)
				{
		        	if (dataBook.isDeleteCascade())
		        	{
		        		dataBook.deleteAllDataRows();
		        	}
		        	else
		        	{
		        		dataBook.restoreAllRows();
		        	}
				}
			}
		}
        
        boolean bResult = dpCurrentDataPage.delete(iSelectedRowIndex);
        
        if (!bResult)
        {
        	// if marked as deleted, then store the Object[] in MemDataBook
        	oaStorage = dpCurrentDataPage.getDataRowStorage(iSelectedRowIndex);
        }
		
		setDetailsChangedInMasterBook();
		invokeRepaintListeners();			
		
		return bResult;
	}

	/**
	 * It sets the values without throwing an event.
	 * 
	 * @param pColumnNames		the column names to use. 
	 * @param pValues			the values to use
	 * @throws ModelException	if the value couldn't set into the memory.
	 */
	protected void setValuesInternal(String[] pColumnNames, Object[] pValues) throws ModelException
	{		
		if (pColumnNames.length != pValues.length)
		{
			throw new ModelException("Number of columns and values are different!");
		}
		
		for (int i = 0; i < pColumnNames.length; i++)
		{
			// set the DataRow value
			super.setValueDRInternal(pColumnNames[i], pValues[i]);
		}		
	} 
	
	/**
	 * It sets in the master IDataBook, that this detail has changed with the notifyDetailChanged()
	 * method.
	 *
	 * @throws ModelException		if the DetailChanged couldn't work.
	 */
	private void setDetailsChangedInMasterBook() throws ModelException
	{
		if (rdMasterReference != null)
		{
			// if selfjoined and not the root node, then set in all parent rows DetailChanged!
			if (isSelfJoined() && getTreePath().length() > 0)
			{
				// set the detail change state correct.
				boolean bChanged = dpCurrentDataPage.getChangedDataRows().length > 0;
				
				TreePath tpCurrentTreePath = getTreePath();
		
				while (tpCurrentTreePath.length() > 0)
				{
					MemDataPage dpParent = (MemDataPage)getDataPage(tpCurrentTreePath.getParentPath());
					
					dpParent.setDetailChanged(tpCurrentTreePath.getLast(), bChanged);
					tpCurrentTreePath = tpCurrentTreePath.getParentPath();
					
					bChanged = dpParent.getChangedDataRows().length > 0;
				}
			}

			// inform master of the change
			ReferenceDefinition dbParentReference = isSelfJoined() ? getRootReference() : getMasterReference();

			if (dbParentReference != null)
			{
				IDataBook dbMaster = dbParentReference.getReferencedDataBook();
				dbMaster.notifyDetailChanged();
			}
		}
	}
	
	/**
	 * After a insert of a new detail row, the master reference columns will be copied
	 * from the master to this detail row. 
	 * 
	 * @param pOnlyNull			if true, it will only be copied if the master columns == null.
	 * @throws ModelException	if the values couldn't copied 
	 */
	private void copyMasterColumnsToCurrentDetail(boolean pOnlyNull) throws ModelException
	{
		if (rdMasterReference != null)
		{
			Object[] oaDetailValues = getValues(rdMasterReference.getColumnNames());
			boolean bNull = true;
			if (pOnlyNull) 
			{
				if (oaDetailValues != null)
				{
					for (int i = 0, iSize = oaDetailValues.length; bNull && i < iSize; i++)
					{
						if (oaDetailValues[i] != null)
						{
							bNull = false;
						}
					}
				}
			}
			
			// just copy it, if the master columns in detail are null.
			if (!pOnlyNull || pOnlyNull & bNull) 
			{
				Object[] oaMasterValues = dpCurrentDataPage.getMasterDataRow().getValues(rdMasterReference.getReferencedColumnNames());
				setValuesInternal(rdMasterReference.getColumnNames(), oaMasterValues);
				
				if (isSelfJoined() && rdTreeRootReference != null)
				{
					oaMasterValues = rdTreeRootReference.getReferencedDataBook().getValues(rdTreeRootReference.getReferencedColumnNames());
					setValuesInternal(rdTreeRootReference.getColumnNames(), oaMasterValues);
				}
			}
		}
	}		
	
	/**
	 * Its saves all required changes, if the write back isolation level is DATAROW.
	 * It determines the root IDataBook and call saveDetails(). 
	 * 
	 * @param pExcludeDataBook	the IDataBook to exclude in the save process
	 * @throws ModelException	if the changes couldn't saved.
	 */
	private void saveDataRowLevel(IDataBook pExcludeDataBook) throws ModelException
	{
		int iOldSelectedRow = getSelectedRow();
		
		// get Root Master first
        IDataBook dbRoot = this;
        ReferenceDefinition dbParentReference = dbRoot.isSelfJoined() ? dbRoot.getRootReference() : dbRoot.getMasterReference();

        // if this databook has WritebackIsolationLevel DATA_ROW, it has to ensure that all Details and all Masters 
        // with WritebackIsolationLevel DATA_ROW are saved too, because in DATA_ROW level only one datarow over all 
        // should be edited.
        // if this databook has WritebackIsolationLevel DATA_SOURCE, only all details with WritebackIsolationLevel DATA_ROW
        // are saved. The master is explizitly not saved.
        // in general it is guaranteed, if a databook with WritebackIsolationLevel DATA_ROW is saved, all its details,
        // ignoring the WritebackIsolationLevel, are saved.
        while (dbRoot.getWritebackIsolationLevel() == WriteBackIsolationLevel.DATA_ROW && dbParentReference != null)
		{
			dbRoot = dbParentReference.getReferencedDataBook();
			
			dbParentReference = dbRoot.isSelfJoined() ? dbRoot.getRootReference() : dbRoot.getMasterReference();
		}

		// then save all details to this root DataBook, except the current DataBook
		((MemDataBook)dbRoot).saveDataRowLevelDetails(pExcludeDataBook);
		if (iOldSelectedRow != getSelectedRow())
		{
			setSelectedRow(iOldSelectedRow);
		}
	}
	
	/**
	 * It saves all required changes, if the write back isolation level is DATA_ROW.
	 * It searches the current row and then hierarchical down over all details to the last detail if there is a 
	 * data book with WriteBackIsolationLevel DATA_ROW.
	 * 
	 * @param pExcludeDataBook	the IDataBook to exclude in the save process
	 * @throws ModelException	if the changes couldn't saved.
	 */
	private void saveDataRowLevelDetails(IDataBook pExcludeDataBook) throws ModelException
	{		
		if (getWritebackIsolationLevel() == WriteBackIsolationLevel.DATA_ROW)
		{
			saveDetails(pExcludeDataBook);
		}
		else if (auDetailDataBooks != null)
		{
			for (int i = 0, size = auDetailDataBooks.size(); i < size; i++)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook != null)
				{
					((MemDataBook)dataBook).saveDataRowLevelDetails(pExcludeDataBook);
				}
			}
		}
	}
	
	/**
	 * It saves all required changes ignoring the write back isolation level.
	 * It saves current row and then hierarchical down over all details to the last detail.
	 * 
	 * @param pExcludeDataBook	the IDataBook to exclude in the save process
	 * @throws ModelException	if the changes couldn't saved.
	 */
	private void saveDetails(IDataBook pExcludeDataBook) throws ModelException
	{		
		if (this != pExcludeDataBook)
		{
			saveSelectedRowIntern(); 
			// just save all current selectedRow of all DataBooks in this hierarchy is enough!!!!
			//saveAllRows();
		}
		
		if (auDetailDataBooks != null)
		{
			for (int i = 0, size = auDetailDataBooks.size(); i < size; i++)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook != null)
				{
					((MemDataBook)dataBook).saveDetails(pExcludeDataBook);
				}
			}
		}
	}
	
	/**
	 * It restores all details, if cascade delete. 
     * Otherwise it tries to save the details, if WriteBackIsolationLevel == DATA_ROW.
     *  	
	 * @throws ModelException if the save or restore fails.
	 */
	private void restoreSaveAllDetails() throws ModelException
	{
		if (auDetailDataBooks != null)
		{
			for (int i = 0, size = auDetailDataBooks.size(); i < size; i++)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook != null)
				{
		        	if (dataBook.isDeleteCascade())
		        	{
		        		dataBook.restoreAllRows();
		        		((MemDataBook)dataBook).restoreSaveAllDetails();
		        	}
		        	else
		        	{
		        		if (dataBook.getWritebackIsolationLevel() == WriteBackIsolationLevel.DATA_ROW)
		        		{
		        			dataBook.saveAllRows();
		        		}
		        		((MemDataBook)dataBook).restoreSaveAllDetails();
		        	}
				}
			}
		}
	}
	
	/**
	 * Stores the selection for a correct, reload, resort, ...
	 * 
	 * @param pSelectionMode 	the SelectionMode to use. 
	 * @throws ModelException if getSelectedRow throw an exception.
	 */
	private void handleStoreSelection(SelectionMode pSelectionMode) throws ModelException
	{
		if (dpCurrentDataPage != null)
		{
			if (pSelectionMode == SelectionMode.CURRENT_ROW)
			{
				if (getSelectedRow() < 0) // Mark that the databook is deselected.
				{
					if (dpCurrentDataPage.getRowCount() == 0)
					{
						setStoredSelection(null);
					}
					else
					{
						setStoredSelection(this);
					}
				}
				else
				{
					// save current DataRow, PK cols or all
					String[] asPKColumns = rdRowDefinition.getPrimaryKeyColumnNames();
					if (asPKColumns == null)
					{
						asPKColumns = rdRowDefinition.getColumnNames();
					}
					if (asPKColumns != null && asPKColumns.length > 0)
					{
						setStoredSelection(getDataRow(iSelectedRowIndex).createDataRow(asPKColumns));
					}
				}
			}
			else
			{
				setStoredSelection(null);
			}
		}
	}
	
	/**
	 * It restores all detail rows.
	 * 
	 * @throws ModelException	if the change couldn't restored in memory.
	 */
	private void restoreAllDetails() throws ModelException
	{
		if (auDetailDataBooks != null)
		{
			for (int i = 0, size = auDetailDataBooks.size(); i < size; i++)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook != null)
				{
					dataBook.restoreAllRows();
				}
			}
		}
		
		// if self joined && inserting, then remove detail pages.
		if (isSelfJoined())
		{
    		// no Writeback enabled, just remove the DataPage from Storage to remove all details
			if (treePath != null)
			{
				removeDataPage(null, treePath.getChildPath(getSelectedRow()));
			}
			else
			{
				removeDataPage(null, null);
			}
		}		
	}

	/**
	 * It restores the changes on the selected row and remove the change state of 
	 * this row.
	 * 
	 * @throws ModelException	if the change couldn't restored in memory.
	 */
	private void restoreRow() throws ModelException
	{		
		IDataRow drOld = getDataRow(iSelectedRowIndex);
		if (eventBeforeRestore != null)
		{
			eventBeforeRestore.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_RESTORE, drOld));
		}

		if (isInserting())
		{
			// #376 - restore row fails in DATASOURCE level with more then one detail level with isInserting rows
			// restoreDetails first, then delete row from page.
			restoreAllDetails();

			dpCurrentDataPage.restore(iSelectedRowIndex);
			setSelectedRowInternal(iSelectedRowIndex - 1);
			setDetailsChangedInMasterBook();
			
			if (eventAfterRestore != null)
			{
				eventAfterRestore.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_RESTORE, drOld));
			}
			if (eventAfterRowSelected != null)
			{
				eventAfterRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_ROW_SELECTED, drOld));
			}
		}
		else
		{
			dpCurrentDataPage.restore(iSelectedRowIndex);
			oaStorage = dpCurrentDataPage.getDataRowStorage(iSelectedRowIndex);
	
			setDetailsChangedInMasterBook();
			
			if (eventAfterRestore != null)
			{
				eventAfterRestore.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_RESTORE, drOld));
			}
		}
	}
	
	/**
	 * It stores the changes in memory as stored. That means the changes will be made and
	 * the change state of the current DataRow (the DataBook it self) will be removed.
	 * 
	 * @throws ModelException	if a ModelException occurs.
	 */
	protected void store() throws ModelException
	{
		if (!dpCurrentDataPage.store(iSelectedRowIndex))
		{
			oaStorage = dpCurrentDataPage.getDataRowStorage(iSelectedRowIndex);
		}
		
		setDetailsChangedInMasterBook();
	}

	/** 
	 * It clears all mem filter sorts in the MemDataPages.
	 * 
	 * @throws ModelException if a ModelException occurs.
	 */
	protected void clearFilterSortInMemDataPages() throws ModelException
	{
		if (eventBeforeReload != null)
		{
			eventBeforeReload.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_RELOAD, null));
		}
		
		if (getMasterReference() != null)
		{
			Enumeration<MemDataPage> enumDataPages = htDataPagesCache.elements();
			while (enumDataPages.hasMoreElements())
			{
				enumDataPages.nextElement().clear();
			}
		}
		else
		{
			if (!isOutOfSync())
			{
				dpCurrentDataPage.clear();
			}
		}		

		// correctSelectedRow has to be before AFTER_RELOAD event, to prevent undefined states in event.
		if (!isOutOfSync())
		{
			correctSelectedRow();
		}
		if (eventAfterReload != null)
		{
			eventAfterReload.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_RELOAD, null));
		}
	}

	/**
	 * Sets the databook readonly but does not automatically save.
	 * 
	 * @param pReadOnly <code>true</code> to set the databook readonly, <code>false</code> otherwise
	 */
	public void setReadOnlyWithoutSave(boolean pReadOnly)
	{
		bReadOnly = pReadOnly;
		
		invokeRepaintListeners();
	}
	
}	// MemDataBook
