/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 16.04.2008 - [HM] - creation
 * 07.11.2010 - [JR] - addListener by index
 * 05.12.2010 - [JR] - ignore SilentAbortException
 *                   - logging implemented
 * 22.12.2012 - [JR] - #454: en/disable event dispatching    
 * 13.03.2013 - [JR] - getLastDispatchedObject, getCurrentDispatchObject implemented
 * 27.05.2013 - [JR] - getWrappedException implemented               
 */
package javax.rad.util;

import java.lang.ref.WeakReference;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.lang.reflect.UndeclaredThrowableException;
import java.util.List;

import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.Reflective;
import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.LoggerFactory;

/**
 * Platform and technology independent event handler.
 * It is designed for use with UI elements and non UI elements.
 * There can be used with any Listener Interface. and implicit forwarded
 * to any function.  
 * <code>
 *   button.eventAction().addListener(
 *     new ActionListener()
 *     {
 *       public void actionPerformed(ActionEvent pEvent)
 *       {
 *         doSave();
 *       }
 *     });
 * </code>
 * <code>
 *   button.eventAction().addListener(this, "doSave");
 * </code>
 * 
 * @author Martin Handsteiner
 * 
 * @param <L> the Listener type
 */
public class EventHandler<L>
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the logger. */
	private static ILogger logger = LoggerFactory.getInstance(EventHandler.class);
	
	/** The event type class. */
	private Class<L> listenerType;
	/** The listener method. */
	private Method listenerMethod;
	/** The listener method. */
	private Class[] parameterTypes;

	/** The listeners. */
	private List<ListenerHandler> listeners = null;
	/** The default listener. */
	private ListenerHandler defaultListener = null;
	
	/** the last dispatched object. */
	private static WeakReference<Object> wrefLastDispatchedObject = null;

	/** the current dispatch object. */
	private static WeakReference<Object> wrefCurrentDispatchObject = null;
	
	/** Is first dispatch. */
	private boolean isFirstDispatch = true;
	/** whether event dispatching should be dispatched. */
	private boolean bDispatchEvents = true;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Constructs a new EventHandler, the listener type may only have 1 method.
	 *  
	 * @param pListenerType the listener type interface.
	 * @param pParameterTypes parameter types to check additional.
	 */
	public EventHandler(Class<L> pListenerType, Class... pParameterTypes)
	{
		this(pListenerType, null, pParameterTypes);
	}
	
	/**
	 * Constructs a new EventHandler.
	 *  
	 * @param pListenerType the listener type interface.
	 * @param pListenerMethodName the method to be called inside the interface.
	 * @param pParameterTypes parameter types to check additional.
	 */
	public EventHandler(Class<L> pListenerType, String pListenerMethodName, Class... pParameterTypes)
	{
		if (!pListenerType.isInterface())
		{
			throw new IllegalArgumentException("The listener type class has to be an interface!");
		}
		
		listenerMethod = findMethodByName(pListenerType, pListenerMethodName);
		
		listenerType = pListenerType;
		
		if (pParameterTypes == null || pParameterTypes.length > 0)
		{
			parameterTypes = pParameterTypes;
		}
		else
		{
			parameterTypes = null;
		}
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Searches the given method.
	 * 
	 * @param pListenerType the interface to search in.
	 * @param pListenerMethodName the method to search for.
	 * @return the method, if found.
	 * @throws IllegalArgumentException if the method does not exist. 
	 */
	private Method findMethodByName(Class<L> pListenerType, String pListenerMethodName)
	{
		Method[] methods = pListenerType.getMethods();
		if (pListenerMethodName == null)
		{
			if (methods.length == 1)
			{
				return methods[0];
			}
			throw new IllegalArgumentException("Listener class " + pListenerType.getName() + " has more than 1 method, the listener method has to be specified!");
		}
		else
		{
			for (int i = 0; i < methods.length; i++)
			{
				if (pListenerMethodName.equals(methods[i].getName()))
				{
					return methods[i];
				}
			}
			throw new IllegalArgumentException("Listener method " + pListenerMethodName + " not found in class " + pListenerType.getName() + "!");
		}
	}
	
	/**
	 * Adds a listener.
	 * If the Listener is already added, it is removed before, and added at the end.
	 * 
	 * @param pListener the listener.
	 */
	public void addListener(L pListener)
	{
		addListener(pListener, -1);
	}
	
	/**
	 * Adds a listener at a given index.
	 * If the Listener is already added, it is removed before, and added at the new position.
	 * 
	 * @param pListener the listener.
	 * @param pIndex the index to add.
	 */
	public void addListener(L pListener, int pIndex)
	{
		if (listeners == null)
		{
			listeners = new ArrayUtil<ListenerHandler>();
		}
		
		int index = listeners.indexOf(pListener);
		
		if (index >= 0)
		{
			listeners.remove(index);
			
			if (pIndex > index)
			{
				pIndex--;
			}
		}
		
		ListenerHandler handler;
		if (Proxy.isProxyClass(pListener.getClass()))
		{
			handler = (ListenerHandler)Proxy.getInvocationHandler(pListener);
		}
		else
		{
			handler = new ListenerHandler(pListener);
		}
		if (pIndex < 0)
		{
			listeners.add(handler);
		}
		else
		{
			listeners.add(pIndex, handler);
		}
	}
	
	/**
	 * Adds a listener.
	 * If the Listener is already added, it is removed before, and added at the end.
	 * 
	 * @param pListener the listener object.
	 * @param pMethodName the method name.
	 */
	public void addListener(Object pListener, String pMethodName)
	{
		addListener(pListener, pMethodName, -1);
	}

	/**
	 * Adds a listener at a given position.
	 * If the Listener is already added, it is removed before, and added at the new position.
	 * 
	 * @param pListener the listener object.
	 * @param pMethodName the method name.
	 * @param pIndex the index to add
	 */
	public void addListener(Object pListener, String pMethodName, int pIndex)
	{
		if (listeners == null)
		{
			listeners = new ArrayUtil<ListenerHandler>();
		}
		
		ListenerHandler handler = new ListenerHandler(pListener, pMethodName);
		
		int index = listeners.indexOf(handler);
		
		if (index >= 0)
		{
			listeners.remove(index);
			
			if (pIndex > index)
			{
				pIndex--;
			}
		}
		
		if (pIndex < 0)
		{
			listeners.add(handler);
		}
		else
		{
			listeners.add(pIndex, handler);
		}
	}	
	
	/**
	 * Removes the listener at the position.
	 * 
	 * @param pIndex the position.
	 */
	public void removeListener(int pIndex)
	{
		if (listeners != null)
		{
			listeners.remove(pIndex);
			
			if (listeners.isEmpty())
			{
				listeners = null;
			}
		}
	}
	
	/**
	 * Removes all listener methods added with the given object, or the listener, if an interface was added.
	 * 
	 * @param pListener the listener.
	 */
	public void removeListener(Object pListener)
	{
		if (listeners != null)
		{
			while (listeners.remove(pListener))
			{
				// Do Nothing
			}
			
			if (listeners.isEmpty())
			{
				listeners = null;
			}
		}
	}
	
	/**
	 * Removes the listener added with the method.
	 * 
	 * @param pListener the listener object.
	 * @param pMethodName the method name.
	 */
	public void removeListener(Object pListener, String pMethodName)
	{
		if (listeners != null)
		{
			listeners.remove(new ListenerHandler(pListener, pMethodName));
			
			if (listeners.isEmpty())
			{
				listeners = null;
			}
		}
	}
	
	/**
	 * Removes all known listeners.
	 */
	public void removeAllListeners()
	{
		if (listeners != null)
		{
			listeners = null;
		}
	}

	/**
	 * Gets the count of listeners.
	 * 
	 * @return the count of listeners.
	 */
	public int getListenerCount()
	{
		if (listeners == null)
		{
			return 0;
		}
		else
		{
			return listeners.size();
		}
	}
	
	/**
	 * Gets the listener at the position.
	 * 
	 * @param pIndex the position.
	 * @return all listeners.
	 */
	public L getListener(int pIndex)
	{
		if (listeners == null)
		{
			throw new IndexOutOfBoundsException("Index: " + pIndex + ", Size: 0");
		}
		else
		{
			return listeners.get(pIndex).listenerInterface;
		}
	}
	
	/**
	 * Gets all listeners.
	 * 
	 * @return all listeners.
	 */
	public L[] getListeners()
	{
		if (listeners == null)
		{
			return (L[])Array.newInstance(listenerType, 0);		
		}
		else
		{
			L[] result = (L[])Array.newInstance(listenerType, listeners.size());
			for (int i = listeners.size() - 1; i >= 0; i--)
			{
				result[i] = listeners.get(i).listenerInterface;
			}
			
			return result;
		}
	}
	
	/**
	 * Sets the default listener.
	 * 
	 * @param pListener the listener.
	 */
	public void setDefaultListener(L pListener)
	{
		if (pListener == null)
		{
			defaultListener = null;
		}
		else if (defaultListener == null || !defaultListener.equals(pListener))
		{
			defaultListener = new ListenerHandler(pListener);
		}
	}
	
	/**
	 * Sets the default listener.
	 * 
	 * @param pListener the listener object.
	 * @param pMethodName the method name.
	 */
	public void setDefaultListener(Object pListener, String pMethodName)
	{
		if (pListener == null)
		{
			defaultListener = null;
		}
		else if (defaultListener == null || !defaultListener.equals(pListener))
		{
			defaultListener = new ListenerHandler(pListener, pMethodName);
		}
	}
	
	/**
	 * Gets the default listener.
	 * 
	 * @return the default listener.
	 */
	public L getDefaultListener()
	{
		if (defaultListener == null)
		{
			return null;
		}
		else
		{
			return defaultListener.listenerInterface;
		}
	}

	/**
	 * Dispatches the given events to all listeners.
	 * 
	 * @param pEventParameter the event parameter.
	 * @return the return value of the deaultListener, if it is called, or null if dispatching is disabled or 
	 *         no listeners were called
	 * @throws Throwable if an exception occurs.
	 */
	public Object dispatchEvent(Object... pEventParameter) throws Throwable
	{
		if (bDispatchEvents)
		{
			if (isFirstDispatch)
			{
				isFirstDispatch = false;

				try
				{
					if (listeners != null)
					{
						//copy, because it is possible that a listener removes itself.
						List<ListenerHandler> liHandlers = new ArrayUtil<ListenerHandler>(listeners);

						int iSize = liHandlers.size();
						
						if (iSize > 0)
						{
							setLastDispatchedObject();
							
							if (pEventParameter != null && pEventParameter.length > 0)
							{
								wrefCurrentDispatchObject = new WeakReference<Object>(pEventParameter[0]);
							}
						}

						try
						{
							for (int i = 0; i < iSize; i++)
							{
								liHandlers.get(i).dispatchEvent(pEventParameter);
							}
						}
						finally
						{
							setLastDispatchedObject();
						}
					}
					else if (defaultListener != null)
					{
						setLastDispatchedObject();
						
						if (pEventParameter != null && pEventParameter.length > 0)
						{
							wrefCurrentDispatchObject = new WeakReference<Object>(pEventParameter[0]);
						}

						try
						{
							return defaultListener.dispatchEvent(pEventParameter);
						}
						finally
						{
							setLastDispatchedObject();
						}
					}
					isFirstDispatch = true;
				}
				catch (Throwable pThrowable)
				{
					logger.debug(pThrowable);
					
					isFirstDispatch = true;
					
					throw getWrappedExceptionAllowSilent(pThrowable);
				}
			}
			
			return null;
		}
		
		return null;
	}
	
	/**
	 * Creates a new listener interface for calling the given method for the given object.
	 * 
	 * @param pListener the object.
	 * @param pMethodName the method.
	 * @return the Interface.
	 */
	public L createListener(Object pListener, String pMethodName)
	{
		return new ListenerHandler(pListener, pMethodName).listenerInterface;
	}
	
	/**
	 * Sets whether event dispatching is en- or disabled.
	 * 
	 * @param pEnabled <code>true</code> to enable dispatching, <code>false</code> to ignore dispatching
	 */
	public void setDispatchEventsEnabled(boolean pEnabled)
	{
		bDispatchEvents = pEnabled;
	}
	
	/**
	 * Gets whether event dispatching is enabled.
	 * 
	 * @return <code>true</code> if event dispatching is enabled, <code>false</code> if it's disabled
	 */
	public boolean isDispatchEventsEnabled()
	{
		return bDispatchEvents;
	}
	
	/**
	 * Sets the current object as last dispatched objects.
	 */
	private void setLastDispatchedObject()
	{
		if (wrefCurrentDispatchObject != null)
		{
			wrefLastDispatchedObject = wrefCurrentDispatchObject;
			
			wrefCurrentDispatchObject = null;
		}
	}
	
	/**
	 * Gets the last dispatched object.
	 * 
	 * @return the object
	 */
	public static Object getLastDispatchedObject()
	{
		if (wrefLastDispatchedObject != null)
		{
			return wrefLastDispatchedObject.get();
		}
		
		return null;
	}
	
	/**
	 * Gets the current dispatch object.
	 * 
	 * @return the object or <code>null</code> if no object was found
	 */
	public static Object getCurrentDispatchObject()
	{
		if (wrefCurrentDispatchObject != null)
		{
			return wrefCurrentDispatchObject.get();
		}
		
		return null;
	}
	
	/**
	 * Gets the cause of an exception that is a wrapper exception, like {@link InvocationTargetException}.
	 * 
	 * @param pCause the wrapper exception
	 * @return the wrapped(inner) exception
	 */
	public static Throwable getWrappedException(Throwable pCause)
	{
		Throwable th = pCause;
		
		while ((th instanceof InvocationTargetException
			    || th instanceof UndeclaredThrowableException
			    || th instanceof SilentAbortException)
			    && th.getCause() != null)
		{
			th = th.getCause();
		}
		
		return th;
	}
	
	/**
	 * Gets the cause of an exception that is a wrapper exception, like {@link InvocationTargetException}.
	 * This methods doesn't unwrap exception that were wrapped with {@link SilentAbortException}.
	 * 
	 * @param pCause the wrapper exception
	 * @return the wrapped(inner) exception
	 */
	public static Throwable getWrappedExceptionAllowSilent(Throwable pCause)
	{
		Throwable th = pCause;
		
		while ((th instanceof InvocationTargetException
				|| th instanceof UndeclaredThrowableException) 
				&& th.getCause() != null)
		{
			th = th.getCause();
		}
		
		return th;
	}
	
	//****************************************************************
	// Subclass definition
	//****************************************************************

	/**
	 * Generic Listener that calls reflective the given method.
	 * The Method is searched in the order:
	 * <code>
	 *   1) public void methodName(E pEvent);
	 *   2) public void methodName();
	 * </code>
	 * If the method does not exist, a NoSuchMethodException is thrown.
	 *   
	 * @author Martin Handsteiner
	 */
	private class ListenerHandler implements InvocationHandler
	{
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Class members
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/** The listener interface. */
		private L listenerInterface;
		
		/** The listener. */
		private Object listener;
		/** The listener. */
		private Method method;
		/** True, if the method should be call with event. */
		private boolean callWithParameter;
		
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Initialization
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
		 * Constructs a new ListenerHandler with an listener interface.
		 * @param pListenerInterface the listener interface.
		 */
		public ListenerHandler(L pListenerInterface)
		{
			listenerInterface = pListenerInterface;
			listener = null;
		}
		
		/**
		 * Constructs a new ListenerHandler with a proxy listener interface.
		 * @param pListener the listener.
		 * @param pMethodName the method name.
		 */
		public ListenerHandler(Object pListener, String pMethodName)
		{
			Class listenerClass = pListener.getClass();
			Class[] paramTypes;
			if (parameterTypes == null)
			{
				paramTypes = listenerMethod.getParameterTypes();
			}
			else
			{
				paramTypes = parameterTypes;
			}
			try
			{
				method = listenerClass.getMethod(pMethodName, paramTypes);
				callWithParameter = true;
			}
			catch (NoSuchMethodException pNoSuchMethodException1)
			{
				try
				{
					method = Reflective.getMethod(listenerClass, pMethodName, paramTypes);
					callWithParameter = true;
				}
				catch (NoSuchMethodException pNoSuchMethodException2)
				{
					try
					{
						method = listenerClass.getMethod(pMethodName);
						callWithParameter = false;
					}
					catch (NoSuchMethodException pNoSuchMethodException3)
					{
						throw new IllegalArgumentException("Method " + pMethodName + " not found in class " + listenerClass.getName() + "!");
					}
				}
			}
			listener = pListener;
			listenerInterface = (L)Proxy.newProxyInstance(pListener.getClass().getClassLoader(), 
					  									  new Class[] {listenerType},
					  									  this);
		}

		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Interface implementation
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
		 * {@inheritDoc}
		 */
		public Object invoke(Object pProxy, Method pMethod, Object[] pArguments) throws Throwable
		{
			try
			{
				if (pMethod.getName() == listenerMethod.getName())
				{
					if (callWithParameter)
					{
						return method.invoke(listener, pArguments);
					}
					else
					{
						return method.invoke(listener);
					}
				}
				else if (pMethod.getName() == "hashCode")
				{
					return Integer.valueOf(listener.hashCode());
				}
				else if (pMethod.getName() == "equals")
				{
					return Boolean.valueOf(listenerInterface == pArguments[0] || listener.equals(pArguments[0]));
				}
				else if (pMethod.getName() == "toString")
				{
					return listener.toString() + "." + method.getName();
				}
			}
			catch (Throwable pThrowable)
			{
				logger.debug(pThrowable);

				throw getWrappedException(pThrowable);
			}
			throw new UnsupportedOperationException("The call of method " + pMethod.getName() + " is not supported!");
		}

		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Overwritten methods
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
		 * {@inheritDoc}
		 */
		@Override
		public int hashCode()
		{
			if (listener == null)
			{
				return listenerInterface.hashCode();
			}
			else
			{
				return listener.hashCode();
			}
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public boolean equals(Object pObject)
		{
			if (pObject.getClass() == ListenerHandler.class)
			{
				ListenerHandler handler = (ListenerHandler)pObject;
				
				return listener == handler.listener && method.equals(handler.method);
			}
			else
			{
				return listenerInterface == pObject || (listener != null && listener == pObject);
			}
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public String toString()
		{
			if (listener == null)
			{
				return listenerInterface.toString();
			}
			else
			{
				return listener.toString();
			}
		}

		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// User-defined methods
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		
		/**
		 * Invokes the listener or the proxy interface.
		 * 
		 * @param pArguments the Arguments.
		 * @return the result.
		 * @throws Throwable if an error occurs.
		 */
		public Object dispatchEvent(Object... pArguments) throws Throwable
		{
			if (listener == null)
			{
				return listenerMethod.invoke(listenerInterface, pArguments);
			}
			else
			{
				if (callWithParameter)
				{
					return method.invoke(listener, pArguments);
				}
				else
				{
					return method.invoke(listener);
				}
			}
		}

	}	// ListenerHandler
	
}	// EventHandler
