/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 17.02.2009 - [HM] - creation
 */
package javax.rad.type.bean;

import java.lang.ref.WeakReference;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.UndeclaredThrowableException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.WeakHashMap;

import javax.rad.type.AbstractType;

import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.LoggerFactory;

/**
 * The <code>BeanType</code> is a wrapper for dynamic/generic beans and POJOs. With this
 * class you can set/get properties without get/set methods.
 * 
 * @author Martin Handsteiner
 */
public class BeanType extends AbstractBeanType<Object>
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The logger for protocol the performance. */
	private static ILogger logger = LoggerFactory.getInstance(BeanType.class);

	/** The property cache. */
	private static WeakHashMap<Class, WeakReference<BeanType>> beanTypeCache = new WeakHashMap<Class, WeakReference<BeanType>>();

	/** The bean class. */
	protected transient Class beanClass;
	
	/** The get methods. */
	protected transient Method[] getMethods;
	
	/** The set methods. */
	protected transient Method[] setMethods;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Constructs a new <code>BeanType</code>.
	 */
	public BeanType()
	{
		super(null, (String[])null);
	}
	
	/**
	 * Constructs a new <code>BeanType</code> with given property names.
	 * 
	 * @param pPropertyNames the bean properties.
	 */
	public BeanType(String[] pPropertyNames)
	{
		this(null, pPropertyNames);
	}
	
	/**
	 * Constructs a new <code>BeanType</code> with given class name and property list.
	 * 
	 * @param pClassName the class name.
	 * @param pPropertyNames the bean properties.
	 */
	public BeanType(String pClassName, String[] pPropertyNames)
	{
		super(pClassName, pPropertyNames);
		
		try
		{
			beanClass = Class.forName(pClassName);
			
			BeanType beanType = getBeanType(beanClass);
			
			getMethods = new Method[propertyNames.length];
			setMethods = new Method[propertyNames.length];
			
			for (int i = 0; i < propertyNames.length; i++)
			{
				int index = beanType.getPropertyIndex(propertyNames[i]);
				
				if (index >= 0)
				{
					getMethods[i] = beanType.getMethods[index];
					setMethods[i] = beanType.setMethods[index];
				}
			}
		}
		catch (Exception pException)
		{
			beanClass = null;
			getMethods = null;
			setMethods = null;
		} 
	}
	
	/**
	 * Constructs a new <code>BeanType</code> with given {@link PropertyDefinition}s.
	 * 
	 * @param pPropertyDefinitions the bean properties.
	 */
	public BeanType(PropertyDefinition[] pPropertyDefinitions)
	{
		this(null, pPropertyDefinitions);
	}	
		
		
	/**
	 * Constructs a new <code>BeanType</code> with given class name and {@link PropertyDefinition}s.
	 * 
	 * @param pClassName the class name.
	 * @param pPropertyDefinitions the bean properties.
	 */
	public BeanType(String pClassName, PropertyDefinition[] pPropertyDefinitions)
	{
		super(pClassName, pPropertyDefinitions);
		
		beanClass = null;
		getMethods = null;
		setMethods = null;
	}
	
	/**
	 * Constructs a new <code>BeanType</code> from a POJO.
	 * 
	 * @param pBeanClass the bean class.
	 */
	protected BeanType(Class pBeanClass)
	{
		beanClass = pBeanClass;
		className = beanClass.getName();
		
		List<String> result = new ArrayList<String>();
		HashMap<String, PropertyDefinition> beanProps = new HashMap<String, PropertyDefinition>();
		HashMap<String, Method> getMeth = new HashMap<String, Method>();
		HashMap<String, Method> setMeth = new HashMap<String, Method>();

		Method[] methods = pBeanClass.getMethods();
		
		for (int i = 0; i < methods.length; i++)
		{
			Method method = methods[i];
			
			if (!Modifier.isStatic(method.getModifiers()) && method.getParameterTypes().length == 0)
			{
				String methodName = method.getName();
				
				String propertyName = null;
				if (methodName.startsWith("get"))
				{
					propertyName = methodName.substring(3);
				}
				else if (methodName.startsWith("is"))
				{
					propertyName = methodName.substring(2);
				}
				if (propertyName != null)
				{
					try
					{
						Method setMethod = pBeanClass.getMethod("set" + propertyName, method.getReturnType());
						if (!Modifier.isStatic(setMethod.getModifiers()))
						{
							PropertyDefinition propertyDefinition = createPropertyDefinition(
									Character.toLowerCase(propertyName.charAt(0)) + propertyName.substring(1),
									method.getReturnType());
							result.add(propertyDefinition.getName()); // property name is interned.
							beanProps.put(propertyDefinition.getName(), propertyDefinition);
							getMeth.put(propertyDefinition.getName(), method);
							setMeth.put(propertyDefinition.getName(), setMethod);
						}
					}
					catch (NoSuchMethodException pNoSuchMethodException)
					{
						// Do nothing
					}
				}
			}
		}
		propertyNames = result.toArray(new String[result.size()]);
		
		Arrays.sort(propertyNames);
		
		propertyDefinitions = new PropertyDefinition[propertyNames.length];
		getMethods = new Method[propertyNames.length];
		setMethods = new Method[propertyNames.length];
		
		for (int i = 0; i < propertyNames.length; i++)
		{
			String propertyName = propertyNames[i];
			propertyDefinitions[i] = beanProps.get(propertyName);
			getMethods[i] = getMeth.get(propertyName);
			setMethods[i] = setMeth.get(propertyName);
		}
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface Implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	// IType
	
	/**
	 * {@inheritDoc}
	 */
	public Class getTypeClass()
	{
		if (beanClass == null)
		{
			return Bean.class;
		}
		else
		{
			return beanClass;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public Object valueOf(Object pObject)
	{
		throw new IllegalArgumentException("The value " + pObject + " cannot be converted to an instance of " + getTypeClass().getName() + "!");
	}

	/**
	 * {@inheritDoc}
	 */
	public int compareTo(Object object1, Object object2)
	{
		// TODO Auto-generated method stub
		return 0;
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Gets a singleton instance of BeanType for the given bean class.
	 * @param pBeanClass the POJO class.
	 * @return the POJO bean type.
	 */
	public static BeanType getBeanType(Class pBeanClass)
	{
		WeakReference<BeanType> weakPojoBeanClass = beanTypeCache.get(pBeanClass);
        
        BeanType beanType;
        if (weakPojoBeanClass == null)
        {
        	beanType = null;
        }
        else
        {
        	beanType = weakPojoBeanClass.get();
        }
		if (beanType == null)
		{
			beanType = new BeanType(pBeanClass);
			
			beanTypeCache.put(pBeanClass, new WeakReference(beanType));
		}
		return beanType;
	}
	
	/**
	 * Gets the BeanType for the given bean.
	 * @param pObject the bean.
	 * @return the bean type.
	 */
	public static BeanType getBeanType(Object pObject)
	{
		if (pObject instanceof IBean)
		{
			return (BeanType)((IBean)pObject).getBeanType();
		}
		else
		{
			return getBeanType(pObject.getClass());
		}
	}
	
	/**
	 * Creates the PropertyDefinition.
	 * 
	 * @param pPropertyName the property name.
	 * @param pPropertyType the property type.
	 * @return the PropertyDefinition
	 */
	protected PropertyDefinition createPropertyDefinition(String pPropertyName, Class pPropertyType)
	{
		return new PropertyDefinition(pPropertyName, AbstractType.getTypeFromClass(pPropertyType), pPropertyType);
	}

	/**
	 * Creates a new instance of the bean.
	 * If it is possible, the pojo is returned.
	 * If not, a instance of Bean is returned. 
	 * 
	 * @return the pojo or a bean
	 */
	public Object newInstance()
	{
		try
		{
			return beanClass.newInstance();
		}
		catch (Throwable pThrowable)
		{
			logger.info("newInstance ", className, " failed!", pThrowable);
			
			return new Bean(this);
		}
	}
	
	/**
	 * Gets the value for a bean.
	 * 
	 * @param pObject the bean.
	 * @param pPropertyIndex the property index.
	 * @return pValue the value of the property index.
	 */
	public Object get(Object pObject, int pPropertyIndex)
	{
		if (pObject instanceof AbstractBean)
		{
			return ((AbstractBean)pObject).get(pPropertyIndex);
		}
		else if (pObject instanceof IBean)
		{
			return ((IBean)pObject).get(propertyNames[pPropertyIndex]);
		}
		else
		{
			try
			{
				Method method = getMethods[pPropertyIndex];
				// If the property does not exist, the value is not set (silently) for more compatibility between
				// Client - Server communication.
				if (method != null) 
				{
					return getMethods[pPropertyIndex].invoke(pObject);
				}
				else
				{
					return null;
				}
			}
			catch (Throwable pThrowable)
			{
				while (pThrowable instanceof InvocationTargetException
						|| pThrowable instanceof UndeclaredThrowableException)
				{
					pThrowable = pThrowable.getCause();
				}
				if (pThrowable instanceof RuntimeException)
				{
					throw (RuntimeException)pThrowable;
				}
				else
				{
					throw new RuntimeException(pThrowable);
				}
			}
			
		}
	}

	/**
	 * Sets the value for a bean.
	 * 
	 * @param pObject the bean.
	 * @param pPropertyIndex the property index.
	 * @param pValue the value of the property index.
	 */
	public void put(Object pObject, int pPropertyIndex, Object pValue)
	{
		if (pObject instanceof AbstractBean)
		{
			((AbstractBean)pObject).put(pPropertyIndex, pValue);
		}
		else if (pObject instanceof IBean)
		{
			((IBean)pObject).put(propertyNames[pPropertyIndex], pValue);
		}
		else
		{
			try
			{
				// UNKNOWN_TYPE is default, and has no type conversion or validation.
				if (propertyDefinitions != null) 
				{
					pValue = propertyDefinitions[pPropertyIndex].getType().validatedValueOf(pValue);
				}
				Method method = setMethods[pPropertyIndex];
				// If the property does not exist, the value is not set silently for more compatibility between
				// Client - Server communication.
				if (method == null)
				{
					logger.debug("put ", propertyNames[pPropertyIndex], " does not exist in this VM, it is silent ignored!");
				}
				else
				{
					method.invoke(pObject, pValue);
				}
			}
			catch (Throwable pThrowable)
			{
				while (pThrowable instanceof InvocationTargetException
						|| pThrowable instanceof UndeclaredThrowableException)
				{
					pThrowable = pThrowable.getCause();
				}
				if (pThrowable instanceof RuntimeException)
				{
					throw (RuntimeException)pThrowable;
				}
				else
				{
					throw new RuntimeException(pThrowable);
				}
			}
		}
	}
	
	/**
	 * Gets a value from a bean.
	 * 
	 * @param pObject the bean.
	 * @param pPropertyName the property name.
	 * @return the value of the property name.
	 */
	public Object get(Object pObject, String pPropertyName)
	{
		int index = getPropertyIndex(pPropertyName);
		if (index < 0)
		{
			return null;
		}
		else
		{
			return get(pObject, index);
		}
	}
	
	/**
	 * Sets the value for a bean.
	 * 
	 * @param pObject the bean.
	 * @param pPropertyName the property name.
	 * @param pValue the value of the property name.
	 */
	public void put(Object pObject, String pPropertyName, Object pValue)
	{
		int index = getPropertyIndex(pPropertyName);
		if (index < 0)
		{
			throw new IllegalArgumentException("The property [" + pPropertyName + "] does not exist!");
		}
		
		put(pObject, index, pValue);
	}
	
	/**
	 * Clones a bean.
	 * 
	 * @param pObject the bean.
	 * @return the cloned bean.
	 */
	public Object clone(Object pObject)
	{
   		Object result = newInstance();

   		for (int i = 0, count = getPropertyCount(); i < count; i++)
   		{
   			try
   			{
   				put(result, i, get(pObject, i));
   			}
   			catch (Exception ex)
   			{
   				// Do nothing
   			}
   		}

   		return result;
	}
	
}	// BeanType
