/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 03.04.2009 - [HM] - creation
 */
package javax.rad.type.bean;

import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.LoggerFactory;

/**
 * The <code>Bean</code> is a bean with dynamic properties. It's a sort of dynamic/generic
 * bean without special get/set implementation.
 * 
 * @author Martin Handsteiner
 */
public class Bean extends AbstractBean<BeanType>
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The logger for protocol the performance. */
	private static ILogger logger = LoggerFactory.getInstance(Bean.class);

	/** The values for the properties. */
	private Object values = null;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates an empty <code>Bean</code>.
	 */
	public Bean()
	{
		this(new BeanType());
	}
	
	/**
	 * Creates a <code>Bean</code> with a {@link BeanType}.
	 * 
	 * @param pBeanType the bean type.
	 */
	public Bean(BeanType pBeanType)
	{
		super(pBeanType);
		
		values = null;
	}
	
	/**
	 * Creates a <code>Bean</code> with an object.
	 * 
	 * @param pObject the object.
	 */
	public Bean(Object pObject)
	{
		super(BeanType.getBeanType(pObject.getClass()));
		
		values = pObject;
	}
	
	/**
	 * Creates a <code>Bean</code> with a bean class.
	 * 
	 * @param pBeanClass the bean class.
	 */
	public Bean(Class pBeanClass)
	{
		super(BeanType.getBeanType(pBeanClass));
		
		values = null;
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public void put(int pIndex, Object pValue)
	{
		if (values == null)
		{
			if (beanType.beanClass == null)
			{
				values = new Object[beanType.getPropertyCount()];
			}
			else
			{
				try
				{
					values = beanType.beanClass.newInstance();
				}
				catch (Throwable pThrowable)
				{
					logger.info("newInstance ", beanType.className, " failed!", pThrowable);
					
					values = new Object[beanType.getPropertyCount()];
				}
			}
		}

		pValue = beanType.getPropertyDefinition(pIndex).getType().validatedValueOf(pValue);
		
		if (values.getClass() == Object[].class)
		{
			Object[] val = (Object[])values;

			if (pIndex >= val.length)
			{
				Object[] newVal = new Object[beanType.getPropertyCount()];
				System.arraycopy(val, 0, newVal, 0, val.length);
				
				values = newVal;
				val = newVal;
			}
			
			val[pIndex] = pValue;
		}
		else
		{
			beanType.put(values, pIndex, pValue);
		}
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Gets the Object that stores the values.
	 * It depends on the bean type, whether it is a pojo or an object array.
	 * 
	 * @return the values storage.
	 */
	public Object getObject()
	{
		return values;
	}
	
	/**
	 * Gets the value of a property. The property is identified by the index
	 * from the property list.
	 * 
	 * @param pIndex the property index
	 * @return the value of the property or <code>null</code> if the property was not found
	 */
	public Object get(int pIndex)
	{
		if (values == null)
		{
			return null;
		}
		else if (values.getClass() == Object[].class)
		{
			Object[] val = (Object[])values;
			
			if (pIndex < val.length)
			{
				return val[pIndex];
			}
			else
			{
				return null;
			}
		}
		else
		{
			return beanType.get(values, pIndex);
		}
	}

}	// Bean
