/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [RH] - creation
 * 02.11.2008 - [RH] - conversion of object to storage and back removed; compare use == for equals ->Internalize
 * 13.11.2008 - [RH] - clone moved to DataType() 
 * 13.03.2010 - [JR] - #88: getTypeIdentifier implemented
 */
package javax.rad.model.datatype;

import java.sql.Types;

import javax.rad.model.ModelException;

/**
 * A <code>BinaryDataType</code> is the data type class of a 
 * binary <code>ColumnDefinition</code>.
 *  
 * @author Roland Hrmann
 */
// TODO [RH] hashcode ist falsch.
public class BinaryDataType extends DataType
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the type identifier. */
	public static final int TYPE_IDENTIFIER = Types.BINARY;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Constructs a <code>BinaryDataType</code> with the maximum <code>Integer</code> size.
	 */
	public BinaryDataType()
	{
		setSize(Integer.MAX_VALUE);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public int getTypeIdentifier()
	{
		return TYPE_IDENTIFIER;
	}

	/**
	 * {@inheritDoc}
	 */
	public Class getTypeClass()
	{
		return byte[].class;
	}
	
    /**
     * {@inheritDoc}
     */
    public Object convertAndCheckToTypeClass(Object pObject) throws ModelException
	{
		byte[] baObject = (byte[])convertToTypeClass(pObject);
    	
		if (baObject == null)
		{
			return null;
		}    

		if (baObject.length > getSize())
		{
			throw new ModelException("Binary too large! - length from " +
					                 baObject.length + " to " + getSize());
		}
		
    	return baObject;
	}
    
	/**
	 * {@inheritDoc}
	 */
	public String convertToString(Object pObject)
	{
		if (pObject == null)
		{
			return null;
		}
		
		return new String((byte[])pObject);
	}    

	/**
	 * {@inheritDoc}
	 */
	public Object convertToTypeClass(Object pObject) throws ModelException
	{
		if (pObject == null)
		{
			return null;
		}
		else if (pObject instanceof byte[])
		{
			byte[] baObject = (byte[])pObject;
			if (baObject.length == 0)
			{
				return null;
			}
			return baObject;
		}
		else if (pObject instanceof String)
		{
			String sObject = (String)pObject;
			if (sObject.length() == 0)
			{
				return null;
			}
			return sObject.getBytes();
		}
		throw new ModelException("Conversion failed! Type not supported ! from " +  
				pObject.getClass().getName() + " to " + getTypeClass().getName());
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int compareTo(Object pObject1, Object pObject2)
	{
		// possible because of Internalize.intern()
		if (pObject1 == pObject2)
		{
			return 0;
		}		
		if (pObject1 != null && pObject2 != null)
		{
			if (pObject1 instanceof byte[])
			{
				try
				{
					byte[] ba1 = (byte[])pObject1;
					byte[] ba2 = (byte[])convertToTypeClass(pObject2);
					
					if (ba1.length < ba2.length)
					{
						return -1;
					}
					else if (ba1.length > ba2.length)
					{
						return 1;
					} 
					for (int i = 0; i < ba1.length; i++)
					{
						if (ba1[i] < ba2[i])
						{
							return -1;
						} 
						else if (ba1[i] > ba2[i])
						{
							return 1;
						} 
					}
					return 0;
				}
				catch (ModelException modelException)
				{
					return -1;
				}
			}
		}
		return super.compareTo(pObject1, pObject2);
	}

	/** 
	 * {@inheritDoc}
	 */
	@Override
	public BinaryDataType clone()
	{
		return (BinaryDataType)super.clone();
	}

} 	// BinaryDataType
