/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 17.11.2008 - [HM] - creation
 */
package javax.rad.genui.celleditor;

import java.util.WeakHashMap;

import javax.rad.genui.UIFactoryManager;
import javax.rad.model.IDataRow;
import javax.rad.model.ui.ICellEditor;
import javax.rad.model.ui.ICellEditorHandler;
import javax.rad.model.ui.ICellEditorListener;
import javax.rad.ui.IResource;
import javax.rad.ui.celleditor.IStyledCellEditor;

/**
 * Platform and technology independent editor.
 * It is designed for use with AWT, Swing, SWT, JSP, JSF, ... .
 * 
 * @author Martin Handsteiner
 * 
 * @param <CE> an instance of ICellEditor.
 */
public class UICellEditor<CE extends IStyledCellEditor> implements IResource,
                                                                   IStyledCellEditor
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** the first resource. */
	private CE iResource;
	
	/** The CellEditor resource. */
	private WeakHashMap<Class, CE> whmpResources = new WeakHashMap<Class, CE>();

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	  
	/**
	 * Constructs a new <code>UICellEditor</code>.
	 * 
	 * @param pCEResource the CellEditor resource.
	 * @see ICellEditor  
	 */
	protected UICellEditor(CE pCEResource)
	{
		iResource = pCEResource;
		
		synchronized(whmpResources)
		{
			whmpResources.put(UIFactoryManager.getFactory().getClass(), pCEResource);
		}
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	  
	/**
	 * {@inheritDoc}
	 */
	public final CE getResource()
	{
		CE ce;
		
		synchronized(whmpResources)
		{
			ce = whmpResources.get(UIFactoryManager.getFactory().getClass());
		}
		
		if (ce == null)
		{
			ce = UIFactoryManager.cloneResource(iResource);
			
			if (ce == null || ce == iResource)
			{
				throw new RuntimeException("Unsupported object type: " + iResource.getClass().getName());
			}
			
			synchronized (whmpResources)
			{
				whmpResources.put(UIFactoryManager.getFactory().getClass(), ce);
			}
		}

		return ce;
	}	
	
	/**
	 * {@inheritDoc}
	 */
	public int getHorizontalAlignment()
    {
    	return getResource().getHorizontalAlignment();
    }

	/**
	 * {@inheritDoc}
	 */
	public void setHorizontalAlignment(int pHorizontalAlignment)
    {
		getResource().setHorizontalAlignment(pHorizontalAlignment);
    }

	/**
	 * {@inheritDoc}
	 */
	public int getVerticalAlignment()
    {
    	return getResource().getVerticalAlignment();
    }

	/**
	 * {@inheritDoc}
	 */
	public void setVerticalAlignment(int pVerticalAlignment)
    {
		getResource().setVerticalAlignment(pVerticalAlignment);
    }
	
	/**
	 * {@inheritDoc}
	 */
	public ICellEditorHandler createCellEditorHandler(ICellEditorListener pCellEditorListener, 
			                                          IDataRow pDataRow, 
			                                          String pColumnName)
	{
		return getResource().createCellEditorHandler(pCellEditorListener, pDataRow, pColumnName);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isDirectCellEditor()
	{
		return getResource().isDirectCellEditor();
	}

    /**
     * Gets the default <code>ICellEditor</code> for the given class.
     * This function should always return an editor.
     * It should look for best matching editor with Class.isAssignableFrom.
     *
     * @param pClass the class type to be edited.
     * @return the <code>ICellEditor</code>
     * @see ICellEditor
     */
    public static ICellEditor getDefaultCellEditor(Class pClass)
    {
    	return UIFactoryManager.getFactory().getDefaultCellEditor(pClass);
    }
   
    /**
     * Sets the default <code>ICellEditor</code> for the given class.
     * This function should always return an editor.
     * It should look for best matching editor with Class.isAssignableFrom.
     * If the given ICellEditor is null, it is removed as editor for the given class.
     *
     * @param pClass the class type to be edited.
     * @param pCellEditor the <code>ICellEditor</code>
     * @see ICellEditor
     */
    public static void setDefaultCellEditor(Class pClass, ICellEditor pCellEditor)
    {
    	UIFactoryManager.getFactory().setDefaultCellEditor(pClass, pCellEditor);
    }
    
}	// UICellEditor
