/*
 * Copyright 2013 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 22.08.2013 - [JR] - creation
 */
package javax.rad.genui;

import java.util.WeakHashMap;

import javax.rad.ui.IResource;

/**
 * The <code>UIFactoryResource</code> holds the resource per factory. It creates a resource
 * copy if the resource was initialized as static resource.
 * 
 * @author Ren Jahn
 *
 * @param <UI> the resource type
 */
public class UIFactoryResource<UI extends IResource> extends UIResource<UI>
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The resource. */
	private WeakHashMap<Class, UI> whmpResources = new WeakHashMap<Class, UI>();

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of <code>UIFactoryResource</code>.
	 * 
	 * @param pResource the resource
	 */
	public UIFactoryResource(UI pResource)
	{
		super(pResource);
		
		synchronized (whmpResources)
		{
			whmpResources.put(UIFactoryManager.getFactory().getClass(), pResource);
		}
	}
	
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Overwritten methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Object getResource()
	{
		UI ui = getUIResource();
		
		if (ui == null)
		{
			return null;
		}
		else
		{
			return ui.getResource();
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public final UI getUIResource()
	{
		UI res;
		
		synchronized (whmpResources)
		{
			res = whmpResources.get(UIFactoryManager.getFactory().getClass());
		}
		
		if (res == null)
		{
			res = UIFactoryManager.cloneResource(uiResource);
			
			if (res == null || res == uiResource)
			{
				throw new RuntimeException("Unsupported object type: " + uiResource.getClass().getName());
			}
			
			synchronized (whmpResources)
			{
				whmpResources.put(UIFactoryManager.getFactory().getClass(), res);
			}
		}
		
		return res;
	}
	
}	// UIFactoryResource
