/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [HM] - creation
 */
package com.sibvisions.util;

import java.lang.ref.WeakReference;

/**
 * Intern functionality for all immutable Objects.
 * It's about 2 times faster than String.intern().
 * 
 * @author Martin Handsteiner
 */
public final class Internalize
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the minimum size of the internal object cache. */
	private static final int MIN_SIZE = 63719; //9337; //63719;//63709; //64969;
	
	/** the maximum size of the internal object cache. */
	private static final int MAX_SIZE = 200000;
	
	/** out of memory detected. */
	private static boolean outOfMemoryDetectedOrMaxSizeReached = false;

	/** rehash counter. */
	private static long lastRehashCheck = -1;

	/** size used. */
	private static int size = 0;

	/** the object cache. */
	private static WeakReference<?>[] cache = new WeakReference<?>[MIN_SIZE];

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Invisible constructor, because the <code>Internalize</code> class is a utility class.
	 */
	private Internalize()
	{
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Internalizes any Immutable Object. The function is synchronized.
	 * 
	 * @param <T> any immutable Object.
	 * @param pObject the Object to internalize.
	 * @return the internalized Object.
	 */
	public static final synchronized <T> T intern(T pObject)
	{
		return internNoSync(pObject);
	}
	
	/**
	 * Internalizes any Immutable Object. The function is not synchronized.
	 * It is provided for fast usage, in loops. The implementation has to be like:
	 * <pre>
	 *   synchronized(Internalize.class)
	 *   {
	 *      ... 
	 *        ... = Internalize.internNoSync(...);
	 *      ...
	 *   }
	 * </pre>
	 * 
	 * @param <T> any immutable Object.
	 * @param pObject the Object to internalize.
	 * @return the internalized Object.
	 */
	public static final <T> T internNoSync(T pObject)
	{
		if (pObject == null)
		{
			return pObject;
		}
		else
		{
			int index = (pObject.hashCode() & 0x7fffffff) % cache.length;
			int newFreeIndex = -1;

			WeakReference<?> result = cache[index];
			Object resultObject = null;
			while (result != null && !pObject.equals(resultObject = result.get()))
			{
				// Implicit fast clearing of unused References.
				// Access of evasion strategy gets faster.
				if (resultObject == null)
				{
					if (newFreeIndex < 0)
					{
						newFreeIndex = index;
					}
					cache[index] = null;
					size--;
				}
				if (++index == cache.length)
				{
					index = 0;
				}
				result = cache[index];
			}
			if (result == null) 
			{
				if (outOfMemoryDetectedOrMaxSizeReached)
				{
					// Out of Memory detected.
					// Now Internalize tries to use as less as possible memory. 
					// Rehash need less than 20 ms, To ensure application performance only every two seconds
					// free memory is checked.
					if (System.currentTimeMillis() > lastRehashCheck + 2000)
					{
						rehashAndClearUnused();
					}
				}
				else
				{
					// new Free Place in hash array for storing Object
					if (newFreeIndex >= 0)
					{
						cache[newFreeIndex] = new WeakReference<Object>(pObject, null);
					}
					else
					{
						cache[index] = new WeakReference<Object>(pObject, null);
					}
					size++;
					if (size > cache.length / 2)
					{
						rehashAndClearUnused();
					}
				}
				return pObject;
			}
			else
			{
				// new Free Place in hash array for storing Object
				if (newFreeIndex >= 0)
				{
					cache[index] = null;
					cache[newFreeIndex] = result;
				}
				return (T)resultObject;
			}
		}
	}

	/**
	 * Rehash the cache if it is to small or to large.
	 */ 
	private static final void rehashAndClearUnused()
	{
		// First of all drop all empty references
		WeakReference<?> result;
		for (int i = 0; i < cache.length; i++)
		{
			result = cache[i];
			if (result != null && result.get() == null)
			{
				cache[i] = null;
				size--;
			}
		}
		// check new length
		int newLength = cache.length;
		if (size > newLength / 2)
		{
			newLength *= 3;
		}
		else
		{
			outOfMemoryDetectedOrMaxSizeReached = false;
			while (newLength > MIN_SIZE && size < newLength / 8)
			{
				newLength /= 3;
			}
		}
		if (newLength > MAX_SIZE || outOfMemoryDetectedOrMaxSizeReached)
		{
			outOfMemoryDetectedOrMaxSizeReached = true;
			lastRehashCheck = System.currentTimeMillis();
		}
		else if (newLength != cache.length)
		{
			try
			{
				WeakReference<?>[] newCache = new WeakReference<?>[newLength];
				
				// do rehash
				for (int i = 0; i < cache.length; i++)
				{
					result = cache[i];
					if (result != null)
					{
						Object object = result.get();
						// Prevent NullPointer caused from threading
						if (object == null)
						{
							size--;
						}
						else
						{
							int index = (object.hashCode() & 0x7fffffff) % newCache.length;
							while (newCache[index] != null)
							{
								if (++index == newCache.length)
								{
									index = 0;
								}
							}
							newCache[index] = result;
						}
					}
				}
				cache = newCache;
			}
			catch (OutOfMemoryError pOutOfMemoryError)
			{
				outOfMemoryDetectedOrMaxSizeReached = true;
				lastRehashCheck = System.currentTimeMillis();
			}
		}
	}
	
}	// Internalize
