/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 21.11.2008 - [JR] - creation
 * 04.12.2008 - [JR] - showError implemented
 * 06.12.2008 - [JR] - createInstance: handled InvocationTargetException
 * 11.12.2008 - [JR] - showError: printStackTrace
 * 26.05.2009 - [JR] - showError moved to SwingFactory
 *                   - renamed to ApplicationUtil and moved to the ui package
 * 24.06.2009 - [JR] - getJNLPCodebase implemented (test JNLP access)
 * 06.10.2009 - [JR] - get/setRegistryKey implemented   
 * 14.11.2009 - [JR] - getRegistryApplicationName implemented     
 * 18.03.2011 - [JR] - getRegistryKey: null check for application parameter   
 * 07.11.2011 - [JR] - createApplication with custom class loader
 * 28.03.2012 - [JR] - #568: check exception [mac os fix]     
 * 28.08.2013 - [JR] - #782: getRegistryApplicationName now returns the URL as slash separated value   
 */
package com.sibvisions.rad.ui;

import java.io.IOException;
import java.io.InputStream;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

import javax.rad.application.IApplication;
import javax.rad.application.ILauncher;

import com.sibvisions.util.Reflective;
import com.sibvisions.util.type.ResourceUtil;
import com.sibvisions.util.xml.XmlNode;
import com.sibvisions.util.xml.XmlWorker;

/**
 * The <code>ApplicationUtil</code> is a utility for application launchers.
 * <ul>
 *   <li>Application creation</li>
 *   <li>Config loading</li>
 *   <li>Parameter handling</li>
 *   <li>Registry access</li>
 * </ul>
 * 
 * @author Ren Jahn
 */
public final class ApplicationUtil
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Invisible constructor, because the <code>ApplicationUtil</code> class is a utility class.
	 */
	private ApplicationUtil()
	{
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Creates an instance of an {@link IApplication}.
	 * 
	 * @param pLauncher the launch configuration
	 * @param pClassName the full qualified class name of the desired class
	 * @return a new {@link IApplication} instance 
	 * @throws Throwable if the instance can not be created
	 */
	public static IApplication createApplication(ILauncher pLauncher, String pClassName) throws Throwable
	{
		return (IApplication)Reflective.construct(pClassName, pLauncher);
	}

	/**
	 * Creates an instance of an {@link IApplication}.
	 * 
	 * @param pLauncher the launch configuration
	 * @param pClassLoader the class loader
	 * @param pClassName the full qualified class name of the desired class
	 * @return a new {@link IApplication} instance 
	 * @throws Throwable if the instance can not be created
	 */
	public static IApplication createApplication(ILauncher pLauncher, ClassLoader pClassLoader, String pClassName) throws Throwable
	{
		return (IApplication)Reflective.construct(pClassLoader, pClassName, pLauncher);
	}
	
	/**
	 * Gets the parsed xml configuration.
	 * 
	 * @param pLauncher the launch configuration
	 * @param pName the configuration name or path
	 * @return the parsed configuration, or <code>null</code> if the config was not found
	 * @throws Exception xml parse error
	 */
	public static XmlNode getConfig(ILauncher pLauncher, String pName) throws Exception
	{
		InputStream isConfig = ResourceUtil.getResourceAsStream(pName);
		
		if (isConfig != null)
		{
			try
			{
				XmlWorker xmw = new XmlWorker();
				
				return xmw.read(isConfig);
			}
			finally
			{
				if (isConfig != null)
				{
					try
					{
						isConfig.close();
					}
					catch (IOException ioe)
					{
						//egal
					}
				}
			}
		}
		
		return null;
	}
	
	/**
	 * Replaces a parameter placeholder with the desired parameter.
	 * The placeholder looks like the following: [SERVER]
	 * The placeholder will be replaced with the value of the SERVER parameter.
	 * 
	 * @param pValue the value of an application parameter
	 * @param pLauncher the launcher
	 * @return the value with replaced placeholder(s)
	 */
	public static String replaceParameter(String pValue, ILauncher pLauncher)
	{
		if (pValue != null)
		{
			String sReplaceValue;
			
			String sReplaceParam;
			
			int iParamStart;
			int iParamEnd = 0;
			
			//check parameter replacement
			while ((iParamStart = pValue.indexOf("[", iParamEnd)) >= 0)
			{
				iParamEnd = pValue.indexOf(']', iParamStart);
				
				if (iParamEnd > iParamStart)
				{
					sReplaceParam = pValue.substring(iParamStart + 1, iParamEnd);
					
					//-> be careful with parameter definition -> recursive calls possible!
					sReplaceValue = pLauncher.getParameter(sReplaceParam);
					
					if (sReplaceValue != null)
					{
						pValue = pValue.replace("[" + sReplaceParam + "]", sReplaceValue);
						
						iParamEnd = iParamStart + sReplaceValue.length();
					}
				}
			}
			
			return pValue;
		}

		return null;
	}	
	
	/**
	 * Sets the value for an application specific registry key.
	 * 
	 * @param pApplication the application name
	 * @param pKey the key to set
	 * @param pValue the value to set
	 * @throws SecurityException if a {@link SecurityManager} is present and denies the registry access
	 * @throws RuntimeException if it's not possible to save the registry key
	 */
	public static void setRegistryKey(String pApplication, String pKey, String pValue)
	{
		Preferences pref = Preferences.userRoot().node(pApplication);

		//#568
		try
		{
			if (pValue == null)
			{
				pref.remove(pKey);
			}
			else
			{
				pref.put(pKey, pValue);
			}
		}
		catch (IllegalStateException ise)
		{
			//should not happen, but is possible on older MacOSX versions
			ise.printStackTrace();
			
			return;
		}
		
		try
		{
			pref.sync();
		}
		catch (BackingStoreException bse)
		{
			throw new RuntimeException("Error saving registry key: " + pApplication + " " + pKey, bse);
		}
	}

	/**
	 * Gets the value for an application specific registry key.
	 * 
	 * @param pApplication the application name
	 * @param pKey the key to get
	 * @return the value for the <code>pKey</code> or <code>null</code> if the application or key is not visible
	 * @throws SecurityException if a {@link SecurityManager} is present and denies the registry access
	 */
	public static String getRegistryKey(String pApplication, String pKey)
	{
		Preferences pref = Preferences.userRoot();

		try
		{
			if (pApplication != null && pref.nodeExists(pApplication))
			{
				return pref.node(pApplication).get(pKey, null);
			}
			
			return null;
		}
		catch (BackingStoreException bse)
		{
			return null;
		}
	}
	
	/**
	 * Gets the application name out of the launcher. If the serverbase param is configured, then the
	 * value of this parameter will be the prefix for the application.
	 * 
	 * @param pLauncher the launcher with properties for the application
	 * @return the application name prefixed with the serverbase
	 */
	public static String getRegistryApplicationName(ILauncher pLauncher)
	{
		String sName = pLauncher.getParameter(ILauncher.PARAM_CODEBASE);
		
		if (sName == null)
		{
			sName = pLauncher.getParameter(ILauncher.PARAM_SERVERBASE);
		}
    	
    	if (sName != null)
    	{
    		sName = sName.replace("://", "/").replace(":", "/");
    		
    		if (sName.endsWith("/"))
    		{
    			return sName + pLauncher.getParameter(ILauncher.PARAM_APPLICATIONNAME);
    		}
    		else
    		{
    			return sName + "/" + pLauncher.getParameter(ILauncher.PARAM_APPLICATIONNAME); 
    		}
    	}
    	else
    	{
    		return pLauncher.getParameter(ILauncher.PARAM_APPLICATIONNAME);
    	}
	}
	
}	// ApplicationUtil
