/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [JR] - creation
 * 05.10.2008 - [JR] - changePassword implemented
 * 28.09.2009 - [JR] - ISession instead of AbstractSession
 * 04.10.2009 - [JR] - changePassword: old password is required
 * 07.10.2009 - [JR] - AutoLogin support
 * 14.10.2009 - [JR] - prepared statements cleanup
 *                   - added missing commit calls
 *                   - added missing rollback calls
 *                   - openConnection: setAutoCommit(false)
 * 21.10.2009 - [JR] - "KEY" renamed to "LOGINKEY" for table AUTOLOGIN (reserved keyword in derby db)
 * 23.10.2009 - [HM] - openConnection: used DBAccess to check the driver name
 *            - [JR] - openConnection: exception handling (NullPointer) 
 * 14.11.2009 - [JR] - #7
 *                     changePassword: validatePassword called 
 * 06.06.2010 - [JR] - changePassword: don't check password during update (paranoid)
 *                   - #132: encryption support     
 * 20.06.2010 - [JR] - #137: initStatements() implemented to cache prepared statements  
 * 23.10.2010 - [JR] - AutoLogin is not required
 * 01.12.2010 - [JR] - #219: used DBCredentials and supported datasource      
 * 11.02.2011 - [JR] - openConnection: fixed private member access through derived classes
 * 06.05.2011 - [JR] - #344: custom access controller support  
 * 11.05.2011 - [JR] - release implemented       
 * 18.06.2011 - [JR] - close AutoLogin statements when an exception occurs during statement creation
 * 31.07.2011 - [JR] - #446: getConnection implemented and table/column names cached
 * 22.09.2011 - [JR] - #475: removeAccess implemented  
 * 22.11.2011 - [JR] - #515: use DBAccess to crete the connection (takes all optimizations e.g. transaction isolation level)
 * 07.12.2011 - [JR] - renamed initStatementsIntern to initStatements and made it protected
 * 16.02.2013 - [JR] - #634: initStatements: removed alias of delete statements 
 * 26.02.2013 - [JR] - #642: getCredentials(ISession) implemented for derived classes
 * 08.09.2013 - [JR] - #788: check environment in getAccessController                                              
 */
package com.sibvisions.rad.server.security;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.UUID;

import javax.rad.application.ILauncher;
import javax.rad.remote.ChangePasswordException;
import javax.rad.remote.IConnectionConstants;
import javax.rad.server.IConfiguration;
import javax.rad.server.ISession;

import com.sibvisions.rad.persist.jdbc.DBAccess;
import com.sibvisions.rad.persist.jdbc.DBCredentials;
import com.sibvisions.rad.server.AbstractSession;
import com.sibvisions.rad.server.config.Configuration;
import com.sibvisions.rad.server.config.DBObjects;
import com.sibvisions.rad.server.config.Zone;
import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.LoggerFactory;
import com.sibvisions.util.xml.XmlNode;

/**
 * The <code>DBSecurityManager</code> uses a database to validate/authenticate users. 
 * It requires the following information to establish a database connection:
 * <ul>
 *   <li>driver (jdbc driver classname)</li>
 *   <li>url (jdbc connect url)</li>
 *   <li>username (database username)</li>
 *   <li>database (database password)</li>
 * </ul>
 * 
 * To use automatic login the session property:<br/>
 * <code>IConnectionConstants.PREFIX_CLIENT + "login.auto"</code> should be set to <code>true</code> when the user logs on.
 * After a successful logon the property: <code>IConnectionConstants.PREFIX_CLIENT + "login.key"</code> will be set to
 * a unique login key. The client should store the key in its local registry. When the property 
 * <code>IConnectionConstants.PREFIX_CLIENT + "login.key"</code> is set before opening the connection, then the user will be logged in
 * if the login is possible! 
 * 
 * @author Ren Jahn
 */
public class DBSecurityManager extends AbstractSecurityManager
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** the name of the users table. */
	protected static final String TABLE_USERS = "USERS";
	
	/** the name of the autologin table. */
	protected static final String TABLE_AUTOLOGIN = "AUTOLOGIN";
	
	/** the name of the accessrules table. */
	protected static final String VIEW_ACCESSRULES = "V_ACCESSRULES";

	
	/** the logger. */
	private static ILogger log = LoggerFactory.getInstance(DBSecurityManager.class);

	/** the database credentials. */
	private DBCredentials credentials = null;
	
	/** database connection. */
	private Connection con = null;
	
	/** the check autologin statement. */
	private PreparedStatement psAutoLogin;
	
	/** the insert autologin statement. */
	private PreparedStatement psInsertAutoLogin;

	/** the delete autologin statement with key. */
	private PreparedStatement psDeleteAutoLoginKey;
	
	/** the delete autologin statement with username. */
	private PreparedStatement psDeleteAutoLoginUser;

	/** the user query by id. */
	private PreparedStatement psUserId;
	
	/** the user query by username. */
	private PreparedStatement psUserName;
	
	/** the accessrule query statement. */
	private PreparedStatement psAccessRule;
	
	/** the change password statement. */
	private PreparedStatement psChangePwd;
	
	/** the change password statement which unsets change password flag. */
	private PreparedStatement psChangePwdUnset;

	/** the name of the users table. */
	String sUsersTable;
	/** the ID column name of the users table. */
	String sUsersId;   
	/** the USERNAME column name of the users table. */
	String sUsersName;
	/** the CHANGE_PASSWORD column name of the users table. */
	String sUsersChgPwd;
	/** the PASSWORD column name of the users table. */
	String sUsersPwd;
	
	/** the name of the autologin table. */
	String sAutoLoginTable;
	/** the USER_ID column name of the autologin table. */
	String sAutoLoginId;
	/** the LOGINKEY column name of the autologin table. */
	String sAutoLoginKey;

	/** the name of the accessrules table. */
	String sAccessTable;
	/** the USERNAME column name of the accessrules table. */
	String sAccessUser;
	
	/** the last modified date of the configuration. */
	private long lConfigModified = -1;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public synchronized void validateAuthentication(ISession pSession) throws Exception
	{
		String sApplication = pSession.getApplicationName();
		String sUserName = pSession.getUserName();
		
		IConfiguration cfgSession = pSession.getConfig();

		ResultSet resUser = null;
		
		boolean bAutoLogin = false;

		openConnection(pSession);

		try
		{
			//Check if the user will be connected through automatic login key
			String sAutoKey = (String)pSession.getProperty(IConnectionConstants.PREFIX_CLIENT + "login.key");
			
			if (psAutoLogin != null)
			{
				
				if (sAutoKey != null)
				{
					ResultSet res = null;
					
					try
					{
						//check if the authentication key is present!
						psAutoLogin.clearParameters();
						psAutoLogin.setString(1, sAutoKey);
						
						res = psAutoLogin.executeQuery();
						
						if (res.next())
						{
							//get the user information for validation!
							psUserId.clearParameters();
							psUserId.setBigDecimal(1, res.getBigDecimal(1));
							
							resUser = psUserId.executeQuery();
							
							bAutoLogin = true;
						}
					}
					catch (Exception e)
					{
						log.debug(e);
					}
					finally
					{
						if (res != null)
						{
							try
							{
								res.close();
							}
							catch (SQLException sqle)
							{
								//nothing to be done
							}
						}
					}
				}
			}
			
			if (resUser == null)
			{
				psUserName.clearParameters();
				psUserName.setString(1, sUserName);
				
				resUser = psUserName.executeQuery();
			}
								
			validateUser(pSession, resUser);

			if (!bAutoLogin)
			{
				//Password check
				String sPassword;
				try
				{
					sPassword = resUser.getString(DBObjects.getColumnName(cfgSession, TABLE_USERS, "PASSWORD"));
				}
				catch (SQLException sqle)
				{
					sPassword = null;
				}
				
				if (!isPasswordValid(pSession, sPassword))
				{
					throw new SecurityException("Invalid password for '" + sUserName + "' and application '" + sApplication + "'");
				}
			}

			//Change Password check
			String sChangePwd;
			try
			{
				sChangePwd = resUser.getString(DBObjects.getColumnName(cfgSession, TABLE_USERS, "CHANGE_PASSWORD"));
			}
			catch (SQLException sqle)
			{
				sChangePwd = null;
			}
			
			if (isChangePassword(pSession, sChangePwd))
			{
				//change password immediately
				throw new ChangePasswordException("Please change your password");
			}
			
			boolean bAutoLoginEnabled = Boolean.valueOf((String)pSession.getProperty(IConnectionConstants.PREFIX_CLIENT + "login.auto")).booleanValue();
			
			//check if the user allows automatic login with login key
			if (sAutoKey == null && bAutoLoginEnabled
				&& psAutoLogin != null && psDeleteAutoLoginUser != null && psInsertAutoLogin != null)
			{
				sAutoKey = UUID.randomUUID().toString();
				
				try
				{
					BigDecimal bdUserId = resUser.getBigDecimal(DBObjects.getColumnName(cfgSession, TABLE_USERS, "ID"));
					
					//cleanup old autologin keys
					psDeleteAutoLoginUser.clearParameters();
					psDeleteAutoLoginUser.setBigDecimal(1, bdUserId);
					if (psDeleteAutoLoginUser.execute())
					{
						psDeleteAutoLoginUser.getResultSet().close();
					}
					
					//insert new autologin key
					psInsertAutoLogin.clearParameters();
					psInsertAutoLogin.setBigDecimal(1, bdUserId);
					psInsertAutoLogin.setString(2, sAutoKey);
					if (psInsertAutoLogin.execute())
					{
						psInsertAutoLogin.getResultSet().close();
					}
					
					con.commit();

					pSession.setProperty(IConnectionConstants.PREFIX_CLIENT + "login.key", sAutoKey);
				}
				catch (Exception e)
				{
					try
					{
						con.rollback();
					}
					catch (SQLException sqle)
					{
						log.error(sqle);
					}
					
					log.error(e);
				}
			}
			
			if (bAutoLogin)
			{
				//set the username when autologin 
				if (pSession instanceof AbstractSession)
				{
					try
					{
						((AbstractSession)pSession).setUserName(resUser.getString(DBObjects.getColumnName(cfgSession, TABLE_USERS, "USERNAME")));
					}
					catch (SQLException sqle)
					{
						throw new SecurityException("USERNAME column for application '" + sApplication + "' was not found!");
					}
				}
			}
			else if (!bAutoLoginEnabled)
			{
				//unset the key
				pSession.setProperty(IConnectionConstants.PREFIX_CLIENT + "login.key", null);
			}
		}
		catch (SQLException sqle)
		{
			log.error(sqle);
			
			throw new SecurityException("Authentication for user '" + sUserName + "' and application '" + sApplication + "' is not possible");
		}
		finally
		{
			if (resUser != null)
			{
				try
				{
					resUser.close();
				}
				catch (Throwable th)
				{
					//ignore
				}
				finally
				{
					resUser = null;
				}
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void changePassword(ISession pSession) throws Exception
	{
		String sOldPwd = (String)pSession.getProperty(IConnectionConstants.OLDPASSWORD);
		String sNewPwd = (String)pSession.getProperty(IConnectionConstants.NEWPASSWORD);
		
		//-------------------------------------------------
		// Password validation
		//-------------------------------------------------
		
		validatePassword(pSession, sOldPwd, sNewPwd);
		
		//-------------------------------------------------
		// Change password
		//-------------------------------------------------

		String sApplication = pSession.getApplicationName();
		String sUserName = pSession.getUserName();
		
		IConfiguration cfgSession = pSession.getConfig();

		
		openConnection(pSession);
		
		ResultSet resUser = null;		
		
		try
		{
			psUserName.clearParameters();
			psUserName.setString(1, sUserName);
			
			resUser = psUserName.executeQuery();
			
			validateUser(pSession, resUser);
			
			//Old Password check
			String sPassword;
			try
			{
				sPassword = resUser.getString(DBObjects.getColumnName(cfgSession, TABLE_USERS, "PASSWORD"));
			}
			catch (SQLException sqle)
			{
				sPassword = null;
			}

			//check current session password and the database password (double check)
			//autologin: no password available -> don't check the session password!
			if (((pSession.getProperty(IConnectionConstants.PREFIX_CLIENT + "login.key") != null && pSession.getPassword() == null)
					 //no need for encryption because both passwords are plain text!
				  || comparePassword(cfgSession, pSession.getPassword(), sOldPwd))
				    //changed order because of encrypted "sPassword" (maybe)
				&& comparePassword(cfgSession, sOldPwd, sPassword))
			{
				//check change_password column and disable the check, if the column doesn't exist
				boolean bChangePwd;
				try
				{
					resUser.getString(DBObjects.getColumnName(cfgSession, TABLE_USERS, "CHANGE_PASSWORD"));
					bChangePwd = true;
				}
				catch (Throwable the)
				{
					//missing field!
					bChangePwd = false;
				}

				PreparedStatement ps;
				
				if (bChangePwd)
				{
					ps = psChangePwdUnset;
				}
				else
				{
					ps = psChangePwd;
				}
				
				ps.clearParameters();
				ps.setString(1, getEncryptedPassword(cfgSession, sNewPwd));
				ps.setString(2, sUserName);

				if (ps.execute())
				{
					ps.getResultSet().close();
				}
				
				con.commit();

				if (ps.getUpdateCount() != 1)
				{
					throw new SecurityException("User '" + sUserName + "' was not found for application '" + sApplication + "'");
				}
			}
			else
			{
				throw new SecurityException("Invalid password for '" + sUserName + "' and application '" + sApplication + "'");
			}
		}
		catch (SQLException sqle)
		{
			log.debug(sqle);
			
			try
			{
				con.rollback();
			}
			catch (SQLException ex)
			{
				log.error(ex);
			}
			
			throw new SecurityException("Error while changing password of '" + sUserName + "' for application '" + sApplication + "'");
		}
		finally
		{
			if (resUser != null)
			{
				try
				{
					resUser.close();
				}
				catch (Throwable th)
				{
					//ignore
				}
				finally
				{
					resUser = null;
				}
			}
		}		
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void logout(ISession pSession)
	{
		if (Boolean.valueOf((String)pSession.getProperty("userlogout")).booleanValue())
		{			
			try
			{
				String sAutoKey = (String)pSession.getProperty(IConnectionConstants.PREFIX_CLIENT + "login.key");
				
				if (sAutoKey != null)
				{
					openConnection(pSession);
	
					psDeleteAutoLoginKey.clearParameters();
					psDeleteAutoLoginKey.setString(1, sAutoKey);
					if (psDeleteAutoLoginKey.execute())
					{
						psDeleteAutoLoginKey.getResultSet().close();
					}
					
					con.commit();
					
					//unset the key
					pSession.setProperty(IConnectionConstants.PREFIX_CLIENT + "login.key", null);
				}
			}
			catch (Exception e)
			{
				try
				{
					con.rollback();
				}
				catch (SQLException sqle)
				{
					log.error(sqle);
				}

				log.error(e);
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized IAccessController getAccessController(ISession pSession) throws Exception
	{
		IConfiguration cfgSession = pSession.getConfig();

		ResultSet res = null;
		
		openConnection(pSession);
		
		//NO access rules available -> no controller!
		if (psAccessRule == null)
		{
			return null;
		}
		
		try
		{
			psAccessRule.clearParameters();
			psAccessRule.setObject(1, pSession.getUserName());

			res = psAccessRule.executeQuery();

			String sEnvironment = (String)pSession.getProperty(IConnectionConstants.PREFIX_CLIENT + ILauncher.PARAM_ENVIRONMENT);
			String sYesValue = DBObjects.getYesValue(cfgSession);
			
			IAccessController accessControl = createAccessController(pSession);

			boolean bValidScreen;
			
			String sLifeCycleName = DBObjects.getColumnName(cfgSession, VIEW_ACCESSRULES, "LIFECYCLENAME");
			String sEnvDesktop = null;
			String sEnvWeb = null;
			String sEnvMobile = null;
			
			String sColName;
			
			boolean bUseEnv = false;
			
			if (sEnvironment != null)
			{
				sEnvDesktop = DBObjects.getColumnName(cfgSession, VIEW_ACCESSRULES, "ENV_DESKTOP");
				sEnvWeb     = DBObjects.getColumnName(cfgSession, VIEW_ACCESSRULES, "ENV_WEB");
				sEnvMobile  = DBObjects.getColumnName(cfgSession, VIEW_ACCESSRULES, "ENV_MOBILE");

				ResultSetMetaData rsmd = res.getMetaData();

				for (int i = 1, anz = rsmd.getColumnCount(); i <= anz && !bUseEnv; i++)
				{
					sColName = rsmd.getColumnName(i);
	
					if (sColName.equals(sEnvDesktop)
						|| sColName.equals(sEnvWeb)
						|| sColName.equals(sEnvMobile))
					{
						bUseEnv = true;
					}
				}
			}
			
			while (res.next())
			{
				bValidScreen = true;
				
				if (bUseEnv)
				{
					if (sEnvironment.equals(ILauncher.ENVIRONMENT_DESKTOP))
					{
						if (!sYesValue.equals(res.getString(sEnvDesktop)))
						{
							bValidScreen = false;
						}
					}
					else if (sEnvironment.equals(ILauncher.ENVIRONMENT_WEB))
					{
						if (!sYesValue.equals(res.getString(sEnvWeb)))
						{
							bValidScreen = false;
						}
					}
					else if (sEnvironment.equals(ILauncher.ENVIRONMENT_MOBILE))
					{
						if (!sYesValue.equals(res.getString(sEnvMobile)))
						{
							bValidScreen = false;
						}
					}
				}
				
				if (bValidScreen)
				{
					accessControl.addAccess(res.getString(sLifeCycleName));
				}
			}
			
			return accessControl;
		}
		catch (SQLException sqle)
		{
			log.debug(sqle);
			
			//no access control!
			return null;
		}
		finally
		{
			if (res != null)
			{
				try
				{
					res.close();
				}
				catch (Throwable th)
				{
					//nothing to be done
				}
				finally
				{
					res = null;
				}
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void release()
	{
		try
		{
			closeConnection();
		}
		catch (Exception e)
		{
			log.error(e);
		}
	}	
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void finalize() throws Throwable 
	{
		//save a call and don't invoke release
		if (con != null)
		{
			try
			{
				con.close();
			}
			catch (Throwable th)
			{
				//ignore
			}
		}

		super.finalize();
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Opens a database connection to the database of an application.
	 * 
	 * @param pSession the session for which the connection should be opened
	 * @return a new or reused connection to the database
	 * @throws Exception if the application zone is invalid or the connection can not be opened 
	 * @throws IllegalArgumentException if the database configuration is invalid (parameters are missing, ...) 
	 */
	protected Connection openConnection(ISession pSession) throws Exception
	{
		String sApplication = pSession.getApplicationName();
		
		DBCredentials dbcred = getCredentials(pSession);
		
		
		if (dbcred == null)
		{
			throw new IllegalArgumentException("Database credentials were not found!"); 
		}
		
		//Validation

		if (dbcred.getDriver() == null)
		{
			DBAccess dba = DBAccess.getDBAccess(dbcred.getUrl());
			
			if (dba != null)
			{
				dbcred = new DBCredentials(dba.getDriver(), dbcred.getUrl(), dbcred.getUserName(), dbcred.getPassword());
				
				if (dbcred.getDriver() == null)
				{
					throw new IllegalArgumentException("Parameter 'driver' is missing for application '" + sApplication + "'");
				}
			}
			else
			{
				throw new IllegalArgumentException("Parameter 'url' is missing for application '" + sApplication + "'");
			}
		}

		if (dbcred.getUrl() == null)
		{
			throw new IllegalArgumentException("Parameter 'url' is missing for application '" + sApplication + "'");
		}

		if (dbcred.getUserName() == null)
		{
			throw new IllegalArgumentException("Parameter 'username' is missing for application '" + sApplication + "'");
		}

		if (dbcred.getPassword() == null)
		{
			throw new IllegalArgumentException("Parameter 'password' is missing for application '" + sApplication + "'");
		}

		//Connection check
		boolean bOpenCon = false;
		
		long lModified;
		
		IConfiguration cfgSession = pSession.getConfig();
		
		if (cfgSession instanceof Zone)
		{
			lModified = ((Zone)cfgSession).getFile().lastModified();
		}
		else
		{
			//maybe a special IConfiguration implementation (paranoid mode)
			lModified = Configuration.getApplicationZone(sApplication).getFile().lastModified();
		}
		
		if (con == null || !credentials.equals(dbcred) || lConfigModified != lModified)
		{
			//if the settings was changed, the connection need to be re-established
			bOpenCon = true;
		}
		else
		{
			//check if it's possible to reuse the connection
			
			Statement stmt = null;

			ResultSet res = null;
			
			try
			{
				stmt = con.createStatement();
				
				res = stmt.executeQuery("select ID from " + DBObjects.getTableName(cfgSession, TABLE_USERS));
				res.next();
			}
			catch (Throwable th)
			{
				bOpenCon = true;
			}
			finally
			{
				if (res != null)
				{
					try
					{
						res.close();
					}
					catch (Throwable th)
					{
						//nothing to be done
					}
				}
				
				if (stmt != null)
				{
					try
					{
						stmt.close();
					}
					catch (Throwable th)
					{
						//nothing to be done
					}
				}
			}
		}
		
		if (bOpenCon)
		{
			closeConnection();
			
			try
			{
				Class.forName(dbcred.getDriver());
			}
			catch (ClassNotFoundException e)
			{
				throw new ClassNotFoundException("JDBC driver '" + dbcred.getDriver() + "' for application '" + sApplication + "' was not found!", e);
			}
			
			try
			{				
				//Use DBAccess to use JVx default connection settings
				DBAccess dba = DBAccess.getDBAccess(dbcred);
				dba.open();
				
				con = dba.getConnection();
				con.setAutoCommit(false);

				initStatements(cfgSession);
			}
			catch (SQLException sqle)
			{
				closeConnection();
				
				throw new Exception("Can not open database connection with '" + dbcred.getUrl() + "' for application '" + sApplication + "'", sqle);
			}

			//Cache values for later validation
			credentials = dbcred;
			
			lConfigModified = lModified;
		}
		
		return con;
	}
	
	/**
	 * Close all statements and the connection.
	 * 
	 * @throws Exception if one statement can not be closed
	 */
	protected void closeConnection() throws Exception
	{
		if (con != null)
		{
			//close ALL internal prepared statements
			Object obj;
			int iModifier;

			//don't use getClass() because derived classes do not have access to private fields!
			for (Field field : DBSecurityManager.class.getDeclaredFields())
			{
				iModifier = field.getModifiers();
				
				if (!Modifier.isFinal(iModifier) && !Modifier.isStatic(iModifier))
				{
					try
					{
						obj = field.get(this);
						if (obj != null && obj instanceof PreparedStatement)
						{
							try 
							{ 
								((PreparedStatement)obj).close(); 
							} 
							catch (Exception e) 
							{
								//nothing to be done
							}
						}
					}
					catch (Exception e)
					{
						log.debug(field.getName(), e);
					}
				}
			}
			
			try
			{
				con.close();
			}
			catch (Throwable th)
			{
				//ignore
			}
			finally
			{
				con = null;
			}
		}
	}
	
	/**
	 * Initializes all prepared statements.
	 * 
	 * @param pConfig the session configuration
	 * @throws Exception if an exception occurs during statement creation
	 */
	private void initStatements(IConfiguration pConfig) throws Exception
	{
		sUsersTable  = DBObjects.getTableName(pConfig, TABLE_USERS);
		sUsersId     = DBObjects.getColumnName(pConfig, TABLE_USERS, "ID");   
		sUsersName   = DBObjects.getColumnName(pConfig, TABLE_USERS, "USERNAME");
		sUsersChgPwd = DBObjects.getColumnName(pConfig, TABLE_USERS, "CHANGE_PASSWORD");
		sUsersPwd    = DBObjects.getColumnName(pConfig, TABLE_USERS, "PASSWORD");
		
		sAutoLoginTable = DBObjects.getTableName(pConfig, TABLE_AUTOLOGIN);
		sAutoLoginId    = DBObjects.getColumnName(pConfig, TABLE_AUTOLOGIN, "USER_ID");
		sAutoLoginKey   = DBObjects.getColumnName(pConfig, TABLE_AUTOLOGIN, "LOGINKEY");

		sAccessTable = DBObjects.getTableName(pConfig, VIEW_ACCESSRULES);
		sAccessUser  = DBObjects.getColumnName(pConfig, VIEW_ACCESSRULES, "USERNAME");

		initStatements(con);
	}
	
	/**
	 * Initializes all prepared statements with current table and column names.
	 * 
	 * @param pConnection the connection to use
	 * @throws Exception if an exception occurs during statement creation
	 */
	protected void initStatements(Connection pConnection) throws Exception
	{
		// USER
		
		psUserId = pConnection.prepareStatement("select * from " + sUsersTable + " u where u." + sUsersId + " = ?");
		
		psUserName = pConnection.prepareStatement("select * from " + sUsersTable + " u where u." + sUsersName + " = ?");
		
		psChangePwd = pConnection.prepareStatement("update " + sUsersTable + " u set u." + sUsersPwd + " = ? " + " where u." + sUsersName + " = ?");
		
		psChangePwdUnset = pConnection.prepareStatement("update " + sUsersTable + 
						    	                        " u set u." + sUsersPwd + " = ?, " +
							                                   "u." + sUsersChgPwd + " = 'N' " +
							                            " where u." + sUsersName + " = ?");
		
		// AUTOLOGIN

		try
		{
			psAutoLogin = pConnection.prepareStatement("select al." + sAutoLoginId + " from " + sAutoLoginTable + " as al where al." + sAutoLoginKey + " = ?");
			
			psInsertAutoLogin = pConnection.prepareStatement("insert into " + sAutoLoginTable + "(" + sAutoLoginId + ", " + sAutoLoginKey + ") values (?, ?)");
	
			//don't use alias' in delete statements (Mysql doesn't like it, e.g. http://dev.mysql.com/doc/refman/5.5/en/delete.html)
			//correct syntax for mysql: delete a1 from table a1 where a1.key = value
			psDeleteAutoLoginKey = pConnection.prepareStatement("delete from " + sAutoLoginTable + " where " + sAutoLoginKey + " = ?");
			
			psDeleteAutoLoginUser = pConnection.prepareStatement("delete from " + sAutoLoginTable + " where " + sAutoLoginId + " = ?");
		}
		catch (SQLException sqle)
		{
			if (psAutoLogin != null)
			{
				try
				{
					psAutoLogin.close();
				}
				catch (Exception e)
				{
					//nothing to be done
				}
				
				psAutoLogin = null;
			}
			
			if (psInsertAutoLogin != null)
			{
				try
				{
					psInsertAutoLogin.close();
				}
				catch (Exception e)
				{
					//nothing to be done
				}
				
				psInsertAutoLogin = null;
			}
			
			if (psDeleteAutoLoginKey != null)
			{
				try
				{
					psDeleteAutoLoginKey.close();
				}
				catch (Exception e)
				{
					//nothing to be done
				}
				
				psDeleteAutoLoginKey = null;
			}

			if (psDeleteAutoLoginUser != null)
			{
				try
				{
					psDeleteAutoLoginUser.close();
				}
				catch (Exception e)
				{
					//nothing to be done
				}
				
				psDeleteAutoLoginUser = null;
			}
		}
		
		// ACCESS
		
		try
		{
			psAccessRule = pConnection.prepareStatement("select * " + 
                                                        "from " + sAccessTable +
                                                       " where " + sAccessUser + " = ?");
		}
		catch (SQLException sqle)
		{
			//nothing to be done
		}
	}
	
	/**
	 * Checks if a user is allowed to authenticate. That means, if the user is active and the login
	 * is valid for the current date. 
	 * 
	 * @param pSession the session which needs access
	 * @param pUserInfo the user information from the database
	 * @throws Exception if the user is not allowed to authenticate
	 */
	private void validateUser(ISession pSession, ResultSet pUserInfo) throws Exception
	{
		String sApplication = pSession.getApplicationName();
		String sUserName = pSession.getUserName();

		if (pUserInfo.next())
		{
			IConfiguration cfgSession = pSession.getConfig();
			
			String sActive;
			try
			{
				sActive = pUserInfo.getString(DBObjects.getColumnName(cfgSession, TABLE_USERS, "ACTIVE"));
			}
			catch (SQLException sqle)
			{
				sActive = null;
			}
			
			if (isActive(pSession, sActive))
			{
				Timestamp tsmpFrom;
				try
				{
					tsmpFrom = pUserInfo.getTimestamp(DBObjects.getColumnName(cfgSession, TABLE_USERS, "VALID_FROM"));
				}
				catch (SQLException sqle)
				{
					tsmpFrom = null;
				}
				
				Timestamp tsmpTo;
				try
				{
					tsmpTo = pUserInfo.getTimestamp(DBObjects.getColumnName(cfgSession, TABLE_USERS, "VALID_TO"));
				}
				catch (SQLException sqle)
				{
					tsmpTo = null;
				}
				
				if (!isValid(pSession, tsmpFrom, tsmpTo))
				{
					throw new SecurityException("User '" + sUserName + "' is expired for application '" + sApplication + "'");
				}
			}
			else
			{
				throw new SecurityException("User '" + sUserName + "' is inactive for application '" + sApplication + "'");
			}
		}
		else
		{
			throw new SecurityException("User '" + sUserName + "' was not found for application '" + sApplication + "'");
		}
	}
	
	/**
	 * Checks if a user is active.
	 * 
	 * @param pSession the session which needs access
	 * @param pActive the active flag or <code>null</code> if the flag is not available
	 * @return <code>true</code> if the active flag is missing or the flag equals the yes value
	 * @throws Exception if the configuration of the session is invalid
	 */
	protected boolean isActive(ISession pSession, String pActive) throws Exception
	{
		//not configured -> ignore
		if (pActive == null)
		{
			return true;
		}
		
		return DBObjects.getYesValue(pSession.getConfig()).equals(pActive);
	}
	
	/**
	 * Checks if a user is valid.
	 * 
	 * @param pSession the session which needs access
	 * @param pFrom the from date/time or <code>null</code> for undefined
	 * @param pTo the to date/time or <code>null</code> for undefined
	 * @return <code>true</code> if the from/to combination is possible, <code>false</code> otherwise
	 */
	protected boolean isValid(ISession pSession, Timestamp pFrom, Timestamp pTo)
	{
		long lNow = System.currentTimeMillis();
		
		return ((pFrom == null || pFrom.getTime() <= lNow)  
			    && (pTo == null || pTo.getTime() > lNow));
	}
	
	/**
	 * Checks if the user password is valid.
	 * 
	 * @param pSession the session which needs access
	 * @param pPassword the confirmation password (encrypted or plain text)
	 * @return <code>true</code> if the user password is valid
	 * @throws Exception if the password validation failed (e.g. encryption problems)
	 */
	protected boolean isPasswordValid(ISession pSession, String pPassword) throws Exception
	{
		return comparePassword(pSession.getConfig(), pSession.getPassword(), pPassword);
	}
	
	/**
	 * Checks if the change password flag is set.
	 * 
	 * @param pSession the session which needs access 
	 * @param pChangePassword the change password flag or <code>null</code> if the flag is not available
	 * @return <code>true</code> if the change password flag is set or <code>false</code> if the flag is
	 *         <code>null</code> or is not set
	 * @throws Exception if the configuration of the session is invalid
	 */
	protected boolean isChangePassword(ISession pSession, String pChangePassword) throws Exception
	{
		//not configured -> ignore
		if (pChangePassword == null)
		{
			return false;
		}
		
		return DBObjects.getYesValue(pSession.getConfig()).equals(pChangePassword);
	}

	/**
	 * Creates an access controller for a {@link ISession}.
	 * 
	 * @param pSession the session which requests the access controller
	 * @return the access controller
	 */
	protected IAccessController createAccessController(ISession pSession)
	{
		String sAccCtrl = pSession.getConfig().getProperty("/application/securitymanager/accesscontroller");
		
		if (sAccCtrl != null && sAccCtrl.trim().length() > 0)
		{
			try
			{
				Class<?> clazz = Class.forName(sAccCtrl);
				
				return (IAccessController)clazz.newInstance();
			}
			catch (ClassNotFoundException cnfe)
			{
				throw new SecurityException("Access controller '" + sAccCtrl + "' was not found!");
			}
			catch (InstantiationException ie)
			{
				throw new SecurityException("Can't instantiate access controller '" + sAccCtrl + "'!");
			}
			catch (IllegalAccessException iae)
			{
				throw new SecurityException("Access controller '" + sAccCtrl + "' not accessible!");
			}
		}
		
		return new DBAccessController();
	}

	/**
	 * Gets the configured database credentials for the given session.
	 * 
	 * @param pSession the session 
	 * @return the configured credentials
	 * @see #getCredentials(IConfiguration)
	 */
	protected DBCredentials getCredentials(ISession pSession)
	{
		return getCredentials(pSession.getConfig());
	}
	
	/**
	 * Gets the configured database credentials from a given configuration. This method handles
	 * credentials, set in the security manager and credentials configured as datasource.
	 * 
	 * @param pConfig the configuration 
	 * @return the configured credentials
	 */
	public static DBCredentials getCredentials(IConfiguration pConfig)
	{
		try
		{
			XmlNode xmnDb = pConfig.getNode("/application/securitymanager/database");
			
			if (xmnDb == null)
			{
				return DataSourceHandler.createDBCredentials(pConfig, "default");
			}
			else
			{
				XmlNode xmnDs = xmnDb.getNode("/datasource"); 
			
				if (xmnDs == null)
				{
					return DataSourceHandler.createDBCredentials(xmnDb);
				}
				else
				{
					return DataSourceHandler.createDBCredentials(pConfig, xmnDs.getValue());
				}
			}
		}
		catch (Exception e)
		{
			log.error(e);
		}
		
		return null;
	}
	
	/**
	 * Gets the current connection to the database. The connection is validated to ensure that it is usable.
	 * 
	 * @return the connection for the security manager or <code>null</code> if the security manager did
	 *         not open a connection
	 * @throws Exception if db access fails
	 */
	public Connection getConnection() throws Exception
	{
		if (con != null)
		{
			boolean bReopen = false;
			
			//check if it's possible to reuse the connection
			
			Statement stmt = null;

			ResultSet res = null;
			
			try
			{
				stmt = con.createStatement();
				
				res = stmt.executeQuery("select ID from " + sUsersTable);
				res.next();
			}
			catch (Throwable th)
			{
				bReopen = true;
			}
			finally
			{
				if (res != null)
				{
					try
					{
						res.close();
					}
					catch (Throwable th)
					{
						//nothing to be done
					}
				}
				
				if (stmt != null)
				{
					try
					{
						stmt.close();
					}
					catch (Throwable th)
					{
						//nothing to be done
					}
				}
			}
			
			if (bReopen)
			{
				//re-open the connection with cached credentials 
				try
				{			
					DBAccess dba = DBAccess.getDBAccess(credentials);

					con = dba.getConnection();
					con.setAutoCommit(false);
					
					initStatements(con);
				}
				catch (SQLException sqle)
				{
					closeConnection();
					
					throw new Exception("Can not open database connection with '" + credentials.getUrl() + "'", sqle);
				}
			}
		}
		
		return con;
	}
	
	/**
	 * Gets the connection to the database. The connection is validated to ensure that it is usable.
	 * 
	 * @param pSession the session that wants access to the database
	 * @return the connection
	 * @throws Exception if db access fails
	 */
	public Connection getConnection(ISession pSession) throws Exception
	{
		return openConnection(pSession);
	}	
	
	//****************************************************************
	// Subclass definition
	//****************************************************************

	/**
	 * The <code>DBAccessController</code> controls the access to server side objects based on the
	 * configuration in the database.
	 * 
	 * @author Ren Jahn
	 */
	public static final class DBAccessController implements IAccessController
	{
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Class members
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/** the allowed lifecycle objects. */
		private ArrayUtil<String> auAllowedLCO = null;
		
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Interface implementation
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
		 * {@inheritDoc}
		 */
		public boolean isAllowed(String pLifeCycleName)
		{
			//all explicite allowed lifecycle objects are accessible
			if (pLifeCycleName != null && auAllowedLCO != null)
			{
				return auAllowedLCO.contains(pLifeCycleName);
			}
			
			return false;
		}
		
		/**
		 * {@inheritDoc}
		 */
		public void addAccess(String pLifecycleName)
		{
			if (pLifecycleName == null)
			{
				return;
			}
			
			if (auAllowedLCO == null)
			{
				auAllowedLCO = new ArrayUtil<String>();
			}

			if (!auAllowedLCO.contains(pLifecycleName))
			{
				auAllowedLCO.add(pLifecycleName);
			}
		}

		/**
		 * {@inheritDoc}
		 */
		public void removeAccess(String pLifeCycleName)
		{
			if (pLifeCycleName == null || auAllowedLCO == null)
			{
				return;
			}
			
			auAllowedLCO.remove(pLifeCycleName);
		}
		
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// User-defined methods
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
		 * Gets the list of allowed life-cycle names.
		 * 
		 * @return the full qualified lice-cycle object names
		 */
		public String[] getAllowedLifeCycleNames()
		{
			if (auAllowedLCO == null)
			{
				return new String[0];
			}
			else
			{
				return auAllowedLCO.toArray(new String[auAllowedLCO.size()]);
			}
		}
		
	}	// DBAccessController
	
}	// DBSecurityManager
