/*
 * Copyright 2011 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 11.12.2011 - [JR] - creation
 */
package com.sibvisions.rad.server.http.rest;

import java.util.concurrent.ConcurrentMap;

import org.restlet.ext.jackson.JacksonRepresentation;
import org.restlet.representation.Representation;
import org.restlet.resource.Get;
import org.restlet.resource.Post;
import org.restlet.resource.Put;
import org.restlet.resource.ServerResource;

import com.sibvisions.rad.server.DirectServerSession;
import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.LoggerFactory;

/**
 * The <code>CallServerResource</code> allows action or object calls.
 * 
 * @author Ren Jahn
 */
public class CallServerResource extends ServerResource
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the logger. */
	private static ILogger logger = LoggerFactory.getInstance(CallServerResource.class);
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Executes an object or action call without parameters.
	 * 
	 * @throws Throwable if call fails because the method is not available or the call throws an Exception
	 * @return the call result in JSON representation
	 */
	@Get
	public Representation call() throws Throwable
	{
		try
		{
			DirectServerSession session = ((RESTSession)getRequest().getClientInfo().getUser()).getSession();
			
			ConcurrentMap<String, Object> cmpAttrib = getRequest().getAttributes();
	
			String sObjectName = (String)cmpAttrib.get(RESTAdapter.PARAM_OBJECT_NAME);
			String sActionName = (String)cmpAttrib.get(RESTAdapter.PARAM_ACTION_NAME);
	
			Object oResult;
			
			if (sObjectName != null)
			{
				oResult = session.call(sObjectName, sActionName);
			}
			else
			{
				oResult = session.callAction(sActionName);
			}
			
			return new JacksonRepresentation(oResult);
		}
		catch (Throwable th)
		{
			logger.debug(th);
			
			throw th;
		}
	}

	/**
	 * Executes an object or action call with or without parameters.
	 * 
	 * @param pParameter the call parameter
	 * @return call result in JSON representation
	 * @throws Throwable if call fails because the method is not available or the call throws an Exception
	 */
	@Put
	@Post
	public Representation call(Representation pParameter) throws Throwable
	{
		try
		{
			Object[] oParams;
			
			if (pParameter != null)
			{
				oParams = JSONUtil.getObject(pParameter, Object[].class);
			}
			else
			{
				oParams = null;
			}
			
			DirectServerSession session = ((RESTSession)getRequest().getClientInfo().getUser()).getSession();
			
			ConcurrentMap<String, Object> cmpAttrib = getRequest().getAttributes();
	
			String sObjectName = (String)cmpAttrib.get(RESTAdapter.PARAM_OBJECT_NAME);
			String sActionName = (String)cmpAttrib.get(RESTAdapter.PARAM_ACTION_NAME);
	
			Object oResult;
			
			if (sObjectName != null)
			{
				oResult = session.call(sObjectName, sActionName, oParams);
			}
			else
			{
				oResult = session.callAction(sActionName, oParams);
			}
			
			return new JacksonRepresentation(oResult);
		}
		catch (Throwable th)
		{
			logger.debug(th);
			
			throw th;
		}
	}
	
}	// CallServerResource
