/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [JR] - creation
 * 05.10.2008 - [JR] - setNewPassword implemented
 * 30.01.2009 - [JR] - check and use client compression support for results
 * 01.02.2009 - [JR] - compression property checked for stream compression
 * 03.02.2009 - [JR] - don't close streams: called flush or finish(-> zip)
 * 04.04.2009 - [JR] - process: used session.getProperties().getChanges(String.class) instead of session.getChangedProperties()
 * 13.05.2009 - [JR] - reflective ObjectProvider instantiation   
 * 25.05.2009 - [JR] - getProperty, setProperty, getCallBackResults, setAndCheckAlive now throws Throwable
 * 27.05.2009 - [JR] - implemented IDirectServer  
 * 04.10.2009 - [JR] - setNewPassword: old password as parameter      
 * 07.10.2009 - [JR] - IResponse.setProperty support  
 * 28.10.2009 - [JR] - getPropertyIntern: only String supported [BUGFIX] 
 * 14.11.2009 - [JR] - create monitoring object through the constructor, otherwise it doesn't receive events
 *                     from the SessionManager because it will be used too late! [BUGFIX]
 * 28.01.2010 - [JR] - removed serializer caching (serializers may not be stateless)
 * 23.02.2010 - [JR] - #18: 
 *                     * getPropertyIntern: use IConnectionConstants.PROPERTY_CLASSES for property transfer
 *                     * getPropertiesIntern: return only IConnectionConstants.PROPERTY_CLASSES properties
 * 02.03.2010 - [JR] - used List instead of ArrayUtil for property transfer 
 * 25.12.2010 - [JR] - #228: fixed objectprovider detection 
 * 29.12.2010 - [JR] - #231: sessionmanager creation with config property
 * 31.07.2011 - [JR] - #16: prepareException used     
 * 22.09.2011 - [JR] - #476: send changed request properties if no connection is available
 * 21.11.2012 - [JR] - #535: changed objectprovider and sessionmanager name to .../name because we'll use sub tags
 * 28.02.2013 - [JR] - #643: 
 *                     * validateCallBack implemented
 *                     * executeCallBack and executeActionCallBack returns Object (if callback is null)
 * 15.10.2013 - [JR] - DefaultSessionManager.setControllerInterval called                                 
 */
package com.sibvisions.rad.server;

import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;
import java.util.Map.Entry;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import javax.naming.InitialContext;
import javax.rad.remote.IConnection;
import javax.rad.remote.IConnectionConstants;
import javax.rad.remote.event.ICallBackListener;
import javax.rad.server.AbstractObjectProvider;
import javax.rad.server.ISession;
import javax.rad.server.ResultObject;

import com.sibvisions.rad.remote.ISerializer;
import com.sibvisions.rad.server.config.Configuration;
import com.sibvisions.rad.server.security.AbstractSecurityManager;
import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.ChangedHashtable;
import com.sibvisions.util.Reflective;
import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.LoggerFactory;
import com.sibvisions.util.type.CommonUtil;
import com.sibvisions.util.type.StringUtil;

/**
 * The <code>Server</code> is the general remote server implementation.
 * It's independent of the communication protocol and handles client requests.
 * <p/>
 * The configuration of the server will be made in the <pre>server.xml</pre> file.
 * It contains the database connect information.
 * 
 * @author Ren Jahn
 */
public class Server implements IDirectServer
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** server logger. */
	private ILogger log;
	
	/** the singleton server instance. */
	private static Server instance = null; 
	
	/** the session manager. */
	private DefaultSessionManager sessman = null;
	
	/** the object provider. */
	private AbstractObjectProvider objectprov = null;
	
	/** the monitoring object. */
	private Monitoring monitoring = null;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new <code>Server</code> instance.
	 */
	public Server()
	{
		String sLogFactory;
		
		try
		{
			sLogFactory = Configuration.getServerZone().getProperty("logfactory");
		}
		catch (Exception e)
		{
			sLogFactory = null;
		}
		
		LoggerFactory.init(sLogFactory);
		
		log = LoggerFactory.getInstance(Server.class);
		
		try
		{
			String sSessMan = Configuration.getServerZone().getProperty("/server/sessionmanager/class");
			
			sessman = (DefaultSessionManager)Reflective.construct(sSessMan, this);
			
			log.debug("Use ", sSessMan, " as SessionManager");
		}
		catch (Throwable th)
		{
			log.debug("NO PROBLEM! Use default SessionManager", th);
			
			sessman = new DefaultSessionManager(this);
		}
		
		try
		{
			String sInterval = Configuration.getServerZone().getProperty("/server/sessionmanager/controllerInterval");
			
			if (!StringUtil.isEmpty(sInterval))
			{
				DefaultSessionManager.setControllerInterval(Long.parseLong(sInterval));
			}
		}
		catch (Throwable th)
		{
			log.debug(th);
		}
		
		try
		{
			String sObjProvider = Configuration.getServerZone().getProperty("/server/objectprovider/class");
		
			objectprov = (AbstractObjectProvider)Reflective.construct(sObjProvider, this);
			
			log.debug("Use ", sObjProvider, " as ObjectProvider");
		}
		catch (Throwable th)
		{
			log.debug("NO PROBLEM! Use default ObjectProvider", th);
			
			objectprov = new DefaultObjectProvider(this); 
		}
		
		monitoring = new Monitoring(this);			
	}
	
	/**
	 * Gets the current server instance as singleton.
	 * 
	 * @return the singleton server instance
	 */
	public static synchronized Server getInstance()
	{
		if (instance == null)
		{
			try
			{
				InitialContext cxt = new InitialContext();
	
				if (cxt != null) 
				{
					instance = (Server)cxt.lookup("java:/comp/env/jvx/server");
				}
			}
			catch (Exception ex)
			{
				LoggerFactory.getInstance(Server.class).debug("Server is not configured as JNDI resource", ex);
			}

			//no JNDI server available
			if (instance == null)
			{
				instance = new Server();
			}
		}
		
		return instance;
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	//IServer

	/**
	 * {@inheritDoc}
	 */
	public final DefaultSessionManager getSessionManager()
	{
		return sessman;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public final AbstractObjectProvider getObjectProvider()
	{
		return objectprov;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public Object createSession(ChangedHashtable<String, Object> pProperties) throws Throwable
	{
		Throwable thReturn = null;
		Object oReturn = null;

		long lStart = System.currentTimeMillis();
		
		
		try
		{
			oReturn = createSessionIntern(null, null, pProperties);

			return oReturn;
		}
		catch (Throwable th)
		{
			thReturn = AbstractSecurityManager.prepareException(th);
			
			throw th;
		}
		finally
		{
			log(null, 
				IConnection.OBJ_SESSION, 
				IConnection.MET_SESSION_CREATE, 
				new Object[] {pProperties}, 
				null, 
				oReturn,
				thReturn,
				System.currentTimeMillis() - lStart);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public Object createSubSession(Object pSessionId, 
			                       ChangedHashtable<String, Object> pProperties) throws Throwable
	{
		Throwable thReturn = null;
		Object oReturn = null;
		
		long lStart = System.currentTimeMillis();
		
		
		try
		{
			oReturn = createSubSessionIntern(null, sessman.get(pSessionId), pProperties);
			
			return oReturn;
		}
		catch (Throwable th)
		{
			thReturn = AbstractSecurityManager.prepareException(th);
			
			throw th;
		}
		finally
		{
			log(pSessionId, 
				IConnection.OBJ_SESSION, 
				IConnection.MET_SESSION_SUBSESSION_CREATE, 
				new Object[] {pProperties}, 
				null, 
				oReturn, 
				thReturn,
				System.currentTimeMillis() - lStart);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void destroySession(Object pSessionId)
	{
		long lStart = System.currentTimeMillis();
		
		
		try
		{
			destroySessionIntern(pSessionId);
		}
		finally
		{
			log(pSessionId, 
				IConnection.OBJ_SESSION, 
				IConnection.MET_SESSION_DESTROY, 
				null, 
				null, 
				null,
				null,
				System.currentTimeMillis() - lStart);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public Object execute(Object pSessionId, String pObjectName, String pMethod, Object... pParams) throws Throwable
	{
		Throwable thReturn = null;
		
		Object oReturn = null;
		
		long lStart = System.currentTimeMillis();

		
		try
		{
			oReturn = executeIntern(sessman.get(pSessionId), new Call(null, pObjectName, pMethod, pParams));
			
			return oReturn;
		}
		catch (Throwable th)
		{
			thReturn = AbstractSecurityManager.prepareException(th);
			
			throw th;
		}
		finally
		{
			log(pSessionId, 
				pObjectName, 
				pMethod, 
				pParams, 
				null, 
				oReturn,
				thReturn,
				System.currentTimeMillis() - lStart);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void executeCallBack(Object pSessionId, 
			                    Object pCallBackId, 
			                    String pObjectName, 
			                    String pMethod, 
			                    Object... pParams) throws Throwable
	{
		Throwable thReturn = null;
		
		long lStart = System.currentTimeMillis();

		
		try
		{
			AbstractSession session = sessman.get(pSessionId);
			
			validateCallBack(session, pObjectName);
			
			Call call = new Call(pCallBackId, pObjectName, pMethod, pParams);
			call.setForceCallBack(true);
			
			executeIntern(session, call);
		}
		catch (Throwable th)
		{
			thReturn = AbstractSecurityManager.prepareException(th);
			
			throw th;
		}
		finally
		{
			log(pSessionId, 
				pObjectName, 
				pMethod, 
				pParams, 
				pCallBackId, 
				null,
				thReturn,
				System.currentTimeMillis() - lStart);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void executeCallBack(Object pSessionId, 
			                    ICallBackListener pCallBackListener, 
			                    String pObjectName, 
			                    String pMethod, 
			                    Object... pParams) throws Throwable
	{
		Throwable thReturn = null;
		
		long lStart = System.currentTimeMillis();

		
		try
		{
			AbstractSession session = sessman.get(pSessionId);
			
			validateCallBack(session, pObjectName);
			
			Call call = new Call(pCallBackListener, pObjectName, pMethod, pParams);
			call.setForceCallBack(true);
			
			executeIntern(session, call);
		}
		catch (Throwable th)
		{
			thReturn = AbstractSecurityManager.prepareException(th);
			
			throw th;
		}
		finally
		{
			log(pSessionId, 
				pObjectName, 
				pMethod, 
				pParams, 
				pCallBackListener, 
				null,
				thReturn,
				System.currentTimeMillis() - lStart);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public Object executeAction(Object pSessionId, String pAction, Object... pParams) throws Throwable
	{
		Throwable thReturn = null;
		Object oReturn = null;
		
		long lStart = System.currentTimeMillis();

		
		try
		{
			oReturn = executeIntern(sessman.get(pSessionId), new Call(null, null, pAction, pParams));
			
			return oReturn;
		}
		catch (Throwable th)
		{
			thReturn = AbstractSecurityManager.prepareException(th);
			
			throw th;
		}
		finally
		{
			log(pSessionId, 
				null, 
				pAction,
				pParams, 
				null, 
				oReturn,
				thReturn,
				System.currentTimeMillis() - lStart);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void executeActionCallBack(Object pSessionId, Object pCallBackId, String pAction, Object... pParams) throws Throwable
	{
		Throwable thReturn = null;
		
		long lStart = System.currentTimeMillis();

		
		try
		{
			AbstractSession session = sessman.get(pSessionId);
			
			validateCallBack(session, null);
			
			Call call = new Call(pCallBackId, null, pAction, pParams);
			call.setForceCallBack(true);
			
			executeIntern(session, call);
		}
		catch (Throwable th)
		{
			thReturn = AbstractSecurityManager.prepareException(th);
			
			throw th;
		}
		finally
		{
			log(pSessionId, 
				null, 
				pAction, 
				pParams, 
				pCallBackId, 
				null,
				thReturn,
				System.currentTimeMillis() - lStart);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void executeActionCallBack(Object pSessionId, ICallBackListener pCallBackListener, String pAction, Object... pParams) throws Throwable
	{
		Throwable thReturn = null;
		
		long lStart = System.currentTimeMillis();

		
		try
		{
			AbstractSession session = sessman.get(pSessionId);
			
			validateCallBack(session, null);
			
			Call call = new Call(pCallBackListener, null, pAction, pParams);
			call.setForceCallBack(true);
			
			executeIntern(session, call);
		}
		catch (Throwable th)
		{
			thReturn = AbstractSecurityManager.prepareException(th);
			
			throw th;
		}
		finally
		{
			log(pSessionId, 
				null, 
				pAction, 
				pParams, 
				pCallBackListener, 
				null,
				thReturn,
				System.currentTimeMillis() - lStart);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setProperty(Object pSessionId, String pName, Object pValue) throws Throwable
	{
		Throwable thReturn = null;

		long lStart = System.currentTimeMillis();
		
		try
		{
			setPropertyIntern(sessman.get(pSessionId), pName, pValue);
		}
		catch (Throwable th)
		{
			thReturn = AbstractSecurityManager.prepareException(th);
			
			throw th;
		}
		finally
		{
			log(pSessionId, 
				IConnection.OBJ_SESSION, 
				IConnection.MET_SESSION_SET_PROPERTY, 
				new Object[] {pName, pValue}, 
				null, 
				null,
				thReturn,
				System.currentTimeMillis() - lStart);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public Object getProperty(Object pSessionId, String pName) throws Throwable
	{
		Throwable thReturn = null;
		
		Object oResult = null;
		
		long lStart = System.currentTimeMillis();
		
		try
		{
			oResult = getPropertyIntern(sessman.get(pSessionId), pName);
		}
		catch (Throwable th)
		{
			thReturn = AbstractSecurityManager.prepareException(th);
			
			throw th;
		}
		finally
		{
			log(pSessionId, 
				IConnection.OBJ_SESSION, 
				IConnection.MET_SESSION_GET_PROPERTY, 
				null, 
				null, 
				oResult,
				thReturn,
				System.currentTimeMillis() - lStart);
		}
		
		return oResult;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public ChangedHashtable<String, Object> getProperties(Object pSessionId) throws Throwable
	{
		Throwable thReturn = null;
		
		ChangedHashtable<String, Object> htResult = null;
		
		long lStart = System.currentTimeMillis();
		
		try
		{
			htResult = getPropertiesIntern(sessman.get(pSessionId));
		}
		catch (Throwable th)
		{
			thReturn = AbstractSecurityManager.prepareException(th);
			
			throw th;
		}
		finally
		{
			log(pSessionId, 
				IConnection.OBJ_SESSION, 
				IConnection.MET_SESSION_GET_PROPERTIES, 
				null, 
				null, 
				htResult,
				thReturn,
				System.currentTimeMillis() - lStart);
		}
		
		return htResult;
	}

	/**
	 * {@inheritDoc}
	 */
	public List<ResultObject> getCallBackResults(Object pSessionId) throws Throwable
	{
		Throwable thReturn = null;

		ArrayUtil<ResultObject> auResult = null;

		long lStart = System.currentTimeMillis();
		
		
		try
		{
			 auResult = getCallBackResultsIntern(sessman.get(pSessionId));
		}
		catch (Throwable th)
		{
			thReturn = AbstractSecurityManager.prepareException(th);

			throw th;
		}
		finally
		{
			log(pSessionId, 
				IConnection.OBJ_SESSION, 
				"getCallBackResults", 
				new Object[] {pSessionId}, 
				null, 
				auResult,
				thReturn,
				System.currentTimeMillis() - lStart);
		}
		
		return auResult;
	}

	/**
	 * {@inheritDoc}
	 */
	public Object[] setAndCheckAlive(Object pSessionId, Object... pSubSessionId) throws Throwable
	{
		Throwable thReturn = null;
		
		Object[] oResult = null;
		
		long lStart = System.currentTimeMillis();
		
		try
		{
			oResult = setAndCheckAliveIntern(sessman.get(pSessionId), pSubSessionId);
		}
		catch (Throwable th)
		{
			thReturn = AbstractSecurityManager.prepareException(th);
			
			throw th;
		}
		finally
		{
			log(pSessionId, 
				IConnection.OBJ_SESSION, 
				IConnection.MET_SESSION_SETCHECKALIVE, 
				pSubSessionId, 
				null, 
				oResult,
				thReturn,
				System.currentTimeMillis() - lStart);
		}
		
		return oResult;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setNewPassword(Object pSessionId, String pOldPassword, String pNewPassword) throws Throwable
	{
		Throwable thReturn = null;
		
		long lStart = System.currentTimeMillis();
		
		try
		{
			setNewPasswordInternal(sessman.get(pSessionId), pOldPassword, pNewPassword);
		}
		catch (Throwable th)
		{
			thReturn = AbstractSecurityManager.prepareException(th);
			
			throw th;
		}
		finally
		{
			log(pSessionId, 
				IConnection.OBJ_SESSION, 
				IConnection.MET_SESSION_SET_NEW_PASSWORD, 
				new Object[] {"****"}, 
				null, 
				null,
				thReturn,
				System.currentTimeMillis() - lStart);
		}
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Processes client requests which uses the communication protocol.
	 * 
	 * @param pRequest the request
	 * @param pResponse the response
	 * @return the accessed session or <code>null</code> if the session is not available
	 * @throws Exception if a problem occurs while accessing the in- or output stream
	 */
	public ISession process(IRequest pRequest, IResponse pResponse) throws Exception
	{
		AbstractSession session = null;
		
		InputStream in;
		
		OutputStream out;
		
		ISerializer serializer = null;
		
		ArrayUtil<ResultObject> auResult = null;
		
		int iCallCount;
		
		ChangedHashtable<String, Object> chtProperties = null;

		
		try
		{
			//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			// REQUEST
			//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

			in = pRequest.getInputStream();

		    //Request-Header (same info as in AbstractSerializedConnection.java)
		    //
		    //<STREAM-IDENTIFIER>       1Byte  (A = Acknowledge; E = Established) 
		    //<OPTION-FLAG-1>           1Byte  (0x01 = COMPRESSED; 0x02 = UNCOMPRESSED>
		    // - <SERIALIZER-CLASSNAME> xBytes (only with A, via DataOutputStream)
		    // - <SESSION-ID>           xBytes (via DataOutputStream)
		    // - <CALL-COUNT>           xBytes (via ISerializer)
		    // - <CALL-PARAMETER>       xBytes (via ISerializer)
		    //    - [Objectname, Method, Parameter, CallBack-ID]
			
			char chStreamID = (char)in.read();
			
			if (chStreamID != IConnection.FLAG_ACKNOWLEDGE && chStreamID != IConnection.FLAG_ESTABLISHED)
			{
				throw new IOException("Invalid stream identifier '" + chStreamID + "'");
			}

			int iMode = in.read();
			
			InputStream isContent;
			
			//Check compression option
			if (iMode == IConnection.MODE_COMPRESSED)
			{
				isContent = new GZIPInputStream(in);
			}
			else
			{
				isContent = in;
			}
			
			DataInputStream disContent = new DataInputStream(isContent);

			String sSerializer = null;

			//The serializer classname is only available when the connection will be opened!
			if (chStreamID == IConnection.FLAG_ACKNOWLEDGE)
			{
				sSerializer = disContent.readUTF();
				
				serializer = (ISerializer)Class.forName(sSerializer).newInstance();
			}

			//Read session id
			Object oSessionId = disContent.readUTF();
						
			//null session id will be transmitted as empty string
			if (((String)oSessionId).trim().length() == 0)
			{
				oSessionId = null;
			}

			if (oSessionId != null)
			{
				session = sessman.get(oSessionId);
				
				serializer = session.getSerializer();
			}
			
			//Now a serializer must be available. Either through a new connection command (acknowledge) or
			//from an already created session
			if (serializer == null)
			{
				throw new SecurityException("Invalid serializer '" + sSerializer + "'");
			}
			
			iCallCount = ((Integer)serializer.read(disContent)).intValue();
			
			auResult = new ArrayUtil<ResultObject>(iCallCount);
			
			Throwable thReturn;
			
			Object oReturn;
			Object[] oCommand;
			
			long lStart;

			for (int i = 0; i < iCallCount; i++)
			{
				lStart = System.currentTimeMillis();
				
				oCommand = (Object[])serializer.read(disContent);
				
				//Request parameter validation
				if (oCommand.length == 4
					&& (oCommand[0] == null || oCommand[0].getClass() == String.class)
					&& oCommand[1].getClass() == String.class
					&& (oCommand[2] == null || oCommand[2] instanceof Object[]))
				{
					thReturn = null;
					oReturn = null;

					try
					{
						oReturn = process(pRequest,
										  serializer,
										  session, 
										  (String)oCommand[0], 
										  (String)oCommand[1], 
										  (Object[])oCommand[2], 
										  (Object)oCommand[3],
										  chtProperties);
					}
					catch (Throwable th)
					{
						thReturn = th;
						
						throw th;
					}
					finally
					{
						log(session != null ? session.getId() : null, 
							(String)oCommand[0], 
							(String)oCommand[1], 
							(Object[])oCommand[2], 
							(Object)oCommand[3], 
							oReturn,
							thReturn,
							System.currentTimeMillis() - lStart);
					}

					//Special handling of properties
					if (IConnection.OBJ_SESSION.equals((String)oCommand[0]) 
						&& IConnection.MET_SESSION_SET_PROPERTY.equals((String)oCommand[1]))
					{
						List<Object[]> auProperties = (List<Object[]>)oReturn;

						if (auProperties != null)
						{
							//Init properties for later use
							chtProperties = new ChangedHashtable<String, Object>();
							
							Object[] oProp;
							
							for (int j = 0, anzj = auProperties.size(); j < anzj; j++)
							{
								oProp = auProperties.get(j);
								
								chtProperties.put((String)oProp[0], oProp[1], false);
							}
						}
					}
					else
					{
						//No properties -> access the session
						if (session == null) 
						{
							//save the session instance for the case that more than one call was submitted 
							//within a single request
							session = sessman.get(oReturn);
						}
					
						//Properties will be handled seperately, so don't add to the result list!
						auResult.add(new ResultObject(IConnection.TYPE_CALL_RESULT, oReturn));
					}
				}
				else
				{
					log.debug("Invalid request parameters!", oCommand);
					
					throw new SecurityException("Invalid request parameters!");
				}
			}
		}
		catch (Throwable th)
		{
			log.error(th);
			
			if (auResult == null)
			{
				auResult = new ArrayUtil<ResultObject>(1);				
			}
			
			auResult.add(new ResultObject(IConnection.TYPE_CALL_ERROR, AbstractSecurityManager.prepareException(th)));
		}
		finally
		{
			try
			{
				pRequest.close();
			}
			catch (Throwable th)
			{
				log.error(th);
			}
			
			//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			// RESPONSE
			//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

			out = pResponse.getOutputStream();
			
		    //Response-Header (same as in AbstractSerializedConnection.java)
		    //
		    //<STREAM-IDENTIFIER>  1Byte  (E = Established; B = Broken) 
		    //<OPTION-FLAG-1>      1Byte  (0x01 = COMPRESSED; 0x02 = UNCOMPRESSED>
		    // - <RESULT-COUNT>    xBytes (only with E, via ISerializer)
		    // - <RESULT-TYPE>     1Byte  (only with E, via ISerializer)		    
		    // - <RESULT-OBJECT>   xBytes (only with E, via ISerializer)		    
		    // - <RESULT-TEXT>     xBytes (only with B, via DataInputStream)

			if (serializer == null)
			{
				//Without serializer the only possibility to send back an error messages is serialization 
				//via DataOutputStream e.g when the session timed out
				
				out.write(IConnection.FLAG_BROKEN);
				out.write(IConnection.MODE_UNCOMPRESSED);
	    		
	    		DataOutputStream dosContent  = new DataOutputStream(out);
	    		
				try
				{
					Throwable thResult;
					
					//Only the first Exception will be sent to the client. Normally in that special case,
					//thats enough information
	        		for (int i = 0, anz = auResult.size(), iExceptionCount = 0; i < anz && iExceptionCount == 0; i++)
	        		{
	        			if (auResult.get(i).getType() == IConnection.TYPE_CALL_ERROR)
	        			{
	        				//When the stream is broken, serialize the exception via DataOutputStream!
	        				
	        				thResult = (Throwable)auResult.get(i).getObject(); 
	        				
		        			if (thResult != null)
		        			{
		        				StackTraceElement[] stack = thResult.getStackTrace();
		        				
		        				dosContent.writeUTF(thResult.getClass().getName());
		        				dosContent.writeUTF((String)CommonUtil.nvl(thResult.getMessage(), ""));
		        				
		        				if (stack != null)
		        				{
			        				dosContent.write(stack.length);
			        				
			        				for (int j = 0, anzStack = stack.length; j < anzStack; j++)
			        				{
			        					dosContent.writeUTF((String)CommonUtil.nvl(stack[j].getClassName(), ""));
			        					dosContent.writeUTF((String)CommonUtil.nvl(stack[j].getMethodName(), ""));
			        					dosContent.writeUTF((String)CommonUtil.nvl(stack[j].getFileName(), ""));
			        					dosContent.writeInt(stack[j].getLineNumber());
			        				}
		        				}
		        				else
		        				{
		        					dosContent.write(0);
		        				}
		        			}
		        			else
		        			{
		        				dosContent.writeUTF("");
		        			}
		        			
		        			iExceptionCount++;
	        			}
	        		}
				}
				finally
				{
	        		dosContent.flush();
				}
			}
			else
			{
				ArrayUtil<ResultObject> auOrderedResult = null;

				List<Entry<String, Object>> liProperties;
				
				//Send the properties, if available!
				//Here we have two solutions:
				//a) the session is created -> use session properties
				//b) the session creation fails -> use request properties

				if (session != null)
				{
					liProperties = session.getProperties().getChanges(IConnectionConstants.PROPERTY_CLASSES);
				}
				else if (chtProperties != null)
				{
					liProperties = chtProperties.getChanges(IConnectionConstants.PROPERTY_CLASSES);
				}
				else
				{
					liProperties = null;
				}
				
				if (liProperties != null && liProperties.size() > 0)
				{
			    	//use a transferable object-type!
			    	List<Object[]> liTransferProperties = new ArrayUtil<Object[]>(liProperties.size());

			    	for (Entry<String, Object> entry : liProperties)
			    	{
			    		liTransferProperties.add(new Object[] {entry.getKey(), entry.getValue()});
			    		
			    		//forward the property to the response!
						pResponse.setProperty(entry.getKey(), entry.getValue());
			    	}
					
					if (auOrderedResult == null)
					{
						auOrderedResult = new ArrayUtil<ResultObject>();
					}
					
					auOrderedResult.add(new ResultObject(IConnection.TYPE_PROPERTY_RESULT, liTransferProperties));
				}
				
				//Send callback results in any available response!
				if (session != null)
				{
					//handle the callback results
					
					if (auOrderedResult == null)
					{
						auOrderedResult = new ArrayUtil<ResultObject>();
					}

					ArrayUtil<ResultObject> auCallBack = getCallBackResultsIntern(session);
					
					if (auCallBack != null)
					{
						auOrderedResult.addAll(auCallBack);
					}
				}
				
				//Add all available results to the end of the response
				
				if (auResult != null)
				{
					if (auOrderedResult == null)
					{
						auOrderedResult = new ArrayUtil<ResultObject>();
					}
					
					auOrderedResult.addAll(auResult);
				}
				
				iCallCount = auOrderedResult != null ? auOrderedResult.size() : 0;
	
				ByteArrayOutputStream baosContent = new ByteArrayOutputStream(IConnection.COMPRESSION_BYTES);
				DataOutputStream dosContent  = new DataOutputStream(baosContent);
	    		
				try
				{
		    		serializer.write(dosContent, Integer.valueOf(iCallCount));
		    		
		    		if (auOrderedResult != null)
		    		{
		        		for (ResultObject roReturn : auOrderedResult)
		        		{
		        			serializer.write(dosContent, Byte.valueOf(roReturn.getType()));
		        			serializer.write(dosContent, roReturn.getObject());
		
		        			//add the callback-id to the result. That's important for the client
		        			//to refer the right callback worker
		        			if (roReturn.getType() == IConnection.TYPE_CALLBACK_ERROR 
		        				|| roReturn.getType() == IConnection.TYPE_CALLBACK_RESULT)
		        			{
		        				serializer.write(dosContent, roReturn.getCallBackId());
		        			}
		        		}
		    		}
				}
				finally
				{
					baosContent.close();
					dosContent.close();
				}
	
				out.write(IConnection.FLAG_ESTABLISHED);
	    		
				//Compression will be used when the client-support for compression is enabled
				boolean bCompressionSupported = session != null 
				                                //getProperties().get -> don't change the access/alive time because the session will not time out!
				                                && Boolean.parseBoolean((String)session.getProperties().get(IConnectionConstants.COMPRESSION));

	    		if (bCompressionSupported && baosContent.size() > IConnection.COMPRESSION_BYTES)
	    		{
	    			out.write(IConnection.MODE_COMPRESSED);
			    	
				    GZIPOutputStream zosContent = new GZIPOutputStream(out);
				    zosContent.write(baosContent.toByteArray());
				    zosContent.flush();
				    zosContent.finish();
	    		}
	    		else
	    		{
	    			out.write(IConnection.MODE_UNCOMPRESSED);
	    			out.write(baosContent.toByteArray());
	    		}
			}
    		
			out.flush();
			
			try
			{
				pResponse.close();
			}
			catch (Throwable th)
			{
				log.error(th);
			}
		}
		
		return session;
	}
	
	/**
	 * Executes a remote method call request.
	 *
	 * @param pRequest the request which executes the command
	 * @param pSerializer the serializer implementation from the request
	 * @param pSession the session
	 * @param pObjectName server object name/alias
	 * @param pMethod method name which should be called
	 * @param pParams parameters for the method call
	 * @param pCallBackId identifier for asynchronous calls
	 * @param pProperties the session properties
	 * @return result of method call or null if it's an asynchronous method call
	 * @throws Throwable if an error occurs during execution
	 */
	private Object process(IRequest pRequest,
						   ISerializer pSerializer,
						   AbstractSession pSession, 
						   String pObjectName, 
						   String pMethod, 
						   Object[] pParams, 
						   Object pCallBackId,
						   ChangedHashtable<String, Object> pProperties) throws Throwable
	{
		if (pCallBackId != null)
		{
			validateCallBack(pSession, pObjectName);
		}
		
		//Special handling of the session managers
		if (pObjectName != null && IConnection.OBJ_SESSION.equals(pObjectName))
		{
			if (IConnection.MET_SESSION_CREATE.equals(pMethod))
			{
				if (pSession != null)
				{
					//Sessions can't be created more than once!
					throw new SecurityException("Session is already open!");
				}
				else
				{
					return createSessionIntern
					(
						pRequest, 
						pSerializer, 
						pProperties
					);
				}
			}		
			else if (IConnection.MET_SESSION_SUBSESSION_CREATE.equals(pMethod))
			{
				if (pSession != null)
				{
					//Sessions can't be created more than once!
					throw new SecurityException("Session is already open!");
				}
				else
				{
					return createSubSessionIntern(pRequest, sessman.get(pParams[0]), pProperties);
				}
			}
			else if (IConnection.MET_SESSION_SET_PROPERTY.equals(pMethod) && pSession == null)
			{
				//If the session is not available yet, cache the properties because the next operation
				//should be the create statement
				return (List<Object[]>)pParams[0];
			}
			else if (pSession != null)
			{
				if (IConnection.MET_SESSION_SETCHECKALIVE.equals(pMethod))
				{
					return setAndCheckAliveIntern(pSession, pParams);
				}
				else if (IConnection.MET_SESSION_DESTROY.equals(pMethod))
				{
					destroySessionIntern(pSession.getId());
					
					return null;
				}
				else if (IConnection.MET_SESSION_GET_PROPERTY.equals(pMethod))
				{
					return getPropertyIntern(pSession, (String)pParams[0]);
				}
				else if (IConnection.MET_SESSION_GET_PROPERTIES.equals(pMethod))
				{
					return getPropertiesIntern(pSession);
				}
			}
		}
	
		//Forward the rest direct to the session
		return executeIntern(pSession, new Call(pCallBackId, pObjectName, pMethod, pParams));
	}
	
	/**
	 * Executes a remote method call request through the session.
	 *
	 * @param pSession the session
	 * @param pCall the call information
	 * @return result of method call or null if it's an asynchronous method call
	 * @throws Throwable if an error occurs during execution
	 */	
	private Object executeIntern(AbstractSession pSession, Call pCall) throws Throwable
	{
		//no Session -> no call
		if (pSession != null)
		{
			return pSession.execute(pCall);
		}
		
		throw new SecurityException("No session for call '" + pCall.formatMethod() + "'");		
	}
		
	/**
	 * Validate if call backs are allowed with given session or objectname.
	 * 
	 * @param pSession the session
	 * @param pObjectName the object name
	 * @throws SecurityException if call back is not allowed
	 */
	private void validateCallBack(AbstractSession pSession, String pObjectName)
	{
		//Callbacks are bound to a session! Not possible without!
		if (pSession == null)
		{
			throw new SecurityException("Call back is not allowed!");
		}
		
		//Special handling of the session managers
		if (pObjectName != null && IConnection.OBJ_SESSION.equals(pObjectName))
		{
			throw new SecurityException("Call back is not allowed!");
		}
	}
	
	/**
	 * Creates a new session through the session manager.
	 *
	 * @param pRequest the request which creates the session
	 * @param pSerializer the serializer implementation for the new session
	 * @param pProperties the initial session properties
	 * @return session identifier of newly created <code>Session</code>
	 * @throws Throwable if the session can not be created
	 */
	private Object createSessionIntern(IRequest pRequest, 
								 	   ISerializer pSerializer, 
									   ChangedHashtable<String, Object> pProperties) throws Throwable
	{
		return sessman.createSession(pRequest, pSerializer, pProperties);
	}
	
	/**
	 * Creates a sub session through the session manager.
	 * 
	 * @param pRequest the request which creates the sub session
	 * @param pSession a valid session
	 * @param pProperties the initial session properties
	 * @return session identifier of newly created <code>SubSession</code>
	 * @throws Throwable if the session can not be created
	 */
	private Object createSubSessionIntern(IRequest pRequest, 
										  AbstractSession pSession, 
										  ChangedHashtable<String, Object> pProperties) throws Throwable
	{
		return sessman.createSubSession(pRequest, pSession, pProperties);
	}
	
	/**
	 * Destroyes a session through the session manager.
	 * 
	 * @param pSessionId session identifier
	 */	
	private void destroySessionIntern(Object pSessionId)
	{
		sessman.destroy(pSessionId);
	}

	/**
	 * Sets a session property.
	 * 
	 * @param pSession the session
	 * @param pName the property name
	 * @param pValue the value of the property or <code>null</code> to delete the property
	 */
	private void setPropertyIntern(AbstractSession pSession, String pName, Object pValue)
	{
		pSession.setProperty(pName, pValue);
	}
	
	/**
	 * Gets the value of a session property.
	 * 
	 * @param pSession the session
	 * @param pName the property name
	 * @return the value of the property or <code>null</code> if the property is not available
	 */
	private Object getPropertyIntern(AbstractSession pSession, String pName)
	{
		Object o = pSession.getProperty(pName);

		if (o != null)
		{
			for (int i = 0; i < IConnectionConstants.PROPERTY_CLASSES.length; i++)
			{
				if (IConnectionConstants.PROPERTY_CLASSES[i].isAssignableFrom(o.getClass()))
				{
					return o;
				}
			}
		}
		
		return null;
	}
	
	/**
	 * Gets a clone of accessible session properties.
	 * 
	 * @param pSession the session
	 * @return a {@link ChangedHashtable} with the property names and values
	 */
	private ChangedHashtable<String, Object> getPropertiesIntern(AbstractSession pSession)
	{
		List<Entry<String, Object>> liValues = pSession.getProperties().getMapping(IConnectionConstants.PROPERTY_CLASSES);
		
		ChangedHashtable<String, Object> chtProps = new ChangedHashtable<String, Object>();
		
		Object oValue;
		
		for (Entry<String, Object> entry : liValues)
		{
			oValue = entry.getValue();
			
			if (oValue != null)
			{
				chtProps.put(entry.getKey(), oValue, false);
			}
		}
		
		return chtProps;
	}
	
	/**
	 * Returns all available objects of asynchronous executions for a session.
	 * 
	 * @param pSession the session
	 * @return result objects or null if there are no result objects for the session
	 */
	private ArrayUtil<ResultObject> getCallBackResultsIntern(AbstractSession pSession)
	{
		if (pSession != null)
		{
			return pSession.removeCallBackResults();
		}
		
		return null;
	}

	/**
	 * Sets the alive state for a session and validates the alive
	 * state of sub sessions.
	 * 
	 * @param pSession the master sessin
	 * @param pSubSessionId a list of sub sessions
	 * @return the list of invalid sub sessions
	 */
	private Object[] setAndCheckAliveIntern(AbstractSession pSession, Object[] pSubSessionId)
	{
		//it's sufficient to set the alive of the main session, because the sub sessions
		//returns the alive from the parent session, and have not an explicit alive time
		pSession.setLastAliveTime(System.currentTimeMillis());
		
		if (pSubSessionId == null || pSubSessionId.length == 0)
		{
			return null;
		}
		else
		{
			ArrayUtil<Object> auInvalidSessions = new ArrayUtil<Object>();

			for (int i = 0, anz = pSubSessionId.length; i < anz; i++)
			{
				try
				{
					//don't perform the check if the session is a SubSession and is sub
					//of the pSession (not imporant because it's only a check and no special
					//access)
					sessman.get(pSubSessionId[i]); 
				}
				catch (RuntimeException re)
				{
					auInvalidSessions.add(pSubSessionId[i]);
				}
			}
			
			if (auInvalidSessions.size() > 0)
			{
				Object[] oInvalidSessions = new Object[auInvalidSessions.size()];
				auInvalidSessions.toArray(oInvalidSessions);
				
				return oInvalidSessions;
			}
			else
			{
				return null;
			}
		}
	}
	
	/**
	 * Sets a new password for the user of a session.
	 * 
	 * @param pSession the session
	 * @param pOldPassword the old password
	 * @param pNewPassword the new password
	 * @throws Throwable if an error occurs during execution
	 */
	private void setNewPasswordInternal(AbstractSession pSession, String pOldPassword, String pNewPassword) throws Throwable
	{
		if (pSession instanceof MasterSession)
		{
			((MasterSession)pSession).setNewPassword(pOldPassword, pNewPassword);
		}
		else
		{
			throw new NoSuchMethodException("Unknown method 'setNewPassword'");
		}
	}

	/**
	 * Returns the monitoring object for this server.
	 * 
	 * @return the monitoring object
	 */
	public final Monitoring getMonitoring()
	{
		return monitoring;
	}
	
	/**
	 * Logs a remote call.
	 * 
	 * @param pSessionId the session identifier
	 * @param pObjectName server object name/alias
	 * @param pMethod method name which should be called
	 * @param pParams parameters for the method call
	 * @param pCallBackId identifier for asynchronous calls
	 * @param pResult result of a method call or null if it's an asynchronous method call
	 * @param pError the exception if the method call throws an exception
	 * @param pDuration the duration of the call
	 */
	private final void log(Object pSessionId, 
			               String pObjectName, 
			               String pMethod, 
			               Object[] pParams, 
			               Object pCallBackId, 
			               Object pResult, 
			               Throwable pError,
			               long pDuration)
	{
		log.debug("SESSION-ID: ",
		          pSessionId,
		      	  "\nOBJECTNAME: ",
		      	  pObjectName,
		      	  "\nMETHOD:     ",
		      	  pMethod,
		      	  "\nPARAMS:     ",
		      	  pParams,
		      	  "\nCALLBACK:   ",
		      	  pCallBackId,
				  "\nRESULT:     ",
				  pResult,
				  "\nEXCEPTION:  ",
				  pError,
				  pError != null ? "DURATION:   " : "\nDURATION:   ",
				  Long.valueOf(pDuration),
				  " ms");
	}
	
}	// Server
