/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 12.10.2008 - [RH] - creation
 * 11.11.2008 - [RH] - delete details before master; insert master before details added
 * 28.04.2009 - [RH] - interface reviewed, javadoc review, NLS removed 
 * 30.04.2009 - [RH] - use of IStorage over an AbstractConnection implemented  
 * 04.05.2009 - [RH] - getClientColumnMetaData() added for CSV Export
 * 07.05.2009 - [RH] - AbstractConnection moved to RemoteDataSource
 * 05.10.2009 - [JR] - memSort, memFilter: methods made public
 * 23.11.2009 - [RH] - ColumnMetaData is replaced with the MetaData class
 * 04.12.2009 - [RH] - setWritebackEnabled(false) if no PrimaryKey columns found on server
 * 28.12.2009 - [JR] - open: catch exception when an exception occurs with auto link columns (no UI mode)
 * 22.02.2010 - [JR] - #67: open: concat name and subStorages to access the sub storage
 * 06.03.2010 - [JR] - getClientMetaData -> getMetaData, getClientColumnMetaData -> getColumnMetaData
 * 25.03.2010 - [JR] - #103: initServerMetaData: retrieve cached metadata and set them into the datasource
 *                   - #92: createNewRow: set default values from metadata  
 * 28.03.2010 - [JR] - #47: open: allowed value check and automatic choice cell editor support
 * 07.10.2010 - [JR] - #117: cache storage fall back to storage without cache
 * 23.02.2011 - [RH] - #199: if also a MasterReference is set over the same columns, then we set this column invisible default.
 *                           over setAutoLinkInvisibleForMasterReferences() can this features be disabled.
 * 06.03.2011 - [JR] - #115: 
 *                     * open: moved allowed values to MemDataBook
 *                     * createNewRow: moved default value to MemDataBook 
 * 09.06.2011 - [RH] - #387 - RemoteDataBook open doesn't check if the name is null
 * 19.08.2011 - [JR] - #459: setMetaDataCacheEnabled
 */
package com.sibvisions.rad.model.remote;

import java.util.Hashtable;
import java.util.Map.Entry;

import javax.rad.genui.UIFactoryManager;
import javax.rad.genui.celleditor.UILinkedCellEditor;
import javax.rad.model.ColumnDefinition;
import javax.rad.model.IDataRow;
import javax.rad.model.IDataSource;
import javax.rad.model.ModelException;
import javax.rad.model.RowDefinition;
import javax.rad.model.datatype.BigDecimalDataType;
import javax.rad.model.datatype.StringDataType;
import javax.rad.model.reference.ReferenceDefinition;
import javax.rad.persist.ColumnMetaData;
import javax.rad.persist.DataSourceException;
import javax.rad.persist.MetaData;
import javax.rad.persist.MetaData.Feature;
import javax.rad.remote.AbstractConnection;

import com.sibvisions.rad.model.mem.MemDataBook;
import com.sibvisions.rad.model.mem.MemDataPage;
import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.LoggerFactory;

/**
 * The <code>RemoteDataBook</code> is a storage independent table, and handles all operations
 * based on the the MemDatabook base class.
 * It communicates to the IStorage and uses the storage methods to fetch, insert
 * update, delete the data on server.
 * 
 * @see com.sibvisions.rad.model.mem.MemDataBook
 * @see javax.rad.model.IDataPage
 * @see javax.rad.model.IDataBook
 * @see javax.rad.model.IRowDefinition
 * @see javax.rad.model.IChangeableDataRow
 * 
 * @author Roland Hrmann
 */
public class RemoteDataBook extends MemDataBook
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The logger for protocol the performance. */
	private static ILogger logger = LoggerFactory.getInstance(RemoteDataBook.class);

	/** the cache for fast access to caching storages. */
	private static Hashtable<String, Boolean> htCachingStorage = new Hashtable<String, Boolean>();
	
	/** It holds the column names of the server column meta data in the client RowDefinition. */
	private String[] saMetaDataColumnNames;
	
	/** It holds the column names of the server column meta data in the client RowDefinition. */
	private String[] saBlockFetchColumnNames;
	
	/** It holds the column names of the server column meta data in the client RowDefinition. */
	private String[] saBlockFetchReferencedColumnNames;
	
	/** stores the block fetches. */
	private Hashtable<Object, Integer> htFetchedRowsForBlock = null;
	
	/** The meta data for this RemoteDataBook. */
	private MetaData mdMetaData;
	
	/** The correct column indexes of the fetched data, after opening the data book. */
	private int[] iaFetchColumnIndexes;

	/** The correct column indexes of the master reference. */
	private int[] iaMasterColumnIndexes;

	/** Indicates whether the server object is a caching storage. */
	private boolean bCachingStorage = true;

	/** whether the meta data cache is enabled (default: <code>true</code>). */
	private boolean bMetaDataCacheEnabled = true;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of <code>RemoteDataBook</code>.
	 */
	public RemoteDataBook()
	{
		super();
		
		setMemSort(false);
		setMemFilter(false);
		setDeleteCascade(false);
		setWritebackEnabled(true);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected boolean isDataPageRefetchPossible()
	{
		return saBlockFetchColumnNames == null;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected MemDataPage createDataPage(IDataRow pMasterDataRow)
	{
		return new RemoteDataPage(this, pMasterDataRow);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDataSource(IDataSource pDataSource) throws ModelException
	{
		if (!(pDataSource instanceof RemoteDataSource))
		{
			throw new ModelException("It's not an RemoteDataSource! -> it have to be one!");
		}
		
		super.setDataSource(pDataSource);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public RemoteDataSource getDataSource()
	{
		return (RemoteDataSource)super.getDataSource();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void open() throws ModelException
	{
		if (!isOpen())
		{
			if (getConnection() == null)
			{
				throw new ModelException("The remote connection to the server IStorage is null!");
			}			

			// #387 - RemoteDataBook open doesn't check if the name is null 
			if (getName() == null)
			{
				throw new ModelException("DataBook Name is null!");				
			}

			try 
			{
				initServerMetaData();
			}
			catch (DataSourceException dataSourceException)
			{
				if (rdRowDefinition == null  || rdRowDefinition.getColumnCount() == 0)
				{
					throw new ModelException("RowDefintion can't init with meta data from storage!",
			 				 				 dataSourceException);
				}
			}

			if (mdMetaData != null)
			{
				//Features

				if (!mdMetaData.isSupported(Feature.Filter))
				{
					setMemFilter(true);
				}
				
				if (!mdMetaData.isSupported(Feature.Sort))
				{
					setMemSort(true);
				}

				if (!mdMetaData.isSupported(Feature.WriteBack))
				{
					setWritebackEnabled(false);
				}
				
				//Columns
				
				ColumnMetaData[] cmd = mdMetaData.getColumnMetaData();
				saMetaDataColumnNames = new String[cmd.length];
				
				if (rdRowDefinition == null)
				{
					rdRowDefinition = new RowDefinition();
				}

				String sName = getName();
				
				ColumnDefinition coldef;
				
				for (int i = 0; i < cmd.length; i++)
				{
					if (rdRowDefinition.getColumnDefinitionIndex(cmd[i].getName()) < 0)
					{
						coldef = ColumnMetaData.createColumnDefinition(cmd[i]);
						
						// All server columns are writeable from the view of a client. Thats because this columns are written to the server.
						// If the server writes this changes to the storage depend on the writable state of the column from the server and
						// not from the client.
						coldef.setWritable(true);
						
						rdRowDefinition.addColumnDefinition(coldef);
						
						if (UIFactoryManager.getFactory() != null)
						{
							if (cmd[i].getLinkReference() != null)
							{
								// Create Reference RemoteDatabook, ReferenceDefinition, LinkCellEditor
								RemoteDataBook rdbReferencedDataBook = new RemoteDataBook();
								rdbReferencedDataBook.setDataSource(getDataSource());
								
								// if the reference Storage is relative, the name of this object has to be added.
								String referencedStorage = cmd[i].getLinkReference().getReferencedStorage();
								if (referencedStorage.startsWith("."))
								{
									referencedStorage = sName + referencedStorage;
								}
								rdbReferencedDataBook.setName(referencedStorage);
								rdbReferencedDataBook.open();
								
								ReferenceDefinition referenceDefinition = new ReferenceDefinition();
								referenceDefinition.setReferencedDataBook(rdbReferencedDataBook);
								referenceDefinition.setReferencedColumnNames(cmd[i].getLinkReference().getReferencedColumnNames());
								referenceDefinition.setColumnNames(cmd[i].getLinkReference().getColumnNames());
								
								UILinkedCellEditor linkedCellEditor = new UILinkedCellEditor();
								linkedCellEditor.setLinkReference(referenceDefinition);
								linkedCellEditor.setValidationEnabled(
										!(coldef.getDataType() instanceof StringDataType
										&& cmd[i].isWritable()));
								
								if (coldef.getDataType() instanceof BigDecimalDataType)
								{
									linkedCellEditor.setHorizontalAlignment(UILinkedCellEditor.ALIGN_RIGHT);
								}
								
								coldef.getDataType().setCellEditor(linkedCellEditor);		
							}
						}
					}
					saMetaDataColumnNames[i] = cmd[i].getName();
				}
				iaFetchColumnIndexes = new int[cmd.length];
				
				for (int i = 0; i < cmd.length; i++)
				{
					iaFetchColumnIndexes[i] = rdRowDefinition.getColumnDefinitionIndex(cmd[i].getName());
				}
				
				if (rdRowDefinition.getPrimaryKeyColumnNames() == null)
				{
					String[] saPKs = mdMetaData.getPrimaryKeyColumnNames();
					if (saPKs == null)
					{
						setWritebackEnabled(false);
					}
					else
					{
						rdRowDefinition.setPrimaryKeyColumnNames(saPKs);						
					}
				}
			}
			if (saBlockFetchColumnNames == null)
			{
				iaMasterColumnIndexes = null;
				saBlockFetchReferencedColumnNames = null;
			}
			else if (getMasterReference() == null)
			{
				throw new ModelException("Block fetch is only allowed if a master reference is set!");
			}
			else if (ArrayUtil.containsAll(getMasterReference().getColumnNames(), saBlockFetchColumnNames))
			{
				String[] masterColumnNames = getMasterReference().getColumnNames();
				String[] masterReferencedColumnNames = getMasterReference().getReferencedColumnNames();
				iaMasterColumnIndexes = new int[masterColumnNames.length];
				
				for (int i = 0; i < iaMasterColumnIndexes.length; i++)
				{
					iaMasterColumnIndexes[i] = rdRowDefinition.getColumnDefinitionIndex(masterColumnNames[i]);
				}

				saBlockFetchReferencedColumnNames = new String[saBlockFetchColumnNames.length];
				for (int i = 0; i < saBlockFetchColumnNames.length; i++)
				{
					saBlockFetchReferencedColumnNames[i] = masterReferencedColumnNames[ArrayUtil.indexOf(masterColumnNames, saBlockFetchColumnNames[i])];
				}
			}
			else
			{
				throw new ModelException("Block fetch have to be a subset of the master reference columns!");
			}
			super.open();
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void close()
	{
		if (isOpen())
		{
			super.close();
		
			htFetchedRowsForBlock = null;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void executeLockAndRefetch() throws ModelException
	{
		if (isWritebackEnabled())
		{
			try 
			{			
				IDataRow drOld = getOriginalRow(); 
				if (drOld == null)
				{
					drOld = getDataRow(getSelectedRow());
				}
				
				long lMillis = System.currentTimeMillis();
				
				Object[] oResult = (Object[])getConnection().call(getName(), "refetchRow", drOld.getValues(saMetaDataColumnNames));
					
				logger.debug("acConnection.refetchRow(", getName(), ",", drOld.getValues(saMetaDataColumnNames), ") in ",
							 Long.valueOf(System.currentTimeMillis() - lMillis), "ms");
					
				if (oResult != null)
	 			{
	 				setValuesInternal(saMetaDataColumnNames, oResult);
	 			}
			}
			catch (Throwable throwable)
			{
				throw new ModelException("Execute Lock and Refetch failed!", throwable);
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void executeInsert() throws ModelException
	{
		if (isWritebackEnabled())
		{
			int pRowIndex = getSelectedRow();
			
			try 
			{
				
				long lMillis = System.currentTimeMillis();
				
				Object[] oResult = (Object[])getConnection().call(getName(), "insert", getDataRow(pRowIndex).getValues(saMetaDataColumnNames));
					
				logger.debug("acConnection.insert(", getName(), ",", getDataRow(pRowIndex), ") in ",
							 Long.valueOf(System.currentTimeMillis() - lMillis), "ms");

	 			if (oResult != null)
	 			{
	 				setValuesInternal(saMetaDataColumnNames, oResult);
	 			}
			}
			catch (Throwable throwable)
			{
				setSelectedRow(pRowIndex);
				
				throw new ModelException("Execute Insert failed!", throwable);
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void executeUpdate() throws ModelException
	{
		if (isWritebackEnabled())
		{		
			int pRowIndex = getSelectedRow();
			
			try 
			{
				long lMillis = System.currentTimeMillis();
				
				Object[] oResult = (Object[])getConnection().call(getName(), 
						                                          "update", 
						                                          getOriginalRow().getValues(saMetaDataColumnNames), 
					                                              getDataRow(pRowIndex).getValues(saMetaDataColumnNames));
					
				logger.debug("acConnection.update(", getName(), ",", getOriginalRow(), ",",
						     getDataRow(pRowIndex), ") in ",
						     Long.valueOf(System.currentTimeMillis() - lMillis), "ms");

	 			if (oResult != null)
	 			{
	 				setValuesInternal(saMetaDataColumnNames, oResult);
	 			}
			}
			catch (Throwable throwable)
			{
				setSelectedRow(pRowIndex);
				
				throw new ModelException("Execute Update failed!", throwable);
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void executeDelete() throws ModelException
	{
		if (isWritebackEnabled())
		{		
			int pRowIndex = getSelectedRow();			
	        			
			try
			{
				long lMillis = System.currentTimeMillis();
				
				getConnection().call(getName(), "delete", getOriginalRow().getValues(saMetaDataColumnNames));
					
				logger.debug("acConnection.delete(", getName(), ",", getOriginalRow().getValues(saMetaDataColumnNames), ") in ",
							 Long.valueOf(System.currentTimeMillis() - lMillis), "ms");
			}
			catch (Throwable throwable)
			{
				setSelectedRow(pRowIndex);

				throw new ModelException("Execute Delete failed!", throwable);
			}				
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void executeRefresh()  throws ModelException
	{
		super.clear();
		
		htFetchedRowsForBlock = null;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMemSort(boolean pMemSort)
	{
		if (isOpen() && !mdMetaData.isSupported(Feature.Sort))
		{
			return;
		}
		
		super.setMemSort(pMemSort);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isMemSort()
	{
		//changed method visibility to public
		
		return super.isMemSort();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMemFilter(boolean pMemFilter)
	{
		if (isOpen() && !mdMetaData.isSupported(Feature.Filter))
		{
			return;
		}
		
		super.setMemFilter(pMemFilter);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isMemFilter()
	{
		//changed method visibility to public
		
		return super.isMemFilter();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setWritebackEnabled(boolean pWritebackEnabled)
	{
		if (isOpen() && !mdMetaData.isSupported(Feature.WriteBack))
		{
			return;
		}

		super.setWritebackEnabled(pWritebackEnabled);
	}	

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isWritebackEnabled()
	{
		//changed method visibility to public

		return super.isWritebackEnabled();
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	
	
	/**
	 * Returns the block fetch columns.
	 * The block fetch columns have to be a subset of the master link columns.
	 * The rows are fetched with these columns, and sorted into the corresponding pages.
	 * 
	 * @return the block fetch columns.
	 */
	public String[] getBlockFetchColumnNames()
	{
		return saBlockFetchColumnNames;
	}
	
	/**
	 * Sets the block fetch columns.
	 * The block fetch columns have to be a subset of the master link columns.
	 * The rows are fetched with these columns, and sorted into the corresponding pages.
	 * 
	 * @param pBlockFetchColumnNames the block fetch columns.
	 * @throws ModelException if the data book is already open.
	 */
	public void setBlockFetchColumnNames(String[] pBlockFetchColumnNames) throws ModelException
	{
		if (isOpen())
		{
			throw new ModelException("It's not allowed on open DataBooks!");			
		}
		
		saBlockFetchColumnNames = pBlockFetchColumnNames;
	}	
	
	/**
	 * Sets the metadata cache enabled. If it is disabled, every open performs a remote call.
	 * 
	 * @param pCacheEnabled <code>true</code> to enable the cache
	 */
	public void setMetaDataCacheEnabled(boolean pCacheEnabled)
	{
		bMetaDataCacheEnabled = pCacheEnabled;
	}
	
	/**
	 * Gets whether metadata cache is enabled.
	 * 
	 * @return <code>true</code> if the cache is enabled, <code>false</code> otherwise
	 */
	public boolean isMetaDataCacheEnabled()
	{
		return bMetaDataCacheEnabled;
	}
	
	/**
	 * It gets the meta data from the Server for this RemoteDataBook.
	 * 
	 * @throws DataSourceException	if an error during communication or Storage interaction occur.
	 */
	private void initServerMetaData() throws DataSourceException
	{
		try
		{
			RemoteDataSource dataSource = getDataSource();
			
			String sDataSourceName = dataSource.getConnection().getLifeCycleName();
			String sName = getName(); 
			
			String sCacheKey = sDataSourceName + "." + sName;
			
			if (bMetaDataCacheEnabled)
			{
				//try to access cached meta-data
				mdMetaData = dataSource.getMetaData(sCacheKey);
			}
			else
			{
				mdMetaData = null;
			}

			if (mdMetaData == null)
			{
				Boolean bCache = htCachingStorage.get(sCacheKey);
				
				if (bCache != null)
				{
					bCachingStorage = bCache.booleanValue();
				}
				else
				{
					//try to access a cached storage
					bCachingStorage = true;
				}
				
				long lMillis = System.currentTimeMillis();
				
				if (bCachingStorage)
				{
					try
					{
						//make a request to the server
						Object[] objResult = getConnection().call(new String[] {sName, sName},
						                                          new String[] {"getMetaData", "getMetaDataFromCache"}, 
						                                          new Object[][] { {sDataSourceName, sName}, 
								                                                   {sDataSourceName}});
		
						mdMetaData = (MetaData)objResult[0];

						if (bMetaDataCacheEnabled)
						{
							//save available MetaData
							Hashtable<String, MetaData> htMetaDataCache = (Hashtable<String, MetaData>)objResult[1];
							
							if (htMetaDataCache != null)
							{
								for (Entry<String, MetaData> entry : htMetaDataCache.entrySet())
								{
									dataSource.putMetaData(entry.getKey(), entry.getValue());
								}
							}
						}
						
						logger.debug("getMetaDataFromCache(", sDataSourceName, ", ", sName, ") in ", Long.valueOf(System.currentTimeMillis() - lMillis), "ms");
					}
					catch (NoSuchMethodException nsme)
					{
						bCachingStorage = false;
					}
				}
				
				if (!bCachingStorage)
				{
					mdMetaData = (MetaData)getConnection().call(sName, "getMetaData");
					
					if (bMetaDataCacheEnabled)
					{
						dataSource.putMetaData(sCacheKey, mdMetaData);
					}
					
					logger.debug("getMetaData(", sName, ") in ", Long.valueOf(System.currentTimeMillis() - lMillis), "ms");
				}
				
				//cache information for later use
				htCachingStorage.put(sCacheKey, Boolean.valueOf(bCachingStorage));
			}
			else
			{				
				logger.debug("use cached metadata: ", sName);
			}
		}
		catch (Throwable throwable)
		{
			throw new DataSourceException("Init ServerMetaData failed!", throwable);
		}
	}

	/**
	 * Returns the server meta data column names.
	 *
	 * @return the server meta data column names.
	 */
	protected String[] getMetaDataColumnNames()
	{
		return saMetaDataColumnNames;
	}
	
	/**
	 * Returns the correct column indexes of the fetched data, after opening the data book.
	 *
	 * @return the correct column indexes of the fetched data, after opening the data book.
	 */
	protected int[] getFetchColumnIndexes()
	{
		return iaFetchColumnIndexes;
	}
	
	/**
	 * Returns the correct master column indexes of the fetched data, after opening the data book.
	 *
	 * @return the correct master column indexes of the fetched data, after opening the data book.
	 */
	protected int[] getMasterColumnIndexesForBlock()
	{
		return iaMasterColumnIndexes;
	}
	
	/**
	 * Returns the amount of fetched rows per block.
	 *
	 * @param pMasterDataRow the master data row.
	 * @return the amount of fetched rows per block.
	 * @throws ModelException if it fails.
	 */
	protected int getFetchedRowsForBlock(IDataRow pMasterDataRow) throws ModelException
	{
		if (htFetchedRowsForBlock == null)
		{
			return 0;
		}
		else
		{
			Integer rows = htFetchedRowsForBlock.get(pMasterDataRow.createDataRow(saBlockFetchReferencedColumnNames));
			if (rows == null)
			{
				return 0;
			}
			else
			{
				return rows.intValue();
			}
		}
	}
	
	/**
	 * Sets the amount of fetched rows per block.
	 *
	 * @param pMasterDataRow the master data row.
	 * @param pFetchedRows the amount of fetched rows per block.
	 * @throws ModelException if it fails.
	 */
	protected void setFetchedRowsForBlock(IDataRow pMasterDataRow, int pFetchedRows) throws ModelException
	{
		if (htFetchedRowsForBlock == null)
		{
			htFetchedRowsForBlock = new Hashtable<Object, Integer>();
		}
		htFetchedRowsForBlock.put(pMasterDataRow.createDataRow(saBlockFetchReferencedColumnNames), Integer.valueOf(pFetchedRows));
	}
	
	/**
	 * Returns the AbstractConnection of the RemoteDataSource for this StorageDataSource.
	 * 
	 * @return the AbstractConnection of the RemoteDataSource for this StorageDataSource.
	 */
	protected AbstractConnection getConnection()
	{
		RemoteDataSource rds = getDataSource();
		
		if (rds == null)
		{
			return null;
		}
		
		return rds.getConnection();
	}
	
	/**
	 * Returns whether the server storage is a cached storage or a storage without cache.
	 * 
	 * @return <code>true</code> if the server storage uses a cache, <code>false</code> otherwise
	 */
	protected boolean isCachingStorage()
	{
		return bCachingStorage;
	}

}	// RemoteDataBook
