/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 08.04.2009 - [JR] - creation
 * 03.09.2009 - [JR] - getName implemented
 * 04.08.2009 - [JR] - getContent with InputStreamReader implemented
 * 27.10.2010 - [JR] - delete, zip implemented
 * 10.11.2010 - [JR] - getContent(String), getContent(File) implemented
 * 02.12.2010 - [JR] - deleteEmpty: boolean parameter added
 * 06.12.2010 - [JR] - copyFile: support directory target
 * 10.12.2010 - [JR] - move implemented
 * 18.02.2011 - [JR] - unzip implemented
 * 10.03.2011 - [JR] - replace implemented
 * 06.04.2011 - [JR] - listZipEntries implemented
 * 12.04.2011 - [JR] - getDirectory implemented
 * 02.06.2011 - [JR] - used BufferedOutputStream where FileInputStream is used
 * 04.06.2011 - [JR] - save creates directories
 * 19.06.2011 - [JR] - save with InputStream as parameter
 * 17.11.2011 - [JR] - moveDirectory: 
 *                     * delete directory
 *                     * keep works now recursive
 */
package com.sibvisions.util.type;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.nio.channels.FileChannel;
import java.util.Calendar;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.zip.CRC32;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.FileSearch;

/**
 * The <code>FileUtil</code> contains file and filename dependent 
 * utility methods.
 * 
 * @author Ren Jahn
 */
public final class FileUtil
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class Members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** File or directory copy mode. */
	public enum CopyMode
	{
		/** Backup existing files. */
		Backup,
		/** Overwrite existing files. */
		Overwrite,
		/** Keep existing files and copy only new files. */
		Keep
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Invisible constructor because <code>FileUtil</code> is a utility
	 * class.
	 */
	private FileUtil()
	{
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Copies the content from a stream into a file.
	 * 
	 * @param pIn the input stream
	 * @param pTarget the output file
	 * @return the number of written bytes
	 * @throws IOException if it's not possible to read from the <code>InputStream</code>
	 *                     or write to the <code>File</code>
	 */
	public static long copy(InputStream pIn, File pTarget) throws IOException
	{
		BufferedOutputStream bos = null;
		
		try
		{
			bos = new BufferedOutputStream(new FileOutputStream(pTarget));
			
			long length = 0;
			int iLen;
			
			byte[] byContent = new byte[4096];
			
			while ((iLen = pIn.read(byContent)) >= 0)
			{
				bos.write(byContent, 0, iLen);
				length += iLen;
			}

			bos.flush();
			
			return length;
		}
		finally
		{
			if (bos != null)
			{
				try
				{
					//closes the filestream also
					bos.close();
				}
				catch (Exception e)
				{
					//nothing to be done
				}
			}
		}
	}
	
	/**
	 * Copies the content of an <code>InputStream</code> to the desired <code>OutputStream</code>.
	 * 
	 * @param pIn input stream with content
	 * @param pOut output stream
	 * @return the length of the InputStream.
	 * @throws IOException if it's not possible to read from the <code>InputStream</code>
	 *                     or write to the <code>OutputStream</code> 
	 */
	public static long copy(InputStream pIn, OutputStream pOut) throws IOException
	{
		return copy(pIn, false, pOut, false);
	}

	/**
	 * Copies the content of an <code>InputStream</code> to the desired <code>OutputStream</code>.
	 * 
	 * @param pIn input stream with content
	 * @param pCloseIn true, if the input stream
	 * @param pOut output stream
	 * @param pCloseOut true, if the output stream
	 * @return the length of the InputStream.
	 * @throws IOException if it's not possible to read from the <code>InputStream</code>
	 *                     or write to the <code>OutputStream</code> 
	 */
	public static long copy(InputStream pIn, boolean pCloseIn, OutputStream pOut, boolean pCloseOut) throws IOException
	{
		long length = 0;
		int iLen;
		
		byte[] byContent = new byte[4096];
		
		while ((iLen = pIn.read(byContent)) >= 0)
		{
			pOut.write(byContent, 0, iLen);
			length += iLen;
		}

		pOut.flush();

		if (pCloseIn)
		{
			pIn.close();
		}
		if (pCloseOut)
		{
			pOut.close();
		}
		return length;
	}

	/**
	 * Copy a file or directory to another file or directory.
	 * 
	 * @param pSource the source file
	 * @param pTarget the target file
	 * @param pPatterns the include/exclude patterns. This parameter is only used for directory copy.
	 * @throws IOException if the copy process failed
	 */
	public static void copy(File pSource, File pTarget, String... pPatterns) throws IOException
	{
		copy(pSource, pTarget, CopyMode.Overwrite, pPatterns);
	}
	
	/**
	 * Copies one file into another file. If the source file is a directory, then all files in
	 * the directory will be copied to the target. 
	 * 
	 * @param pSource source file
	 * @param pTarget target file
	 * @param pMode the copy mode
	 * @param pPatterns include/exclude file pattern(s)
	 * @throws IOException if it's not possible to copy the file or directory
	 */
	public static void copy(File pSource, File pTarget, CopyMode pMode, String... pPatterns) throws IOException
	{
		if (pSource.isFile())
		{
			copyFile(pSource, pTarget, pMode);
		}
		else if (pSource.isDirectory())
		{
			copyDirectory(pSource, pTarget, pMode, pPatterns);
		}
		else
		{
			throw new IOException("File or directory is required: '" + pSource.getAbsolutePath() + "'!");
		}
	}
	
	/**
	 * Copies one file into another file using <code>java.nio</code> Channels.
	 * 
	 * @param pSource source file
	 * @param pTarget target file
	 * @param pMode the copy mode. If the target file should be backuped, then a timestamp 
	 *              will be added to the filename.
	 * @throws IOException if it's not possible to copy the file
	 */
	private static void copyFile(File pSource, File pTarget, CopyMode pMode) throws IOException
	{
		if (pTarget.exists())
		{
			if (pMode == CopyMode.Keep)
			{
				return;
			}
			else if (pMode == CopyMode.Backup)
			{
				String sDate = DateUtil.format(Calendar.getInstance().getTime(), "dd_MM_yyyy_HH_mm_ss");
				
				String sExt = getExtension(pTarget.getName());
				
				File fiNew = new File(pTarget.getParent(), removeExtension(pTarget.getName()) + "_" + sDate + (sExt != null ? "." + sExt : ""));
				
				if (!pTarget.renameTo(fiNew))
				{
					throw new IOException("Can't rename file '" + pTarget.getAbsolutePath() + "' to '" + fiNew.getAbsolutePath() + "'!");
				}
			}
		}
		
		FileInputStream fisSource = null;
		FileOutputStream fosTarget = null;
		
		FileChannel fcSource = null;	
		FileChannel fcTarget = null;
		
		try
		{
			if (!pTarget.exists())
			{
				File fiParent = pTarget.getParentFile();
				
				if (fiParent != null)
				{
					fiParent.mkdirs();
				}
			}
			else if (pTarget.isDirectory())
			{
				pTarget = new File(pTarget, pSource.getName());
			}
			
	    	fisSource = new FileInputStream(pSource);
	    	fosTarget = new FileOutputStream(pTarget);
			
			fcSource = fisSource.getChannel();	
	    	fcTarget = fosTarget.getChannel();
	    	
	    	long length = fcSource.size();
			
	    	fcSource.transferTo(0, length, fcTarget);
		}
		finally
		{
			if (fcSource != null)
			{
				try
				{
					fcSource.close();
				}
				catch (IOException ioe)
				{
					//egal
				}
			}
			
			if (fisSource != null)
			{
	    		try
	    		{
	    			fisSource.close();
	    		}
	    		catch (IOException ioe)
	    		{
	    			//egal
	    		}
			}
			
			if (fcTarget != null)
			{
				try
				{
					fcTarget.close();
				}
				catch (IOException ioe)
				{
					//egal
				}
			}
	
			if (fosTarget != null)
			{
	    		try
	    		{
	    			fosTarget.close();
	    		}
	    		catch (IOException ioe)
	    		{
	    			//egal
	    		}
			}
		}
	}
	
	/**
	 * Copie the files and sub directories from one directory to another directory.
	 * 
	 * @param pSource the source directory
	 * @param pTarget the target directory
	 * @param pMode the copy mode
	 * @param pPatterns the include/exclude patterns
	 * @throws IOException if the copy process failed
	 */
	private static void copyDirectory(File pSource, File pTarget, CopyMode pMode, String... pPatterns) throws IOException
	{
		FileSearch fsearch = new FileSearch();
		fsearch.search(pSource, true, pPatterns);
		
		File fiOldDir;
		File fiNewDir;

        int iLen = pSource.getAbsolutePath().length();
		
		for (Entry<String, List<String>> entry : fsearch.getFilesPerDirectory().entrySet())
		{
			fiOldDir = new File(entry.getKey());
			fiNewDir = new File(pTarget, entry.getKey().substring(iLen));

			if (!fiNewDir.exists() && !fiNewDir.mkdirs())
			{
				throw new IOException("Can't create directory '" + fiNewDir.getAbsolutePath() + "'!");
			}
			
			for (String file : entry.getValue())
			{
				copyFile(new File(fiOldDir, file), new File(fiNewDir, file), pMode);
			}
		}
		
		//create empty directories
		for (String sDir : fsearch.getFoundDirectories())
		{
			fiNewDir = new File(pTarget, sDir.substring(iLen));
			
			if (!fiNewDir.exists())
			{
				fiNewDir.mkdirs();
			}
		}
	}

	/**
	 * Moves the given source file to another location. If the target file exists, it will be overwritten.
	 * 
	 * @param pSource the source file
	 * @param pTarget the target location
	 * @throws IOException if an error occurs during moving
	 */
	public static void move(File pSource, File pTarget) throws IOException
	{
		move(pSource, pTarget, CopyMode.Overwrite);
	}
	
	/**
	 * Moves the given source file to another location. The given copy mode defines the operation if the 
	 * target file already exists.
	 * 
	 * @param pSource the source file
	 * @param pTarget the target location
	 * @param pMode the copy mode
	 * @throws IOException if an error occurs during moving
	 */
	public static void move(File pSource, File pTarget, CopyMode pMode) throws IOException
	{
		if (pSource.equals(pTarget))
		{
			return;
		}
		
		if (pSource.isFile())
		{
			moveFile(pSource, pTarget, pMode);
		}
		else if (pSource.isDirectory())
		{
			moveDirectory(pSource, pTarget, pMode);
		}
		else
		{
			throw new IOException("File or directory is required: '" + pSource.getAbsolutePath() + "'!");
		}
	}
	
	/**
	 * Moves the given source file to another location.
	 * 
	 * @param pSource the source file
	 * @param pTarget the target location
	 * @param pMode the copy mode
	 * @throws IOException if the move operation fails
	 */
	private static void moveFile(File pSource, File pTarget, CopyMode pMode) throws IOException
	{
		if (pTarget.isDirectory())
		{
			pTarget = new File(pTarget, pSource.getName());
		}

		if (pTarget.exists())
		{
			//It is not possible to move a file to a directory
			if (pTarget.isDirectory())
			{
				throw new IOException("Can't move '" + pSource.getAbsolutePath() + "' to directory '" + pTarget.getAbsolutePath() + "'!");
			}

			if (pMode == null || pMode == CopyMode.Overwrite)
			{
				if (!pTarget.delete())
				{
					throw new IOException("Can't overwrite '" + pTarget.getAbsolutePath() + "'!");
				}
			}
			else if (pMode == CopyMode.Keep)
			{
				return;
			}
			else
			{
				String sDate = DateUtil.format(Calendar.getInstance().getTime(), "dd_MM_yyyy_HH_mm_ss");
				
				String sExt = getExtension(pTarget.getName());
				
				File fiNew = new File(pTarget.getParent(), removeExtension(pTarget.getName()) + "_" + sDate + (sExt != null ? "." + sExt : ""));
				
				if (!pTarget.renameTo(fiNew))
				{
					throw new IOException("Can't rename file '" + pTarget.getAbsolutePath() + "' to '" + fiNew.getAbsolutePath() + "'!");
				}
			}
		}
		else
		{
			File fiParent = pTarget.getParentFile();
			
			if (fiParent != null)
			{
				fiParent.mkdirs();
			}
		}
		
		if (!pSource.renameTo(pTarget))
		{
			throw new IOException("Can't move '" + pSource.getAbsolutePath() + "' to '" + pTarget.getAbsolutePath() + "'!");
		}
	}
	
	/**
	 * Moves the given source director to another location.
	 * 
	 * @param pSource the source directory
	 * @param pTarget the target directory
	 * @param pMode the copy mode
	 * @throws IOException if the move operation fails
	 */
	private static void moveDirectory(File pSource, File pTarget, CopyMode pMode) throws IOException
	{
		if (pTarget.exists())
		{
			if (pTarget.isFile())
			{
				throw new IOException("Can't move '" + pSource.getAbsolutePath() + "' to file '" + pTarget.getAbsolutePath() + "'!");
			}
			
			if (pMode == null || pMode == CopyMode.Overwrite || pMode == CopyMode.Keep)
			{
				//don't delete existing directory and copy new files and overwrite existing files
				FileUtil.copyDirectory(pSource, pTarget, pMode);
				
				FileUtil.delete(pSource);
				
				return;
			}	
			else
			{
				String sDate = DateUtil.format(Calendar.getInstance().getTime(), "dd_MM_yyyy_HH_mm_ss");
				
				File fiNew = new File(pTarget.getAbsolutePath() + "_" + sDate);
				
				if (!pTarget.renameTo(fiNew))
				{
					throw new IOException("Can't rename file '" + pTarget.getAbsolutePath() + "' to '" + fiNew.getAbsolutePath() + "'!");
				}
			}
		}
		else
		{
			pTarget.mkdirs();
		}
		
		if (!pSource.renameTo(pTarget))
		{
			throw new IOException("Can't move '" + pSource.getAbsolutePath() + "' to '" + pTarget.getAbsolutePath() + "'!");
		}
	}

	/**
	 * Reads the content of an <code>InputStream</code> into a byte array and closes the stream after reading.
	 * <p>
	 * @param pStream the input stream
	 * @return the content of the stream or null if an error occurs
	 */
	public static byte[] getContent(InputStream pStream)
	{
		return getContent(pStream, true);
	}
	
	/**
	 * Reads the content of an <code>InputStream</code> into a byte array.
	 * <p>
	 * @param pStream the input stream
	 * @param pAutoClose whether the input stream should be closed after reading
	 * @return the content of the stream or null if an error occurs
	 */
	public static byte[] getContent(InputStream pStream, boolean pAutoClose)
	{
	    if (pStream != null)
	    {
	    	BufferedInputStream bis;
	    	
	    	if (pStream instanceof BufferedInputStream)
	    	{
	    		bis = (BufferedInputStream)pStream;
	    	}
	    	else
	    	{
	    		bis = new BufferedInputStream(pStream);	    		
	    	}
	    	
	    	ByteArrayOutputStream baos = new ByteArrayOutputStream();
	    	
	    	byte[] byTmp = new byte[8192];
	
		    int iLen;

		    
		    try
	        {
		        while ((iLen = bis.read(byTmp)) >= 0)
		        {
		            baos.write(byTmp, 0, iLen);
		        }
		
		        baos.close();
		        
		        return baos.toByteArray();
	        }
	        catch (IOException ioe)
	        {
	        	return null;
	        }
	        finally
	        {
	        	if (pAutoClose)
	        	{
	        		try
	        		{
	        			bis.close();
	        		}
	        		catch (IOException ioex)
	        		{
	        			//nothing to be done
	        		}
	        	}
	        }
	    }
	
	    return null;
	}
	
	/**
	 * Gets the content from a file.
	 * 
	 * @param pFile the file
	 * @return the content of the file or null if an error occurs 
	 * @throws IOException if the file does not exist
	 */
	public static byte[] getContent(File pFile) throws IOException
	{
		if (pFile == null)
		{
			throw new IOException("File not found: null");
		}
		
		return getContent(pFile.getAbsolutePath());
	}
	
	/**
	 * Gets the content from a file.
	 * 
	 * @param pFileName the path to the file
	 * @return the content of the file or null if an error occurs 
	 * @throws IOException if the file does not exist
	 */
	public static byte[] getContent(String pFileName) throws IOException
	{
		if (pFileName == null)
		{
			throw new IOException("File not found: null");
		}

		return getContent(new FileInputStream(pFileName), true);
	}

	/**
	 * Reads the content of an <code>InputStreamReader</code> into a byte array and closes the stream after reading.
	 * <p>
	 * @param pReader the input stream reader
	 * @return the content of the reader or null if an error occurs
	 */
	public static byte[] getContent(InputStreamReader pReader)
	{
		return getContent(pReader, true);
	}
	
	/**
	 * Reads the content of an <code>InputStreamReader</code> into a byte array.
	 * <p>
	 * @param pReader the input stream reader
	 * @param pAutoClose whether the input stream should be closed after reading
	 * @return the content of the reader or null if an error occurs
	 */
	public static byte[] getContent(InputStreamReader pReader, boolean pAutoClose)
	{
	    if (pReader != null)
	    {
			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			OutputStreamWriter osw;
			
			try
			{
				osw = new OutputStreamWriter(baos, pReader.getEncoding());
			}
			catch (UnsupportedEncodingException use)
			{
				use.printStackTrace();
				
				return null;
			}
			
		    int iLen;

	        char[] ch = new char[8192];

	        
	        try
	        {
		        while ((iLen = pReader.read(ch)) >= 0)
		        {
		        	osw.write(ch, 0, iLen);
		        }
	        	
		        osw.flush();
	        	osw.close();

	        	baos.close();
	        	
		        return baos.toByteArray();
	        }
	        catch (IOException ioe)
	        {
	        	return null;
	        }
	        finally
	        {
		        if (pAutoClose)
		        {
		        	try
		        	{
		        		pReader.close();
		        	}
		        	catch (IOException ioex)
		        	{
		        		//nothing to be done
		        	}
		        }
	        }
	    }
	
	    return null;
	}

	/**
	 * Removes the file extension from a file name.
	 * 
	 * @param pPath the filename
	 * @return the filename without extension
	 */
	public static String removeExtension(String pPath)
	{
		if (pPath != null)
		{
			int iPos = pPath.lastIndexOf('.');
			
			if (iPos > 0)
			{
				return pPath.substring(0, iPos);
			}
		}
		
		return pPath;
	}

	/**
	 * Gets the extension of an absolute file path.
	 * 
	 * @param pPath the absolute path of a file
	 * @return the extension (text behind the last '.') or <code>null</code> if the path is <code>null</code>
	 *         or has no extension
	 */
	public static String getExtension(String pPath)
	{
		if (pPath == null)
		{
			return null;
		}
		
		int iPos = pPath.lastIndexOf('.');
		
		if (iPos >= 0)
		{
			return pPath.substring(iPos + 1);
		}
		
		return null;
	}
	
	/**
	 * Gets a file that does not exist.
	 * If the given file already exists, the file name built in the way "nameWithoutExtension[number].extension".
	 * 
	 * @param pFile the desired file name
	 * @return a file that does not exist.
	 */
	public static File getNotExistingFile(File pFile)
	{
		String extension = getExtension(pFile.getName());
		String nameWithoutExtension = removeExtension(pFile.getName());
	
		int i = 1;
		while (pFile.exists())
		{
			pFile = new File(pFile.getParent(), nameWithoutExtension + "[" + i + "]." + extension);
			i++;
		}
		return pFile;
	}

	/**
	 * Gets the name of a file from an absolute path.
	 * 
	 * @param pAbsolutePath the absolute path for a file
	 * @return the name of the file without path
	 */
	public static String getName(String pAbsolutePath)
	{
		if (pAbsolutePath == null)
		{
			return null;
		}
		
		int iPos = pAbsolutePath.lastIndexOf('/');
		
		if (iPos < 0)
		{
			iPos = pAbsolutePath.lastIndexOf('\\');
		}
		
		if (iPos >= 0)
		{
			return pAbsolutePath.substring(iPos + 1);
		}
		
		return pAbsolutePath;
	}
	
	/**
	 * Gets the directory for a path. It searchs the last path separator and cuts off all characters
	 * behind, e.g. /home/a/b/c returns /home/a/b
	 * 
	 * @param pAbsolutePath a path
	 * @return the parent directory path or <code>null</code> if no path separator was found
	 */
	public static String getDirectory(String pAbsolutePath)
	{
		if (pAbsolutePath == null)
		{
			return null;
		}
		
		int iPos = pAbsolutePath.lastIndexOf('/');
		
		if (iPos < 0)
		{
			iPos = pAbsolutePath.lastIndexOf('\\');
		}

		if (iPos > 0)
		{
			return pAbsolutePath.substring(0, iPos);
		}
		
		return null;
	}
	
	/**
	 * Formats the given path as directory. That means a trailing slash will be added if the pat
	 * is not already a directory.
	 *  
	 * @param pPath the path
	 * @return the directory path
	 */
	public static String formatAsDirectory(String pPath)
	{
		if (pPath == null)
		{
			return null;
		}
		
		char ch = pPath.charAt(pPath.length() - 1);
		
		if (ch == '\\' || ch == '/')
		{
			return pPath;
		}
		else
		{
			return pPath + "/";
		}
	}

    /**
     * Delete a file or directory.
     * 
     * @param pSource the file or directory
     * @param pPattern the patterns
     * @return true if the directory was deleted, <code>false</code> otherwise
     */
    public static boolean delete(File pSource, String...pPattern)
    {
        FileSearch fsearch = new FileSearch();
        
        fsearch.search(pSource, true, pPattern);


        //delete files
        for (String file : fsearch.getFoundFiles())
        {
            if (!new File(file).delete())
            {
                return false;
            }
        }

        //delete directories (from bottom to top)
        List<String> liDir = fsearch.getFoundDirectories();
        
        for (int i = liDir.size() - 1; i >= 0; i--)
        {
        	if (!new File(liDir.get(i)).delete())
        	{
        		return false;
        	}
        }
        
        if (pSource.exists() && !pSource.delete())
        {
        	return false;
        }
        
        return true;
    }	
    
    /**
     * Delete empty directories, recursive.
     * 
     * @param pDirectory the start directories
     * @param pPattern the include/exclude pattern
     * @return <code>true</code> if all empty directories were deleted, <code>false</code>
     *         otherwise
     */
    public static boolean deleteEmpty(File pDirectory, String... pPattern)
    {
        FileSearch fsearch = new FileSearch();
    	fsearch.search(pDirectory, true, pPattern);

        //delete directories (from bottom to top)
        List<String> liDir = fsearch.getFoundDirectories();

        File fiDir;
        
        String[] sPaths;
        
        for (int i = liDir.size() - 1; i >= 0; i--)
        {
        	fiDir = new File(liDir.get(i));
        	
        	sPaths = fiDir.list();
        	
        	if (sPaths != null && sPaths.length == 0 && !fiDir.delete())
        	{
        		return false;
        	}
        }
        
        return true;	
    }
    
    /**
     * Deletes directories if they are empty.
     * 
     * @param pDirectory a list of directories
     * @return <code>true</code> if all empty directories were deleted, <code>false</code> if an empty
     *         directory can not be deleted
     */
    public static boolean deleteIfEmpty(File... pDirectory)
    {
        String[] sPaths;

        for (File dir : pDirectory)
        {
        	sPaths = dir.list();
        	
        	if (sPaths != null && sPaths.length == 0 && !dir.delete())
        	{
        		return false;
        	}
        }
        
        return true;
    }

    /**
     * Creates a zip package. 
     * <p>
     * @param pOut the output stream
     * @param pMode {@link ZipOutputStream#STORED} or {@link ZipOutputStream#DEFLATED}
     * @param pSources the list of sources
     * @throws IOException if a problem occurs during zip creation
     */
    public static void zip(OutputStream pOut, int pMode, String... pSources) throws IOException
    {
        FileInputStream in = null;

        ZipOutputStream zos = new ZipOutputStream(pOut);
        BufferedOutputStream bos = new BufferedOutputStream(zos);

        CRC32 crc = null;

        ZipEntry ze;

        FileSearch fsearch = new FileSearch();

        File fCutOff;
        File fCurrent;

        byte[] by = new byte[4096];

        int iLength;
        int iCutOff;


        zos.setMethod(pMode);

        try
        {
	        for (String sSource : pSources)
	        {
	        	fsearch.clear();
	        	fsearch.search(sSource, true);
	        
	        	fCutOff = new File(sSource);
	        	
	        	//directory: remove the whole path
	        	//file: remove the parent path
		        if (fCutOff.isDirectory())
		        {
		            iCutOff = fCutOff.getAbsolutePath().length() + 1;
		        }
		        else
		        {
		            iCutOff = fCutOff.getParent().length() + 1;
		        }
	
		        //add found files
		        for (String sFile : fsearch.getFoundFiles())
		        {
		            fCurrent = new File(sFile);
		
		            if (pMode == ZipOutputStream.STORED)
		            {
		                crc = new CRC32();
		                
		                try
		                {
			                in = new FileInputStream(fCurrent);
			
			                while ((iLength = in.read(by)) >= 0)
			                {
			                    crc.update(by, 0, iLength);
			                }
		                }
			            finally
			            {
			            	if (in != null)
			            	{
				            	try
				            	{
				            		in.close();
				            	}
				            	catch (Exception e)
				            	{
				            		//nothing to be done
				            	}
			            	}
			            }
		            }
		
		            //configure zip entry
		            ze = new ZipEntry(fCurrent.getAbsolutePath().substring(iCutOff).replace('\\', '/'));
		
		            if (pMode == ZipOutputStream.STORED)
		            {
		                ze.setSize(fCurrent.length());
		                ze.setTime(fCurrent.lastModified());
		                ze.setCrc(crc.getValue());
		            }
		
		            zos.putNextEntry(ze);
		
		            try
		            {
			            in = new FileInputStream(fCurrent);
			
			            while ((iLength = in.read(by)) >= 0)
			            {
			                bos.write(by, 0, iLength);
			            }
		            }
		            finally
		            {
		            	if (in != null)
		            	{
			            	try
			            	{
			            		in.close();
			            	}
			            	catch (Exception e)
			            	{
			            		//nothing to be done
			            	}
		            	}
		            }
		
		            bos.flush();
		            zos.closeEntry();
		        }
	        }
        }
        finally
        {
        	try
        	{
        		bos.close();
        	}
        	catch (Exception e)
        	{
        		//nothing to be done
        	}
        }

        //cleanup
        by  = null;
        zos = null;
        bos = null;
        in  = null;
        fsearch = null;
    }
    
	/**
	 * Extracts the content of a zip archive.
	 * 
	 * @param pArchive the zip archive
	 * @param pTarget where the content should go
	 * @throws IOException if an error occurs during extraction
	 * @see #unzip(InputStream, File)
	 */
    public static void unzip(InputStream pArchive, String pTarget) throws IOException
    {
    	unzip(pArchive, new File(pTarget));
    }
    
	/**
	 * Extracts the content of a zip archive.
	 * 
	 * @param pArchive the zip archive
	 * @param pTarget where the content should go
	 * @throws IOException if an error occurs during extraction
	 */
	public static void unzip(InputStream pArchive, File pTarget) throws IOException
	{
		File fiCurrent;
		
		int iLen;
		
		byte[] byData = new byte[4096];
		
		
		fiCurrent = pTarget;
		
		if (fiCurrent.exists() || fiCurrent.mkdirs())
		{
			BufferedOutputStream bosCurrent = null;

			ZipInputStream zis = null;
			
			try
			{
				zis = new ZipInputStream(pArchive);
	
				ZipEntry zeCurrent;
				
				while ((zeCurrent = zis.getNextEntry()) != null)
	    		{
	    			fiCurrent = new File(pTarget, zeCurrent.getName());
	    			
	    			//create directories if needed
	    			if (zeCurrent.isDirectory())
	    			{
	    				fiCurrent.mkdirs();
	    			}
	    			else
	    			{
	    				try
	    				{
		        			//extract
		        			bosCurrent = new BufferedOutputStream(new FileOutputStream(fiCurrent));
		        			
		        			while ((iLen = zis.read(byData)) != -1)
		        			{
		        				bosCurrent.write(byData, 0, iLen);
		        			}
	    				}
	    				finally
	    				{
	    					if (bosCurrent != null)
	    					{
		    					try
		    					{
		    						bosCurrent.close();
		    						bosCurrent = null;
		    					}
		    					catch (Exception e)
		    					{
		    						//nothing to be done
		    					}
	    					}
	    				}
	    			}
	    		}
			}
			finally
			{
				if (zis != null)
				{
					try
					{
						zis.close();
					}
					catch (Exception e)
					{
						//nothing to be done
					}
				}
			}
		}
		else
		{
			throw new IOException("Can not create target directory: " + pTarget.getAbsolutePath());
		}
	}

	/**
	 * Gets a list of all entries from a zip archive.
	 * 
	 * @param pArchive the stream for the zip archive
	 * @return the list of entries
	 * @throws IOException if the archive does not contain zip content or a read error occurs
	 */
    public static List<String> listZipEntries(InputStream pArchive) throws IOException
    {
		ZipInputStream zis = null;
		
		try
		{
			zis = new ZipInputStream(pArchive);

			List<String> liEntries = new ArrayUtil<String>();

			ZipEntry zeCurrent;
			
			while ((zeCurrent = zis.getNextEntry()) != null)
			{
				liEntries.add(zeCurrent.getName());
			}
			
			return liEntries;
		}
		finally
		{
			if (zis != null)
			{
				try
				{
					zis.close();
				}
				catch (Exception e)
				{
					//nothing to be done
				}
			}
		}
    }
	
    /**
     * Saves the given content into a file.
     * 
     * @param pFile the output file
     * @param byContent the data
     * @throws IOException if an error occurs during output
     */
    public static void save(File pFile, byte[] byContent) throws IOException
    {
    	ByteArrayInputStream bais = new ByteArrayInputStream(byContent); 
    	
    	try
    	{
    		save(pFile, bais);
    	}
    	finally
    	{
    		if (bais != null)
    		{
    			try
    			{
    				bais.close();
    			}
    			catch (Exception e)
    			{
    				// nothing to be done
    			}
    		}
    	}
    }
    
    /**
     * Saves the given stream into a file.
     * 
     * @param pFile the output file
     * @param pInput the input stream
     * @throws IOException if an error occurs during output
     */
    public static void save(File pFile, InputStream pInput) throws IOException
    {
    	BufferedOutputStream bos = null;
    	
    	try
    	{
    		if (!pFile.getParentFile().exists())
    		{
    			pFile.getParentFile().mkdirs();
    		}
    		
    		bos = new BufferedOutputStream(new FileOutputStream(pFile));
    		
    		int iLen;
    		
    		byte[] byData = new byte[4096];
    		
    		while ((iLen = pInput.read(byData)) >= 0)
    		{
	    		bos.write(byData, 0, iLen);
    		}
    		
    		bos.flush();
    		bos.close();
    		
    		bos = null;
    	}
    	finally
    	{
    		if (bos != null)
    		{
    			try
    			{
    				bos.close();
    			}
    			catch (Exception e)
    			{
    				// nothing to be done
    			}
    		}
    	}
    }

    /**
     * Replaces key/value mappings read from the given input file and writes the result to the given
     * output file. This method requires ASCII files. If the given input and output file are the same,
     * then this methods caches the data in a temporary file or in memory before writing the result to
     * the output file. An input file with a filesize smaller than 1M is cached in memory.
     * 
     * @param pInput the input file
     * @param pOutput the output file
     * @param pMapping the key/value mapping e.g. ${PARAM1} = JVx
     * @param pEncoding the encoding for the input/output files
     * @throws IOException if a read or write error occurs
     */
    public static void replace(File pInput, File pOutput, Hashtable<String, String> pMapping, String pEncoding) throws IOException
    {
    	replace(pInput, pOutput, pMapping, pEncoding, pEncoding);
    }   
    
    /**
     * Replaces key/value mappings read from the given input file and writes the result to the given
     * output file. This method requires ASCII files. If the given input and output file are the same,
     * then this methods caches the data in a temporary file or in memory before writing the result to
     * the output file. An input file with a filesize smaller than 1M is cached in memory.
     * 
     * @param pInput the input file
     * @param pOutput the output file
     * @param pMapping the key/value mapping e.g. ${PARAM1} = JVx
     * @param pInEncoding the encoding of the input file
     * @param pOutEncoding the encoding for the output file
     * @throws IOException if a read or write error occurs
     */
    public static void replace(File pInput, File pOutput, Hashtable<String, String> pMapping, String pInEncoding, String pOutEncoding) throws IOException
    {
		FileInputStream fis = null;
		OutputStream    os = null;
		
		File fiTemp = null;
		
		try
		{
			fis = new FileInputStream(pInput);

			boolean bTempWrite;

			if (pInput == pOutput || pInput.equals(pOutput))
			{
				bTempWrite = true;
				
				if (pInput.length() <= 1024000L)
				{
					os = new ByteArrayOutputStream();
				}
				else
				{
					fiTemp = File.createTempFile(FileUtil.class.getName(), "jvx");
					
					os = new FileOutputStream(fiTemp);
				}
			}
			else
			{
				bTempWrite = false;
				os = new FileOutputStream(pOutput);
			}
			
			replace(fis, os, pMapping, pInEncoding, pOutEncoding);
			
			fis.close();
			fis = null;
			
			if (bTempWrite)
			{
				//try to rename the file -> fast variant
				if (fiTemp != null)
				{
					if (pInput.delete())
					{
						if (fiTemp.renameTo(pInput))
						{
							bTempWrite = false;
						}
					}
				}
				
				if (bTempWrite)
				{
					if (os instanceof ByteArrayOutputStream)
					{
						BufferedOutputStream bos = null;
						
						try
						{
							bos = new BufferedOutputStream(new FileOutputStream(pOutput));
							bos.write(((ByteArrayOutputStream)os).toByteArray());
						}
						finally
						{
							if (bos != null)
							{
								try
								{
									bos.close();
								}
								catch (Exception e)
								{
									//nothing to be done
								}
							}
						}
					}
					else
					{
						copy(fiTemp, pOutput);
					}
				}
			}
		}
		finally
		{
			if (fis != null)
			{
				try
				{
					fis.close();
				}
				catch (Exception e)
				{
					//nothing to be done
				}
			}

			if (os != null)
			{
				try
				{
					os.close();
				}
				catch (Exception e)
				{
					//nothing to be done
				}
			}
			
			if (fiTemp != null)
			{
				if (!fiTemp.delete())
				{
					fiTemp.deleteOnExit();
				}
			}
		}
    }
    
    /**
     * Replaces key/value mappings read from the given input stream and writes the result to the given
     * output stream. This method requires ASCII streams.
     * 
     * @param pInput the input stream
     * @param pOutput the output stream
     * @param pMapping the key/value mapping e.g. ${PARAM1} = JVx
     * @param pInEncoding the encoding of the input stream
     * @param pOutEncoding the encoding for the output stream
     * @throws IOException if a read or write error occurs
     */
    public static void replace(InputStream pInput, OutputStream pOutput, Hashtable<String, String> pMapping, String pInEncoding, String pOutEncoding) throws IOException
    {
    	BufferedReader brInput = null;
    	BufferedWriter bwOutput = null;
    	
    	try
    	{
			//Input
			InputStreamReader isr;
			
			if (pInEncoding != null)
			{
				isr = new InputStreamReader(pInput, pInEncoding);
			}
			else
			{
				isr = new InputStreamReader(pInput);
			}
			
			brInput = new BufferedReader(isr);
	
			//Output
			
			OutputStreamWriter oswTemplate;
			
			if (pOutEncoding != null)
			{
				oswTemplate = new OutputStreamWriter(pOutput, pOutEncoding);
			}
			else
			{
				oswTemplate = new OutputStreamWriter(pOutput);
			}
			
			
			bwOutput = new BufferedWriter(oswTemplate);
			
			
			String sLineSeparator = System.getProperty("line.separator");
			
	        String sLine = brInput.readLine();
	        while (sLine != null)
	        {
        		for (Map.Entry<String, String> entry : pMapping.entrySet())
        		{
        			sLine = StringUtil.replace(sLine, entry.getKey(), entry.getValue());
        		}
	        	
	        	bwOutput.write(sLine);
	        	
	        	sLine = brInput.readLine();

	        	//last line!
	        	if (sLine != null)
	        	{
		        	bwOutput.write(sLineSeparator);
	        	}
	        }
	        
	    	bwOutput.write(sLineSeparator);
    	}
    	finally
    	{
			if (brInput != null)
			{
				try
				{
					brInput.close();
				}
				catch (Exception e)
				{
					//nothing to be done
				}
			}

			if (bwOutput != null)
			{
				try
				{
					bwOutput.close();
				}
				catch (Exception e)
				{
					//nothing to be done
				}
			}
    	}
    }
    
	/**
	 * Fast like search in path with wildcard(**, * and ?) support.
	 *  
	 * @param pSource any string
	 * @param pSearch search pattern with or without wildcards.
	 * @return <code>true</code> if, and only if, the string matches the pattern
	 */
	public static boolean like(String pSource, String pSearch) 
	{
	    if (pSource == null && pSearch == null) 
	    {
	    	return true;
	    }
	    else if (pSource == null || pSearch == null) 
	    {
	    	return false;
	    }
	    else 
	    {
	    	return like(pSource, 0, pSource.length(), pSearch, 0, pSearch.length());
	    }
	}
    
	/**
	 * Sub function of like. For performance reasons, the original Strings are not modified,
	 * The start and end defines the region to search.
	 * 
	 * @param pSource	the source string.
	 * @param pSrcStart	the start index of the source region.
	 * @param pSrcEnd	the end index of the source region.
	 * @param pSearch	the search string.
	 * @param pStart  	the start index of the search region.
	 * @param pEnd		the end index of the search region.
	 * @return true, if the like matches.
	 */
	private static boolean like(String pSource, int pSrcStart, int pSrcEnd, 
			                    String pSearch, int pStart,    int pEnd)
	{
		int pos = pSrcStart;
	  	for (int i = pStart; i < pEnd; i++) 
	  	{
	  		char ch = pSearch.charAt(i);
	  		if (ch == '*') 
	  		{
	  			if (i + 1 < pEnd && pSearch.charAt(i + 1) == '*')
	  			{	  				
		  			if (i + 1 == pEnd - 1)
	  				{
	  					return true;
	  				}
	  				
		  			int nStart = i + 2;
		  			
		  			while (pos < pSrcEnd) 
		  			{ 
		  				if (like(pSource, pos, pSrcEnd, pSearch, nStart, pEnd))
		  				{
		  					return true;
		  				}
		  				pos++;
		  			}
		  			return false;
	  			}
	  			else
	  			{
		  			int nStart = i + 1;
		  			
		  			int iSrcEnd = pSource.indexOf('/', pos);
		  			
		  			if (iSrcEnd < 0)
		  			{
			  			iSrcEnd = pSource.indexOf('\\', pos);
		  			}
		  			
		  			if (iSrcEnd < 0)
		  			{
			  			if (i == pEnd - 1) 
			  			{
			  				return true;
			  			}
		  				
		  				iSrcEnd = pSrcEnd;
		  			}
		  			else
		  			{
		  				iSrcEnd++;
		  			}
		  			
		  			int iEnd = pSearch.indexOf('/', nStart);
		  			
		  			if (iEnd < 0)
		  			{
		  				iEnd = pSearch.indexOf('\\', nStart);
		  			}
		  			
		  			if (iEnd < 0)
		  			{
		  				iEnd = pEnd;
		  			}
		  			else
		  			{
		  				iEnd++;
		  			}
		  			
		  			while (pos < iSrcEnd) 
		  			{ 
		  				if (like(pSource, pos, iSrcEnd, pSearch, nStart, iEnd))
		  				{
		  					if (like(pSource, iSrcEnd, pSrcEnd, pSearch, iEnd, pEnd))
		  					{
		  						return true;
		  					}
		  				}
		  				pos++;
		  			}
		  			return false;
	  			}
	  		}
	  		else if (pos == pSrcEnd) 
	  		{
	  			return false;
	  		}
	  		else 
	  		{
	  			if (ch != '?' && ch != pSource.charAt(pos)) 
	  			{
	  				return false;
	  			}
	  			pos++;
	  		}
	  	}
	  	return pos == pSrcEnd;
	}    
    
}	// FileUtil
