/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 26.10.2009 - [HM] - creation
 * 13.02.2010 - [JR] - format with Date and long parameter implemented
 */
package com.sibvisions.util.type;

import java.text.DateFormat;
import java.text.DateFormatSymbols;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Locale;

/**
 * The <code>DateUtil</code> is a utility class for date conversion and for formatting dates
 * as string.
 *  
 * @author Martin Handsteiner
 */
public class DateUtil
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The date format. */
	private SimpleDateFormat dateFormat;
	
	/** The locale base for creation. */
	private Locale creationLocale = null;
	/** The pattern base for creation. */
	private String creationPattern = null;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** 
	 * Constructs a new instance of <code>DateUtil</code> with default date format.
	 */
	public DateUtil()
	{
		this(null);
	}

	/** 
	 * Constructs a new instance of <code>DateUtil</code> that supports empty Strings and null values.
	 * 
	 * @param pDateFormat the formatter that should support empty Strings and null values
	 */
	public DateUtil(DateFormat pDateFormat)
	{
		setDateFormat(pDateFormat);
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Parses the date from text.
	 * 
	 * @param pText the text.
	 * @return the parsed date.
     * @throws ParseException if there is an error in the conversion
	 */
	public Date parse(String pText) throws ParseException
	{
		if (pText == null || pText.length() == 0)
		{
			return null;
		}
		else
		{
			if (creationLocale != null)
			{
				setDateFormatIntern(LocaleUtil.getDefault(), creationPattern);
			}
			
			return parseStringIntern(pText);
		}
	}

	/**
	 * Formats the date to text.
	 * 
	 * @param pDate the date.
	 * @return the formatted text.
	 */
	public String format(Date pDate)
	{
		if (pDate == null)
		{
			return null;
		}
		else
		{
			if (creationLocale != null)
			{
				setDateFormatIntern(LocaleUtil.getDefault(), creationPattern);
			}
			
			return dateFormat.format(pDate);
		}
	}

	/**
	 * Gets the date format.
	 * 
	 * @return the date format.
	 */
	public DateFormat getDateFormat()
	{
		return dateFormat;
	}

	/**
	 * Sets the new date format, if something was changed.
	 * @param pLocale the locale
	 * @param pDatePattern the pattern
	 */
	private void setDateFormatIntern(Locale pLocale, String pDatePattern)
	{
		if (pDatePattern == null)
		{
			if (pLocale != creationLocale || creationPattern != null)
			{
				dateFormat = (SimpleDateFormat)DateFormat.getDateTimeInstance(DateFormat.LONG, DateFormat.SHORT, LocaleUtil.getDefault());
						
				creationLocale = pLocale;
				creationPattern = pDatePattern;
			}
		}
		else
		{
			if (pLocale != creationLocale || !pDatePattern.equals(creationPattern))
			{
				dateFormat = new SimpleDateFormat(pDatePattern, LocaleUtil.getDefault());
				
				creationLocale = pLocale;
				creationPattern = pDatePattern;
			}
		}
	}
	
	/**
	 * Gets the date format.
	 * 
	 * @param pDateFormat the date format.
	 */
	public void setDateFormat(DateFormat pDateFormat)
	{
		if (pDateFormat == null)
		{
			setDateFormatIntern(LocaleUtil.getDefault(), null);
		}
		else if (pDateFormat instanceof SimpleDateFormat)
		{
			dateFormat = (SimpleDateFormat)pDateFormat;
			creationLocale = null;
			creationPattern = null;
		}
		else
		{
			throw new IllegalArgumentException("Only SimpleDateFormat is supported!");
		}
	}

	/**
	 * Gets the date format pattern.
	 * 
	 * @return the date format pattern.
	 */
	public String getDatePattern()
	{
		return dateFormat.toPattern();
	}

	/**
	 * Gets the date format pattern.
	 * 
	 * @param pDatePattern the date format pattern.
	 */
	public void setDatePattern(String pDatePattern)
	{
		if (pDatePattern == null)
		{
			setDateFormat(null);
		}
		else
		{
			setDateFormatIntern(LocaleUtil.getDefault(), pDatePattern);
		}
	}

	/**
	 * Gets the best transformation of the date part for the pattern.
	 * 
	 * @param pPart the date part
	 * @param pPattern the pattern
	 * @param pSymbols the date format symbols
	 * @return the best transformation of the date part for the pattern
	 */
	private String getPart(String pPart, String pPattern, DateFormatSymbols pSymbols)
	{
		try
		{
			int number = Integer.parseInt(pPart);
			if (pPattern.startsWith("MMM"))
			{
				if (pPattern.length() == 3)
				{
					return pSymbols.getShortMonths()[number - 1];
				}
				else
				{
					return pSymbols.getMonths()[number - 1];
				}
			}
		}
		catch (Exception ex)
		{
			if (pPattern.startsWith("M"))
			{
				String lowerPart = pPart.toLowerCase();
				for (int i = 0; i < 12; i++)
				{
					if (pSymbols.getMonths()[i].toLowerCase().startsWith(lowerPart) || pSymbols.getShortMonths()[i].toLowerCase().startsWith(lowerPart))
					{
						int mon = i + 1;
						if (pPattern.length() == 2 && mon < 10)
						{
							return "0" + mon;
						}
						else if (pPattern.length() <= 2)
						{
							return String.valueOf(mon);
						}
						else if (pPattern.length() == 3)
						{
							return pSymbols.getShortMonths()[i];
						}
						else
						{
							return pSymbols.getMonths()[i];
						}
					}
				}
			}
		}
		return pPart;
	}
	
	/**
	 * Parses the date parts and the seperator parts.
	 * 
	 * @param pText the date
	 * @return the date parts and the seperator parts
	 */
	private List<String>[] getParsed(String pText)
	{
		List<String> datePart = new ArrayList<String>();
		List<String> seperatorPart = new ArrayList<String>();
		
		int len = pText.length();
		int pos = 0;
		boolean isLetterOrDigit = Character.isLetterOrDigit(pText.charAt(0));
		for (int i = 1; i < len; i++)
		{
			char ch = pText.charAt(i);
			if (isLetterOrDigit != Character.isLetterOrDigit(ch))
			{
				String part = pText.substring(pos, i);
				if (isLetterOrDigit)
				{
					datePart.add(part);
				}
				else
				{
					seperatorPart.add(part);
				}
				pos = i;
				isLetterOrDigit = !isLetterOrDigit;
			}
		}
		String part = pText.substring(pos);
		if (isLetterOrDigit)
		{
			datePart.add(part);
		}
		else
		{
			seperatorPart.add(part);
		}

		List[] result = new List[2];
		result[0] = datePart;
		result[1] = seperatorPart;
		return result;
	}
	
    /**
     * Parses <code>text</code> returning an Date. Some
     * formatters may return null.
     *
     * @param pText String to convert
     * @return Date representation of text
     * @throws ParseException if there is an error in the conversion
     */
	private Date parseStringIntern(String pText) throws ParseException
	{
		try
		{
			return dateFormat.parse(pText);
		}
		catch (ParseException parseException)
		{
			try
			{
				DateFormatSymbols symbols = dateFormat.getDateFormatSymbols();
				
				String pattern = dateFormat.toPattern();
	
				GregorianCalendar cal = new GregorianCalendar();
			    cal.set(Calendar.HOUR_OF_DAY, 0);
			    cal.set(Calendar.MINUTE, 0);
			    cal.set(Calendar.SECOND, 0);
			    cal.set(Calendar.MILLISECOND, 0);
	
			    String referenceDate = dateFormat.format(cal.getTime());
				
			    List<String>[] parsedPattern = getParsed(pattern);
			    List<String>[] parsedDate = getParsed(pText);
			    List<String>[] parsedReference = getParsed(referenceDate);
			    
			    int datePartCount = parsedDate[0].size();
			    int patternSeperatorCount = parsedPattern[1].size();
			    StringBuffer result = new StringBuffer();
			    for (int i = 0; i < parsedPattern[0].size(); i++)
			    {
			    	if (i < datePartCount)
			    	{
			    		result.append(getPart(parsedDate[0].get(i), parsedPattern[0].get(i), symbols));
			    	}
			    	else
			    	{
			    		result.append(parsedReference[0].get(i));
			    	}
			    	if (i < patternSeperatorCount)
			    	{
			    		result.append(parsedPattern[1].get(i));
			    	}
			    }
			    return dateFormat.parse(result.toString());
			}
			catch (Exception exception)
			{
				// its a wrong dateformat!
			}
		    throw new ParseException("Wrong Dateformat", 0);
		}
	}

	/**
	 * Formats a time string.
	 * 
	 * @param pDate the time (in millis since 01.01.1970 00:00) value to be formatted into a time string
	 * @param pFormat the format 
	 * @return the formatted date/time string
	 * @see SimpleDateFormat
	 */
	public static String format(long pDate, String pFormat)
	{
		return format(new Date(pDate), pFormat);
	}
	
	/**
	 * Formats a Date into a date/time string.
	 * 
	 * @param pDate the time value to be formatted into a time string
	 * @param pFormat the format 
	 * @return the formatted date/time string
	 * @see SimpleDateFormat
	 */
	public static String format(Date pDate, String pFormat)
	{
		SimpleDateFormat sdf = new SimpleDateFormat(pFormat);
		
		return sdf.format(pDate);
	}
	
	/**
	 * Creates a new date instance.
	 * 
	 * @param pDay the day of month
	 * @param pMonth the month
	 * @param pYear the year
	 * @param pHour the hour of day
	 * @param pMinutes the minutes
	 * @param pSeconds the seconds
	 * @return the date
	 */
	public static Date getDate(int pDay, int pMonth, int pYear, int pHour, int pMinutes, int pSeconds)
	{
		Calendar cal = GregorianCalendar.getInstance();
		
		cal.set(Calendar.DAY_OF_MONTH, pDay);
		cal.set(Calendar.MONTH, pMonth - 1);
		cal.set(Calendar.YEAR, pYear);
		cal.set(Calendar.HOUR_OF_DAY, pHour);
		cal.set(Calendar.MINUTE, pMinutes);
		cal.set(Calendar.SECOND, pSeconds);
		cal.set(Calendar.MILLISECOND, 0);
		
		return cal.getTime();
	}
	
}	// DateUtil
