/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 08.04.2009 - [JR] - creation
 * 15.06.2009 - [JR] - encodeURL replaced with encodeURLPart
 *                     decodeURLPart implemented
 * 23.06.2009 - [JR] - implemented encodeHex with byte[] as parameter   
 * 02.08.2009 - [JR] - encodeURLPart: UTF-8 encoding      
 * 03.08.2009 - [JR] - encodeURLPart: ISO-8859-1 (as defined in RFC. If UTF-8, then the application server e.g. 
 *                     tomcat has to use URIEncoding="UTF-8")
 * 08.03.2012 - [JR] - encodeHex(InputStream) implemented                                  
 */
package com.sibvisions.util.type;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;

/**
 * The <code>CodecUtil</code> contains methods for encode and decode operations.
 * 
 * @author Ren Jahn
 */
public final class CodecUtil
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Invisible constructor because <code>CodecUtil</code> is a utility
	 * class.
	 */
	private CodecUtil()
	{
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Encodes the part for an URL with UTF-8 encoding.
	 * 
	 * @param pPart the url part
	 * @return the encoded part
	 */
	public static String encodeURLPart(String pPart)
	{
		StringBuffer sbfUrl = new StringBuffer();
		
		int iChar;

		try
		{
			byte[] by = pPart.getBytes("ISO-8859-1");
			
			for (int i = 0, anz = by.length; i < anz; i++)
			{
				iChar = by[i];
				
				if ((iChar >= 'a' && iChar <= 'z')
					|| (iChar >= 'A' && iChar <= 'Z')
					|| (iChar >= '0' && iChar <= '9')
					|| iChar == '$'
					|| iChar == '-'
					|| iChar == '_'
					|| iChar == '.'
					|| iChar == '!'
					|| iChar == '\''
					|| iChar == '('
					|| iChar == ')'
					|| iChar == ','
					|| iChar == '/')
				{
					sbfUrl.append((char)iChar);
				}
				else
				{
					sbfUrl.append('%');
					sbfUrl.append(String.valueOf(Integer.toHexString(((iChar & 0xff) + 256))).substring(1));
				}
			}
			
			return sbfUrl.toString();
		}
		catch (Exception e)
		{
			return null;
		}
	}
	
	/**
	 * Encodes a plain value for an URL query parameter with ISO-8859-1.
	 * 
	 * @param pValue the plain query parameter value
	 * @return the encoded value or <code>null</code> if ISO-8859-1 is not supported
	 */
	public static String encodeURLParameter(String pValue)
	{
		return encodeURLParameter(pValue, "ISO-8859-1");
	}
	
	/**
	 * Encodes a plain value for an URL query parameter with a specific charset.
	 * 
	 * @param pValue the plain query parameter value
	 * @param pEncoding the charset e.g. UTF-8
	 * @return the encoded value or <code>null</code> if <code>pEncoding</code> is not supported
	 */
	public static String encodeURLParameter(String pValue, String pEncoding)
	{
		try
		{
			return URLEncoder.encode(pValue, pEncoding);
		}
		catch (UnsupportedEncodingException use)
		{
			return null;
		}
	}
	
	/**
	 * Decodes the part of an URL with UTF-8.
	 * 
	 * @param pURL the url part
	 * @return the decoded part
	 */
	public static String decodeURLPart(String pURL)
	{
		try
		{
			String sHex;
			
			char ch;
			
			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			
			for (int i = 0, anz = pURL.length(); i < anz;)
			{
				ch = pURL.charAt(i);

				i++;
				
				if (ch == '%')
				{
					sHex = pURL.substring(i, i + 2);
					
					i += 2;

					baos.write((byte)((Integer.parseInt(sHex, 16) - 256) & 0xff));
				}
				else
				{
					baos.write(ch);
				}
			}
			
			baos.flush();
			baos.close();
			
			return new String(baos.toByteArray(), "ISO-8859-1");
		}
		catch (Exception e)
		{
			return null;
		}
	}
	
	/**
	 * Decodes an encoded value from an URL query parameter with ISO-8859-1.
	 * 
	 * @param pValue the encoded query parameter value
	 * @return the plain value or <code>null</code> if ISO-8859-1 is not supported
	 */
	public static String decodeURLParameter(String pValue)
	{
		return decodeURLParameter(pValue, "ISO-8859-1");
	}
	
	/**
	 * Decodes an encoded value from an URL query parameter with a specific charset.
	 * 
	 * @param pValue the encoded query parameter value
	 * @param pEncoding the charset e.g. UTF-8
	 * @return the plain value or <code>null</code> if <code>pEncoding</code> is not supported
	 */
	public static String decodeURLParameter(String pValue, String pEncoding)
	{
		try
		{
			return URLDecoder.decode(pValue, pEncoding);
		}
		catch (UnsupportedEncodingException use)
		{
			return null;
		}
	}
	
	/**
	 * Encodes a text to raw html.
	 * 
	 * @param pText the text
	 * @return the html encoded text
	 */
	public static String encodeHtml(String pText)
	{
		if (pText == null)
		{
			return null;
		}
		
		String sText = pText.replace("&", "&amp;");
		sText = sText.replace("\"", "&quot;");
		sText = sText.replace("/", "&frasl;");
		sText = sText.replace("<", "&lt;");
		sText = sText.replace(">", "&gt;");
		sText = sText.replace("", "&ouml;");
		sText = sText.replace("", "&uuml;");
		sText = sText.replace("", "&auml;");
		sText = sText.replace("", "&Ouml;");
		sText = sText.replace("", "&Uuml;");
		sText = sText.replace("", "&Auml;");
		sText = sText.replace("", "&szlig;");
		
		return sText;
	}

	/**
	 * Encodes a text to a hex encoded string.
	 * 
	 * @param pText the plain text
	 * @return the encoded text
	 * @throws UnsupportedEncodingException if utf-8 encoding is not supported
	 */
	public static String encodeHex(String pText) throws UnsupportedEncodingException
	{
		if (pText == null)
		{
			return null;
		}
		else
		{
			return encodeHex(pText.getBytes("UTF-8"));
		}
	}
	
	/**
	 * Encodes a list of bytes to a hex encoded string.
	 * 
	 * @param pContent the list of bytes
	 * @return the encoded text
	 */
	public static String encodeHex(byte[] pContent)
	{
		if (pContent == null)
		{
			return null;
		}

		StringBuilder sbEncoded = new StringBuilder();
		
		for (int i = 0, anz = pContent.length; i < anz; i++)
		{
			sbEncoded.append(String.valueOf(Integer.toHexString(((pContent[i] & 0xff) + 256))).substring(1));
		}

		return sbEncoded.toString();
	}
	
	/**
	 * Encodes a stream to a hex encoded string.
	 * 
	 * @param pStream the stream
	 * @return the encoded text
	 * @throws IOException if stream reading fails
	 */
	public static String encodeHex(InputStream pStream) throws IOException
	{
    	BufferedInputStream bis;
    	
    	if (pStream instanceof BufferedInputStream)
    	{
    		bis = (BufferedInputStream)pStream;
    	}
    	else
    	{
    		bis = new BufferedInputStream(pStream);	    		
    	}
    	
    	byte[] byTmp = new byte[8192];

	    int iLen;
	    
    	StringBuilder sbEncoded = new StringBuilder();
    	
        while ((iLen = bis.read(byTmp)) >= 0)
        {
    		for (int i = 0; i < iLen; i++)
    		{
    			sbEncoded.append(String.valueOf(Integer.toHexString(((byTmp[i] & 0xff) + 256))).substring(1));
    		}
        }
        
        return sbEncoded.toString();
	}

	/**
	 * Decodes a hex, utf-8 encoded text to a string.
	 * 
	 * @param pHex the decoded text
	 * @return the plain text
	 * @throws UnsupportedEncodingException if utf-8 encoding is not supported
	 */
	public static String decodeHex(String pHex) throws UnsupportedEncodingException
	{
		if (pHex == null)
		{
			return null;
		}

		return new String(decodeHexAsBytes(pHex), "UTF-8");
	}
	
	/**
	 * Decodes a hex, utf-8 encoded text to a string.
	 * 
	 * @param pHex the decoded text
	 * @return the plain text
	 * @throws UnsupportedEncodingException if utf-8 encoding is not supported
	 */
	public static byte[] decodeHexAsBytes(String pHex) throws UnsupportedEncodingException
	{
		if (pHex == null)
		{
			return null;
		}
		
		byte[] byDecoded = new byte[pHex.length() / 2];
		
		for (int i = 0, j = 0, anz = pHex.length(); i < anz; i += 2, j++)
		{
			byDecoded[j] = (byte)(Integer.parseInt(pHex.substring(i, i + 2), 16));
		}
		
		return byDecoded;
	}

}	// CodecUtil
