/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [HM] - creation
 * 05.10.2008 - [JR] - setBackground: fixed NullPointerException if color parameter is null
 * 04.12.2008 - [JR] - removed getResourceToAdd
 * 18.03.2011 - [JR] - #313: component moved/resized implemented
 * 22.07.2011 - [JR] - get/setBounds: check parent in hierarchy
 */
package com.sibvisions.rad.ui.awt.impl;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.image.BufferedImage;

import javax.rad.ui.IAlignmentConstants;
import javax.rad.ui.IColor;
import javax.rad.ui.IComponent;
import javax.rad.ui.IContainer;
import javax.rad.ui.ICursor;
import javax.rad.ui.IDimension;
import javax.rad.ui.IFactory;
import javax.rad.ui.IFont;
import javax.rad.ui.IImage;
import javax.rad.ui.IPoint;
import javax.rad.ui.IRectangle;
import javax.rad.ui.event.ComponentHandler;
import javax.rad.ui.event.KeyHandler;
import javax.rad.ui.event.MouseHandler;
import javax.rad.ui.event.UIComponentEvent;
import javax.rad.ui.event.UIKeyEvent;
import javax.rad.ui.event.UIMouseEvent;

import com.sibvisions.util.type.ImageUtil;

/**
 * A <em>component</em> is an object having a graphical representation
 * that can be displayed on the screen and that can interact with the
 * user.
 * 
 * @author Martin Handsteiner
 * @param <C> AWT Component implementation
 */
public class AwtComponent<C extends Component> extends AwtResource<C> 
                                               implements IComponent, 
                                                          IAlignmentConstants, 
                                                          MouseListener, 
                                               			  KeyListener,
                                               			  ComponentListener
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Class members
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** Container where the component is added. */
	protected IContainer parent	= null;
	
	/** The factory that created this component. */
	protected IFactory factory = null;
	
	/** The Event Source. */
	protected IComponent eventSource = this;

	/** EventHandler for mouePressed. */
	private MouseHandler eventMousePressed = null;
	/** EventHandler for mouseReleased. */
	private MouseHandler eventMouseReleased = null;
	/** EventHandler for mouseClicked. */
	private MouseHandler eventMouseClicked = null;
	/** EventHandler for mouseEntered. */
	private MouseHandler eventMouseEntered = null;
	/** EventHandler for mouseExited. */
	private MouseHandler eventMouseExited = null;
	
	/** EventHandler for keyPressed. */
	private KeyHandler  eventKeyPressed = null;
	/** EventHandler for keyReleased. */
	private KeyHandler  eventKeyReleased = null;
	/** EventHandler for keyTyped. */
	private KeyHandler  eventKeyTyped = null;
	
	/** EventHandler for componentMoved. */
	protected ComponentHandler eventComponentMoved = null;
	/** EventHandler for componentResized. */
	protected ComponentHandler eventComponentResized = null;
	
	/** X Alignment. */
	private int	horizontalAlignment;
	
	/** Y Alignment. */
	private int	verticalAlignment;

	/** whether the mouse listener was added. */
	private boolean bMouseListener = false;
	/** whether the key listener was added. */
	private boolean bKeyListener = false;
	/** whether the component listener was added. */
	private boolean bComponentListener = false;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Initialization
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Create a new instance of <code>AwtComponent</code>.
	 * 
	 * @param pComponent AWT Component implementation.
	 */
	protected AwtComponent(C pComponent)
	{
		super(pComponent);
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Interface implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public String getName()
	{
		return resource.getName();
	}

	/**
	 * {@inheritDoc}
	 */
	public void setName(String pName)
	{
		resource.setName(pName);
	}

	/**
	 * {@inheritDoc}
	 */
	public IFactory getFactory()
	{
		return factory;
	}

	/**
	 * {@inheritDoc}
	 */
	public int getHorizontalAlignment()
	{
		return horizontalAlignment;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setHorizontalAlignment(int pHorizontalAlignment)
	{
		horizontalAlignment = pHorizontalAlignment;
	}

	/**
	 * {@inheritDoc}
	 */
	public int getVerticalAlignment()
	{
		return verticalAlignment;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setVerticalAlignment(int pVerticalAlignment)
	{
		verticalAlignment = pVerticalAlignment;
	}

	/**
	 * {@inheritDoc}
	 */
	public IDimension getPreferredSize()
	{
		Dimension preferredSize = resource.getPreferredSize();
		if (preferredSize == null)
		{
			return null;
		}
		else
		{
			return new AwtDimension(preferredSize);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void setPreferredSize(IDimension pPreferredSize)
	{
		if (pPreferredSize == null)
		{
			resource.setPreferredSize(null);
		}
		else
		{
			resource.setPreferredSize((Dimension)pPreferredSize.getResource());
		}
		resource.invalidate();
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isPreferredSizeSet()
	{
		return resource.isPreferredSizeSet();
	}

	/**
	 * {@inheritDoc}
	 */
	public IDimension getMinimumSize()
	{
		Dimension minimumSize = resource.getMinimumSize();
		if (minimumSize == null)
		{
			return null;
		}
		else
		{
			return new AwtDimension(minimumSize);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void setMinimumSize(IDimension pMinimumSize)
	{
		if (pMinimumSize == null)
		{
			resource.setMinimumSize(null);
		}
		else
		{
			resource.setMinimumSize((Dimension)pMinimumSize.getResource());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isMinimumSizeSet()
	{
		return resource.isMinimumSizeSet();
	}

	/**
	 * {@inheritDoc}
	 */
	public IDimension getMaximumSize()
	{
		Dimension maximumSize = resource.getMaximumSize();
		if (maximumSize == null)
		{
			return null;
		}
		else
		{
			return new AwtDimension(maximumSize);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void setMaximumSize(IDimension pMaximumSize)
	{
		if (pMaximumSize == null)
		{
			resource.setMaximumSize(null);
		}
		else
		{
			resource.setMaximumSize((Dimension)pMaximumSize.getResource());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isMaximumSizeSet()
	{
		return resource.isMaximumSizeSet();
	}

	/**
	 * {@inheritDoc}
	 */
	public IColor getBackground()
	{
		Color color = resource.getBackground();
		if (color == null)
		{
			return null;
		}
		else
		{
			return new AwtColor(color);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void setBackground(IColor pBackground)
	{
		if (pBackground == null)
		{
			resource.setBackground(null);
		}
		else
		{
			resource.setBackground((Color)pBackground.getResource());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isBackgroundSet()
	{
		return resource.isBackgroundSet();
	}

	/**
	 * {@inheritDoc}
	 */
	public IColor getForeground()
	{
		Color color = resource.getForeground();
		if (color == null)
		{
			return null;
		}
		else
		{
			return new AwtColor(color);
		}
    }

	/**
	 * {@inheritDoc}
	 */
	public void setForeground(IColor pForeground)
	{
		if (pForeground == null)
		{
			resource.setForeground(null);
		}
		else
		{
			resource.setForeground((Color)pForeground.getResource());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isForegroundSet()
	{
		return resource.isForegroundSet();
	}

	/**
	 * {@inheritDoc}
	 */
	public ICursor getCursor()
	{
		Cursor cursor = resource.getCursor();
		if (cursor == null)
		{
			return null;
		}
		else
		{
			return new AwtCursor(cursor);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void setCursor(ICursor pCursor)
	{
		if (pCursor == null)
		{
//			resource.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
			resource.setCursor(null);
		}
		else
		{
			resource.setCursor((Cursor)pCursor.getResource());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isCursorSet()
	{
		return resource.isCursorSet();
	}

	/**
	 * {@inheritDoc}
	 */
	public IFont getFont()
	{
		Font font = resource.getFont();
		if (font == null)
		{
			return null;
		}
		else
		{
			return new AwtFont(font);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void setFont(IFont pFont)
	{
		if (pFont == null) 
		{
			resource.setFont(null);
		}
		else
		{
			resource.setFont((Font)pFont.getResource());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isFontSet()
	{
		return resource.isFontSet();
	}

	/**
	 * {@inheritDoc}
	 */
	public String getToolTipText()
	{
		return null;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setToolTipText(String pText)
	{
		// do nothing
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setFocusable(boolean pFocusable)
	{
		resource.setFocusable(pFocusable);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isFocusable()
	{
		return resource.isFocusable();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void requestFocus()
    {
		resource.requestFocus();
    }

	/**
	 * {@inheritDoc}
	 */
	public IContainer getParent()
	{
		return parent;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setParent(IContainer pParent)
	{
		if (pParent == null)
		{
			if (parent != null && parent.indexOf(this) >= 0)
			{
				throw new IllegalArgumentException("Can't unset parent, because this component is still added!");
			}
		}
		else
		{
			if (pParent.indexOf(this) < 0)
			{
				throw new IllegalArgumentException("Can't set parent, because this component is not added!");
			}
		}
  
		parent = pParent;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isVisible()
	{
		return resource.isVisible();
	}

	/**
	 * {@inheritDoc}
	 */
	public void setVisible(boolean pVisible)
	{
		resource.setVisible(pVisible);
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isEnabled()
	{
		return resource.isEnabled();
	}

	/**
	 * {@inheritDoc}
	 */
	public void setEnabled(boolean pEnable)
	{
		resource.setEnabled(pEnable);
	}

	/**
	 * {@inheritDoc}
	 */
	public IPoint getLocationRelativeTo(IComponent pComponent)
	{
		Point point = resource.getLocation();
		if (pComponent != null)
		{
			Container logicalParent = (Container)pComponent.getResource();
			Container physicalParent = resource.getParent();

			if (logicalParent.isShowing() && logicalParent != physicalParent && physicalParent != null)
			{
				Point logicalLoc = logicalParent.getLocationOnScreen();
				Point physicalLoc = physicalParent.getLocationOnScreen();

				point.x += physicalLoc.x - logicalLoc.x;
				point.y += physicalLoc.y - logicalLoc.y;
			}
		}
		return new AwtPoint(point);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setLocationRelativeTo(IComponent pComponent, IPoint pLocation)
	{
		Point point = (Point)pLocation.getResource();
		if (pComponent != null)
		{
			Container logicalParent = (Container)pComponent.getResource();
			Container physicalParent = resource.getParent();

			if (logicalParent.isShowing() && logicalParent != physicalParent && physicalParent != null)
			{
				Point logicalLoc = logicalParent.getLocationOnScreen();
				Point physicalLoc = physicalParent.getLocationOnScreen();

				point.x -= physicalLoc.x - logicalLoc.x;
				point.y -= physicalLoc.y - logicalLoc.y;
			}
		}
		resource.setLocation(point);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IPoint getLocation()
	{
		return getLocationRelativeTo(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	public void setLocation(IPoint pLocation)
	{
		setLocationRelativeTo(parent, pLocation);
	}

	/**
	 * {@inheritDoc}
	 */
	public IDimension getSize()
	{
		return new AwtDimension(resource.getSize());
	}

	/**
	 * {@inheritDoc}
	 */
	public void setSize(IDimension pSize)
	{
		resource.setSize((Dimension)pSize.getResource());
		
		if (!resource.isShowing())
		{
			validate(resource);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public IRectangle getBounds()
	{
		Rectangle bounds = resource.getBounds();
		if (parent != null)
		{
			Container logicalParent = (Container)parent.getResource();
			Container physicalParent = resource.getParent();

			if (logicalParent.isShowing() && physicalParent != null && logicalParent != physicalParent)
			{
				Point logicalLoc = logicalParent.getLocationOnScreen();
				Point physicalLoc = resource.getParent().getLocationOnScreen();

				bounds.x += physicalLoc.x - logicalLoc.x;
				bounds.y += physicalLoc.y - logicalLoc.y;
			}
		}
		return new AwtRectangle(bounds);
	}

	/**
	 * {@inheritDoc}
	 */
	public void setBounds(IRectangle pBounds)
	{
		Rectangle bounds = (Rectangle)pBounds.getResource();
		if (parent != null)
		{
			Container logicalParent = (Container)parent.getResource();
			Container physicalParent = resource.getParent();

			if (logicalParent.isShowing() && physicalParent != null && logicalParent != physicalParent)
			{
				Point logicalLoc = logicalParent.getLocationOnScreen();
				Point physicalLoc = resource.getParent().getLocationOnScreen();

				bounds.x -= physicalLoc.x - logicalLoc.x;
				bounds.y -= physicalLoc.y - logicalLoc.y;
			}
		}
		resource.setBounds(bounds);

		if (!resource.isShowing())
		{
			validate(resource);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public IComponent getEventSource()
	{
		return eventSource;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setEventSource(IComponent pEventSource)
	{
		eventSource = pEventSource;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public MouseHandler eventMousePressed()
	{
		if (eventMousePressed == null)
		{
			eventMousePressed = new MouseHandler("mousePressed");
			
			addMouseListener();
		}
		return eventMousePressed;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public MouseHandler eventMouseReleased()
	{
		if (eventMouseReleased == null)
		{
			eventMouseReleased = new MouseHandler("mouseReleased");
			
			addMouseListener();
		}
		return eventMouseReleased;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public MouseHandler eventMouseClicked()
	{
		if (eventMouseClicked == null)
		{
			eventMouseClicked = new MouseHandler("mouseClicked");
			
			addMouseListener();
		}
		return eventMouseClicked;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public MouseHandler eventMouseEntered()
	{
		if (eventMouseEntered == null)
		{
			eventMouseEntered = new MouseHandler("mouseEntered");
			
			addMouseListener();
		}
		return eventMouseEntered;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public MouseHandler eventMouseExited()
	{
		if (eventMouseExited == null)
		{
			eventMouseExited = new MouseHandler("mouseExited");
			
			addMouseListener();
		}
		return eventMouseExited;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public KeyHandler eventKeyPressed()
	{
		if (eventKeyPressed == null)
		{
			eventKeyPressed = new KeyHandler("keyPressed");
			
			addKeyListener();
		}
		return eventKeyPressed;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public KeyHandler eventKeyReleased()
	{
		if (eventKeyReleased == null)
		{
			eventKeyReleased = new KeyHandler("keyReleased");

			addKeyListener();
		}
		return eventKeyReleased;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public KeyHandler eventKeyTyped()
	{
		if (eventKeyTyped == null)
		{
			eventKeyTyped = new KeyHandler("keyTyped");
			
			addKeyListener();
		}
		return eventKeyTyped;
	}
	
    /**
	 * {@inheritDoc}
	 */
    public ComponentHandler eventComponentMoved()
    {
		if (eventComponentMoved == null)
		{
			eventComponentMoved = new ComponentHandler("componentMoved");
			
			addComponentListener();
		}
		return eventComponentMoved;
    }
	
    /**
	 * {@inheritDoc}
	 */
    public ComponentHandler eventComponentResized()
    {
		if (eventComponentResized == null)
		{
			eventComponentResized = new ComponentHandler("componentResized");
			
			addComponentListener();
		}
		return eventComponentResized;
    }	
	
	/**
	 * {@inheritDoc}
	 */
	public IImage capture(int pWidth, int pHeight)
	{
		return new AwtImage(createImage(resource, pWidth, pHeight));
	}
	
	/**
	 * {@inheritDoc}
	 */
    public void mousePressed(MouseEvent pMouseEvent)
    {
    	if (eventMousePressed != null)
    	{
    		eventMousePressed.dispatchEvent(new UIMouseEvent(eventSource, 
    													     UIMouseEvent.MOUSE_PRESSED, 
    													     pMouseEvent.getWhen(), 
    													     pMouseEvent.getModifiers(), 
    													     pMouseEvent.getX(), 
    													     pMouseEvent.getY(), 
    													     pMouseEvent.getClickCount(), 
    													     pMouseEvent.isPopupTrigger()));
    	}
    }

	/**
	 * {@inheritDoc}
	 */
    public void mouseReleased(MouseEvent pMouseEvent)
    {
    	if (eventMouseReleased != null)
    	{
    		eventMouseReleased.dispatchEvent(new UIMouseEvent(eventSource, 
    														  UIMouseEvent.MOUSE_RELEASED, 
    														  pMouseEvent.getWhen(), 
    														  pMouseEvent.getModifiers(), 
    														  pMouseEvent.getX(), 
    														  pMouseEvent.getY(), 
    														  pMouseEvent.getClickCount(), 
    														  pMouseEvent.isPopupTrigger()));
    	}
    }

	/**
	 * {@inheritDoc}
	 */
    public void mouseClicked(MouseEvent pMouseEvent)
    {
    	if (eventMouseClicked != null)
    	{
    		eventMouseClicked.dispatchEvent(new UIMouseEvent(eventSource, 
    	  												     UIMouseEvent.MOUSE_CLICKED, 
    	  												     pMouseEvent.getWhen(), 
    	  												     pMouseEvent.getModifiers(), 
    	  												     pMouseEvent.getX(), 
    	  												     pMouseEvent.getY(), 
    	  												     pMouseEvent.getClickCount(), 
    	  												     pMouseEvent.isPopupTrigger()));
    	}
    }

	/**
	 * {@inheritDoc}
	 */
    public void mouseEntered(MouseEvent pMouseEvent)
    {
    	if (eventMouseEntered != null)
    	{
    		eventMouseEntered.dispatchEvent(new UIMouseEvent(eventSource, 
    														 UIMouseEvent.MOUSE_ENTERED, 
    														 pMouseEvent.getWhen(), 
    														 pMouseEvent.getModifiers(), 
    														 pMouseEvent.getX(), 
    														 pMouseEvent.getY(), 
    														 pMouseEvent.getClickCount(), 
    														 pMouseEvent.isPopupTrigger()));
    	}
    }

	/**
	 * {@inheritDoc}
	 */
    public void mouseExited(MouseEvent pMouseEvent)
    {
    	if (eventMouseExited != null)
    	{
    		eventMouseExited.dispatchEvent(new UIMouseEvent(eventSource, 
    													    UIMouseEvent.MOUSE_EXITED, 
    													    pMouseEvent.getWhen(), 
    													    pMouseEvent.getModifiers(), 
    													    pMouseEvent.getX(), 
    													    pMouseEvent.getY(), 
    													    pMouseEvent.getClickCount(), 
    													    pMouseEvent.isPopupTrigger()));
    	}
    }

	/**
	 * {@inheritDoc}
	 */
    public void keyPressed(KeyEvent pKeyEvent)
    {
    	if (eventKeyPressed != null)
    	{
    		eventKeyPressed.dispatchEvent(new UIKeyEvent(eventSource, 
    													 UIKeyEvent.KEY_PRESSED, 
    													 pKeyEvent.getWhen(), 
    													 pKeyEvent.getModifiers(), 
    													 pKeyEvent.getKeyCode(), 
    													 pKeyEvent.getKeyChar()));
    	}
    }

	/**
	 * {@inheritDoc}
	 */
    public void keyReleased(KeyEvent pKeyEvent)
    {
    	if (eventKeyReleased != null)
    	{
    		eventKeyReleased.dispatchEvent(new UIKeyEvent(eventSource, 
     													  UIKeyEvent.KEY_RELEASED, 
     													  pKeyEvent.getWhen(), 
     													  pKeyEvent.getModifiers(), 
    													  pKeyEvent.getKeyCode(), 
    													  pKeyEvent.getKeyChar()));
    	}
    }
    
	/**
	 * {@inheritDoc}
	 */
    public void keyTyped(KeyEvent pKeyEvent)
    {
    	if (eventKeyTyped != null)
    	{
    		eventKeyTyped.dispatchEvent(new UIKeyEvent(eventSource, 
   													   UIKeyEvent.KEY_TYPED, 
   													   pKeyEvent.getWhen(), 
   													   pKeyEvent.getModifiers(), 
   													   pKeyEvent.getKeyCode(), 
   													   pKeyEvent.getKeyChar()));
    	}
    }
    
	/**
	 * {@inheritDoc}
	 */
	public void componentHidden(ComponentEvent e)
	{
	}

	/**
	 * {@inheritDoc}
	 */
	public void componentShown(ComponentEvent e)
	{
	}

	/**
	 * {@inheritDoc}
	 */
	public void componentMoved(ComponentEvent e)
	{
    	if (eventComponentMoved != null)
    	{
    		eventComponentMoved.dispatchEvent(new UIComponentEvent(eventSource, 
    															   UIComponentEvent.COMPONENT_MOVED, 
    															   System.currentTimeMillis(),
    															   0));
    	}
	}

	/**
	 * {@inheritDoc}
	 */
	public void componentResized(ComponentEvent e)
	{
    	if (eventComponentResized != null)
    	{
    		eventComponentResized.dispatchEvent(new UIComponentEvent(eventSource, 
    															     UIComponentEvent.COMPONENT_RESIZED, 
    															     System.currentTimeMillis(),
    															     0));
    	}
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * Sets the components factory implementation.
     * 
     * @param pFactory the {@link IFactory}
     * @see AwtFactory
     */
	public void setFactory(IFactory pFactory)
	{
		factory = pFactory;
	}
	
	/**
	 * Adds a mouse listener for this component, if not already added.
	 */
	private void addMouseListener()
	{
		if (!bMouseListener)
		{
			bMouseListener = true;
			resource.addMouseListener(this);
		}
	}
	
	/**
	 * Adds a key listener for this component, if not already added.
	 */
	private void addKeyListener()
	{
		if (!bKeyListener)
		{
			bKeyListener = true;
			resource.addKeyListener(this);
		}
	}
    
	/**
	 * Adds a component listener for this component, if not already added.
	 */
	private void addComponentListener()
	{
		if (!bComponentListener)
		{
			bComponentListener = true;
			resource.addComponentListener(this);
		}
	}

	/**
	 * Layouts the component.
	 * @param pComponent the component.
	 */
	public void validate(Component pComponent)
	{
		pComponent.doLayout();
		
		if (pComponent instanceof Container)
		{
			Container cont = (Container)pComponent;
			
			for (int i = 0; i < cont.getComponentCount(); i++)
			{
				validate(cont.getComponent(i));
			}
		}
	}
	
	/**
	 * Creates an image from a component.
	 * 
	 * @param pComponent the component
	 * @param pWidth the expected image width
	 * @param pHeight the expected image height
	 * @return the component image
	 */
	protected Image createImage(Component pComponent, int pWidth, int pHeight)
	{
		BufferedImage image = new BufferedImage(pComponent.getWidth(), pComponent.getHeight(), BufferedImage.BITMASK);
		
		pComponent.validate();
		pComponent.paint(image.getGraphics());

		return ImageUtil.getScaledImage(image, pWidth, pHeight, false);
	}
	
}	// AwtComponent
