/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 08.05.2009 - [JR] - creation
 * 04.10.2009 - [JR] - setProperty: security for CLIENT properties
 * 18.11.2009 - [JR] - #33: WrappedSession: put implemented
 * 02.03.2011 - [JR] - #297: implemented addObject, removeObject, getObject and renamed addObject to putObject
 */
package com.sibvisions.rad.server;

import java.util.Hashtable;

import javax.rad.remote.IConnection;
import javax.rad.remote.IConnectionConstants;
import javax.rad.server.IConfiguration;
import javax.rad.server.IServer;
import javax.rad.server.ISession;
import javax.rad.server.InjectObject;
import javax.rad.server.SessionContext;

/**
 * The <code>SessionContextImpl</code> is an internal {@link SessionContext} implementation for
 * the {@link AbstractSession}.
 * 
 * @author Ren Jahn
 */
final class SessionContextImpl extends SessionContext
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the associated session. */
	private WrappedSession wsessCurrent;
	
	/** the name of the object from which the method will be called. */
	private String sObjectName = null;
	
	/** the name of the method which will be called. */
	private String sMethodName = null;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of <code>SessionContextImpl</code> for a specific
	 * {@link ISession}.
	 * 
	 * @param pSession the associated session for this {@link SessionContext}
	 */
	SessionContextImpl(AbstractSession pSession)
	{
		wsessCurrent = new WrappedSession(pSession);
		
		setCurrentInstance(this);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Abstract methods implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void destroy()
	{
		setCurrentInstance(null);

		wsessCurrent = null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ISession getSession()
	{
		return wsessCurrent;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public IConfiguration getSessionConfig()
	{
		if (wsessCurrent == null)
		{
			return null;
		}
		
		return wsessCurrent.getConfig();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IConfiguration getServerConfig()
	{
		if (wsessCurrent == null)
		{
			return null;
		}
		
		return wsessCurrent.getServerConfig();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public IConnection getServerConnection()
	{
		if (wsessCurrent == null)
		{
			return null;
		}
		
		return new DirectServerConnection((IDirectServer)wsessCurrent.getServer());
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getObjectName()
	{
		return sObjectName;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMethodName()
	{
		return sMethodName;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public InjectObject putObject(InjectObject pObject)
	{
		return wsessCurrent.session.putObject(pObject);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public InjectObject removeObject(InjectObject pObject)
	{
		return wsessCurrent.session.removeObject(pObject);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public InjectObject getObject(String pName)
	{
		return wsessCurrent.session.getObject(pName);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // User-defined methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Sets the name of the current object from which the method will be called.
	 * 
	 * @param pObjectName the object name or <code>null</code> if the object is unknown
	 * @see #getObjectName()
	 */
	void setObjectName(String pObjectName)
	{
		sObjectName = pObjectName;
	}
	
	/**
	 * Sets the name of the method which will be called.
	 * 
	 * @param pMethodName the method name or <code>null</code> if the method is unknown
	 * @see #getMethodName()
	 */
	void setMethodName(String pMethodName)
	{
		sMethodName = pMethodName;
	}
	
	//****************************************************************
	// Subclass definition
	//****************************************************************
	
	/**
	 * The <code>WrappedSession</code> allows restricted access to the 
	 * {@link AbstractSession}.
	 * 
	 * @author Ren Jahn
	 */
	private static final class WrappedSession implements ISession
	{
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Class members
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/** the "hidden" session. */
		private final AbstractSession session;
		
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Initialization
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		
		/**
		 * Creates a new instance of <code>WrappedSession</code> for an {@link ISession}.
		 * 
		 * @param pSession an {@link ISession} implementation
		 */
		private WrappedSession(AbstractSession pSession)
		{
			session = pSession;
		}
		
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Interface implementation
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		
		/**
		 * {@inheritDoc}
		 */
		public Object getId()
		{
			return session.getId();
		}

		/**
		 * {@inheritDoc}
		 */
		public String getLifeCycleName()
		{
			return session.getLifeCycleName();
		}
		
		/**
		 * {@inheritDoc}
		 */
		public String getApplicationName()
		{
			return session.getApplicationName();
		}
		
		/**
		 * {@inheritDoc}
		 */
		public String getUserName()
		{
			return session.getUserName();
		}

		/**
		 * {@inheritDoc}
		 */
		public String getPassword()
		{
			return session.getPassword();
		}
		
		/**
		 * {@inheritDoc}
		 */
		public Object getProperty(String pName)
		{
			return session.getProperty(pName);
		}
		
		/**
		 * {@inheritDoc}
		 */
		public Hashtable<String, Object> getProperties()
		{
			return (Hashtable<String, Object>)session.getProperties().clone();
		}
		
		/**
		 * {@inheritDoc}
		 */
		public void setProperty(String pName, Object pValue)
		{
			if (pName != null && pName.startsWith(IConnectionConstants.PREFIX_CLIENT))
			{
				throw new SecurityException("It's not allowed to change client properties through the SessionContext!");
			}
			
			session.setProperty(pName, pValue);
		}

		/**
		 * {@inheritDoc}
		 */
		public long getStartTime()
		{
			return session.getStartTime();
		}

		/**
		 * {@inheritDoc}
		 */
		public long getLastAccessTime()
		{
			return session.getLastAccessTime();
		}

		/**
		 * {@inheritDoc}
		 */
		public long getLastAliveTime()
		{
			return session.getLastAliveTime();
		}

		/**
		 * {@inheritDoc}
		 */
		public void setAliveInterval(long pAliveInterval)
		{
			session.setAliveInterval(pAliveInterval);
		}

		/**
		 * {@inheritDoc}
		 */
		public long getAliveInterval()
		{
			return session.getAliveInterval();
		}

		/**
		 * {@inheritDoc}
		 */
		public boolean isAlive(long pAccessTime)
		{
			return session.isAlive(pAccessTime);
		}

		/**
		 * {@inheritDoc}
		 */
		public void setMaxInactiveInterval(int pMaxInactiveInterval)
		{
			session.setMaxInactiveInterval(pMaxInactiveInterval);
		}
		
		/**
		 * {@inheritDoc}
		 */
		public int getMaxInactiveInterval()
		{
			return session.getMaxInactiveInterval();
		}

		/**
		 * {@inheritDoc}
		 */
		public boolean isInactive(long pAccessTime)
		{
			return session.isInactive(pAccessTime);
		}

		/**
		 * {@inheritDoc}
		 */
		public Object call(String pObjectName, String pMethod, Object... pParams) throws Throwable
		{
			return session.call(pObjectName, pMethod, pParams);
		}
		
		/**
		 * {@inheritDoc}
		 */
		public Object callAction(String pAction, Object... pParams) throws Throwable
		{
			return session.callAction(pAction, pParams);
		}
		
		/**
		 * {@inheritDoc}
		 */
		public Object get(String pObjectName) throws Throwable
		{
			return session.get(pObjectName);
		}
		
		/**
		 * {@inheritDoc}
		 */
		public Object put(String pObjectName, Object pObject) throws Throwable
		{
			return session.put(pObjectName, pObject);
		}

		/**
		 * {@inheritDoc}
		 */
		public IConfiguration getConfig()
		{
			return session.getApplicationZone().getConfig();
		}
		
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// User-defined methods
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
		 * Gets the server configuration.
		 * 
		 * @return the server configuration
		 */
		private IConfiguration getServerConfig()
		{
			return session.getApplicationZone().getServerZone().getConfig();
		}
		
		/**
		 * Gets the referenced server.
		 * 
		 * @return the server
		 */
		private IServer getServer()
		{
			return session.getSessionManager().getServer();
		}

	}	// WrappedSession
	
}	// SessionContextImpl
