/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 07.11.2008 - [JR] - creation
 * 12.02.2009 - [JR] - get: checked InvokationTarget via Reflective.getCause
 * 15.02.2009 - [JR] - extended from Bean and rewritten invoke (only call declared methods)
 *                   - changed recursion detection
 *                   - removed already implemented methods (Bean suppport)
 * 05.03.2010 - [JR] - invoke: support superclass calls
 * 22.10.2010 - [JR] - invoke: stop on superclass Object.class
 * 25.05.2011 - [JR] - #363: implemented ILifeCycleObject
 *                   - initBeanType: created PropertyDefinition with type-class
 * 22.11.2012 - [JR] - #608: ensure call hierarchy (fallback)
 */
package com.sibvisions.rad.server;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Hashtable;

import javax.rad.type.AbstractType;
import javax.rad.type.bean.Bean;
import javax.rad.type.bean.BeanType;
import javax.rad.type.bean.PropertyDefinition;

import com.sibvisions.util.Reflective;
import com.sibvisions.util.type.StringUtil;

/**
 * The <code>GenericBean</code> handles the access to the values
 * of cached members and defined methods. If a member is uninitialized
 * the <code>GenericBean</code> will try to initialize it by calling
 * the init method for the member.<p/>
 * 
 * Examples for using implementing a GenericBean subclass<p/>
 * 
 * The fastest and safest way to use the GenericBean is to implement an init method 
 * for every property. If you have init methods you don't have to implement the get
 * methods, but it's good style to implement both: 
 * <pre>
 * public class Session extends GenericBean
 * {
 *     private DBAccess initDataSource() throws Exception
 *     {
 *         IConfiguration cfgSession = session.getCurrentSessionConfig();
 *         
 *         OracleDBAccess dba = new OracleDBAccess();
 *         
 *         dba.setConnection(cfgSession.getProperty("/application/securitymanager/database/url")); 
 *         dba.setUser(cfgSession.getProperty("/application/securitymanager/database/username"));
 *         dba.setPassword(cfgSession.getProperty("/application/securitymanager/database/password"));
 *         dba.open();
 *         
 *         return dba;
 *     }
 *     
 *     private DBStorage initPerson() throws Exception
 *     {
 *         DBStorage dbsPerson = new DBStorage();
 *        
 *         dbsPerson.setDBAccess(getDataSource());
 *         dbsPerson.setWriteBackTable("V_PERSON");
 *         dbsPerson.setFromClause("V_PERSON");
 *         dbsPerson.open();
 *        
 *         return dbsPerson;
 *     }
 *     
 *     public DBAccess getDataSource()
 *     {
 *         return (DBAccess)get("dataSource");
 *     }
 *     
 *     public DBStorage getPerson()
 *     {
 *         return (DBStorage)get("person");
 *     }
 * }
 * </pre>
 * It's also possible to integrate the initialization into the get method, thats recommended. The 
 * disadvantage of this implementation is that more calls will be made (That's the result of avoiding
 * recursive calls, because getPerson calls get("person") and this calls getPerson again), but you 
 * have the same flexibility as above and you have only one method where your object will be accessed:
 * <pre>
 * public class Session extends GenericBean
 * {
 *     public DBAccess getDataSource() throws Exception
 *     {
 *         OracleDBAccess dba = (OracleDBAccess)get("dataSource");
 *         
 *         if (dba == null)
 *         {
 *             IConfiguration cfgSession = session.getCurrentSessionConfig();
 *             
 *             dba = new OracleDBAccess();
 *         
 *             dba.setConnection(cfgSession.getProperty("/application/securitymanager/database/url")); 
 *             dba.setUser(cfgSession.getProperty("/application/securitymanager/database/username"));
 *             dba.setPassword(cfgSession.getProperty("/application/securitymanager/database/password"));
 *             dba.open();
 *         }
 *         
 *         return dba;
 *     }
 *     
 *     public DBStorage getPerson() throws Exception
 *     {
 *         DBStorage dbsPerson = (DBStorage)get("person");
 *        
 *         if (dbsPerson == null)
 *         {
 *             dbsPerson = new DBStorage();
 *            
 *             dbsPerson.setDBAccess(getDataSource());
 *             dbsPerson.setWriteBackTable("V_PERSON");
 *             dbsPerson.setFromClause("V_PERSON");
 *             dbsPerson.open();
 *         }
 *        
 *         return dbsPerson;
 *     }
 * }
 * </pre>
 * The EJB like implementation looks like the following:
 * <pre>
 * public class Session extends GenericBean
 * {
 *     private DBAccess dba;
 *     
 *     private DBStorage dbsPerson;
 *     
 * 
 *     public DBAccess getDataSource() throws Exception
 *     {
 *         if (dba == null)
 *         {
 *             IConfiguration cfgSession = session.getCurrentSessionConfig();
 *             
 *             dba = new OracleDBAccess();
 *         
 *             dba.setConnection(cfgSession.getProperty("/application/securitymanager/database/url")); 
 *             dba.setUser(cfgSession.getProperty("/application/securitymanager/database/username"));
 *             dba.setPassword(cfgSession.getProperty("/application/securitymanager/database/password"));
 *             dba.open();
 *             
 *             put("dataSource", dba);
 *         }
 *         
 *         return dba;
 *     }
 *     
 *     public DBStorage getPerson() throws Exception
 *     {
 *         if (dbsPerson == null)
 *         {
 *             dbsPerson = new DBStorage();
 *            
 *             dbsPerson.setDBAccess(getDataSource());
 *             dbsPerson.setWriteBackTable("V_PERSON");
 *             dbsPerson.setFromClause("V_PERSON");
 *             dbsPerson.open();
 *         }
 *        
 *         return dbsPerson;
 *     }
 * }
 * </pre>
 * The problem with above implementation is that the objects won't be managed from the expected GenericBean,
 * if use extends from another GenericBean implementation like Session. That's the case because the extended
 * class inherits all methods from the super class and all objects will be created in the inherited class if
 * you call a method. But the objects from the super class should be stored in the super class instance!<br/>
 * <b>We recommend to use the second or first implementation mechanism!</b> <p/>
 * 
 * It's also possible to ignore lazy loading and generic object access. When you call 
 * get("person") you will get another object as dba, when you didn't put the object. 
 * And with this solutions you get the exception before using the object and that's not 
 * always the right place.<br>
 * You can use one of the following mechanism:
 * <pre>
 * public class Session extends GenericBean
 * {
 *     private DBAccess dba = createDataSource();
 *     
 *     private DBStorage dbsPerson = createPerson();
 *     
 * 
 *     public Session() throws Exception
 *     {
 *         //important because the create methods throws Exceptions
 *     }
 * 
 *     //dont set the name to initDataSource, unless you put(...) the instance, because thats the name 
 *     //of an automatic called method
 *     private DBAccess createDataSource() throws Exception
 *     {
 *         IConfiguration cfgSession = session.getCurrentSessionConfig();
 *         
 *         OracleDBAccess dba = new OracleDBAccess();
 *         
 *         dba.setConnection(cfgSession.getProperty("/application/securitymanager/database/url")); 
 *         dba.setUser(cfgSession.getProperty("/application/securitymanager/database/username"));
 *         dba.setPassword(cfgSession.getProperty("/application/securitymanager/database/password"));
 *         dba.open();
 *         
 *         //with this call you can set the method name to initDataSource and you can use get("dataSource")
 *         //and getDataSource without problems
 *         put("dataSource", dba);
 *         
 *         return dba;
 *     }
 *     
 *     //dont set the name to initDataSource, unless you put(...) the instance, because thats the name 
 *     //of an automatic called method
 *     private DBStorage createPerson() throws Exception
 *     {
 *         DBStorage dbsPerson = new DBStorage();
 *        
 *         dbsPerson.setDBAccess(getDataSource());
 *         dbsPerson.setWriteBackTable("V_PERSON");
 *         dbsPerson.setFromClause("V_PERSON");
 *         dbsPerson.open();
 *        
 *         //with this call you can set the method name to initPerson and you can use get("person")
 *         //and getPerson without problems
 *         put("person", dbsPerson);
 *        
 *         return dbsPerson;
 *     }
 *     
 *     public DBAccess getDataSource()
 *     {
 *         return dba;
 *     }
 *     
 *     public DBStorage getPerson()
 *     {
 *         return dbsPerson;
 *     }
 * }
 * </pre>
 * Another way is:
 * <pre>
 * public class Session extends GenericBean
 * {
 *     private DBAccess dba;
 *     
 *     private DBStorage dbsPerson;
 *     
 * 
 *     public Session() throws Exception
 *     {
 *         IConfiguration cfgSession = session.getCurrentSessionConfig();
 *         
 *         OracleDBAccess dba = new OracleDBAccess();
 *         
 *         dba.setConnection(cfgSession.getProperty("/application/securitymanager/database/url")); 
 *         dba.setUser(cfgSession.getProperty("/application/securitymanager/database/username"));
 *         dba.setPassword(cfgSession.getProperty("/application/securitymanager/database/password"));
 *         dba.open();
 *         
 *         DBStorage dbsPerson = new DBStorage();
 *        
 *         dbsPerson.setDBAccess(getDataSource());
 *         dbsPerson.setWriteBackTable("V_PERSON");
 *         dbsPerson.setFromClause("V_PERSON");
 *         dbsPerson.open();
 *     }
 *     
 *     public DBAccess getDataSource()
 *     {
 *         return dba;
 *     }
 *     
 *     public DBStorage getPerson()
 *     {
 *         return dbsPerson;
 *     }
 * }
 * </pre>
 * 
 * @author Ren Jahn
 */
public abstract class GenericBean extends Bean
                                  implements ILifeCycleObject
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the parent bean. */
	private Bean parent = null;

	/** the bean type for properties that are defined in super classes. */
	private BeanType beanTypeSuperClasses = new BeanType();
	
	/** the binding between properties and classes. */
	private Hashtable<PropertyDefinition, Class<?>> htProperties = new Hashtable<PropertyDefinition, Class<?>>();

	/** the property access for recursive call detection. */
	private Hashtable<BeanType, boolean[]> htPropertyAccess = new Hashtable<BeanType, boolean[]>();
	
	/** whether the superclass check was done. */
	private boolean bSuperClassChecked = false;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of <code>GenericBean</code> without a parent.
	 */
	public GenericBean()
	{
		super();
	
		initBeanType(beanType, getClass());
	}
	
	/**
	 * Initalizes the bean type with the property definition of this instance.
	 * 
	 * @param pType the bean type
	 * @param pClass the class
	 */
	private void initBeanType(BeanType pType, Class<?> pClass)
	{
		Method[] methods = pClass.getDeclaredMethods();
		
		String sName;
		
		boolean bGet;
		
		for (Method met : methods)
		{
			if (checkMethod(met))
			{
				sName = met.getName();
				
				bGet = sName.startsWith("get"); 
				
				if ((bGet && sName.length() > 3)
					|| (sName.startsWith("init") && sName.length() > 4))
				{
					if (bGet)
					{
						sName = sName.substring(3);
					}
					else
					{
						sName = sName.substring(4);
					}
					
					sName = StringUtil.formatMemberName(sName);
					
					if (pType.getPropertyIndex(sName) < 0)
					{
						PropertyDefinition propdef = new PropertyDefinition(sName, AbstractType.getTypeFromClass(met.getReturnType()), met.getReturnType());
						
						pType.addPropertyDefinition(propdef);
						htProperties.put(propdef, pClass);
					}
				}
			}
		}
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public void destroy()
	{
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean containsKey(Object pKey)
	{
		if (pKey == null)
		{
			return false;
		}
		
		boolean bFound = super.containsKey(pKey);
		
		if (!bFound)
		{
			if (parent != null)
			{
				return parent.containsKey(pKey);
			}
		}
		
		return bFound;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean containsValue(Object pValue)
	{
		if (pValue == null)
		{
			return false;
		}

		boolean bFound = super.containsValue(pValue);
		
		if (parent != null)
		{
			return parent.containsValue(pValue);
		}
		
		return bFound;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object get(String pName)
	{
		int index = beanType.getPropertyIndex(pName);
		
		//It is possible that a method is available in the parent 
		//but also in the super class beantype (because of the class hierarchy)
		if (index < 0)
		{
			Object oParent;

			boolean bCheckSuperClassMethods = false;			
			
			if (parent != null)
			{
				oParent = parent.get(pName);
				
				if (parent.containsKey(pName))
				{
					return oParent;
				}
				else
				{
					//the property is not in this class and not in the parent hierarchy
					bCheckSuperClassMethods = true;
				}
			}
			else
			{
				bCheckSuperClassMethods = true;
				
				oParent = null;
			}
			
			if (bCheckSuperClassMethods)
			{
				index = beanTypeSuperClasses.getPropertyIndex(pName);
				
				if (index < 0)
				{
					//superclass check?
					//Possible if eg. a MasterConnection uses the LCO of a SubConnection
					//In that case, the Session object is not set as parent because the MasterConnection
					//gets the Application as parent and in that case, initXxx method won't be called
					//if get("xxx") will be invoked.
					//To support this, we check if one of our superclasses
					if (!bSuperClassChecked)
					{
						//not found in parent and not in "this" class, but maybe the method was defined
						//in a superclass of "this" class
						Class<?> clazz = getClass().getSuperclass();
						
						while (clazz != null && clazz != GenericBean.class)
						{
							initBeanType(beanTypeSuperClasses, clazz);
							
							clazz = clazz.getSuperclass();
						}
		
						bSuperClassChecked = true;
						
						index = beanTypeSuperClasses.getPropertyIndex(pName);
		
						if (index < 0)
						{
							return null;
						}
					}
					else
					{
						return null;
					}
				}
				
				//Invocation of methods from superclasses only are possible if this call is a sub call from another (allowed)call
				//It should not be possible to invoke methods that are not defined in the LCO itself!
				boolean bFirstGet = true;

				boolean[] baDefault = htPropertyAccess.get(beanType);
				
				if (baDefault != null)
				{
					for (int i = 0; i < baDefault.length && bFirstGet; i++)
					{
						bFirstGet = !baDefault[i];
					}
				}
				
				if (bFirstGet)
				{
					return null;
				}			
				
				return get(beanTypeSuperClasses, index);
			}
			else
			{
				return null;
			}
		}
		
		return get(beanType, index);
	}
	
	/**
	 * Gets the value for a cached member variable. If the value for the 
	 * member is not cached, it will be created with following rules:
	 * <ul>
	 *   <li>call the init&lt;membername&gt; method</li>
	 *   <li>call the get&lt;membername&gt; method</li>
	 *   <li>delegate to the parent, if available</li>
	 * </ul>
	 * 
	 * @param pIndex the index of the property from the bean type
	 * @return the cached or created value; <code>null</code> if it's not possible
	 *         to create a value
	 * @throws RuntimeException if an error occurs during object creation
	 */
	@Override
	public Object get(int pIndex) 
	{
		return get(beanType, pIndex);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Sets the parent bean for this bean.
	 * 
	 * @param pParent the parent bean
	 */
	public void setParent(Bean pParent)
	{
		parent = pParent;
	}
	
	/**
	 * Gets the parent, if set.
	 * 
	 * @return the parent or <code>null</code> if not set
	 */
	public Bean getParent()
	{
		return parent;
	}
	
	/**
	 * Invokes a method of this object via reflective call. If the method is not declared,
	 * te invocation will be delegated to the parent, if available.
	 * 
	 * @param pMethod the method name
	 * @param pParams the params for the method
	 * @return the return value of the method or <code>null</code> if the method
	 *         doesn't return a value
	 * @throws RuntimeException if the desired method is not available or the method throws
	 *                          an erroror during execution
	 */
	public Object invoke(String pMethod, Object... pParams)
	{
		try
		{
			try
			{
				//if this object has no parent -> search all methods and not only declared
				return Reflective.call(this, parent != null, pMethod, pParams);
			}
			catch (NoSuchMethodException nsme)
			{
				//try only, if the parent is set, because the above call searches all methods if the parent
				//is not set
				if (parent != null)
				{
					Class<?> clsSuper = getClass().getSuperclass();
					Class<?> clsParent = parent.getClass();
					
					while (clsSuper != Object.class && clsSuper != clsParent)
					{
						try
						{
							return Reflective.call(this, clsSuper, true, pMethod, pParams);
						}
						catch (NoSuchMethodException nsmex)
						{
							//try the next superclass 
							clsSuper = clsSuper.getSuperclass();
						}
					}
					
					//last option: delegate to the parent
					if (parent instanceof GenericBean)
					{
						return ((GenericBean)parent).invoke(pMethod, pParams);
					}
				}
				
				throw nsme;
			}
		}
		catch (Throwable th)
		{
			if (th instanceof InvocationTargetException)
			{
				th = ((InvocationTargetException)th).getCause();
			}
			
			if (th instanceof RuntimeException)
			{
				throw (RuntimeException)th;
			}
			else
			{
				throw new RuntimeException(th);
			}
		}
	}
	
	/**
	 * Gets the value for a cached member variable. If the value for the 
	 * member is not cached, it will be created with following rules:
	 * <ul>
	 *   <li>call the init&lt;membername&gt; method</li>
	 *   <li>call the get&lt;membername&gt; method</li>
	 *   <li>delegate to the parent, if available</li>
	 * </ul>
	 * 
	 * @param pBeanType the bean type
	 * @param pIndex the index of the property from the <code>pBeanType</code>
	 * @return the cached or created value; <code>null</code> if it's not possible
	 *         to create a value
	 * @throws RuntimeException if an error occurs during object creation
	 */
	private Object get(BeanType pBeanType, int pIndex)
	{
		boolean[] baCheck = htPropertyAccess.get(pBeanType);
		
		if (baCheck == null)
		{
			baCheck = new boolean[pIndex + 1];
			
			htPropertyAccess.put(pBeanType, baCheck);
		}
		else if (pIndex >= baCheck.length)
		{
			boolean[] baCopy = new boolean[pBeanType.getPropertyCount()];
			
			System.arraycopy(baCheck, 0, baCopy, 0, baCheck.length);
			
			baCheck = baCopy;
			
			htPropertyAccess.put(pBeanType, baCheck);
		}
		
		//avoid recursive calls with the same object name 
		if (baCheck[pIndex])
		{
			return null;
		}
		
		Object oValue = super.get(pIndex);
		
		if (oValue != null)
		{
			return oValue;
		}
		
		try
		{
			baCheck[pIndex] = true;
			
			PropertyDefinition propdef = pBeanType.getPropertyDefinition(pIndex); 
			
			String sPropertyName = propdef.getName();
			
			Class<?> clazz = htProperties.get(propdef);
			
			//first step: try init method
			Method method = getMethod(clazz, "init", sPropertyName);
			
			if (checkMethod(method))
			{
				oValue = Reflective.invoke(this, method);
				
				put(sPropertyName, oValue);
				
				return oValue;
			}
			
			//second step: try get method
			method = getMethod(clazz, "get", sPropertyName);
			
			if (checkMethod(method))
			{
				oValue = method.invoke(this);
				
				put(sPropertyName, oValue);
				
				return oValue;
			}
			
			//third step: delegate to the parent, if possible
			if (parent != null)
			{
				return parent.get(sPropertyName);
			}
			
			return null;
		}
		catch (Throwable th)
		{
			if (th instanceof InvocationTargetException)
			{
				th = ((InvocationTargetException)th).getCause();
			}
			
			if (th instanceof RuntimeException)
			{
				throw (RuntimeException)th;
			}
			else
			{
				throw new RuntimeException(th);
			}
		}
		finally
		{
			baCheck[pIndex] = false;
		}
	}
	
	/**
	 * Gets a method for a member variable.
	 * 
	 * @param pClass the class that contains the method
	 * @param pPrefix the method prefix (init, get, ...)
	 * @param pName the name of the member variable
	 * @return the method or <code>null</code> if the method is not available
	 */
	private Method getMethod(Class<?> pClass, String pPrefix, String pName)
	{
		try
		{
			Class<?> clazz = pClass;
			
			if (clazz == null)
			{
				clazz = getClass();
			}
			
			return clazz.getDeclaredMethod(StringUtil.formatMethodName(pPrefix, pName));
		}
		catch (NoSuchMethodException nsme)
		{
			return null;
		}
	}
	
	/**
	 * Checks if a method is a valid method to get/init members. A method is valid
	 * if it is not static and it has no parameters.
	 * 
	 * @param pMethod the method to check
	 * @return <code>true</code> if the method is a valid member method
	 */
	private boolean checkMethod(Method pMethod)
	{
		if (pMethod == null)
		{
			return false;
		}
		
		int iModifiers = pMethod.getModifiers();
		
		return !Modifier.isStatic(iModifiers) 
		       && (pMethod.getParameterTypes() == null || pMethod.getParameterTypes().length == 0);
		
	}
	
}	// GenericBean
