/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [JR] - creation
 * 10.02.2009 - [JR] - getClassNameMapping: added full qualified class name to mapping [BUGFIX]
 * 11.02.2009 - [JR] - createObject: AExchangeManager included
 * 12.02.2009 - [JR] - TYPE_SESSION splitted to TYPE_SUBSESSION and TYPE_MASTERSESSION
 * 10.05.2009 - [JR] - getObject: support for dot separated object names
 * 18.11.2009 - [JR] - #33: putObject implemented
 * 27.01.2010 - [JR] - invoke implemented
 * 15.02.2010 - [JR] - createInstance: checked GenericBean instance
 * 22.02.2010 - [JR] - #67: getObject: Map support
 *                   - getObject: object not found throws an exception [BUGFIX]
 * 05.03.2010 - [JR] - invoke: user friendly message when the object is null
 * 23.03.2010 - [JR] - #103: getObject now sets the object name and the method
 * 24.03.2010 - [JR] - #105: putObject: dot notation support
 *                   - Map instead of Bean
 * 22.12.2010 - [JR] - initApplicationObject: check ClassNotFoundException to allow missing application objects
 * 25.12.2010 - [JR] - removed final from class 
 * 01.03.2011 - [JR] - initSessionObject: injectObjects after caching the object 
 *                                        (allows LCO access for inject objects constructor)
 * 02.03.2011 - [JR] - #297: updateSessionObject (put object even if it exists -> support object changing)
 * 25.05.2011 - [JR] - getSessionObjectInternal implemented  
 * 26.05.2011 - [JR] - #363: ILifeCycleObject handling  
 * 21.11.2012 - [JR] - #535: check object and method access via IObjectAccessProvider                                          
 */
package com.sibvisions.rad.server;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.rad.server.AbstractObjectProvider;
import javax.rad.server.ISession;
import javax.rad.server.InjectObject;
import javax.rad.server.SessionContext;
import javax.rad.server.event.ISessionListener;
import javax.rad.type.bean.Bean;

import com.sibvisions.rad.server.config.Configuration;
import com.sibvisions.rad.server.security.IObjectAccessController;
import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.Reflective;
import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.LoggerFactory;
import com.sibvisions.util.type.StringUtil;

/**
 * The <code>DefaultObjectProvider</code> manages the remote accessible objects. It compiles
 * source files and offers always the current object.
 * 
 * @author Ren Jahn
 */
public class DefaultObjectProvider extends AbstractObjectProvider
                                   implements ISessionListener
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** the logger instance. */
	private ILogger log = LoggerFactory.getInstance(getClass());
	
	/** the cache for application life cycle objects. */
	private Hashtable<String, Map> htApplicationObjects = null;
	
	/** the cache for session life cycle objects. */
	private Hashtable<Long, Map> htSessionObjects = null;
	
	/** the object access controller. */
	private IObjectAccessController oaController = null;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Creates an instance of <code>AbstractObjectProvider</code>.
	 * 
	 * @param pServer communication server
	 */
	protected DefaultObjectProvider(Server pServer)
	{
		super(pServer);
		
		pServer.getSessionManager().addSessionListener(this);

		//#535
		try
		{
			String sObjectAccess = Configuration.getServerZone().getProperty("/server/objectprovider/accesscontroller");
		
			if (sObjectAccess != null)
			{
				oaController = (IObjectAccessController)Reflective.construct(sObjectAccess);

				log.debug("Use ", sObjectAccess, " as ObjectAccessController");
			}
		}
		catch (Throwable th)
		{
			log.debug("Can't use configured ObjectAccessController!", th);
			
			oaController = null;
		}
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public void sessionCreated(ISession pSession)
	{
	}

	/**
	 * {@inheritDoc}
	 */
	public void sessionDestroyed(ISession pSession)
	{
		if (htSessionObjects != null)
		{
			Map map = htSessionObjects.remove(((AbstractSession)pSession).getObjectId());
			
			if (map instanceof ILifeCycleObject)
			{
				try
				{
					((ILifeCycleObject)map).destroy();
				}
				catch (Throwable th)
				{
					log.debug(th);
				}
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized Object getObject(ISession pSession, String pObjectName) throws Throwable
	{
		Map mapLifeCycle = getSessionObject(pSession);
		
		//search the object name within the life-cycle object
		
		if (pObjectName == null || pObjectName.trim().length() == 0)
		{
			if (mapLifeCycle == null)
			{
				throw new RuntimeException("Unknown object '" + pSession.getLifeCycleName() + "'");
			}
			
			//an action call doesn't need an object name
			return mapLifeCycle;
		}
		else
		{
			//search the desired object
			ArrayUtil<String> auNames = StringUtil.separateList(pObjectName, ".", true);
			
			//get the callable object
			Object oInvoke = mapLifeCycle;
			Object oResult;
			String sObjectName;
			
			StringBuilder sbCurrentObjectName = new StringBuilder();
			
			SessionContextImpl context = ((SessionContextImpl)SessionContext.getCurrentInstance());
			
			String sOriginalMethodName = context.getMethodName();
			
			context.setMethodName(null);
			
			IObjectAccessController controller = getObjectAccessController();
			
			for (int i = 0, anz = auNames.size(); i < anz; i++)
			{
				sObjectName = auNames.get(i);
				
				if (sbCurrentObjectName.length() > 0)
				{
					sbCurrentObjectName.append(".");
				}
				
				sbCurrentObjectName.append(sObjectName);
				
				context.setObjectName(sbCurrentObjectName.toString());
				
				if (i == anz - 1)
				{
					context.setMethodName(sOriginalMethodName);
				}

				if (oInvoke == null)
				{
					throw new RuntimeException("Unknown object '" + sbCurrentObjectName.toString() + "'");
				}
				
				try
				{
					oInvoke = Reflective.call(oInvoke, StringUtil.formatMethodName("get", sObjectName));
				}
				catch (NoSuchMethodException nsme)
				{
					if (oInvoke instanceof Map)
					{
						oResult = ((Map)oInvoke).get(sObjectName);
						
						if (oResult == null && !((Map)oInvoke).containsKey(sObjectName))
						{
							throw new RuntimeException("Unknown object '" + sObjectName + "'");
						}
						
						//use the result!
						oInvoke = oResult;
					}
					else
					{
						throw new RuntimeException("Unknown object '" + sObjectName + "'", nsme);
					}
				}
				
				//#535
				if (controller != null && !controller.isObjectAccessAllowed(this, pSession, mapLifeCycle, sbCurrentObjectName.toString(), oInvoke))
				{
					throw new SecurityException("Access to '" + sObjectName + "' is denied!");
				}
			}
			
			//don't check null, because a getXXX method exists and returns null (maybe expected)!
			return oInvoke;
		}		
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized Object putObject(ISession pSession, String pObjectName, Object pObject) throws Throwable
	{
		if (pObjectName == null)
		{
			return null;
		}
		
		Object objSession;
		
		String sObjectName;
		
		int iPos = pObjectName.lastIndexOf(".");
		
		if (iPos > 0)
		{
			SessionContextImpl context = ((SessionContextImpl)SessionContext.getCurrentInstance()); 
			
			//cache the old method name because the getObject method changes the method name, and in that special
			//case it's not correct to seta method name!
			String sOldMethodName = context.getMethodName();
			
			context.setMethodName(null);
			
			objSession = getObject(pSession, pObjectName.substring(0, iPos));
			
			context.setMethodName(sOldMethodName);
			
			sObjectName = pObjectName.substring(iPos + 1);
		}
		else
		{
			objSession = getSessionObject(pSession);
			 
			sObjectName = pObjectName;
		}
		
		try
		{
			return Reflective.call(objSession, StringUtil.formatMethodName("set", sObjectName), pObject);
		}
		catch (NoSuchMethodException nsme)
		{
			if (objSession instanceof Map)
			{
				return ((Map)objSession).put(sObjectName, pObject);
			}
			else
			{
				throw new RuntimeException("Can't set object '" + pObjectName + "'", nsme);
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public Object invoke(ISession pSession, String pObjectName, String pMethodName, Object... pParams) throws Throwable
	{
		Object obj = getObject(pSession, pObjectName);
		
		//it's possible that a null object will be returned as valid object, but that's not a valid call!
		if (obj == null)
		{
			throw new RuntimeException("The Object '" + pObjectName + "' is known but 'null' was returned!");
		}
		
		//#535
		IObjectAccessController controller = getObjectAccessController();
		
		if (controller != null && !controller.isMethodInvocationAllowed(this, pSession, pObjectName, obj, pMethodName, pParams))
		{
			throw new SecurityException("Invocation of '" + pMethodName + "' is not allowed!");
		}
		
		if (obj instanceof GenericBean)
		{
			//try to call the action generic
			return ((GenericBean)obj).invoke(pMethodName, pParams);
		}
		else
		{
			//call the action by name, and all methods because we are not a generic bean
			return Reflective.call(obj, pMethodName, pParams);
		}
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Server getServer()
	{
		return (Server)super.getServer();
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Gets the life-cycle object for a session. 
	 * 
	 * @param pSession the accessing session 
	 * @return the life-cycle object for the session
	 * @throws Exception if the life-cycle object can not be created
	 */
	protected Map getSessionObject(ISession pSession) throws Exception
	{
		Map mapSession = null;
		
		AbstractSession session = (AbstractSession)pSession;
		
		if (htSessionObjects != null)
		{
			mapSession = htSessionObjects.get(session.getObjectId());
		}
		
		if (mapSession == null)
		{
			mapSession = initSessionObject(session);
		}
		else
		{
			updateSessionObject(session, mapSession);
		}
		
		return mapSession;
	}
	
	/**
	 * Gets te life-cycle object for a session from the cache if it is available. This method does NOT create
	 * a new object if it is not available. Only for internal use.
	 * 
	 * @param pSession the session
	 * @return the life-cycle object or <code>null</code> if the session has no life-cycle object
	 */
	protected Map getSessionObjectInternal(ISession pSession)
	{
		if (htSessionObjects != null)
		{
			return htSessionObjects.get(((AbstractSession)pSession).getObjectId());
		}
		else
		{
			return null;
		}
	}
	
	/**
	 * Creates and initializes a life-cycle object for a session.
	 * 
	 * @param pSession the accessing session 
	 * @return the life-cycle object for the session
	 * @throws Exception if the life-cycle object can not be created
	 */
	private Map initSessionObject(AbstractSession pSession) throws Exception
	{
		if (pSession instanceof SubSession)
		{
			AbstractSession sessParent = ((SubSession)pSession).getMasterSession();
			
			Map mapMaster = null;
			
			
			if (htSessionObjects != null)
			{
				mapMaster = htSessionObjects.get(sessParent.getObjectId());
			}
			
			//If the Master-session didn't execute commands -> the life-cycle object is not
			//available. In that case it's important to create the object.
			//But be careful! We need a SessionContext for the MasterSession, otherwise we have
			//a SessionContext for the SubSession!
			if (mapMaster == null)
			{				
				//object and method name are not known. Of course, we could check if a SessionContext
				//is available, but it's better to give the life-cycle object always consistent values!
				SessionContext context = sessParent.createSessionContext(null, null);
				
				try
				{
					mapMaster = initSessionObject(sessParent);
				}
				finally
				{
					context.release();
				}
			}

			Map mapSub = createInstance(null, pSession, pSession.getLifeCycleName(), mapMaster);
			
			if (htSessionObjects == null)
			{
				htSessionObjects = new Hashtable<Long, Map>();
			}
			
			htSessionObjects.put(pSession.getObjectId(), mapSub);

			//inject after put to Hashtable -> that makes it possible that injected objects have access to
			//the life-cycle object in the constructor
			injectObjects(pSession, mapSub);
			
			return mapSub;
		}
		else
		{
			Map mapApplication = getApplicationObject(pSession);

			Map mapMaster = createInstance(null, pSession, pSession.getLifeCycleName(), mapApplication);
			
			if (htSessionObjects == null)
			{
				htSessionObjects = new Hashtable<Long, Map>();
			}

			htSessionObjects.put(pSession.getObjectId(), mapMaster);

			//inject after put to Hashtable -> that makes it possible that injected objects have access to
			//the life-cycle object in the constructor
			injectObjects(pSession, mapMaster);
			
			return mapMaster;
		}
	}
	
	/**
	 * Updates an existing life-cycle object with current session information.
	 * 
	 * @param pSession the session
	 * @param pMap the associated life-cycle object
	 * @throws Exception if the session access fails
	 */
	private void updateSessionObject(AbstractSession pSession, Map pMap) throws Exception
	{
		List<Entry<String, InjectObject>> liChanges = pSession.getChangedInjectObjects();
		
		if (liChanges != null)
		{
			Entry<String, InjectObject> entry;
			
			InjectObject injobj;
			
			for (int i = 0, anz = liChanges.size(); i < anz; i++)
			{
				entry = liChanges.get(i);

				injobj = entry.getValue();

				if (injobj == null)
				{
					//if the object was not an injected object -> no problem because it will be createad again
					pMap.remove(entry.getKey());
				}
				else
				{
					pMap.put(injobj.getName(), injobj.getObject());
				}
			}
		}
	}
	
	/**
	 * Gets the life-cycle object for an application.
	 * 
	 * @param pSession the accessing session 
	 * @return the life-cycle object for the application
	 * @throws Exception if the life-cycle object can not be created
	 */
	protected Map getApplicationObject(AbstractSession pSession) throws Exception
	{
		Map mapApplication = null;

		String sApplicationName = pSession.getApplicationName(); 
		
		if (htApplicationObjects != null)
		{
			mapApplication = htApplicationObjects.get(sApplicationName);
		}
		
		if (mapApplication == null)
		{
			mapApplication = initApplicationObject(pSession);
		}
		
		return mapApplication;
	}
	
	/**
	 * Creates and initializes a life-cycle object for an application.
	 * 
	 * @param pSession the accessing session
	 * @return the new application object
	 * @throws Exception if the life-cycle object can not be created
	 */
	private Map initApplicationObject(AbstractSession pSession) throws Exception
	{
		String sApplication = pSession.getApplicationZone().getProperty("/application/lifecycle/application");
		
		if (sApplication != null && sApplication.trim().length() > 0)
		{
			try
			{
				Map mapApplication = createInstance(null, pSession, sApplication, null);
				
				if (htApplicationObjects == null)
				{
					htApplicationObjects = new Hashtable<String, Map>();
				}
				
				htApplicationObjects.put(pSession.getApplicationName(), mapApplication);
				
				return mapApplication;
			}
			catch (ClassNotFoundException cnfe)
			{
				log.error("Application object '" + sApplication + "' was not found!", cnfe);

				//don't create a empty Map, to support application loading for new Master sessions
				//e.g. if we change the configuration
				return null;
			}
		}
		else
		{
			return null;
		}
	}

	/**
	 * Creates a new {@link Map} instance with a specific class name and, if possible, sets a parent object.
	 * 
	 * @param pLoader the class loader for instance creation
	 * @param pSession the calling session
	 * @param pInstanceName the full qualified class name for the instance
	 * @param pParent the parent map instance
	 * @return the new instance
	 * @throws Exception if the instance can not be created
	 */
	protected Map createInstance(ClassLoader pLoader, AbstractSession pSession, String pInstanceName, Map pParent) throws Exception
	{
		if (pInstanceName != null)
		{
			//Objekt aus der neuen Klasse erstellen
			Object objInstance;
			
			if (pLoader == null)
			{
				objInstance = Class.forName(pInstanceName).newInstance();
			}
			else
			{
				objInstance = Class.forName(pInstanceName, true, pLoader).newInstance();
			}
			
			if (!(objInstance instanceof Map))
			{
				throw new RuntimeException("The lifecycle object '" + pInstanceName + "' has to be a Map!");
			}
			
			Map mapInstance = (Map)objInstance;
			
			//---------------------------------------------------------------------
			// Use the "members" of the parent 
			//---------------------------------------------------------------------
			
			if (pParent != null)
			{
				if (mapInstance instanceof GenericBean && pParent instanceof Bean)
				{
					((GenericBean)mapInstance).setParent((Bean)pParent);
				}
				else
				{
					log.error("Can't set parent for: ", pInstanceName, " because the life-cycle object is not instance of GenericBean");
				}
			}
			
			return mapInstance;
		}
		else
		{
			throw new ClassNotFoundException("Missing instance name");
		}
	}
	
	/**
	 * Injects the available objects from the session context into the sessions life-cycle object.
	 * 
	 * @param pSession the accessing session
	 * @param pLifeCycleObject the life-cycle object
	 * @throws Exception if the injection configuration is invalid
	 */
	private void injectObjects(AbstractSession pSession, Map pLifeCycleObject) throws Exception
	{
		String sName;
		
		//copy a reference of the inject object to the sessions life-cycle object. That makes it possible
		//to access an injected object with a remote call
		InjectObject injobj;
		
		for (Enumeration<InjectObject> en = pSession.getInjectObjects(); en.hasMoreElements();)
		{
			injobj = en.nextElement();
		
			sName = injobj.getName();
			
			pLifeCycleObject.put(sName, injobj.getObject());
		}
	}
	
	/**
	 * Sets the object access controller.
	 * 
	 * @param pController the controller
	 */
	public void setObjectAccessController(IObjectAccessController pController)
	{
		oaController = pController;
	}
	
	/**
	 * Gets the object access controller.
	 * 
	 * @return the controller
	 */
	public IObjectAccessController getObjectAccessController()
	{
		return oaController;
	}
	
}	// DefaultObjectProvider
