/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 27.05.2009 - [JR] - creation
 * 18.05.2010 - [JR] - create an instance of Reflective to ensure the correct UI thread! 
 */
package com.sibvisions.rad.server;

import javax.rad.remote.IConnection;
import javax.rad.remote.event.CallBackEvent;
import javax.rad.remote.event.ICallBackListener;
import javax.rad.server.ResultObject;

import com.sibvisions.util.Reflective;

/**
 * The <code>Call</code> is an abstract view of a method call. A method
 * call needs information for the execution:
 * <ul>
 *   <li>synchronous or asynchronous calls</li>
 *   <li>an object name</li>
 *   <li>a method name</li>
 *   <li>parameters</li>
 * </ul>
 * @author Ren Jahn
 */
final class Call
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** the reference to the correct UI thread. */
	private Reflective refUIThread = new Reflective();
	
	/** the listener for results. */
	private ICallBackListener listener;
	
	/** the callback for results. */
	private Object oCallBackId;
	
	/** the object name for the call. */
	private String sObjectName;
	
	/** the method name for the call. */
	private String sMethodName;
	
	/** the parameters for the call. */
	private Object[] oParams;
	
	/** the creation time. */
	private long lCreation = System.currentTimeMillis();
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of <code>Call</code>.
	 * 
	 * @param pCallBackListener a callback listener for an asynchronous call or <code>null</code> for a synchronous call
	 * @param pObjectName the object for the call
	 * @param pMethodName the method to call
	 * @param pParams the call parameters
	 */
	Call(ICallBackListener pCallBackListener, String pObjectName, String pMethodName, Object... pParams)
	{
		listener = pCallBackListener;
		
		sObjectName = pObjectName;
		sMethodName = pMethodName;
		oParams = pParams;
	}
	
	/**
	 * Creates a new instance of <code>Call</code>.
	 * 
	 * @param pCallBackId a callback id for an asynchronous call or <code>null</code> for a synchronous call
	 * @param pObjectName the object for the call
	 * @param pMethodName the method to call
	 * @param pParams the call parameters
	 */
	Call(Object pCallBackId, String pObjectName, String pMethodName, Object... pParams)
	{
		oCallBackId = pCallBackId;

		sObjectName = pObjectName;
		sMethodName = pMethodName;
		oParams = pParams;
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Gets the object name for the method call.
	 * 
	 * @return the object name
	 */
	public String getObjectName()
	{
		return sObjectName;
	}
	
	/**
	 * Gets the method name for the call.
	 * 
	 * @return the method name
	 */
	public String getMethodName()
	{
		return sMethodName;
	}
	
	/**
	 * Gets the parameters for the call.
	 * 
	 * @return the parameters
	 */
	public Object[] getParameters()
	{
		return oParams;
	}
	
	/**
	 * Checks if the call is asynchronous (callback) or synchronous.  
	 * 
	 * @return <code>true</code> if the call is asynchronous
	 */
	public boolean isCallBack()
	{
		return listener != null || oCallBackId != null;
	}
	
	/**
	 * Formats the call to a readable format: <code>objectName</code>.<code>methodName</code>
	 * or <code>methodName</code> if the <code>objectName == null</code>. 
	 * 
	 * @return the method format
	 */
	public String formatMethod()
	{
		if (sObjectName == null)
		{
			return sMethodName;
		}
		else
		{
			return sObjectName + "." + sMethodName;
		}
	}
	
	/**
	 * Handles a successfull asynchronous call. If a callback listener was specified, then
	 * it will be notified. The result will be added as callback result to the session if
	 * a callback id was specified. It no callback listener and no callback id was specified
	 * then a {@link RuntimeException} will be thrown.
	 *   
	 * @param pSession the caller session
	 * @param pResult the result from an asynchronous call
	 * @throws RuntimeException if no callback id and no callback listener was found
	 */
	void success(AbstractSession pSession, Object pResult)
	{
		sendResult(pSession, IConnection.TYPE_CALLBACK_RESULT, pResult);
	}

	/**
	 * Handles an exception thrown from an asynchronous call. If a callback listener was specified, then
	 * it will be notified. The exception will be added as callback result to the session if
	 * a callback id was specified. It no callback listener and no callback id was specified
	 * then a {@link RuntimeException} will be thrown.
	 *   
	 * @param pSession the caller session
	 * @param pError the exception from an asynchronous call
	 * @throws RuntimeException if no callback id and no callback listener was found
	 */
	void error(AbstractSession pSession, Throwable pError)
	{
		sendResult(pSession, IConnection.TYPE_CALLBACK_ERROR, pError);
	}
	
	/**
	 * Handles the callback result via invokeLater.
	 * 
	 * @param pSession the caller session
	 * @param pType the result type {@link IConnection#TYPE_CALLBACK_ERROR} or {@link IConnection#TYPE_CALLBACK_RESULT}
	 * @param pResult the result/error object
	 */
	private void sendResult(final AbstractSession pSession, final byte pType, final Object pResult)
	{
		refUIThread.invokeLater(new CallBackWorker(this, pSession, pType, pResult));
	}
	
	//****************************************************************
	// Subclass definition
	//****************************************************************

	/**
	 * The <code>CallBackWorker</code> is a simple {@link Runnable} implementation for
	 * transfering the result of an asynchronous call back to the "client" to the 
	 * UI thread.
	 * 
	 * @author Ren Jahn
	 */
	static final class CallBackWorker implements Runnable
	{
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Class members
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/** the call definition. */
		private Call call;
		
		/** the caller session. */
		private AbstractSession session;
		
		/** the call result. */
		private Object result;
		
		/** the result type. */
		private byte byType;
		
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Initialization
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
		 * Creates a new instance of <code>CallBackWorker</code>.
		 * 
		 * @param pCall the call definition
		 * @param pSession the caller session
		 * @param pType the result type ({@link IConnection#TYPE_CALLBACK_ERROR} or {@link IConnection#TYPE_CALLBACK_RESULT})
		 * @param pResult the result object
		 */
		private CallBackWorker(Call pCall, AbstractSession pSession, byte pType, Object pResult)
		{
			call 	= pCall;
			session = pSession;
			result  = pResult;
			byType  = pType;
		}

		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Interface implementation
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		
		/**
		 * {@inheritDoc}
		 */
		public void run()
		{
			
			if (call.oCallBackId != null)
			{
				session.addCallBackResult(new ResultObject(byType, result, call.oCallBackId));
			}
			else if (call.listener != null)
			{
				call.listener.callBack(new CallBackEvent(call.sObjectName, 
														 call.sMethodName, 
														 IConnection.TYPE_CALLBACK_RESULT == byType ? result : null,
														 (Throwable)(IConnection.TYPE_CALLBACK_ERROR == byType ? result : null),
														 call.lCreation,
														 System.currentTimeMillis()));
			}
			else
			{
				throw new RuntimeException("Callback not possible without callback id or callback listener!");
			}
		}
		
	}	// CallBackWorker
	
}	// Call
