/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 13.05.2009 - [RH] - creation.
 * 23.11.2009 - [RH] - ColumnMetaData && PrimaryKey Column is replaced with MetaData class
 * 02.03.2010 - [RH] - reorganized MetaData -> ServerMetaData, ColumnMetaData -> ServerColumnMetaData
 * 27.03.2010 - [JR] - #92: default value support
 * 29.04.2010 - [JR] - #122: lockRowInternal: check db version
 * 08.05.2010 - [JR] - #127: db version check is not necessary because the db metadata has: supportsSelectForUpdate!
 * 19.11.2010 - [RH] - getUKs, getPKs return Type changed to a <code>Key</code> based result.                                      
 * 06.01.2011 - [JR] - #234: used ColumnMetaDataInfo 
 * 18.11.2011 - [RH] - #510: All XXDBAccess should provide a SQLException format method 
 * 16.01.2011 - [JR] - #537: get_column_name set to false
 */
package com.sibvisions.rad.persist.jdbc;

import java.sql.CallableStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Hashtable;
import java.util.List;

import javax.rad.persist.DataSourceException;

import com.sibvisions.util.type.StringUtil;

/**
 * The <code>HSQLDBAccess</code> is the implementation for Hypersonic SQL databases.<br>
 *  
 * @see com.sibvisions.rad.persist.jdbc.DBAccess
 * 
 * @author Roland Hrmann
 */
public class HSQLDBAccess extends DBAccess
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Constructs a new HSQLDBAccess Object.
	 */
	public HSQLDBAccess()
	{
		super();	
		
		setDriver("org.hsqldb.jdbcDriver");		
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void open() throws DataSourceException
	{
		setDBProperty("get_column_name", "false");
		
		super.open();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Key getPK(String sCatalog, 
         	              String sSchema, 
                          String pTable) throws DataSourceException
    {
		return super.getPK(sCatalog, sSchema, getWritebackTable(pTable));
    }
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ForeignKey> getFKs(String sCatalog, 
      	   						   String sSchema, 
      	   						   String pTable) throws DataSourceException
    {
		return super.getFKs(sCatalog, sSchema, getWritebackTable(pTable));		
    }
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<Key> getUKs(String sCatalog, 
  			 					 String sSchema, 
  			 					 String pTable) throws DataSourceException
    {
		return super.getUKs(sCatalog, sSchema, getWritebackTable(pTable));		    
    }
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public ColumnMetaDataInfo getColumnMetaData(String pFromClause, 
											    String[] pQueryColumns,
								                String pBeforeQueryColumns, 
								                String pWhereClause, 
								                String pAfterWhereClause,
											    String pWritebackTable, 
											    String[] pWritebackColumns) throws DataSourceException
    {
		if (pWritebackColumns != null)
		{
			String[] sWriteBackColumnsUpper = new String[pWritebackColumns.length];
			for (int i = 0; i < sWriteBackColumnsUpper.length; i++)
			{
				sWriteBackColumnsUpper[i] = pWritebackColumns[i].toUpperCase();
			}
			
			pWritebackColumns = sWriteBackColumnsUpper;
		}
		
		return super.getColumnMetaData(pFromClause, 
									   pQueryColumns,
						               pBeforeQueryColumns, 
						               pWhereClause, 
						               pAfterWhereClause,
									   getWritebackTable(pWritebackTable), 
				                       pWritebackColumns);	    
    }
	
	/**
	 * {@inheritDoc}
	 */
	@Override		
	public Object[] insertDatabaseSpecific(String pWritebackTable, String pInsertStatement, ServerMetaData pServerMetaData, 
            Object[] pNewDataRow, String pDummyColumn)  throws DataSourceException
    {
		pNewDataRow = insertAnsiSQL(pWritebackTable, pInsertStatement, pServerMetaData, pNewDataRow, pDummyColumn);
		return hsqlInsertPostProcessing(pServerMetaData, pNewDataRow);
    }	
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Hashtable<String, Object> getDefaultValues(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		return super.getDefaultValues(pCatalog, pSchema, pTable.toUpperCase());
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Object translateDefaultValue(String pColumnName, int pDataType, String pDefaultValue) throws Exception
	{
		//HSQL JDBC returns 'value'
		return super.translateDefaultValue(pColumnName, pDataType, StringUtil.removeQuotes(pDefaultValue, "'"));
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Returns the WritebackTable as toUpper, because of the meta data bug in HSQL.
	 *  
	 * @param pWritebackTable	the write back tabel to use.
	 * @return the WritebackTable as toUpper, because of the meta data bug in HSQL.
	 */
	public String getWritebackTable(String pWritebackTable)
	{	
		if (pWritebackTable != null)
		{
			return pWritebackTable.toUpperCase();
		}
		return null;
    }
	
	/**
	 * Hypersonic SQL specific implementation to get the auto increment value of a PK.
	 * This database supports only one auto increment column in the PK, and doesn't
	 * support the jdbc .getGeneratedKeys() method.
	 * 
	 * @param pServerMetaData	the meta data to use.
	 * @param pNewDataRow		the new IDataRow with the values to insert
	 * @return the new inserted row with the new PK value.
	 * @throws DataSourceException
	 *            if an <code>Exception</code> occur during insert to the storage
	 */
	private Object[] hsqlInsertPostProcessing(ServerMetaData pServerMetaData, Object[] pNewDataRow) throws DataSourceException
	{
		// Refetch generated Primary Key, if a PK is defined
		if (pServerMetaData.getPrimaryKeyColumnNames() != null && pServerMetaData.getPrimaryKeyColumnNames().length > 0)
		{		
			CallableStatement psSelectPK = null;
			ResultSet rsPK = null;
			try 
			{
				psSelectPK = getConnection().prepareCall("CALL IDENTITY()");		
	
				rsPK = psSelectPK.executeQuery();

				if (rsPK.next())
				{		
					// hsqlDB support only one generated Column, we decide it is the first in Primary Key
					if (pServerMetaData.getPrimaryKeyColumnIndices().length > 0)						
					{
	    				pNewDataRow[pServerMetaData.getPrimaryKeyColumnIndices()[0]] = rsPK.getObject(1);						
					}
				}
				
				return pNewDataRow;
			}
			catch (SQLException sqlException)
			{
				throw new DataSourceException("The generated keys couldn't read!", formatSQLException(sqlException));
			}
			finally
			{
				try 
				{
					if (rsPK != null)
					{
						rsPK.close();
					}
					if (psSelectPK != null)
					{
						psSelectPK.close();
					}
				}
				catch (SQLException e)
				{
					throw new DataSourceException("Jdbc statement close failed", formatSQLException(e));
				}				
			}
		}
		return pNewDataRow;
	}
		
} 	// HSQLDBAccess
