/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 05.10.2009 - [RH] - creation.
 * 27.03.2010 - [JR] - #92: default value support         
 * 28.04.2011 - [RH] - #341 -  LikeReverse Condition, LikeReverseIgnoreCase Condition               
 */
package com.sibvisions.rad.persist.jdbc;

import java.util.Hashtable;

import javax.rad.persist.DataSourceException;

import com.sibvisions.util.type.StringUtil;

/**
 * The <code>DerbyDBAccess</code> is the implementation for Derby databases. Client driver version.<br>
 *  
 * @see com.sibvisions.rad.persist.jdbc.DBAccess
 * 
 * @author Roland Hrmann
 */
public class DerbyDBAccess extends DBAccess
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Constructs a new DerbyDBAccess Object.
	 */
	public DerbyDBAccess()
	{
		super();		

		setDriver("org.apache.derby.jdbc.ClientDriver");
		// [RH] [HM] integrate embedded. 
//		setDriverName("org.apache.derby.jdbc.EmbeddedDriver");	
	}
		
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean supportsGetGeneratedKeys()
	{
		return true;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Hashtable<String, Object> getDefaultValues(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		return super.getDefaultValues(pCatalog, pSchema, pTable.toUpperCase());
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Object translateDefaultValue(String pColumnName, int pDataType, String pDefaultValue) throws Exception
	{
		//DB2 JDBC returns 'value'
		return super.translateDefaultValue(pColumnName, pDataType, StringUtil.removeQuotes(pDefaultValue, "'"));
	}

	/** 
	 * {@inheritDoc}
	 */
	@Override
	protected String createReplace(String pSource, String pOld, String pNew) 
	{
		// Derby doesn't support any replace command. -> LikeReverseXXX doesn't support *->% replacement in DB!!!
		return pSource;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public ColumnMetaDataInfo getColumnMetaData(String pFromClause, 
		 							     	    String[] pQueryColumns,
										     	String pBeforeQueryColumns, 
										     	String pWhereClause, 
										     	String pAfterWhereClause,
										     	String pWritebackTable,
										     	String[] pWritebackColumns) throws DataSourceException
    {
		ColumnMetaDataInfo mdInfo = super.getColumnMetaData(pFromClause, pQueryColumns, pBeforeQueryColumns, pWhereClause, 
				                                      	    pAfterWhereClause, pWritebackTable, pWritebackColumns);
		
		String table = mdInfo.getTable();
		if (table != null && !table.startsWith(QUOTE) && !table.endsWith(QUOTE))
		{
			mdInfo.setTable(table.toUpperCase());
		}
		
		return mdInfo;
    }
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
} 	// DerbyDBAccess
