/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 30.05.2009 - [RH] - creation
 * 27.03.2010 - [JR] - #92: default value support 
 * 11.03.2011 - [RH] - #308 - DB specific automatic quoting implemented      
 * 25.07.2011 - [RH] - #441 - DB2 doesn't support catalogs and return it wrong from the meta data.                           
 */
package com.sibvisions.rad.persist.jdbc;

import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Types;
import java.util.Hashtable;
import java.util.List;

import javax.rad.model.datatype.BigDecimalDataType;
import javax.rad.persist.DataSourceException;
import com.sibvisions.util.type.StringUtil;

/**
 * The <code>DB2DBAccess</code> is the implementation for DB2 databases.<br>
 *  
 * @see com.sibvisions.rad.persist.jdbc.DBAccess
 * 
 * @author Roland Hrmann
 */
public class DB2DBAccess extends DBAccess
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Constructs a new DB2DBAccess Object.
	 */
	public DB2DBAccess()
	{
		super();		
		
		setDriver("com.ibm.db2.jcc.DB2Driver");	
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void open() throws DataSourceException
	{
		//JDBC driver correct error messages
		setDBProperty("retrieveMessagesFromServerOnGetMessage", "true");

		super.open();
	}	
	
	/* 
	 * {@inheritDoc}
	 */
	@Override
	public void setUsername(String pUsername)
	{
		super.setUsername(DBAccess.removeQuotes(pUsername));
	}	
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Hashtable<String, Object> getDefaultValues(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		return super.getDefaultValues(pCatalog, pSchema, pTable.toUpperCase());
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Object translateDefaultValue(String pColumnName, int pDataType, String pDefaultValue) throws Exception
	{
		//DB2 JDBC returns 'value'
		return super.translateDefaultValue(pColumnName, pDataType, StringUtil.removeQuotes(pDefaultValue, "'"));
	}
		
	/** 
	 * {@inheritDoc}
	 */
	@Override
	public ColumnMetaDataInfo getColumnMetaData(String pFromClause, 
	     	String[] pQueryColumns,
	     	String pBeforeQueryColumns, 
	     	String pWhereClause, 
	     	String pAfterWhereClause,
	     	String pWritebackTable,
	     	String[] pWritebackColumns) throws DataSourceException
	{	
		ColumnMetaDataInfo cmdi = super.getColumnMetaData(
				pFromClause, pQueryColumns, pBeforeQueryColumns, pWhereClause, pAfterWhereClause, pWritebackTable, pWritebackColumns);
		if (cmdi != null)
		{
			cmdi.setCatalog(null);
		}
		String table = cmdi.getTable();
		if (table != null && !table.startsWith(QUOTE) && !table.endsWith(QUOTE))
		{
			cmdi.setTable(table.toUpperCase());
		}
		return cmdi;
	}
	
	/** 
	 * {@inheritDoc}
	 */
	@Override
	public List<ForeignKey> getFKs(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		List<ForeignKey> lFKs = super.getFKs(pCatalog, pSchema, pTable);
		
		for (int i = 0; i < lFKs.size(); i++)
		{
			ForeignKey fk = lFKs.get(i);
			fk.setPKCatalog(null);
		}
			
		return lFKs;
	}	
	
	/**
	 * {@inheritDoc} 
	 */
	@Override
	protected boolean setDatabaseSpecificType(ResultSetMetaData pMetaData, int pColumnIndex, ServerColumnMetaData pColumnMetaData) throws SQLException
	{
		int iType = pMetaData.getColumnType(pColumnIndex);
		
		if (iType == Types.OTHER) // for DECFLOAT type == Oracle's NUMBER ; scale/precision 0,-1 
		{
			pColumnMetaData.setDataType(BigDecimalDataType.TYPE_IDENTIFIER);
			pColumnMetaData.setPrecision(0);
			pColumnMetaData.setScale(-1);
			pColumnMetaData.setSigned(pMetaData.isSigned(pColumnIndex)); 

	    	return true;
		}
				
		return super.setDatabaseSpecificType(pMetaData, pColumnIndex, pColumnMetaData);
	}	
	
} 	// DB2DBAccess
