/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 09.04.2009 - [RH] - creation
 * 18.04.2009 - [RH] - get/set/DataPage/RowIndex moved to ChangeableDataRow        
 * 31.03.2011 - [RH] - #163 - IChangeableDataRow should support isWritableColumnChanged                          
 */
package com.sibvisions.rad.model.mem;

import java.io.Serializable;

import javax.rad.model.IChangeableDataRow;
import javax.rad.model.IDataPage;
import javax.rad.model.IDataRow;
import javax.rad.model.IRowDefinition;
import javax.rad.model.ModelException;

/**
 * An <code>IChangeableDataRow</code> extends the <code>IDataRow</code> with support for
 * the change state of the row, some supporting methods and an unique ID column.
 * 
 * @see javax.rad.model.IDataRow
 * @see javax.rad.model.IDataPage
 * @see javax.rad.model.IDataBook
 * 
 * @author Roland Hrmann
 */
public class ChangeableDataRow extends DataRow 
                               implements IChangeableDataRow, 
                                          Serializable
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Constants
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** Internal Offset in the AbstractStorage Array for the Inserting, etc.. states. */
	protected static final int    	  INTERNAL_OFFSET     = 3;
	
	/** Internal state for INSERTING. */
	protected static final Integer    INSERTING = Integer.valueOf(1);
	/** Internal state for UPDATING. */
	protected static final Integer    UPDATING  = Integer.valueOf(2);
	/** Internal state for UPDATING, if minimum one writeable column is involved. */
	protected static final Integer    WRITABLE_COLUMN_CHANGED  = Integer.valueOf(3);
	/** Internal state for DELETING. */
	protected static final Integer    DELETING  = Integer.valueOf(4);

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The corresponding<code>IDataPage</code> of the <code>IDataRow</code> in the <code>IDataBook</code>. */
	private transient IDataPage	dpDataPage = null;
	
	/** The row index of the <code>IDataRow</code> in the <code>IDataBook</code>. */
	private transient int		iRowIndex = -1;    
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Internal default constructor for the MemDataBook.
	 */
	protected ChangeableDataRow()
	{	
	}

	/**
	 * Constructs a <code>DataRow</code> with a given <code>IRowDefinition</code>.
	 * 
	 * @param pRowDefinition the <code>IRowDefinition</code>
	 */
	public ChangeableDataRow(IRowDefinition pRowDefinition)
	{
		this(pRowDefinition, null, null, -1);
	}
	
	/**
	 * Constructs a <code>DataRow</code> with a given <code>IRowDefinition</code> 
	 * and initialize it a copy of the <code>Object[]<></code> data.
	 * 
	 * @param pRowDefinition the <code>IRowDefinition</code>
	 * @param pData the <code>Object[]<></code> with data of the <code>DataRow</code>.
	 * @param pDataPage the corresponding<code>IDataPage</code> of the <code>IDataRow</code> in the <code>IDataBook</code>
	 * @param pRowIndex the row index of the <code>IDataRow</code> in the <code>IDataBook</code>
	 */
	protected ChangeableDataRow(IRowDefinition pRowDefinition, Object[] pData, IDataPage pDataPage, int pRowIndex)
	{
		super(pRowDefinition, pData);
		
		dpDataPage = pDataPage;
		iRowIndex  = pRowIndex;		
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


	/**
	 * {@inheritDoc}
	 */
	public IDataPage getDataPage()
	{
		return dpDataPage;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public int getRowIndex()
	{
		return iRowIndex;
	}
		
	/**
	 * {@inheritDoc}
	 */
	public Object getUID()
	{
		int iCount = rdRowDefinition.getColumnCount();
		if (oaStorage == null || oaStorage.length <= iCount)
		{
			return null;
		}			
		return (Integer)oaStorage[iCount];
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isInserting()  throws ModelException
	{
		int iCount = rdRowDefinition.getColumnCount() + 1; 
		if (oaStorage.length <= iCount)
		{
			return false;
		}			
		return oaStorage[iCount] == INSERTING;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isUpdating()  throws ModelException
	{
		int iCount = rdRowDefinition.getColumnCount() + 1;
		if (oaStorage.length <= iCount)
		{
			return false;
		}			
		
		Integer iState = (Integer)oaStorage[iCount];
		
		return iState == UPDATING || iState == WRITABLE_COLUMN_CHANGED;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isDeleting()  throws ModelException
	{
		int iCount = rdRowDefinition.getColumnCount() + 1;
		
		if (oaStorage.length <= iCount)
		{
			return false;
		}			
		return oaStorage[iCount] == DELETING;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isDetailChanged()  throws ModelException
	{
		int iCount = rdRowDefinition.getColumnCount() + 2; 
		
		if (oaStorage.length <= iCount)
		{
			return false;
		}			
		
		Object oState = oaStorage[iCount];
		if (oState == null)
		{
			return false;
		}
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	public IDataRow getOriginalRow() throws ModelException
	{
		// copy the saved original row (the last columns in the Object[]) to the result		
		int iColumnCount = getRowDefinition().getColumnCount();
		if (oaStorage.length < rdRowDefinition.getColumnCount() + 1)
		{
			return null;
		}			
		Object oState = oaStorage[iColumnCount + 1];
		if (oaStorage.length == INTERNAL_OFFSET + iColumnCount * 2)
		{
			Object[] oaResultRow = new Object[iColumnCount];
			System.arraycopy(oaStorage, INTERNAL_OFFSET + iColumnCount, 
					         oaResultRow, 0, 
					         oaResultRow.length);
			
			return new DataRow(getRowDefinition(), oaResultRow);
		}
		else if (oState == DELETING)
		{
			return this;
		}
		else
		{
			return null;
		}
	}	

	/** 
	 * {@inheritDoc}
	 */
	// #163 - IChangeableDataRow should support isWritableColumnChanged 
	public boolean isWritableColumnChanged()  
	{
		int iColCount = rdRowDefinition.getColumnCount() + 1;
		
		if (oaStorage.length <= iColCount)
		{
			return false;
		}			
		return oaStorage[iColCount] == WRITABLE_COLUMN_CHANGED;
	}	

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString()
	{
		return "Changeable" + super.toString() + ", UID=" + getUID();
	}	
		
} 	// ChangeableDataRow
