/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 02.12.2008 - [JR] - creation
 * 08.12.2008 - [JR] - updateAnchors implemented
 * 26.02.2008 - [HM] - getLocalizedMessage replaced with getMessage, because translation
 *                     should work without localization
 * 15.01.2013 - [JR] - #625: added methods                     
 */
package com.sibvisions.rad.application;

import javax.rad.application.IContent;
import javax.rad.genui.UIDimension;
import javax.rad.genui.UIImage;
import javax.rad.genui.UIInsets;
import javax.rad.genui.celleditor.UITextCellEditor;
import javax.rad.genui.component.UIButton;
import javax.rad.genui.component.UIIcon;
import javax.rad.genui.component.UITextArea;
import javax.rad.genui.component.UIToggleButton;
import javax.rad.genui.container.UIGroupPanel;
import javax.rad.genui.container.UIInternalFrame;
import javax.rad.genui.control.UIEditor;
import javax.rad.genui.control.UITable;
import javax.rad.genui.layout.UIFormLayout;
import javax.rad.model.ColumnDefinition;
import javax.rad.model.ColumnView;
import javax.rad.model.IRowDefinition;
import javax.rad.model.ModelException;
import javax.rad.model.datatype.ObjectDataType;
import javax.rad.ui.IAlignmentConstants;
import javax.rad.ui.celleditor.ITextCellEditor;
import javax.rad.ui.component.IButton;
import javax.rad.ui.component.IIcon;
import javax.rad.ui.component.ITextArea;
import javax.rad.ui.component.IToggleButton;
import javax.rad.ui.container.IDesktopPanel;
import javax.rad.ui.control.IEditor;
import javax.rad.ui.control.ITable;
import javax.rad.ui.event.UIActionEvent;
import javax.rad.ui.layout.IFormLayout;
import javax.rad.ui.layout.IFormLayout.IConstraints;

import com.sibvisions.rad.model.mem.MemDataBook;
import com.sibvisions.util.type.CommonUtil;

/**
 * The <code>Error</code> is designed to visualize errors occured 
 * in an application. It displays the whole error chain.
 * 
 * @author Ren Jahn
 */
public class Error extends UIInternalFrame
                   implements IContent
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the message opener. */
	private Object		  	opener;

	/** the layout. */
	private IFormLayout 	foMain;
	
	/** the details panel. */
	private UIGroupPanel 	gpanDetails;
	
	/** the error description. */ 
	private ITextArea 		taMessage;
	
	/** the ok/close button. */
	private IButton 		butOK;
	
	/** the details button. */
	private IToggleButton 	butDetails;
	
	/** the representation of the error chain. */
	private MemDataBook 	mdbError;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of <code>Error</code> for a special desktop.
	 * 
	 * @param pDesktop the desktop for showing the frame
	 * @throws Exception if it's not possible to build the UI
	 */
	public Error(IDesktopPanel pDesktop) throws Exception
	{
		super(pDesktop);
		
		init();
	}

	/**
	 * Builds the UI.
	 * 
	 * @throws Exception if the error table can not be opened
	 */
	private void init() throws Exception
	{
		setTitle("Information");
		setResizable(true);
		setIconImage(null);
		setMaximizable(false);
		setIconifiable(false);
		setModal(true);
		
		setMinimumSize(new UIDimension(200, 140));

		//----------------------------------------------------------------
		// Standard components

		IIcon icon = new UIIcon();
		icon.setImage(UIImage.getImage(UIImage.MESSAGE_ERROR_LARGE));
		icon.setVerticalAlignment(IAlignmentConstants.ALIGN_TOP);

		taMessage = new UITextArea();
		taMessage.setMinimumSize(new UIDimension(0, 70));
		taMessage.setPreferredSize(new UIDimension(470, 70));
		taMessage.setWordWrap(true);
		taMessage.setEditable(false);
		
		butOK = new UIButton();
		butOK.eventAction().addListener(this, "doOk");
		butOK.setText("OK");
		butOK.setDefaultButton(true);
		
		butDetails = new UIToggleButton();
		butDetails.eventAction().addListener(this, "doDetails");
		butDetails.setText("Details");

		//----------------------------------------------------------------
		// Detail components
		
		IRowDefinition rowdef;
		
		mdbError = new MemDataBook();
		mdbError.setName("Error");
		mdbError.setUpdateEnabled(false);
		mdbError.setDeleteEnabled(false);
		
		rowdef = mdbError.getRowDefinition();
		
		rowdef.addColumnDefinition(new ColumnDefinition("ERROR"));
		rowdef.addColumnDefinition(new ColumnDefinition("DETAIL"));
		rowdef.addColumnDefinition(new ColumnDefinition("CAUSE", new ObjectDataType()));
		rowdef.setColumnView(null, new ColumnView("ERROR"));
		rowdef.getColumnDefinition("ERROR").setLabel("Cause(s) of failure");
		
		mdbError.open();
		
		ITable tblError = new UITable();
		tblError.setDataBook(mdbError);
		tblError.setPreferredSize(new UIDimension(0, 90));
		tblError.setMaximumSize(new UIDimension(Integer.MAX_VALUE, 90));
		tblError.setAutoResize(true);
		
		ITextCellEditor cedDetail = new UITextCellEditor();
		cedDetail.setContentType(ITextCellEditor.TEXT_PLAIN_MULTILINE);
		
		IEditor editErrorDetail = new UIEditor();
		editErrorDetail.setCellEditor(cedDetail);
		editErrorDetail.setDataRow(mdbError);
		editErrorDetail.setColumnName("DETAIL");
		editErrorDetail.setPreferredSize(new UIDimension(0, 200));

		//----------------------------------------------------------------
		// Details
		//----------------------------------------------------------------

		UIFormLayout foDetails = new UIFormLayout();
		foDetails.setMargins(new UIInsets(2, 2, 2, 2));
		
		gpanDetails = new UIGroupPanel();
		gpanDetails.setText("Details");
		gpanDetails.setLayout(foDetails);

		gpanDetails.add(tblError, 
				        foDetails.getConstraints(foDetails.getTopMarginAnchor(), 
				        		                 foDetails.getLeftMarginAnchor(), 
				        		                 null, 
				        		                 foDetails.getRightMarginAnchor()));
		gpanDetails.add(editErrorDetail, 
				        foDetails.getConstraints(foDetails.createAnchor(foDetails.getConstraints(tblError).getBottomAnchor(), 5), 
				        		                 foDetails.getLeftMarginAnchor(), 
				        		                 foDetails.getBottomMarginAnchor(), 
				        		                 foDetails.getRightMarginAnchor()));
		
		//----------------------------------------------------------------
		// Standard
		//----------------------------------------------------------------
		
		foMain = new UIFormLayout();
		
		foMain.setVerticalAlignment(IAlignmentConstants.ALIGN_TOP);
		foMain.setMargins(new UIInsets(7, 7, 7, 7));
		
		setLayout(foMain);

		add(icon);
		add(taMessage, 
			foMain.getConstraints(foMain.getTopMarginAnchor(), 
					              foMain.createAnchor(foMain.getConstraints(icon).getRightAnchor(), 5), 
					              null, 
					              foMain.getRightMarginAnchor()));
		add(butOK, 
		    foMain.getConstraints(null, 
		    		              null, 
		    		              foMain.getBottomMarginAnchor(), 
		    		              foMain.getRightMarginAnchor()));
		add(butDetails, 
			foMain.getConstraints(null, 
					              null, 
					              foMain.getBottomMarginAnchor(), 
					              foMain.createAnchor(foMain.getConstraints(butOK).getLeftAnchor(), -5))); 
		//only visible if the user decides
		add(gpanDetails,
			foMain.getConstraints(null, 
								  foMain.getLeftMarginAnchor(), 
								  foMain.getBottomMarginAnchor(), 
								  foMain.getRightMarginAnchor()));

		gpanDetails.setVisible(false);
		
		//make the message resizable
		foMain.getConstraints(taMessage).setBottomAnchor(foMain.createAnchor(foMain.getConstraints(butOK).getTopAnchor(), -7));
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Interface implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public void notifyDestroy()
	{
	}

	/**
	 * {@inheritDoc}
	 */
	public void notifyVisible()
	{
		
	}
	
	/**
	 * {@inheritDoc}
	 */
	public <OP> void setOpener(OP pOpener)
	{
		opener = pOpener;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public <OP> OP getOpener()
	{
		return (OP)opener;
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Adds an error to the list of errors.
	 * 
	 * @param pError the error
	 */
	public void addError(Throwable pError)
	{
		if (pError != null)
		{
			mdbError.setInsertEnabled(true);
			
			Throwable cause = pError;
			
			try
			{
				String sError;
				String sMessage;
				
				do
				{
					sError = cause.getMessage();
					
					if (sError == null || sError.trim().length() == 0)
					{
						sError = cause.getClass().getName();
						sMessage = cause.getClass().getSimpleName();
					}
					else
					{
						sMessage = sError;
					}
					
					mdbError.insert(true);
					mdbError.setValues(new String[] {"ERROR", "DETAIL", "CAUSE"}, 
							           new Object[] {sError, CommonUtil.dump(cause, false), cause});
					
					cause = cause.getCause();
				}
				while (cause != null);
				
				mdbError.saveAllRows();
				
				taMessage.setText(sMessage);
			}
			catch (ModelException me)
			{
				//it's not possible to report the error to the ExceptionHandler,
				//because it produces an error!
				
				debug(me);
			}
			
			mdbError.setInsertEnabled(false);
		}
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Updates the Layout anchors dependent of the visible details.
	 */
	protected void updateAnchors()
	{
		IConstraints consOk = foMain.getConstraints(butOK); 

		if (gpanDetails.isVisible())
		{
			//Details mode

			foMain.getConstraints(taMessage).setBottomAnchor(null);
			
			consOk.setTopAnchor(foMain.createAnchor(foMain.getConstraints(taMessage).getBottomAnchor(), 7));
			consOk.setBottomAnchor(null);

			IConstraints consDetails = foMain.getConstraints(gpanDetails);
			
			consDetails.setTopAnchor(foMain.createAnchor(foMain.getConstraints(butOK).getBottomAnchor(), 5));
		}
		else
		{
			//Simple mode

			consOk.setBottomAnchor(foMain.getBottomMarginAnchor());
			consOk.setTopAnchor(null);

			foMain.getConstraints(taMessage).setBottomAnchor(foMain.createAnchor(consOk.getTopAnchor(), -7));
		}
		
		IConstraints consDetails = foMain.getConstraints(butDetails);

		consDetails.setTopAnchor(consOk.getTopAnchor());
		consDetails.setBottomAnchor(consOk.getBottomAnchor());
	}
	
	/**
	 * Gets the message text area.
	 *  
	 * @return the text area
	 */
	protected ITextArea getTextArea()
	{
		return taMessage;
	}
	
	/**
	 * Gets the OK button.
	 * 
	 * @return the button
	 */
	protected IButton getOKButton()
	{
		return butOK;
	}
	
	/**
	 * Gets the details button.
	 * 
	 * @return the button
	 */
	protected IToggleButton getDetailsButton()
	{
		return butDetails;
	}
	
	/**
	 * Gets the details group panel.
	 * 
	 * @return the group panel
	 */
	protected UIGroupPanel getDetailsGroupPanel()
	{
		return gpanDetails;
	}

	/**
	 * Gets the error message.
	 * 
	 * @return the message
	 */
	public String getMessage()
	{
		return taMessage.getText();
	}
	
	/**
	 * Gets all available errors.
	 * 
	 * @return all reported errors
	 */
	public Throwable[] getErrors()
	{
		try
		{
			Throwable[] thErrors = new Throwable[mdbError.getRowCount()];
			
			for (int i = 0, anz = mdbError.getRowCount(); i < anz; i++)
			{
				thErrors[i] = (Throwable)mdbError.getDataRow(i).getValue("CAUSE");
			}
			
			return thErrors;
		}
		catch (ModelException me)
		{
			throw new RuntimeException(me);
		}
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Actions
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Closes the content.
	 * 
	 * @param pEvent the triggering event
	 */
	public void doOk(UIActionEvent pEvent)
	{
		dispose();
	}

	/**
	 * Shows the details.
	 * 
	 * @param pEvent the triggering event
	 */
	public void doDetails(UIActionEvent pEvent)
	{
		IToggleButton but = (IToggleButton)pEvent.getSource();
		
		gpanDetails.setVisible(but.isSelected());

		updateAnchors();

		pack();
	}
	
}	// Error
