/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 19.02.2010 - [JR] - creation
 * 28.03.2010 - [JR] - #47: default choice cell editor mapped
 * 29.11.2010 - [JR] - getLauncher implemented
 * 07.01.2011 - [JR] - removed static block -> moved to application
 * 11.07.2011 - [RH] - #421 (Vx): ApplicationUtil should have a Boolean ChoiceCellEditor
 * 09.11.2011 - [RH] - #501 (Vx): ApplicationUtil should have a scaleImage method 
 * 15.11.2011 - [JR] - getImageData, convertAlignment implemented
 */
package com.sibvisions.rad.application;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;

import javax.rad.application.ILauncher;
import javax.rad.genui.UIColor;
import javax.rad.genui.UIImage;
import javax.rad.genui.celleditor.UIChoiceCellEditor;
import javax.rad.genui.celleditor.UIImageViewer;
import javax.rad.genui.celleditor.UINumberCellEditor;
import javax.rad.genui.celleditor.UITextCellEditor;
import javax.rad.genui.control.UICellFormat;
import javax.rad.io.IFileHandle;
import javax.rad.model.IDataBook;
import javax.rad.model.IRowDefinition;
import javax.rad.model.ModelException;
import javax.rad.model.datatype.BinaryDataType;
import javax.rad.model.ui.IControl;
import javax.rad.model.ui.IEditorControl;
import javax.rad.model.ui.ITableControl;
import javax.rad.ui.IAlignmentConstants;
import javax.rad.ui.IComponent;
import javax.rad.ui.IImage;
import javax.rad.ui.IImage.ImageType;

import com.sibvisions.util.type.FileUtil;
import com.sibvisions.util.type.ImageUtil;

/**
 * The <code>ApplicationUtil</code> is a utility class which provides methods and constants which will
 * be used from applications.
 * 
 * @author Ren Jahn
 */
public class ApplicationUtil
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Constants
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		
	/** Red cell. */
	public static final UICellFormat RED_CELL 		= new UICellFormat(UIColor.red);
	/** Orange cell. */
	public static final UICellFormat ORANGE_CELL 	= new UICellFormat(UIColor.orange);
	/** Yellow cell. */
	public static final UICellFormat YELLOW_CELL 	= new UICellFormat(UIColor.yellow);

	/** Blue cell. */
	public static final UICellFormat BLUE_CELL 		= new UICellFormat(new UIColor(50, 150, 255));
	/** Green cell. */
	public static final UICellFormat GREEN_CELL 	= new UICellFormat(UIColor.green);
	
	/** Passwort cell editor. */
	public static final UITextCellEditor   PASSWORD_EDITOR 			= new UITextCellEditor(UITextCellEditor.TEXT_PLAIN_PASSWORD);
	/** Centered text cell editor. */
	public static final UITextCellEditor   CENTERED_TEXT_EDITOR		= new UITextCellEditor(IAlignmentConstants.ALIGN_CENTER);
	/** Centered number cell editor. */
	public static final UINumberCellEditor CENTERED_NUMBER_EDITOR	= new UINumberCellEditor(IAlignmentConstants.ALIGN_CENTER);
	/** Multiline (text area) cell editor. */
	public static final UITextCellEditor   MULTILINE_EDITOR 		= new UITextCellEditor(UITextCellEditor.TEXT_PLAIN_WRAPPEDMULTILINE);

	/** Yes/No choice cell editor. */
	public static final UIChoiceCellEditor YESNO_EDITOR 			= createYNChoiceCellEditor();
	
	/** True/False choice cell editor for BooleanDataTypes. */
	public static final UIChoiceCellEditor TRUEFALSE_EDITOR			= createTFChoiceCellEditor();

	/** The Image Viewer. */
	public static final UIImageViewer	   IMAGE_VIEWER				= new UIImageViewer();
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Invisible constructor, because <code>ApplictionUtil</code> is a utility class.
	 */
	protected ApplicationUtil()
	{
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	
	
	/**
	 * Returns a new checkbox cell editor with True, False values.
	 * 
	 * @return  a new checkbox cell editor with True, False values.
	 */
	public static UIChoiceCellEditor createTFChoiceCellEditor()
	{
		return 	new UIChoiceCellEditor(
				new Object [] { Boolean.TRUE, Boolean.FALSE},
	         	new String [] { UIImage.CHECK_YES_SMALL,
				 		        UIImage.CHECK_SMALL},
	                            UIImage.CHECK_SMALL);
	}
	
	/**
	 * Returns a new checkbox cell editor with "Y", "N" values.
	 * 
	 * @return  a new checkbox cell editor with "Y", "N" values.
	 */
	public static UIChoiceCellEditor createYNChoiceCellEditor()
	{
		return new UIChoiceCellEditor(new Object [] {"Y", "N"},
					              	  new String [] {UIImage.CHECK_YES_SMALL, UIImage.CHECK_SMALL},
					              	  UIImage.CHECK_SMALL);
	}	
	
	/**
	 * Gets all visible columns on screen.
	 * 
	 * @param pDataBook the data book.
	 * @return all visible columns on screen.
	 */
	public static String[] getAllVisibleColumns(IDataBook pDataBook)
	{
		IRowDefinition rowDefinition = pDataBook.getRowDefinition();
		
		ArrayList<String> columns = new ArrayList<String>();
		
		IControl[] controls = pDataBook.getControls();
		
		for (int i = 0; i < controls.length; i++)
		{
			IControl control = controls[i];

			if (control instanceof IEditorControl)
			{
				try
				{
					String columnName = ((IEditorControl)control).getColumnName();
					
					if (!columns.contains(columnName) 
							&& !(rowDefinition.getColumnDefinition(columnName).getDataType() instanceof BinaryDataType))
					{
						columns.add(columnName);
					}
				}
				catch (Exception e)
				{
					// Do nothing
				}
			}
			else if (control instanceof ITableControl)
			{
				String[] columnNames = pDataBook.getRowDefinition().getColumnView(ITableControl.class).getColumnNames();
				
				for (int j = 0; j < columnNames.length; j++)
				{
					String columnName = columnNames[j];
					
					try
					{
						if (!columns.contains(columnName)
								&& !(rowDefinition.getColumnDefinition(columnName).getDataType() instanceof BinaryDataType))
						{
							columns.add(columnName);
						}
					}
					catch (ModelException e)
					{
						// Do nothing
					}
				}
			}
		}
		return columns.toArray(new String[columns.size()]);
	}
	
	/**
	 * Gets the launcher for a specific component. The component should be added.
	 * 
	 * @param pComponent any added component
	 * @return the launcher or <code>null</code> if the component is not added
	 */
	public static ILauncher getLauncher(IComponent pComponent)
	{
		IComponent comp = pComponent;
		
		while (comp != null && !(comp instanceof ILauncher))
		{
			comp = comp.getParent();
		}
		
		return (ILauncher)comp;
	}
	
	/**
	 * It reads the file handle and scales the image to the determined width and height. 
	 * 
	 * @param pFileHandle	the FileHandle to use.
	 * @param pWidth		the width to use.
	 * @param pHeight		the heigth to use.
	 * @return the scaled image with the determined width and height.
	 * @throws IOException if the image can't read or scaled.
	 */
	public static byte[] scaleImage(IFileHandle pFileHandle, int pWidth, int pHeight) throws IOException
	{
		String sFormat = FileUtil.getExtension(pFileHandle.getFileName().toLowerCase());

		if ("png".equals(sFormat) || "jpg".equals(sFormat) || "gif".equals(sFormat))
		{
			ByteArrayOutputStream stream = new ByteArrayOutputStream();
			ImageUtil.createScaledImage(pFileHandle.getInputStream(), pWidth, pHeight, true, stream, sFormat);
			stream.close();

			return stream.toByteArray();
		}
		else
		{
			throw new IOException("Image format '" + sFormat + "' not supported. Use 'png', 'jpg' or 'gif'!");
		}
	}	

	/**
	 * Gets the raw image data.
	 * 
	 * @param pImage the image
	 * @return the image data as bytes
	 * @throws IOException if image conversion fails
	 */
	public static byte[] getImageData(IImage pImage) throws IOException
	{
		if (pImage == null)
		{
			return null;
		}
		
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
	
		try
		{
			pImage.saveAs(baos, ImageType.PNG);
			
			return baos.toByteArray();
		}
		finally
		{
			try
			{
				baos.close();
			}
			catch (Exception e)
			{
				//nothing to be done
			}
		}
	}
	
	/**
	 * Converts an alignment "name" to the constant value.
	 * 
	 * @param pAlign the alignment: "left", "right", "top", "bottom", "center", "stretch"
	 * @return the alignment from {@link IAlignmentConstants}
	 */
	public static int convertAlignment(String pAlign)
	{
		if (pAlign != null)
		{
			String sAlign = pAlign.toLowerCase();
			
			if ("left".equals(sAlign))
			{
				return IAlignmentConstants.ALIGN_LEFT;
			}
			else if ("right".equals(sAlign))
			{
				return IAlignmentConstants.ALIGN_RIGHT;
			}
			else if ("top".equals(sAlign))
			{
				return IAlignmentConstants.ALIGN_TOP;
			}
			else if ("bottom".equals(sAlign))
			{
				return IAlignmentConstants.ALIGN_BOTTOM;
			}
			else if ("stretch".equals(sAlign))
			{
				return IAlignmentConstants.ALIGN_STRETCH;
			}
		}
		
		return IAlignmentConstants.ALIGN_CENTER;
	}
	
}	// ApplicationUtil
