/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 09.04.2009 - [RH] - creation
 * 18.04.2009 - [RH] - get/set/DataPage/RowIndex moved to ChangeableDataRow      
 * 31.03.2011 - [RH] - #163 - IChangeableDataRow should support isWritableColumnChanged                            
 */
package javax.rad.model;

/**
 * An <code>IChangeableDataRow</code> extends the <code>IDataRow</code> with support for
 * the change state of the row, belonging DataPage and row index and some supporting 
 * methods and an unique ID column.
 * 
 * @see javax.rad.model.IDataRow
 * @see javax.rad.model.IDataPage
 * @see javax.rad.model.IDataBook
 * 
 * @author Roland Hrmann
 */
public interface IChangeableDataRow extends IDataRow
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Method definitions
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Returns the corresponding<code>IDataPage</code> of the <code>IDataRow</code>.
	 * That returns != null only if the <code>IDataRow</code> is from a <code>IDataBook</code>
	 * 
	 * @return the corresponding<code>IDataPage</code> of the <code>IDataRow</code>.
	 */
	public IDataPage getDataPage();
	
	/**
	 * Returns the row index of the <code>IDataRow</code>s in the <code>IDataPage</code>.
	 * That returns != -1 only if the <code>IDataRow</code> is from a <code>IDataPage</code>
	 * 
	 * @return the row index of the <code>IDataRow</code>s in the <code>IDataPage</code>.
	 */
	public int getRowIndex();
		
	/**
	 * Returns the internal unique identifier for a <code>IChangeableDataRow</code>, which 
	 * isInserting().
	 * <br>
	 * 
	 * @return the internal unique identifier for a <code>IChangeableDataRow</code>, which 
	 *         isInserting().
	 */
	public Object getUID();

    /**
     * Returns if this row is new.
     * 
     * @return if this row is new.
	 * @throws ModelException 
	 * 			  if a exception occurs during synchronize.
     */
    public boolean isInserting() throws ModelException;

    /**
     * Returns if this row is changed.
     * 
     * @return if this row is changed.
	 * @throws ModelException 
	 * 			  if a exception occurs during synchronize.
     */
    public boolean isUpdating() throws ModelException;

    /**
     * Returns if this row is deleted.
     * 
     * @return if this row is deleted.
	 * @throws ModelException 
	 * 			  if a exception occurs during synchronize.
     */
    public boolean isDeleting() throws ModelException;

	/**
	 * Returns true if under this row in the detail DataBook one or more changed 
	 * (isInserting/Updating/Deleting() == true) detail rows existing.
	 * 
	 * @return true if under this row in the detail DataBook one or more changed 
	 *         (isInserting/Updating/Deleting() == true) detail rows existing.
	 * @throws ModelException 
	 * 			  if a exception occurs during synchronize.
	 */
	public boolean isDetailChanged() throws ModelException;	

	/**
	 * Returns the original row, before the first change is made.
	 * 
	 * @return the original row, before the first change is made.
	 * @throws ModelException	if the original row couldn't determined
	 */
	public IDataRow getOriginalRow() throws ModelException;

	/**
	 * Returns true if one isWriteable() column is changed.
	 *  
	 * @return true if one isWriteable() column is changed.
	 */
	// #163 - IChangeableDataRow should support isWritableColumnChanged 
	public boolean isWritableColumnChanged();
	
} 	// IChangeableDataRow
